/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_labels_dialog.h"
#include "sg.h"
#include "sg_layer.h"

static gchar *label_styles[] = {"Numeric",
                                "Text from Column",
                                 NULL};

static gchar *label_formats[] = {"Decimal: 1.000", 
                                 "Scientific: 1.E10", 
                                 "Scientific: 1x10^10", 
                                 NULL};

typedef struct{

  SGplot *plot;
  SGlayer *layer;
  GtkPlotAxis *axis;

  GtkWidget *font_combo;
  GtkWidget *style_combo;
  GtkWidget *format_combo;
  GtkWidget *data_combo;
  GtkWidget *ticks_combo;
  GtkWidget *offset_spin;
  GtkWidget *angle_spin;
  GtkWidget *precision_spin;
  GtkWidget *prefix_entry;
  GtkWidget *suffix_entry;

} SGlabelsdialog;

static SGlabelsdialog labels_dialog[4];
static void     new_style                               (GtkWidget *widget, 
                                                         gpointer data);

void
sg_labels_dialog_update_plot(GtkWidget *notebook)
{
  SGlabelsdialog *dialog;
  SGworksheet *worksheet = NULL;
  GtkPlot *plot;
  GtkPlotAxis *axis;
  GdkColor bg, fg;
  GtkWidget *child;
  GList *list;
  gint page;
  gint tick_labels, tick_values;
  gint i, n = 0;
  gint col = 0;
  gint values_col = -1, labels_col = -1;
  GList *labels = NULL;
  gint the_axis = 0;
  gchar *text;
  gchar *default_label = "";
  gchar *font;
  gint height, angle;

  page = gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook));
  dialog = &labels_dialog[page];
  plot = GTK_PLOT(dialog->layer->real_plot);
  axis = dialog->axis;

  if(axis == plot->left) the_axis = GTK_PLOT_AXIS_LEFT;
  if(axis == plot->right) the_axis = GTK_PLOT_AXIS_RIGHT;
  if(axis == plot->top) the_axis = GTK_PLOT_AXIS_TOP;
  if(axis == plot->bottom) the_axis = GTK_PLOT_AXIS_BOTTOM;

  font = GTK_FONT_COMBO(dialog->font_combo)->psfont->psname;
  height = GTK_FONT_COMBO(dialog->font_combo)->height;

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->style_combo)->list)->selection->data;
  tick_labels = gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->style_combo)->list), child);

  switch(tick_labels){
    case 0:
       axis->custom_labels = FALSE;
       labels = NULL; 
       break;
    case 1:
       axis->custom_labels = TRUE;

       child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->data_combo)->list)->selection->data;
       tick_labels = gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->data_combo)->list), child);

       child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->ticks_combo)->list)->selection->data;
       tick_values = gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->ticks_combo)->list), child);
       tick_values--;

       n = 0;
       list = worksheets;
       while(list){
          worksheet = (SGworksheet *)list->data;
   
          for(i=0; i <= GTK_SHEET(worksheet->sheet)->maxcol; i++){
               if(n == tick_labels) {
                    labels_col = i; 
                    break; 
               }
               n++;
          };

          if(n == tick_labels) break; 
          list = list->next;
       }

       n = 0;
       if(tick_values >= 0)
         list = worksheets;
         while(list){
            worksheet = (SGworksheet *)list->data;
   
            for(i=0; i <= GTK_SHEET(worksheet->sheet)->maxcol; i++){
                 if(n == tick_values) {
                      values_col = i; 
                      break; 
                 }
                 n++;
            };

            if(n == tick_values) break; 
            list = list->next;
         }


       sg_layer_set_tick_labels(dialog->layer, the_axis, worksheet, values_col, labels_col);
       break;
  }

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->format_combo)->list)->selection->data;
  axis->label_style = gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->format_combo)->list), child);

  axis->label_precision = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->precision_spin));
  axis->labels_offset = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->offset_spin));
  angle = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->angle_spin));

  switch(the_axis){
    case GTK_PLOT_AXIS_LEFT:
     if(angle == 90 || angle == 270)
                axis->labels_attr.justification = GTK_JUSTIFY_CENTER; 
     if(angle == 0)
                axis->labels_attr.justification = GTK_JUSTIFY_RIGHT; 
     if(angle == 180)
                axis->labels_attr.justification = GTK_JUSTIFY_LEFT; 
     break;
    case GTK_PLOT_AXIS_RIGHT:
     if(angle == 90 || angle == 270)
                axis->labels_attr.justification = GTK_JUSTIFY_CENTER; 
     if(angle == 0)
                axis->labels_attr.justification = GTK_JUSTIFY_LEFT; 
     if(angle == 180)
                axis->labels_attr.justification = GTK_JUSTIFY_RIGHT; 
     break;
    case GTK_PLOT_AXIS_TOP:
     if(angle == 0 || angle == 180)
                axis->labels_attr.justification = GTK_JUSTIFY_CENTER; 
     if(angle == 90)
                axis->labels_attr.justification = GTK_JUSTIFY_LEFT; 
     if(angle == 270)
                axis->labels_attr.justification = GTK_JUSTIFY_RIGHT; 
     break;
    case GTK_PLOT_AXIS_BOTTOM:
     if(angle == 0 || angle == 180)
                axis->labels_attr.justification = GTK_JUSTIFY_CENTER; 
     if(angle == 90)
                axis->labels_attr.justification = GTK_JUSTIFY_RIGHT; 
     if(angle == 270)
                axis->labels_attr.justification = GTK_JUSTIFY_LEFT; 
  }

  fg = axis->labels_attr.fg;
  bg = axis->labels_attr.bg; 
  gtk_plot_axis_set_labels_attributes(plot, (GtkPlotAxisPos)the_axis,
                                      font,
                                      height,
                                      angle,
                                      &fg,
                                      &bg, 
                                      TRUE, 
                                      axis->labels_attr.justification);

  text = gtk_entry_get_text(GTK_ENTRY(dialog->prefix_entry));
  if(text && strlen(text) > 0)
     gtk_plot_axis_set_labels_prefix(plot, (GtkPlotAxisPos)the_axis, text);
  else
     gtk_plot_axis_set_labels_prefix(plot, (GtkPlotAxisPos)the_axis, NULL);

  text = gtk_entry_get_text(GTK_ENTRY(dialog->suffix_entry));
  if(text && strlen(text) > 0)
     gtk_plot_axis_set_labels_suffix(plot, (GtkPlotAxisPos)the_axis, text);
  else
     gtk_plot_axis_set_labels_suffix(plot, (GtkPlotAxisPos)the_axis, NULL);

  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(dialog->plot->real_canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(dialog->plot->real_canvas));
}


static void 
init_dialog(SGlabelsdialog *dialog)
{
  GtkPSFont *font;

  sg_combo_set_items(GTK_COMBO(dialog->style_combo), label_styles);
  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->style_combo)->list), dialog->axis->custom_labels); 

  sg_combo_set_items(GTK_COMBO(dialog->format_combo), label_formats);
  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->format_combo)->list), dialog->axis->label_style); 

  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->style_combo)->entry), FALSE);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->format_combo)->entry), FALSE);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->data_combo)->entry), FALSE);

  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->ticks_combo)->entry), FALSE);

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->offset_spin), 
                            dialog->axis->labels_offset);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->precision_spin), 
                            dialog->axis->label_precision);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->angle_spin), 
                            dialog->axis->labels_attr.angle);

  if(dialog->axis->labels_prefix)
     gtk_entry_set_text(GTK_ENTRY(dialog->prefix_entry), dialog->axis->labels_prefix);

  if(dialog->axis->labels_suffix)
     gtk_entry_set_text(GTK_ENTRY(dialog->suffix_entry), dialog->axis->labels_suffix);

  font = gtk_psfont_get_font(dialog->axis->labels_attr.font);
  gtk_font_combo_select(GTK_FONT_COMBO(dialog->font_combo),
                        font->family,
                        FALSE,
                        FALSE,
                        dialog->axis->labels_attr.height);


  new_style(NULL, dialog);
}

static void
new_style(GtkWidget *widget, gpointer data)
{
  GtkWidget *child;
  SGlabelsdialog *dialog;
  gint style;

  dialog = (SGlabelsdialog *)data;

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->style_combo)->list)->selection->data;
  style = gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->style_combo)->list), child);

  switch(style){
    case 0:
       gtk_widget_set_sensitive(dialog->format_combo, TRUE);
       gtk_widget_set_sensitive(dialog->data_combo, FALSE);
       gtk_widget_set_sensitive(dialog->ticks_combo, FALSE);
       gtk_widget_set_sensitive(dialog->precision_spin, TRUE);
       break; 
    case 1:
       gtk_widget_set_sensitive(dialog->format_combo, FALSE);
       gtk_widget_set_sensitive(dialog->data_combo, TRUE);
       gtk_widget_set_sensitive(dialog->ticks_combo, TRUE);
       gtk_widget_set_sensitive(dialog->precision_spin, FALSE);
       break; 
  }
}

static GtkWidget * 
sg_build_labels_dialog (SGlayer *layer, GtkPlotAxis *axis, SGlabelsdialog *dialog)
{
  SGworksheet *worksheet;
  GtkWidget *frame;
  GtkWidget *main_box;
  GtkWidget *table, *main_table;
  GtkWidget *label;
  GtkWidget *item;
  GtkAdjustment *adj;
  GList *list;
  gint i;
  gchar text[255];

  dialog->axis = axis;
  dialog->layer = layer;
  dialog->plot = (SGplot *)(layer->parent);

  /* Create widgets */
  main_box = gtk_hbox_new (FALSE, 5);
  gtk_container_set_border_width(GTK_CONTAINER(main_box), 5);

  main_table = gtk_table_new(2, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(main_table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(main_table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(main_table), 5);
  gtk_box_pack_start (GTK_BOX (main_box), main_table, FALSE, FALSE, 0);

/*----------------------------------------------*/
  frame = gtk_frame_new("Format");
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_table_attach_defaults(GTK_TABLE(main_table), frame, 0, 1, 1, 2);

  table = gtk_table_new(8, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);

  label = gtk_label_new("Style");
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
  label = gtk_label_new("Format");
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);
  label = gtk_label_new("Column");
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2, 3);
  label = gtk_label_new("Position");
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 3, 4);

  dialog->style_combo = gtk_combo_new();
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->style_combo, 1, 2, 0, 1);
  dialog->format_combo = gtk_combo_new();
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->format_combo, 1, 2, 1, 2);
  dialog->data_combo = gtk_combo_new();
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->data_combo, 1, 2, 2, 3);
  dialog->ticks_combo = gtk_combo_new();
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->ticks_combo, 1, 2, 3, 4);

  item = gtk_list_item_new_with_label("Row index");
  gtk_widget_show(item);
  gtk_container_add(GTK_CONTAINER(GTK_COMBO(dialog->ticks_combo)->list), item);

  list = worksheets;
  while(list){
     worksheet = (SGworksheet *)list->data;
   
     for(i=0; i <= GTK_SHEET(worksheet->sheet)->maxcol; i++){
       if(!GTK_SHEET(worksheet->sheet)->column[i].name)
          sprintf(text,"%s->%d",worksheet->name,i);
       else
          sprintf(text,"%s->%s",worksheet->name,GTK_SHEET(worksheet->sheet)->column[i].name);

       item = gtk_list_item_new_with_label(text);
       gtk_widget_show(item);
       gtk_container_add(GTK_CONTAINER(GTK_COMBO(dialog->data_combo)->list), item);
       item = gtk_list_item_new_with_label(text);
       gtk_widget_show(item);
       gtk_container_add(GTK_CONTAINER(GTK_COMBO(dialog->ticks_combo)->list), item);
     };
     list = list->next;
  }

  label = gtk_label_new("Precision");
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 4, 5);

  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 8., 1., 1., 0.);
  dialog->precision_spin = gtk_spin_button_new(adj, 0, 0);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->precision_spin), TRUE);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(dialog->precision_spin), 0);
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->precision_spin, 1, 2, 4, 5);

  label = gtk_label_new("Prefix");
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 5, 6);

  dialog->prefix_entry = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->prefix_entry, 1, 2, 5, 6);

  label = gtk_label_new("Suffix");
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 6, 7);

  dialog->suffix_entry = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->suffix_entry, 1, 2, 6, 7);
  

  dialog->font_combo = gtk_font_combo_new();
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->font_combo, 0, 2, 7, 8);

/*----------------------------------------------*/
  frame = gtk_frame_new("Position");
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_table_attach_defaults(GTK_TABLE(main_table), frame, 1, 2, 1, 2);

  table = gtk_table_new(3, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);

  gtk_table_attach_defaults(GTK_TABLE(table),
                            gtk_label_new("Offset"), 0, 1, 0, 1);
  gtk_table_attach_defaults(GTK_TABLE(table),
                            gtk_label_new("Rotate"), 0, 1, 1, 2);

  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 60., 1., 1., 0.);
  dialog->offset_spin = gtk_spin_button_new(adj, 0, 0);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->offset_spin), TRUE);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(dialog->offset_spin), 0);
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->offset_spin, 1, 2, 0, 1);
  
  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 270., 90., 90., 0.);
  dialog->angle_spin = gtk_spin_button_new(adj, 90, 0);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->angle_spin), TRUE);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(dialog->angle_spin), 0);
  gtk_entry_set_editable(GTK_ENTRY(dialog->angle_spin), FALSE);
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->angle_spin, 1, 2, 1, 2);
  
  /* connect signals */
  gtk_signal_connect(GTK_OBJECT(GTK_COMBO(dialog->style_combo)->entry),
                     "changed",
                     GTK_SIGNAL_FUNC(new_style),
                     dialog);

  gtk_widget_show_all(main_box);

  init_dialog(dialog);

  return main_box;
}

GtkWidget *
sg_labels_dialog_new(SGlayer *layer)
{
  GtkWidget *dialog;
  GtkWidget *notebook;
  GtkPlot *plot;

  plot = GTK_PLOT(layer->real_plot);
  notebook = gtk_notebook_new();

  switch(layer->type){
    case SG_LAYER_3D:
      dialog = sg_build_labels_dialog(layer, plot->bottom, &labels_dialog[0]);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new("X"));
      dialog = sg_build_labels_dialog(layer, plot->left, &labels_dialog[1]);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new("Y"));
      dialog = sg_build_labels_dialog(layer, plot->top, &labels_dialog[2]);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new("Z"));
      break;
    case SG_LAYER_2D:
      dialog = sg_build_labels_dialog(layer, plot->left, &labels_dialog[0]);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new("Left"));
      dialog = sg_build_labels_dialog(layer, plot->right, &labels_dialog[1]);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new("Right"));
      dialog = sg_build_labels_dialog(layer, plot->top, &labels_dialog[2]);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new("Top"));
      dialog = sg_build_labels_dialog(layer, plot->bottom, &labels_dialog[3]);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new("Bottom"));
      break;
    case SG_LAYER_POLAR:
      dialog = sg_build_labels_dialog(layer, plot->bottom, &labels_dialog[0]);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new("Angles"));
      dialog = sg_build_labels_dialog(layer, plot->left, &labels_dialog[1]);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new("R"));
      break;

  }
  return notebook;
}
