/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <Python.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_wrap.h"
#include "sg.h"
#include "python/python_main.h"
#include "sg_layer.h"
#include "sg_dataset.h"

static gint set_ticks				(gchar *plot_name, gint nth, 
         					 GtkPlotOrientation orientation,
       						 gdouble major, gint nminor);
static gint set_title				(gchar *plot_name, gint nth, 
         					 GtkPlotOrientation orientation,
       						 gchar *title);

SGworksheet *new_worksheet(gchar *name){
    GList *list;
    SGworksheet *worksheet;
    SGworksheet *data;
    gboolean found = FALSE;

    list = worksheets;
    while(list){
        data = (SGworksheet *)list->data;
        if(!strcmp(data->name, name)){
            found=TRUE;
            break;
        }
        list=list->next;
    }

    if (found)
        return data;
  
    worksheet = sg_project_new_worksheet();
  
    if(name) sg_project_rename_worksheet(worksheet, name);
  
    return worksheet;
}

gint            
update_worksheet                (gchar *name)
{
  SGworksheet *worksheet;
  GtkSheet *sheet;
  gint col;

  worksheet = sg_project_get_worksheet(name);

  if(!worksheet){
    return WRAP_ERROR;
  }

  sheet = GTK_SHEET(worksheet->sheet);

  sg_worksheet_unupdate_exp_range(worksheet,
               sheet->range.row0,sheet->range.rowi,
               sheet->range.col0,sheet->range.coli);

  sg_worksheet_update_exp_range(worksheet,
               sheet->range.row0,sheet->range.rowi,
               sheet->range.col0,sheet->range.coli);

  return WRAP_OK;
}


gint            
set_cell_value                  (gchar *name,
                                 gchar *col_name,
                                 gint row, 
                                 gdouble value)
{
  SGworksheet *worksheet;
  gchar text[80];
  gint col;

  worksheet = sg_project_get_worksheet(name);

  if(!worksheet){
    return WRAP_ERROR;
  }

  col = sg_worksheet_get_column(worksheet, col_name);
  if(col < 0) return WRAP_ERROR;

  g_snprintf(text, 80, "%f\n", value);

  sg_worksheet_cell_set(worksheet, row, col, text, TRUE, TRUE);

  return WRAP_OK;
}

gint            
set_cell_text                   (gchar *name,
                                 gchar *col_name,
                                 gint row,
                                 gchar *text)
{
  SGworksheet *worksheet;
  gint col;

  worksheet = sg_project_get_worksheet(name);

  if(!worksheet){
    return WRAP_ERROR;
  }

  col = sg_worksheet_get_column(worksheet, col_name);
  if(col < 0) return WRAP_ERROR;
  
  sg_worksheet_cell_set(worksheet, row, col, text, TRUE, TRUE);

  return WRAP_OK;
}

gdouble            
get_cell_value                  (gchar *name,
                                 gchar *col_name,
                                 gint row)
{
  SGworksheet *worksheet;
  gint col;
  gdouble value = 0.0f;
  gboolean error;;

  worksheet = sg_project_get_worksheet(name);

  if(!worksheet){
    return 0.0f;
  }

  col = sg_worksheet_get_column(worksheet, col_name);
  if(col < 0) return WRAP_ERROR;

  return sg_worksheet_cell_get_double(worksheet, row, col,&error);
}

gchar *            
get_cell_text                   (gchar *name,
                                 gchar *col_name,
                                 gint row)
{
  SGworksheet *worksheet;
  gchar *text = NULL;
  gint col;

  worksheet = sg_project_get_worksheet(name);

  if(!worksheet){
    return NULL;
  }

  col = sg_worksheet_get_column(worksheet, col_name);
  if(col < 0) return NULL;
  
  text = sg_worksheet_cell_get_text(worksheet, row, col);

  return text;
}


gint            
set_column_name                 (gchar *name,
                                 gchar *col_name,
                                 gchar *new_name)
{
  SGworksheet *worksheet;
  gint col;

  worksheet = sg_project_get_worksheet(name);

  if(!worksheet){
    return WRAP_ERROR;
  }

  col = sg_worksheet_get_column(worksheet, col_name);
  if(col < 0) return WRAP_ERROR;

  sg_worksheet_set_column_name(worksheet, col, new_name);

  return WRAP_OK;
}

gint            
set_column_values               (gchar *name,
                                 gchar *col_name,
                                 gchar *exp,
                                 gint from,
                                 gint to)
{
  SGworksheet *worksheet;
  gint row, col;

  worksheet = sg_project_get_worksheet(name);

  if(!worksheet){
    return WRAP_ERROR;
  }

  col = sg_worksheet_get_column(worksheet, col_name);
  if(col < 0) return WRAP_ERROR;

  gtk_sheet_freeze(GTK_SHEET(worksheet->sheet));
  sg_worksheet_update_column_exp(worksheet, exp, col, from, to);
  gtk_sheet_thaw(GTK_SHEET(worksheet->sheet));

  return WRAP_OK;
}

gint            
set_column_type                 (gchar *name,
                                 gchar *col_name, 
                                 SGcolumntype type)
{
  SGworksheet *worksheet;
  gint row, col;

  worksheet = sg_project_get_worksheet(name);

  if(!worksheet){
    return WRAP_ERROR;
  }

  col = sg_worksheet_get_column(worksheet, col_name);
  if(col < 0) return WRAP_ERROR;

  sg_worksheet_column_set_format(worksheet, col, 
                                 type,
                                 worksheet->column[col].format,
                                 worksheet->column[col].internal,
                                 worksheet->column[col].precision);

  return WRAP_OK;
}

gint            
set_column_numbers              (gchar *name,
                                 gchar *col_name, 
                                 SGcolumninternal internal,
                                 SGcolumnformat format,
                                 gint precision)
{
  SGworksheet *worksheet;
  gint row, col;

  worksheet = sg_project_get_worksheet(name);

  if(!worksheet){
    return WRAP_ERROR;
  }

  col = sg_worksheet_get_column(worksheet, col_name);
  if(col < 0) return WRAP_ERROR;

  sg_worksheet_column_set_format(worksheet, col, 
                                 SG_TYPE_NUMBER,
                                 format,
                                 internal,
                                 precision);

  return WRAP_OK;
}


SGplot*
new_plot(gchar *name, SGlayerType type)
{
  SGplot *plot;
  
  plot = sg_project_new_plot_with_layer(type);

  if(name) sg_project_rename_plot(plot, name);

  return plot;
}

SGlayer*
new_layer(gchar *plot_name, SGlayerType type)
{
  SGplot *plot;
  SGlayer *layer;

  plot = sg_project_get_plot(plot_name);

  if(plot){
      if(type != SG_LAYER_POLAR)
        layer = sg_layer_new(type, .65, .45);
      else
        layer = sg_layer_new(type, .75, .45);

      sg_plot_add_layer(plot, layer, .175, .15);

      return layer;
  }

  return NULL;
}

gint
remove_layer(gchar *plot_name, gint nth)
{
  SGplot *plot;
  SGlayer *layer;

  plot = sg_project_get_plot(plot_name);

  if(plot){
      layer = (SGlayer *)g_list_nth_data(plot->layers, nth - 1);

      if(layer) 
          sg_plot_remove_layer(plot, layer);
      else
          return WRAP_ERROR;
  } else {
      return WRAP_ERROR; 
  }

  return WRAP_OK;
}

gint
move_layer(gchar *plot_name, gint nth, gdouble x, gdouble y)
{
  SGplot *plot;
  SGlayer *layer;

  plot = sg_project_get_plot(plot_name);

  if(plot){
      layer = (SGlayer *)g_list_nth_data(plot->layers, nth - 1);

      if(layer) 
          sg_plot_move_layer(plot, layer, x, y);
      else
          return WRAP_ERROR;
  } else {
      return WRAP_ERROR; 
  }

  return WRAP_OK;
}

gint
resize_layer(gchar *plot_name, gint nth, gdouble w, gdouble h)
{
  SGplot *plot;
  SGlayer *layer;

  plot = sg_project_get_plot(plot_name);

  if(plot){
      layer = (SGlayer *)g_list_nth_data(plot->layers, nth - 1);

      if(layer) 
          sg_plot_resize_layer(plot, layer, w, h);
      else
          return WRAP_ERROR;
  } else {
      return WRAP_ERROR; 
  }

  return WRAP_OK;
}

gint
clear_plot(gchar *plot_name)
{
  SGplot *plot;
  SGlayer *layer;
  SGdataset *dataset;

  plot = sg_project_get_plot(plot_name);

  if(plot){
      sg_plot_clear(plot);
      gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
      gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
  } else {
      return WRAP_ERROR; 
  }

  return WRAP_OK;
}

gint
clear_layer(gchar *plot_name, gint nth)
{
  SGplot *plot;
  SGlayer *layer;

  plot = sg_project_get_plot(plot_name);

  if(plot){
      layer = (SGlayer *)g_list_nth_data(plot->layers, nth - 1);

      if(layer){
          sg_layer_clear(layer); 
          gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
          gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
      } else
          return WRAP_ERROR;
  } else {
      return WRAP_ERROR; 
  }

  return WRAP_OK;
}

gint
set_symbol(gchar *plot_name, gint nth, gint symbol)
{
  SGplot *plot;
  SGlayer *layer;

  plot = sg_project_get_plot(plot_name);

  if(plot){
      layer = (SGlayer *)g_list_nth_data(plot->layers, nth - 1);

      if(layer){
          layer->symbol = symbol;
          if(layer->symbol < 0) 
                     layer->symbol = 0;
          if(layer->symbol >= NUM_SYMBOLS) 
                     layer->symbol = NUM_SYMBOLS - 1;
      } else
          return WRAP_ERROR;
  } else {
      return WRAP_ERROR; 
  }

  return WRAP_OK;
} 

gint
set_symbol_style(gchar *plot_name, gint nth, gint style)
{
  SGplot *plot;
  SGlayer *layer;

  plot = sg_project_get_plot(plot_name);

  if(plot){
      layer = (SGlayer *)g_list_nth_data(plot->layers, nth - 1);

      if(layer){
          layer->symbol_style = style;
          if(layer->symbol_style < 0) 
                     layer->symbol_style = 0;
          if(layer->symbol_style >= NUM_SYMBOL_STYLES) 
                     layer->symbol_style = NUM_SYMBOL_STYLES - 1;
      } else
          return WRAP_ERROR;
  } else {
      return WRAP_ERROR; 
  }

  return WRAP_OK;
} 

gint
set_symbol_color(gchar *plot_name, gint nth, gchar *color_string)
{
  SGplot *plot;
  SGlayer *layer;

  plot = sg_project_get_plot(plot_name);

  if(plot){
      layer = (SGlayer *)g_list_nth_data(plot->layers, nth - 1);

      if(layer){
          GdkColor color;
          gdk_color_parse(color_string, &color);
          gdk_color_alloc(gdk_colormap_get_system(), &color);
          layer->symbol_color = color;
      } else
          return WRAP_ERROR;
  } else {
      return WRAP_ERROR; 
  }

  return WRAP_OK;
} 


gint
set_line_style(gchar *plot_name, gint nth, gint style)
{
  SGplot *plot;
  SGlayer *layer;

  plot = sg_project_get_plot(plot_name);

  if(plot){
      layer = (SGlayer *)g_list_nth_data(plot->layers, nth - 1);

      if(layer){
          layer->line_style = style;
          if(layer->line_style < 0) 
                     layer->line_style = 0;
          if(layer->line_style >= NUM_LINE_STYLES) 
                     layer->line_style = NUM_LINE_STYLES - 1;
      } else
          return WRAP_ERROR;
  } else {
      return WRAP_ERROR; 
  }

  return WRAP_OK;
} 

gint
set_line_color(gchar *plot_name, gint nth, gchar *color_string)
{
  SGplot *plot;
  SGlayer *layer;

  plot = sg_project_get_plot(plot_name);

  if(plot){
      layer = (SGlayer *)g_list_nth_data(plot->layers, nth - 1);

      if(layer){
          GdkColor color;
          gdk_color_parse(color_string, &color);
          gdk_color_alloc(gdk_colormap_get_system(), &color);
          layer->line_color = color;
      } else
          return WRAP_ERROR;
  } else {
      return WRAP_ERROR; 
  }

  return WRAP_OK;
} 


gint
set_connector(gchar *plot_name, gint nth, gint connector)
{
  SGplot *plot;
  SGlayer *layer;

  plot = sg_project_get_plot(plot_name);

  if(plot){
      layer = (SGlayer *)g_list_nth_data(plot->layers, nth - 1);

      if(layer){
          layer->connector = connector;
          if(layer->connector < 0) 
                     layer->connector = 0;
          if(layer->connector >= NUM_CONNECTORS) 
                     layer->connector = NUM_CONNECTORS - 1;
      } else
          return WRAP_ERROR;
  } else {
      return WRAP_ERROR; 
  }

  return WRAP_OK;
} 

gint
plot_exp_xy(gchar *plot_name, gint nth, gchar *expx, gchar *expy, gchar *exp_name)
{
  SGplot *plot;
  SGlayer *layer;
  SGdataset *dataset;
  gchar *exp[9];
  gchar name[80];
  gint i;

  if(!expx || !expy) return WRAP_ERROR;

  for(i = 0; i < 9; i++) exp[i] = NULL;
  exp[0] = expx;
  exp[1] = expy;

  plot = sg_project_get_plot(plot_name);

  if(plot){
      layer = (SGlayer *)g_list_nth_data(plot->layers, nth - 1);

      if(layer){
          SGdataset *clone;
          dataset = sg_project_new_expression(SG_STYLE_LPOINTS, exp);
          clone = sg_dataset_clone(dataset);
          sg_dataset_refresh(clone);
          snprintf(name, 80, "(X,Y)->%s,%s\n", expx, expy);
          sg_dataset_set_name(clone, g_strdup(name));
          sg_layer_add_dataset_default(layer, clone); 
          if (!exp_name)
            gtk_plot_data_set_legend(GTK_PLOT_DATA(clone->real_data), name); 
          else
            gtk_plot_data_set_legend(GTK_PLOT_DATA(clone->real_data), exp_name); 
          gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
          gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
      } else
          return WRAP_ERROR;
  } else {
      return WRAP_ERROR; 
  }

  return WRAP_OK;
}

gint
plot_exp_xy_bars(gchar *plot_name, gint nth, gchar *expx, gchar *expy, gchar *exp_name, 
                 SGdataStyle orient, gfloat bin_width)
{
  SGplot *plot;
  SGlayer *layer;
  SGdataset *dataset;
  GtkPlotData *data;
  gchar *exp[9];
  gchar name[80];
  gint i;

  if(!expx || !expy) return WRAP_ERROR;
  if(orient!= SG_STYLE_VBARS && orient!= SG_STYLE_HBARS) return WRAP_ERROR;

  for(i = 2; i < 9; i++) exp[i] = NULL;
  exp[0] = g_strdup(expx);
  exp[1] = g_strdup(expy);

  plot = sg_project_get_plot(plot_name);

  if(plot){
      layer = (SGlayer *)g_list_nth_data(plot->layers, nth - 1);

      if(layer){
          SGdataset *clone;
          dataset = sg_project_new_expression(orient, exp);
          clone = sg_dataset_clone(dataset);

          data=GTK_PLOT_DATA(clone->real_data);
          GTK_PLOT_BAR(data)->width = bin_width;
          sg_dataset_refresh(clone);
          snprintf(name, 80, "(EXPR)->%s,%s\n", expx, expy);
          sg_dataset_set_name(clone, g_strdup(name));
          sg_layer_add_dataset_default(layer, clone); 
          if (!exp_name)
            gtk_plot_data_set_legend(GTK_PLOT_DATA(clone->real_data), g_strdup(name)); 
          else
            gtk_plot_data_set_legend(GTK_PLOT_DATA(clone->real_data), g_strdup(exp_name)); 
          gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
          gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
      } else
          return WRAP_ERROR;
  } else {
      return WRAP_ERROR; 
  }

  return WRAP_OK;
}


gint
plot_exp(gchar *plot_name, gint nth, gchar *exp, gchar *exp_name)
{
  SGplot *plot;
  SGlayer *layer;
  SGdataset *dataset;
  GdkColor color;
  gchar name[80];
  
  plot = sg_project_get_plot(plot_name);

  if(plot){
      layer = (SGlayer *)g_list_nth_data(plot->layers, nth - 1);

      if(layer){
          SGdataset *clone;

          dataset = sg_project_new_function(exp);
          clone = sg_dataset_clone(dataset);

          sg_layer_add_dataset_default(layer, clone); 
          if (!exp_name){
            snprintf(name, 80, "%s=%s\n", dataset->real_data->name, exp);
            gtk_plot_data_set_legend(clone->real_data, name); 
          }else
            gtk_plot_data_set_legend(GTK_PLOT_DATA(clone->real_data), exp_name); 
          color.red=0;
          color.green=0;
          color.blue=0;
          gdk_color_alloc(gdk_colormap_get_system(), &color);
          gtk_plot_data_set_symbol(GTK_PLOT_DATA(clone->real_data),
                               GTK_PLOT_SYMBOL_NONE,
                               GTK_PLOT_SYMBOL_EMPTY,
                               0,
                               0,
                               &color,
                               &color);
          gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
          gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
      } else
          return WRAP_ERROR;
  } else {
      return WRAP_ERROR; 
  }

  return WRAP_OK;
}

gint
remove_worksheet(gchar *name)
{
  SGworksheet *worksheet;

  worksheet = sg_project_get_worksheet(name);
 
  if(worksheet){
    sg_project_remove_worksheet(worksheet);
  } else {
    return WRAP_ERROR; 
  }

  return WRAP_OK;
} 

gint
hide_zgrids(gchar *plot_name, gint nth)
{
  SGplot *plot;
  SGlayer *layer;

  plot = sg_project_get_plot(plot_name);

  if(plot){
      layer = (SGlayer *)g_list_nth_data(plot->layers, nth - 1);

      if(layer){
          if(layer->type == SG_LAYER_3D){
            GTK_PLOT3D(layer->real_plot)->az->show_major_grid = FALSE;
            GTK_PLOT3D(layer->real_plot)->az->show_minor_grid = FALSE;
          }else
            return WRAP_ERROR;
          gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
          gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
      } else
          return WRAP_ERROR;
  } else {
      return WRAP_ERROR; 
  }

  return WRAP_OK;
}

gint
read_file(gchar *name, gchar *path)
{
  SGworksheet *worksheet;

  if(!name || !path) return WRAP_ERROR;

  worksheet = sg_project_get_worksheet(name);
 
  if(worksheet){
    gboolean veto = TRUE;
    veto = sg_worksheet_file_import_block(worksheet, path);
    if(!veto)
      return WRAP_ERROR;
  } else {
    return WRAP_ERROR; 
  }

  return WRAP_OK;
} 

gchar *get_active_worksheet_name(void)
{
    return active_worksheet->name;
}

gchar *get_active_plot_name(void)
{
  if (!active_plot)
  { if (!plots) return NULL;
    active_plot=(SGplot *)plots->data;
  }
  return active_plot->name;

}

gint get_active_layer(SGplot *plot)
{
    GList *list;
    gint n = 1;

    list = plot->layers;
    while(list){
      if((SGlayer *)list->data == plot->active_layer) break;
      n++;
      list = list->next;
    }
 
    return n;
}

GList *get_worksheet_names(void){

  GList *list=NULL,*sheets=NULL;
  
  sheets=worksheets;
  
  while(sheets){
     gchar *name=((SGworksheet*)(sheets->data))->name;
     list=g_list_append(list,(gpointer)name);
     sheets=sheets->next;
  }  

  return list;
}

GList *get_column_names(SGworksheet *worksheet){

  GList *list=NULL;
  GtkSheetColumn *column;
  gint col;

  for (col=0;col<=GTK_SHEET(worksheet->sheet)->maxcol;col++){
    column = &(GTK_SHEET(worksheet->sheet)->column[col]);
    list=g_list_append(list,(gpointer)column->name);
  }

  return list;
}

GList *get_layer_data_names(SGlayer *layer)
{
  GList *list=NULL,*data=NULL;
  
  data = layer->datasets;
  
  while(data){
     gchar *name=((SGdataset*)(data->data))->real_data->name;
     list=g_list_append(list,(gpointer)name);
     data=data->next;
  }  

  return list;
}

gchar *get_column_exp(SGworksheet *worksheet,gint colnum)
{ if (colnum<GTK_SHEET(worksheet->sheet)->maxcol)
  return worksheet->column[colnum].exp;
  else return NULL;
}

gint get_column_precision(SGworksheet *worksheet,gint colnum)
{ if (colnum<GTK_SHEET(worksheet->sheet)->maxcol)
  return worksheet->column[colnum].precision;
  else return -1;
}


gint
remove_plot(gchar *name)
{
  SGplot *plot;

  plot = sg_project_get_plot(name);
 
  if(plot){
    sg_project_remove_plot(plot);
  } else {
    return WRAP_ERROR; 
  }

  return WRAP_OK;
} 


gint
rename_worksheet(gchar *old_name, gchar *new_name)
{
  SGworksheet *worksheet;

  if(!old_name || !new_name) return WRAP_ERROR;

  worksheet = sg_project_get_worksheet(old_name);
 
  if(worksheet){
    gboolean veto = TRUE;
    veto = sg_project_rename_worksheet(worksheet, new_name);
    if(!veto)
      return WRAP_ERROR;
  } else {
    return WRAP_ERROR; 
  }

  return WRAP_OK;
} 

gint
rename_plot(gchar *old_name, gchar *new_name)
{
  SGplot *plot;

  if(!old_name || !new_name) return WRAP_ERROR;

  plot = sg_project_get_plot(old_name);
 
  if(plot){
    gboolean veto = TRUE;
    veto = sg_project_rename_plot(plot, new_name);
    if(!veto)
       return WRAP_ERROR;
  } else {
    return WRAP_ERROR;
  }

  return WRAP_OK;
} 

gint
show_legends(gchar *plot_name, gint nth)
{
  SGplot *plot;
  SGlayer *layer;

  plot = sg_project_get_plot(plot_name);

  if(plot){
      layer = (SGlayer *)g_list_nth_data(plot->layers, nth - 1);

      if(layer){
          gtk_plot_show_legends(GTK_PLOT(layer->real_plot));
          gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
          gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
      } else
          return WRAP_ERROR;
  } else {
      return WRAP_ERROR; 
  }

  return WRAP_OK;
}

gint
hide_legends(gchar *plot_name, gint nth)
{
  SGplot *plot;
  SGlayer *layer;

  plot = sg_project_get_plot(plot_name);

  if(plot){
      layer = (SGlayer *)g_list_nth_data(plot->layers, nth - 1);

      if(layer){
          gtk_plot_hide_legends(GTK_PLOT(layer->real_plot));
          gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
          gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
      } else
          return WRAP_ERROR;
  } else {
      return WRAP_ERROR; 
  }

  return WRAP_OK;
}

gint
move_legends(gchar *plot_name, gint nth, gdouble x, gdouble y)
{
  SGplot *plot;
  SGlayer *layer;

  plot = sg_project_get_plot(plot_name);

  if(plot){
      layer = (SGlayer *)g_list_nth_data(plot->layers, nth - 1);

      if(layer){
          gtk_plot_legends_move(GTK_PLOT(layer->real_plot), x, y);
          gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
          gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
      } else
          return WRAP_ERROR;
  } else {
      return WRAP_ERROR; 
  }

  return WRAP_OK;
}


gint
set_xrange(gchar *plot_name, gint nth, gdouble xmin, gdouble xmax)
{
  SGplot *plot;
  SGlayer *layer;

  plot = sg_project_get_plot(plot_name);

  if(plot){
      layer = (SGlayer *)g_list_nth_data(plot->layers, nth - 1);

      if(layer){
          if(layer->type == SG_LAYER_3D)
            gtk_plot3d_set_xrange(GTK_PLOT3D(layer->real_plot), xmin, xmax);
          else 
            gtk_plot_set_xrange(GTK_PLOT(layer->real_plot), xmin, xmax);
          gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
          gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
      } else
          return WRAP_ERROR;
  } else {
      return WRAP_ERROR; 
  }

  return WRAP_OK;
}

gint
set_yrange(gchar *plot_name, gint nth, gdouble ymin, gdouble ymax)
{
  SGplot *plot;
  SGlayer *layer;

  plot = sg_project_get_plot(plot_name);

  if(plot){
      layer = (SGlayer *)g_list_nth_data(plot->layers, nth - 1);

      if(layer){
          if(layer->type == SG_LAYER_3D)
            gtk_plot3d_set_yrange(GTK_PLOT3D(layer->real_plot), ymin, ymax);
          else 
            gtk_plot_set_yrange(GTK_PLOT(layer->real_plot), ymin, ymax);
          gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
          gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
      } else
          return WRAP_ERROR;
  } else {
      return WRAP_ERROR; 
  }

  return WRAP_OK;
}


gint
set_zrange(gchar *plot_name, gint nth, gdouble zmin, gdouble zmax)
{
  SGplot *plot;
  SGlayer *layer;

  plot = sg_project_get_plot(plot_name);

  if(plot){
      layer = (SGlayer *)g_list_nth_data(plot->layers, nth - 1);

      if(layer){
          if(layer->type == SG_LAYER_3D)
            gtk_plot3d_set_yrange(GTK_PLOT3D(layer->real_plot), zmin, zmax);
          else 
            return WRAP_ERROR;
          gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
          gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
      } else
          return WRAP_ERROR;
  } else {
      return WRAP_ERROR; 
  }

  return WRAP_OK;
}


gint
set_xticks(gchar *plot_name, gint nth, gdouble major, gint nminor)
{
  return(set_ticks(plot_name, nth, GTK_PLOT_AXIS_X, major, nminor));
}

gint
set_yticks(gchar *plot_name, gint nth, gdouble major, gint nminor)
{
  return(set_ticks(plot_name, nth, GTK_PLOT_AXIS_Y, major, nminor));
}

gint
set_zticks(gchar *plot_name, gint nth, gdouble major, gint nminor)
{
  return(set_ticks(plot_name, nth, GTK_PLOT_AXIS_Z, major, nminor));
}

static gint
set_ticks(gchar *plot_name, gint nth, 
          GtkPlotOrientation orientation, 
          gdouble major, gint nminor)
{
  SGplot *plot;
  SGlayer *layer;

  plot = sg_project_get_plot(plot_name);

  if(plot){
      layer = (SGlayer *)g_list_nth_data(plot->layers, nth - 1);

      if(layer){
          if(layer->type == SG_LAYER_3D)
            gtk_plot3d_axis_set_ticks(GTK_PLOT3D(layer->real_plot), orientation, major, nminor);
          else 
            gtk_plot_axis_set_ticks(GTK_PLOT(layer->real_plot), orientation, major, nminor);
          gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
          gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
      } else
          return WRAP_ERROR;
  } else {
      return WRAP_ERROR; 
  }

  return WRAP_OK;
}

gint
set_xtitle(gchar *plot_name, gint nth, gchar *title)
{
  set_title(plot_name, nth, (GtkPlotOrientation)GTK_PLOT_AXIS_TOP, title);
  return(set_title(plot_name, nth, (GtkPlotOrientation)GTK_PLOT_AXIS_BOTTOM, title));
}

gint
set_ytitle(gchar *plot_name, gint nth, gchar *title)
{
  set_title(plot_name, nth, (GtkPlotOrientation)GTK_PLOT_AXIS_LEFT, title);
  return(set_title(plot_name, nth, (GtkPlotOrientation)GTK_PLOT_AXIS_RIGHT, title));
}

gint
set_ztitle(gchar *plot_name, gint nth, gchar *title)
{
  return(set_title(plot_name, nth, GTK_PLOT_AXIS_Z, title));
}


static gint
set_title(gchar *plot_name, gint nth, 
          GtkPlotOrientation orientation, 
          gchar *title)
{
  SGplot *plot;
  SGlayer *layer;

  plot = sg_project_get_plot(plot_name);

  if(plot){
      layer = (SGlayer *)g_list_nth_data(plot->layers, nth - 1);

      if(layer){
          if(title){
            gtk_plot_axis_set_title(GTK_PLOT(layer->real_plot), (GtkPlotAxisPos)orientation, title);
/*            gtk_plot_axis_show_title(GTK_PLOT(layer->real_plot), (GtkPlotAxisPos)orientation);*/
          } else
            gtk_plot_axis_hide_title(GTK_PLOT(layer->real_plot), (GtkPlotAxisPos)orientation);
          gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
          gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
      } else
          return WRAP_ERROR;
  } else {
      return WRAP_ERROR; 
  }

  return WRAP_OK;
}

gint
show_xgrids(gchar *plot_name, gint nth)
{
  SGplot *plot;
  SGlayer *layer;

  plot = sg_project_get_plot(plot_name);

  if(plot){
      layer = (SGlayer *)g_list_nth_data(plot->layers, nth - 1);

      if(layer){
          if(layer->type == SG_LAYER_3D){
            GTK_PLOT3D(layer->real_plot)->ax->show_major_grid = TRUE;
            GTK_PLOT3D(layer->real_plot)->ax->show_minor_grid = TRUE;
          }else{
            GTK_PLOT(layer->real_plot)->bottom->show_major_grid = TRUE;
            GTK_PLOT(layer->real_plot)->bottom->show_minor_grid = TRUE;
          }
          gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
          gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
      } else
          return WRAP_ERROR;
  } else {
      return WRAP_ERROR; 
  }

  return WRAP_OK;
}

gint
show_ygrids(gchar *plot_name, gint nth)
{
  SGplot *plot;
  SGlayer *layer;

  plot = sg_project_get_plot(plot_name);

  if(plot){
      layer = (SGlayer *)g_list_nth_data(plot->layers, nth - 1);

      if(layer){
          if(layer->type == SG_LAYER_3D){
            GTK_PLOT3D(layer->real_plot)->ay->show_major_grid = TRUE;
            GTK_PLOT3D(layer->real_plot)->ay->show_minor_grid = TRUE;
          }else{
            GTK_PLOT(layer->real_plot)->left->show_major_grid = TRUE;
            GTK_PLOT(layer->real_plot)->left->show_minor_grid = TRUE;
          }
          gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
          gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
      } else
          return WRAP_ERROR;
  } else {
      return WRAP_ERROR; 
  }

  return WRAP_OK;
}

gint
show_zgrids(gchar *plot_name, gint nth)
{
  SGplot *plot;
  SGlayer *layer;

  plot = sg_project_get_plot(plot_name);

  if(plot){
      layer = (SGlayer *)g_list_nth_data(plot->layers, nth - 1);

      if(layer){
          if(layer->type == SG_LAYER_3D){
            GTK_PLOT3D(layer->real_plot)->az->show_major_grid = TRUE;
            GTK_PLOT3D(layer->real_plot)->az->show_minor_grid = TRUE;
          }else
            return WRAP_ERROR;
          gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
          gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
      } else
          return WRAP_ERROR;
  } else {
      return WRAP_ERROR; 
  }

  return WRAP_OK;
}

gint
hide_xgrids(gchar *plot_name, gint nth)
{
  SGplot *plot;
  SGlayer *layer;

  plot = sg_project_get_plot(plot_name);

  if(plot){
      layer = (SGlayer *)g_list_nth_data(plot->layers, nth - 1);

      if(layer){
          if(layer->type == SG_LAYER_3D){
            GTK_PLOT3D(layer->real_plot)->ax->show_major_grid = FALSE;
            GTK_PLOT3D(layer->real_plot)->ax->show_minor_grid = FALSE;
          }else{
            GTK_PLOT(layer->real_plot)->bottom->show_major_grid = FALSE;
            GTK_PLOT(layer->real_plot)->bottom->show_minor_grid = FALSE;
          }
          gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
          gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
      } else
          return WRAP_ERROR;
  } else {
      return WRAP_ERROR; 
  }

  return WRAP_OK;
}

gint
hide_ygrids(gchar *plot_name, gint nth)
{
  SGplot *plot;
  SGlayer *layer;

  plot = sg_project_get_plot(plot_name);

  if(plot){
      layer = (SGlayer *)g_list_nth_data(plot->layers, nth - 1);

      if(layer){
          if(layer->type == SG_LAYER_3D){
            GTK_PLOT3D(layer->real_plot)->ay->show_major_grid = FALSE;
            GTK_PLOT3D(layer->real_plot)->ay->show_minor_grid = FALSE;
          }else{
            GTK_PLOT(layer->real_plot)->left->show_major_grid = FALSE;
            GTK_PLOT(layer->real_plot)->left->show_minor_grid = FALSE;
          }
          gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
          gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
      } else
          return WRAP_ERROR;
  } else {
      return WRAP_ERROR; 
  }

  return WRAP_OK;
}
