//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function CL_init(init_glob)
// Initialization of CelestLab data
//
// Calling Sequence
// CL_init([init_glob])
//
// Description
// <itemizedlist>
// <listitem>
// <p>Initializes CelestLab data. </p>
// <p> These data consist of physical quantities and other constants. </p>
// <p> The data are defined in 4 files (in the <b>%CL_home/data/utils
// </b> directory):</p> 
// <p> - <b>physical_data.scd</b>: physical data. </p> 
// <p> - <b>constants.scd</b>: miscellaneous data 
// (don't normally change). </p> 
// <p> - <b>planetary_data.scd</b>: planetary data. </p> 
// <p> - <b>TAI_UTC.scd</b>: leap seconds between 
// TAI and UTC. </p> 
// <p></p></listitem>
// <listitem>
// <p>The data once read are stored in variables whose names begin with "%CL_".</p> 
// <p> These variables are defined as globals and returned to the caller as locals. </p>
// <p> The data are the following: </p>
// <p> - The data names from <b>physical_data.scd</b> and 
//        <b>constants.scd</b> ("data.xxx = ...") 
//        become %CL_xxx. </p>
// <p> - The planetary data from <b>planetary_data.scd</b>
//       (data.planet = [...]) 
//        are stored in a structure called %CL_planetC. 
//        %CL_planetC.planet contains the fields: {mu: gravitational 
//        constant, eqRad: equatorial radius, obla: oblateness} </p>
// <p> - The data from <b>TAI_UTC.scd</b> are stored 
//        in a matrix "%CL_TAI_UTC" containing the 3 columns defined in the file: 
//        beginning of interval (JD1950.0), end of interval (JD1950.0), number 
//        of leap seconds (seconds). </p>
// </listitem>
// </itemizedlist>
//
// Parameters
// init_glob: (boolean) Specifies if global variables are defined and initialized (%t by default). 
//
// Authors
// CNES - DCT/SB

// ----------------------------------------------------------------------------
// Declarations:
// ----------------------------------------------------------------------------

if(~exists('%CL_home')) then global %CL_home; end;

if(~exists("init_glob", "local")); init_glob = %t; end


// Internal function: Loading of data 
// returns structure containing all the data
function [data] = load_std_data(nomfic)
  data = struct();
  //-- fprintf(%io(2), "CelestLab: loading %s\n", nomfic);  
  err = exec(nomfic, -1, 'errcatch');
  if (err <> 0)
    CL__error("Error loading file: " + nomfic);
  end
endfunction


// Internal function: Loading of planetary constants
// returns structure containing all the data
function [planets] = load_planet_data(nomfic)
  planets = struct();
  data = struct(); 
  //-- fprintf(%io(2), "CelestLab: loading %s\n", nomfic);  
  err = exec(nomfic, -1, 'errcatch');
  if (err <> 0)
    CL__error("Error loading file: " + nomfic);
  end
  names = fieldnames(data); 
  for nom = names'
    val = data(nom); 
    planets(nom).eqRad = val(1);
    planets(nom).obla  = (val(1)-val(2))/val(1);
    planets(nom).mu    = val(3);
  end
endfunction


// Internal function: Loading of leap seconds 
// returns Nx3 matrix
function [tai_utc] = load_leapsec_data(nomfic)
  data = [];
  //-- fprintf(%io(2), "CelestLab: loading %s\n", nomfic);  
  err = exec(nomfic, -1, 'errcatch');
  if (err <> 0)
    CL__error("Error loading file: " + nomfic);
  end
  tai_utc = data; 
endfunction

// ----------------------------------------------------------------------------
// Code:
// ----------------------------------------------------------------------------

// directory where files are stored
DIR = fullfile(%CL_home, "data", "utils")

// liste des variables globales
tab_glob = [];     // tableau des noms
val_glob = list(); // liste des valeurs

// ----------------------------------------------------------------------------
// Loading of standard data
for nomfic = ["physical_data.scd", "constants.scd"]
  data = load_std_data(fullfile(DIR, nomfic));
  names = fieldnames(data); 
  for nom = names'
    tab_glob($+1) = "%CL_" + nom;
    val_glob($+1) = data(nom);
  end
end

// ----------------------------------------------------------------------------
// Loading of planetary data
tab_glob($+1) = "%CL_planetC";
val_glob($+1) = load_planet_data(fullfile(DIR, "planetary_data.scd"));

// ----------------------------------------------------------------------------
// Loading of leap seconds data  
tab_glob($+1) = "%CL_TAI_UTC";
val_glob($+1) = load_leapsec_data(fullfile(DIR, "TAI_UTC.scd"));


// ----------------------------------------------------------------------------
// Data handling: 
// - declaration as global variables (if init_glob == %t)
// - variables returned as locals (using 'return')  

for k = 1:length(val_glob)
  if (init_glob) 
     execstr("clearglobal " + tab_glob(k)); // clears for safety 
     execstr("global " + tab_glob(k));      // declares as global  
  end
  execstr(tab_glob(k) + " = val_glob(k)");  // affects value  
end

str_glob = strcat(tab_glob, ","); 
cmd = "[" + str_glob + "] = return(" + str_glob + ")"; 
execstr(cmd); // returns variables as locals


endfunction
