//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [y] = CL_interpLagrange(xref, yref, x, n, opt)
// Lagrange interpolation
//
// Calling Sequence
// [y] = CL_interpLagrange(xref, yref, x, [n, opt])
//
// Description
// <itemizedlist><listitem>
// <p>Interpolates using Lagrange method.</p>
// <p>Given reference abscissae <b>(xref)</b> sorted in strictly increasing order and corresponding ordinates <b>(yref)</b>, 
// the function computes interpolated ordinates for the given abscissae <b>(x)</b>.</p>
// <p>The number of points to be used for the interpolation <b>(n)</b> should be an even whole number in the range [2,16].</p> 
// <p>The behavior of the function for the allowed range of values for <b>x</b> can be tweaked via the argument <b>opt</b>:</p>
// <p>- opt = 0 : Default mode. The allowed range for <b>x</b> is [xref(1), xref($)]</p>
// <p>- opt = 1 : Strict (= optimal) mode. The allowed range for <b>x</b> is [xref(n/2), xref($-n/2+1)]</p>
// <p></p></listitem>
// <listitem>
// <p>Notes:</p>  
// <p> - The degree of the polynomial is n-1. </p>
// <p> - y = %nan for values of x that are outside the allowed range</p>
// </listitem>
// </itemizedlist>
//
// Parameters
// xref: Reference abscissae, sorted in strictly increasing order (1xN)
// yref: Corresponding ordinates (PxN)
// x: Abscissae, where to interpolate (1xM)
// y: Interpolated values (PxM)
// n: (optional) Number of points for the interpolation (1x1). Must be even, >= 2 et <= 16. Default is 8. 
// opt: (optional) Use mode (1x1). Default is 0
//
// Authors
// CNES - DCT/SB
//
// Examples
// // Example 1: Exact interpolation 
// xref = 1:6; 
// yref = [xref; xref.^2; xref.^3]; 
// x = 0:1:7; 
//
// [y] = CL_interpLagrange(xref, yref, x, n=4); 
// disp(y - [x; x.^2; x.^3]); 
// 
// [y] = CL_interpLagrange(xref, yref, x, n=4, opt=1); 
// disp(y - [x; x.^2; x.^3]); 
//
// // Example 2: Interpolation of sin and cos
// xref = linspace(0,1,101); 
// yref = [sin(2*%pi*xref); cos(2*%pi*xref)];  
// x = linspace(0,1,1001); 
// 
// [y] = CL_interpLagrange(xref, yref, x, opt=0); 
// scf();
// plot(x, y - [sin(2*%pi*x); cos(2*%pi*x)]); 
// 
// [y] = CL_interpLagrange(xref, yref, x, opt=1); 
// scf();
// plot(x, y - [sin(2*%pi*x); cos(2*%pi*x)]); 



  function [Ind, Coefs] = CL_interpLagrange__mat(xref, x, n)
  // ---------------------------
  // fonction interne : 
  // pr-calculs ne dpendant que des abscisses
  // Ind, Coefs => dimension = nxN 
  // (N: nombre de colonnes de x) 
  // ligne i de Ind,Coefs <=> ieme point d'interpolation (1  n)
  // Pour un vecteur donn yref (mme dimension que xref), 
  // la valeur interpole est: 
  // y = somme_I(yref_I * C_I)
  // C_I = prod_K((x - xref_K)/(xref_I - xref_K), K <> I)
  // I : indices des points d'interpolation (n valeurs)
  // I -> colonne de la matrice Ind
  // C_I ->  colonne de la matrice Coefs
  //
  // Les points d'interpolation (indices I, ici 1  n) 
  // sont choisis tels que:
  // xref_1 < xref_2 < xref_n/2 <= x <= xref_n/2+1 < ... xref_n
  // (sauf aux bords)
  // 
  // NB: pas de controle des arguments. 
  // ---------------------------

    N = size(x,2); 
    Nref = size(xref,2); 

    // dsearch(x,X) renvoie k si x est dans ]X(k), X(k+1)]
    // extension de l'intervalle xref pour interpoler au del des bornes
    // => K == -1 si hors intervalle
    // K => vecteur ligne
    K = dsearch(x, [xref(1) - (xref(2) - xref(1)), xref, xref(Nref) + (xref(Nref) - xref(Nref-1))]) - 1; 

    // indices min des points d'interpolation
    // (K reprsente l'indice d'ordre n/2)
    Kmin = K - n/2 + 1; 

    // les n indices (Kmin .. Kmin+n-1) sont forcs dans [1, Nref] 
    I = find(Kmin < 1); 
    Kmin(I) = 1; 
    I = find(Kmin + n - 1 > Nref); 
    Kmin(I) = Nref - n + 1; 

    // colonne de Ind: indices d'interpolation
    Ind = zeros(n, N); 
    for i = 1 : n
      Ind(i,:) = Kmin + i - 1; 
    end
    
    // Valeurs de xref aux indices Ind
    Xref = matrix(xref(Ind), size(Ind)); 

    // pas de rsultat si abscisse hors intervalle 
    I = find(K == -1); 
    Xref(:,I) = %nan; 

    Coefs = zeros(n, N); 
    C1 = ones(n-1, 1); 

    for i = 1 : n
      I = [1:i-1, i+1:n]; 
      Coefs(i,:) = prod((C1 * x - Xref(I,:)) ./        ...
                        (C1 * Xref(i,:) - Xref(I,:)), "r"); 
    end

  endfunction


  // ---------------------------
  // MAIN 
  // ---------------------------

  if (~exists("n", "local")); n = 8; end 
  if (~exists("opt", "local")); opt = 0; end 

  if (opt <> 0 & opt <> 1) 
    CL__error("Invalid value for opt"); 
  end

  if (n < 2 | n > 16 | modulo(n,2) <> 0) 
    CL__error("Invalid value for n"); 
  end

  N = size(x,2); 
  Nref = size(xref,2); 
  P = size(yref,1); 

  if (size(xref,1) <> 1 | size(yref,2) <> Nref)
    CL__error("Wrong input argument size (xref or yref)"); 
  end

  if (N == 0) 
    y = []; 
    return; // <== RETURN
  end

  if (Nref < n)
    CL__error("Insufficient number of values in xref (should be >= n)"); 
  end

  if (find(xref(2:$)-xref(1:$-1) <= 0) <> [])
    CL__error("Abscissae not strictly increasing"); 
  end

  // indices et coefs d'interpolation
  [Ind, Coefs] = CL_interpLagrange__mat(xref, x, n); 

  // interpolation pour chaque ligne de yref
  y = zeros(P,N); 

  for k = 1 : P

    // -- version lisible:
    // yrefk = yref(k,:); 
    // Yrefk = matrix(yrefk(Ind), size(Ind)); 
    // y(k,:) = sum(Yrefk .* Coefs, "r");

    // -- version compacte (sans variables intermdiaires):
    y(k,:) = sum(matrix(yref(k,Ind), size(Ind)) .* Coefs, "r"); 
 
  end

  // Invalidation de certains rsultats suivant option 
  // et valeurs des abscisses

  // n1/n2 : indices min/max o interpolation OK (hors marges)

  if (opt == 0)
    n1 = 1; 
    n2 = Nref;
  else
    n1 = n/2; 
    n2 = Nref - n/2 + 1;
  end

  // eps1/eps2 : marges min/max (> 0)

  m1 = max(n1,2); 
  m2 = min(n2,Nref-1); 
  eps1 = (xref(m1) - xref(m1-1))/100; 
  eps2 = (xref(m2+1) - xref(m2))/100; 

  I = find(x < xref(n1) - eps1 | x > xref(n2) + eps2); 
  y(:,I) = %nan; 

endfunction


