//  Copyright (c) CNES  2011
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [pos2,vel2,jacob] = CL__fr_convPosVel(pos1,vel1,mat,omega,cvel,cjac)
// converts position and velocity and computes the jacobian
//
// Calling Sequence
// [pos2,vel2,jacob] = CL__fr_convPosVel(pos1,vel1,mat,omega,cvel,cjac)
//
// Description
// <itemizedlist><listitem>
// <p>Computes position and velocity relative to frame2 as function of position and velocity relative to frame1: </p> 
// <p>pos2 = mat * pos1 </p> 
// <p>vel2 = mat * (vel1 - omega ^ p1) </p> 
// <p>The velocity is computed if cvel = %t and omega is not empty.</p>
// <p>The jacobian is computed if cjac == %t and omega is not empty.</p>
// <p>jacob = d(pos2,vel2)/d(pos1,vel1), more precisely jacob(i,j,:) = d(pv2(i,:)/d(pv1(j,:))</p>
// <p></p></listitem>
// </itemizedlist>
//
// Parameters
// pos1: Position vector relative to frame1 (with coordinates given in frame1) (3xN)
// vel1: Velocity vector relative to frame1 (with coordinates given in frame1) (3xN)
// mat: Transformation matrix from frame1 to frame2 (pos2 = mat * pos1) (3x3xN)
// omega: (optional) Angular velocity vector of frame2 wrt frame1 with coordinates in frame1 (3xN). Empty by default. 
// cvel: (boolean, optional, NAME_ARG) %t if velocity should be computed (1x1). Default is %t. 
// cjac: (boolean, optional, NAME_ARG) %t if jacobian should be computed (1x1). Default is %f. 
// pos2: Position vector relative to frame2 (with coordinates given in frame2) (3xN)
// vel2: Velocity vector relative to frame2 (with coordinates given in frame2) (3xN)
// jacob: (optional) Jacobian of the transformation (pos1,vel1) to (pos2,vel2) (6x6xN)
//
// Authors
// CNES - DCT/SB
//

// Declarations:

// Code:
if (~exists("omega", "local")); omega = []; end
if (~exists("cvel", "local")); cvel = %t; end
if (~exists("cjac", "local")); cjac = %f; end

// omega == [] => can't compute velocity or jacobian
if (omega == [])
  cvel = %f; 
  cjac = %f;  
end

pos2 = mat * pos1;
vel2 = [];
jacob = [];

if (cvel)
  vel2 = mat * (vel1 - CL_cross(omega, pos1));
end

if (cjac)
  jacob = CL__fr_jacobian(mat, omega);
end

endfunction
