//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [pos_car,vel_car,jacob]=CL_oe_kep2car(kep, mu)
// Keplerian to cartesian orbital elements
//
// Calling Sequence
// [pos_car,vel_car,jacob] = CL_oe_kep2car(kep [,mu])
//
// Description
// <itemizedlist><listitem>
// <p>Converts classical keplerian orbital elements to position and velocity  
// for an elliptic, hyperbolic or parabolic orbit.</p>
// <p>The transformation jacobian is optionally computed.</p>
// <p>See <link linkend="Orbital Elements">Orbital elements</link> for more details</p>
// </listitem>
// </itemizedlist>
//
// Parameters
// kep: classical keplerian parameters [sma;ecc;inc;pom;raan;anm] [m,rad] (6xN)
// mu : (optional) gravitational constant. [m^3/s^2] (default value is %CL_mu)
// pos_car: position [X;Y;Z] [m] (3xN)
// vel_car: velocity [Vx;Vy;Vz] [m/s] (3xN)
// jacob: (optional) transformation jacobian (See <link linkend="Orbital Elements">Orbital elements</link> for more details) (6x6xN)
//
// Bibliography
// 1) CNES - MSLIB FORTRAN 90, Volume V (mv_kep_car)
//
// See also
// CL_oe_car2kep
// CL_oe_kep2cirEqua
//
// Authors
// CNES - DCT/SB
//
// Examples
// // Example 1
// kep=[24464560,0.7311,0.122138,3.10686,1.00681,0.048363]';
// [pos_car,vel_car]=CL_oe_kep2car(kep);
// [kep2]=CL_oe_car2kep(pos_car,vel_car);

// Declarations:
global %CL__PRIV; 
if (~exists("%CL_epsilon")); %CL_epsilon = %CL__PRIV.epsilon; end
if (~exists("%CL_mu")); %CL_mu = %CL__PRIV.mu; end

// Code:
[lhs,rhs]=argn();
if ~exists('mu','local') then mu=%CL_mu; end

select lhs
  case 3
    compute_jacob=%t
  case 2
    compute_jacob=%f
  case 1
    compute_jacob=%f
  else
    CL__error('Invalid number of output arguments')
end

aoup = kep(1,:)
ecc = kep(2,:)
ai = kep(3,:)

eps100 = 100*%eps
N = size(kep,2);
pos_car = zeros(3,N);
vel_car = zeros(3,N);
if (compute_jacob) 
  jacob = zeros(6,6,N);
end

//data tests
i_aoup = find(aoup<=eps100) //semi major axis or parabola parameter <=0
if i_aoup~=[] then CL__error('semi major axis or parabola parameter <=0'); end

i_ecc = find(ecc<0)
if i_ecc~=[] then CL__error('negative eccentricity'); end

i_ai = find( (ai<0)|(ai>%pi) )
if i_ai~=[] then CL__error('negative inclination or bigger than pi'); end

//----------------------
// Keplerian -> cartesien conversion depending on type of orbit
//----------------------

// Elliptic case
ie = find( ecc <= 1-%CL_epsilon.parab )
if ie~=[]
  if compute_jacob
    [pos,vel,jac] = CL__oe_kep2carEllip(kep(:,ie),mu)
    jacob(:,:,ie) = jac
  else
    [pos,vel] = CL__oe_kep2carEllip(kep(:,ie),mu)
  end
  pos_car(:,ie) = pos
  vel_car(:,ie) = vel
end

// Hyperbolic case
ih = find( ecc >= 1+%CL_epsilon.parab )
if ih~=[]
  if compute_jacob
    [pos,vel,jac] = CL__oe_kep2carHyperb(kep(:,ih),mu)
    jacob(:,:,ih) = jac
  else
    [pos,vel] = CL__oe_kep2carHyperb(kep(:,ih),mu)
  end
  pos_car(:,ih) = pos
  vel_car(:,ih) = vel
end

// Parabolic case
ip = find( (ecc>1-%CL_epsilon.parab)&(ecc<1+%CL_epsilon.parab) )
if ip~=[]
  if compute_jacob
    [pos,vel,jac] = CL__oe_kep2carParab(kep(:,ip),mu)
    jacob(:,:,ip) = jac
  else
    [pos,vel] = CL__oe_kep2carParab(kep(:,ip),mu)
  end
  pos_car(:,ip) = pos
  vel_car(:,ip) = vel
end

endfunction

