//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [q] = CL_rot_matrix2quat(M)
// Transformation matrix to quaternion
//
// Calling Sequence
// q = CL_rot_matrix2quat(M)
//
// Description
// <itemizedlist><listitem>
// <p>Computes the quaternion corresponding to a given transformation matrix.</p> 
// <p></p></listitem>
// <listitem>
// <p><b>Notes:</b></p>
// <p> - See <link linkend="CL_rot_angles2matrix">CL_rot_angles2matrix</link> for 
// conventions regarding the definition of the transformation matrix. </p>
// <p> - See <link linkend="Data types">Data types</link> or <link linkend="CL_rot_defQuat">CL_rot_defQuat</link> for more details on quaternions. </p>
// </listitem>
// </itemizedlist>
//
// Parameters
// M : Transformation matrix (3x3xN)
// q : Quaternion (dim N)
//
// Authors
// CNES - DCT/SB
//
// Bibliography
// 1) Mecanique Spatiale - CNES Cepadues 1995, Tome I, 7.2.2.3 Les quaternions
// 2) James R. Wertz, Spacecraft attitude determination and control (volume 73 of Astrophyisics and Space Science Library), D. Reidel Publishing Company, 1980, appendix D-E
//
// See also
// CL_rot_defQuat
// CL_rot_quat2matrix
// CL_rot_angles2matrix
//
// Examples
// ang = CL_deg2rad(10);
// M = CL_rot_angles2matrix(3,ang) // rotation around Z axis
// q = CL_rot_matrix2quat(M)
// M2 = CL_rot_quat2matrix(q)    //same as M1

// Declarations:


// Code:


if (argn(2) <> 1); CL__error("Invalid number of input arguments"); end,

// return [] if argument is empty
if (M == [])
  q = CL_rot_defQuat([]); 
  return; 
end

select typeof(M)
  case "constant"
    [li co]=size(M);
    n = 1;
  case "hypermat"
    [li co n]=size(M);
  else
    CL__error("incorrect matrix as input argument");
end

if (li==3 & co==3) then
  a11=M(1,1,:); a11=a11(:);
  a12=M(1,2,:); a12=a12(:);
  a13=M(1,3,:); a13=a13(:);
  a21=M(2,1,:); a21=a21(:);
  a22=M(2,2,:); a22=a22(:);
  a23=M(2,3,:); a23=a23(:);
  a31=M(3,1,:); a31=a31(:);
  a32=M(3,2,:); a32=a32(:);
  a33=M(3,3,:); a33=a33(:);
else
  CL__error("check input matrix");
end;

q1=zeros(n,1);
q2=zeros(n,1);
q3=zeros(n,1);
q4=zeros(n,1);

V = real( sqrt(1+[a11 - a22 - a33, a22 - a33 - a11, a33 - a11 - a22,  a11 + a22 + a33]) );

[m,k] = max(V,'c');

for kk=unique(k)'
  nn=find(k==kk);
  select kk
    case 1
      q1(nn)=0.5*V(nn,1);
      q2(nn)=(a12(nn)+a21(nn))./(4*q1(nn));
      q3(nn)=(a13(nn)+a31(nn))./(4*q1(nn));
      q4(nn)=(a23(nn)-a32(nn))./(4*q1(nn));
    case 2
      q2(nn)=0.5*V(nn,2);
      q1(nn)=(a12(nn)+a21(nn))./(4*q2(nn));
      q3(nn)=(a23(nn)+a32(nn))./(4*q2(nn));
      q4(nn)=(a31(nn)-a13(nn))./(4*q2(nn));
    case 3
      q3(nn)=0.5*V(nn,3);
      q1(nn)=(a13(nn)+a31(nn))./(4*q3(nn));
      q2(nn)=(a23(nn)+a32(nn))./(4*q3(nn));
      q4(nn)=(a12(nn)-a21(nn))./(4*q3(nn));
    case 4
      q4(nn)=0.5*V(nn,4);
      q1(nn)=(a23(nn)-a32(nn))./(4*q4(nn));
      q2(nn)=(a31(nn)-a13(nn))./(4*q4(nn));
      q3(nn)=(a12(nn)-a21(nn))./(4*q4(nn));
  end

end

q=CL_rot_defQuat(q4',q1',q2',q3');

endfunction
