//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [mean_kep] = CL_ex_meanLyddane(osc_kep, er,mu,zonals)
// Lyddane orbit propagation analytical model (mean elements)
//
// Calling Sequence
// mean_kep = CL_ex_meanLyddane(osc_kep [,er,mu,zonals])
//
// Description
// <itemizedlist><listitem>
// <p>Computes the mean Keplerian elements <b>mean_kep</b> from 
// the osculating Keplerian elements <b>osc_kep</b>.</p>
// <p></p></listitem>
// <listitem> 
// <p>The orbital elements are the following: </p> 
// <p><inlinemediaobject><imageobject><imagedata fileref="kep_par.gif"/></imageobject></inlinemediaobject></p>
// <p></p></listitem>
// <listitem>
// <p>Warnings :</p>
// <p> - This function does not work for inclinations close to the critical inclinations (63.43494882 deg and 116.5650512 deg)</p>
// <p> - This function does not work for eccentricities higher than 0.9</p></listitem>
// </itemizedlist>
//
// Parameters
// osc_kep: Osculating Keplerian elements [sma;ecc;inc;pom;raan;anm] (6xN)
// er: (optional) Equatorial radius [m] (default is %CL_eqRad)
// mu: (optional) Gravitational constant [m^3/s^2] (default value is %CL_mu)
// zonals: (optional) Vector of zonals coefficients J1 to Jn (troncated to J5) to be used (default is %CL_j1jn(1:5)) (Nz x 1)
// mean_kep: Mean Keplerian elements [sma;ecc;inc;pom;raan;anm] (6xN)
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_ex_lyddane
// CL_ex_lyddaneMan
//
// Bibliography
// 1) CNES - MSLIB FORTRAN 90, Volume E (me_lyddane_mean)
//
// Examples
// osc_kep_t1 = [42166712 ; 7.9e-3 ; CL_deg2rad([97.2 ; 89 ; 125 ; 0])];
// mean_kep_t1 = CL_ex_meanLyddane(osc_kep_t1)
// // check 
// [mean_kep_t2,osc_kep_t2] = CL_ex_lyddane(0,mean_kep_t1,0)

//

//************************************************************************
// But:  Calcul des parametres meanens du modele d'extrapolation d'orbite de Lydanne
// ===
//
// Note d'utilisation: les unites de distance sont en metres OBLIGATOIREMENT
// ==================  (pour mu, er, et a)
//
//                     l'eccentricite doit appartenir a [ 0. , 0.9 [
//                     le domaine d'erreur est          < 0. et >= 0.9
//
//                     l'inclinaison  doit appartenir a [ 0. , pi ]
//                     et non proches des inclinaisons critiques definies par
//                     (pm_i_critique_non_retro, pm_i_critique_retro) a pm_eps_i_critique pres.
//************************************************************************

//************************************************************************
// But:  Calcul des parametres meanens du modele d'extrapolation d'orbite de Lydanne
// ===
//
// Note d'utilisation: les unites de distance sont en metres OBLIGATOIREMENT
// ==================  (pour mu, er, et a)
//
//                     l'eccentricite doit appartenir a [ 0. , 0.9 [
//                     le domaine d'erreur est          < 0. et >= 0.9
//
//                     l'inclinaison  doit appartenir a [ 0. , pi ]
//                     et non proches des inclinaisons critiques definies par
//                     (pm_i_critique_non_retro, pm_i_critique_retro) a pm_eps_i_critique pres.
//************************************************************************

// Declarations:
global %CL__PRIV; 
if (~exists("%CL_eqRad")); %CL_eqRad = %CL__PRIV.eqRad; end
if (~exists("%CL_mu")); %CL_mu = %CL__PRIV.mu; end
if (~exists("%CL_j1jn")); %CL_j1jn = %CL__PRIV.j1jn; end

// Code:
if ~exists('er','local') then er=%CL_eqRad; end
if ~exists('mu','local') then mu=%CL_mu; end
if ~exists('zonals','local') then zonals=%CL_j1jn(1:5); end

eps100 = 100 * %eps // epsilon pour test de convergence


// interface de lyddane (parametres adaptes) 
// NB: er,mu,zonals : are those from the upper level 
function [osc_cir] = Calcul_osc(moy_cir)
  t0 = 0; 
  t1 = 0; 
  moy0 = CL_oe_cirEqua2kep(moy_cir);
  [moy1, osc1] = CL_ex_lyddane(t0, moy0, t1, er,mu,zonals);
  osc_cir = CL_oe_kep2cirEqua(osc1);
endfunction


// initialisation des parametres 
osc_cir_ref = CL_oe_kep2cirEqua(osc_kep); // osculateurs de reference (adaptes) 
moy_cir = osc_cir_ref; 
moy_cir(6,:) = CL_rMod(moy_cir(6,:),0,2*%pi);
osc_cir = zeros(osc_kep); 
ecart = zeros(osc_kep); 

// ecarts max admis
ecart_admi = eps100 * (1 + abs(moy_cir)); 


// debut d'iteration
I = 1 : size(osc_kep,2);  // I : indices non OK
itermax = 50; // nombre maximum d'iterations 
iter = 0; // current number of iterations

while (I ~= [] & iter <= itermax) 

  // calcul des osculateurs correspondant aux meanens (a la meme date)
  osc_cir(:,I) = Calcul_osc(moy_cir(:,I));

  // calcul des ecarts 
  ecart(:,I)  = osc_cir_ref(:,I) - osc_cir(:,I);
  ecart(6,I) = CL_rMod(ecart(6,I), -%pi, %pi);

  // mise a jour
  moy_cir(:,I) = moy_cir(:,I) + ecart(:,I);

  I = find( max(abs(ecart) - ecart_admi, 'r') > 0 );

  iter = iter + 1

end

if (I ~= [])
  CL__error("No convergence in computation of mean elements")
end 


// affectation des parametres de sortie
mean_kep = CL_oe_cirEqua2kep(moy_cir);
mean_kep(4:6,:) = CL_rMod(mean_kep(4:6,:), osc_kep(4:6,:)-%pi, osc_kep(4:6,:)+%pi);

endfunction
