/** @file gtkimcontextscim.cpp
 *  @brief immodule for GTK2.
 */

/* 
 * Smart Common Input Method
 * 
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: gtkimcontextscim.cpp,v 1.68.2.1 2004/05/01 13:23:30 suzhe Exp $
 */

#define Uses_SCIM_DEBUG
#define Uses_SCIM_BACKEND
#define Uses_SCIM_SERVER
#define Uses_SCIM_SERVER_MODULE
#define Uses_SCIM_CONFIG
#define Uses_SCIM_CONFIG_MODULE
#define Uses_SCIM_CONFIG_PATH
#define Uses_SCIM_SOCKET_TRANSACTION
#define Uses_SCIM_PANEL
#include <iostream>

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include <gdk/gdk.h>

#include "scim_private.h"
#include "scim.h"

using namespace scim;

#include "gtkimcontextscim.h"

#define GTK_TYPE_IM_CONTEXT_SCIM             _gtk_type_im_context_scim
#define GTK_IM_CONTEXT_SCIM(obj)             (GTK_CHECK_CAST ((obj), GTK_TYPE_IM_CONTEXT_SCIM, GtkIMContextSCIM))
#define GTK_IM_CONTEXT_SCIM_CLASS(klass)     (GTK_CHECK_CLASS_CAST ((klass), GTK_TYPE_IM_CONTEXT_SCIM, GtkIMContextSCIMClass))
#define GTK_IS_IM_CONTEXT_SCIM(obj)          (GTK_CHECK_TYPE ((obj), GTK_TYPE_IM_CONTEXT_SCIM))
#define GTK_IS_IM_CONTEXT_SCIM_CLASS(klass)  (GTK_CHECK_CLASS_TYPE ((klass), GTK_TYPE_IM_CONTEXT_SCIM))
#define GTK_IM_CONTEXT_SCIM_GET_CLASS(obj)   (GTK_CHECK_GET_CLASS ((obj), GTK_TYPE_IM_CONTEXT_SCIM, GtkIMContextSCIMClass))

#define SCIM_MAX_IM_CONTEXT_NUMBER           256

#define SCIM_CONFIG_GTKIMMODULE_PANEL_PROGRAM "/GtkIMModule/PanelProgram"

#ifndef SCIM_PANEL_PROGRAM
  #define SCIM_PANEL_PROGRAM                 "scim-panel-gtk"
#endif

#ifndef SCIM_KEYBOARD_ICON_FILE
  #define SCIM_KEYBOARD_ICON_FILE            (SCIM_ICONDIR "/keyboard.png")
#endif

typedef std::vector<KeyEvent> KeyEventVector;

/* Typedef */
struct _GtkIMContextSCIMImpl
{
    ServerFactoryPointer   factory;
    ServerInstancePointer  si;
    SocketClient           panel;
    SocketTransaction      send;
    String                 charset;
    GdkWindow             *client_window;
    WideString             preedit_string;
    AttributeList          preedit_attrlist;
    gint                   preedit_caret;
    gint                   cursor_x;
    gint                   cursor_y;
    guint                  panel_source_id;
    gboolean               use_preedit;
    gboolean               is_on;
};

struct _GSCIMPanelSource
{
    GSource                source;
    GtkIMContextSCIM      *context;
    GPollFD                fd;
};

typedef struct _GSCIMPanelSource GSCIMPanelSource;

/* Methods declaration */
static void     gtk_im_context_scim_class_init         (GtkIMContextSCIMClass *klass,
                                                        gpointer              *klass_data);
static void     gtk_im_context_scim_init               (GtkIMContextSCIM      *im_context_scim,
                                                        GtkIMContextSCIMClass *klass);
static void     gtk_im_context_scim_finalize           (GObject               *obj);
static void     gtk_im_context_scim_set_client_window  (GtkIMContext          *context,
                                                        GdkWindow             *client_window);
static gboolean gtk_im_context_scim_filter_keypress    (GtkIMContext          *context,
                                                        GdkEventKey           *key);
static void     gtk_im_context_scim_reset              (GtkIMContext          *context);
static void     gtk_im_context_scim_focus_in           (GtkIMContext          *context);
static void     gtk_im_context_scim_focus_out          (GtkIMContext          *context);
static void     gtk_im_context_scim_set_cursor_location(GtkIMContext          *context,
                                                        GdkRectangle          *area);
static void     gtk_im_context_scim_set_use_preedit    (GtkIMContext          *context,
                                                        gboolean               use_preedit);
static void     gtk_im_context_scim_get_preedit_string (GtkIMContext          *context,
                                                        gchar                **str,
                                                        PangoAttrList        **attrs,
                                                        gint                  *cursor_pos);

/* private functions */
static void     launch_panel ();

static bool     panel_connect_server                  (GtkIMContextSCIM      *ic);
static bool     panel_prepare_transaction             (GtkIMContextSCIM      *ic);
static bool     panel_send_request                    (GtkIMContextSCIM      *ic);
static bool     panel_receive_reply                   (GtkIMContextSCIM      *ic);

static void     panel_req_turn_on_panel               (GtkIMContextSCIM      *ic);
static void     panel_req_turn_off_panel              (GtkIMContextSCIM      *ic);
static void     panel_req_update_display              (GtkIMContextSCIM      *ic);
static void     panel_req_update_screen               (GtkIMContextSCIM      *ic);
static void     panel_req_show_help                   (GtkIMContextSCIM      *ic);
static void     panel_req_show_server_menu            (GtkIMContextSCIM      *ic);
static void     panel_req_focus_in                    (GtkIMContextSCIM      *ic);
static void     panel_req_focus_out                   (GtkIMContextSCIM      *ic);
static void     panel_req_update_server_info          (GtkIMContextSCIM      *ic);
static void     panel_req_update_spot_location        (GtkIMContextSCIM      *ic);
static void     panel_req_show_preedit_string         (GtkIMContextSCIM      *ic);
static void     panel_req_hide_preedit_string         (GtkIMContextSCIM      *ic);
static void     panel_req_update_preedit_string       (GtkIMContextSCIM      *ic,
                                                        const WideString      &str,
                                                        const AttributeList   &attrs);
static void     panel_req_update_preedit_caret        (GtkIMContextSCIM      *ic,
                                                        int                    caret);
static void     panel_req_show_aux_string             (GtkIMContextSCIM      *ic);
static void     panel_req_hide_aux_string             (GtkIMContextSCIM      *ic);
static void     panel_req_update_aux_string           (GtkIMContextSCIM      *ic,
                                                        const WideString      &str,
                                                        const AttributeList   &attrs);
static void     panel_req_show_status_string          (GtkIMContextSCIM      *ic);
static void     panel_req_hide_status_string          (GtkIMContextSCIM      *ic);
static void     panel_req_update_status_string        (GtkIMContextSCIM      *ic,
                                                        const WideString      &str,
                                                        const AttributeList   &attrs);
static void     panel_req_show_lookup_table           (GtkIMContextSCIM      *ic);
static void     panel_req_hide_lookup_table           (GtkIMContextSCIM      *ic);
static void     panel_req_update_lookup_table         (GtkIMContextSCIM      *ic,
                                                        const LookupTable     &table);
static void     panel_req_update_full_width_punct     (GtkIMContextSCIM      *ic,
                                                        bool                   full);
static void     panel_req_update_full_width_letter    (GtkIMContextSCIM      *ic,
                                                        bool                   full);

/* Panel Source relates functions */
static gboolean panel_source_prepare                  (GSource               *source,
                                                        gint                  *timeout);
static gboolean panel_source_check                    (GSource               *source);
static gboolean panel_source_dispatch                 (GSource               *source,
                                                        GSourceFunc            callback,
                                                        gpointer               user_data);
static bool     panel_check_connection                (GtkIMContextSCIM      *ic);

/* utility functions */
static void     set_focus_ic                           (GtkIMContextSCIM      *ic);

static bool     commit_key_event                       (GtkIMContextSCIM      *ic,
                                                        const GdkEventKey     &event);

static bool     match_key_event                        (const KeyEventVector  &keys,
                                                        const KeyEvent        &key);

static KeyEvent keyevent_gdk_to_scim                   (const GdkEventKey     &gdkevent);

static void     initialize                             (void);

static void     finalize                               (void);

static void     attach_server_instance                 (GtkIMContextSCIM      *ic);

static bool     open_default_server_factory            (GtkIMContextSCIM      *ic,
                                                        int                    icid);
static void     open_next_server_factory               (GtkIMContextSCIM      *ic);
static void     open_previous_server_factory           (GtkIMContextSCIM      *ic);
static void     open_specific_server_factory           (GtkIMContextSCIM      *ic,
                                                        const String           &uuid);

static String   get_help_info                          (GtkIMContextSCIM      *ic);

/* slot functions */
static void     slot_show_preedit_string               (ServerInstanceBase    *si);
static void     slot_show_status_string                (ServerInstanceBase    *si);
static void     slot_show_aux_string                   (ServerInstanceBase    *si);
static void     slot_show_lookup_table                 (ServerInstanceBase    *si);

static void     slot_hide_preedit_string               (ServerInstanceBase    *si);
static void     slot_hide_status_string                (ServerInstanceBase    *si);
static void     slot_hide_aux_string                   (ServerInstanceBase    *si);
static void     slot_hide_lookup_table                 (ServerInstanceBase    *si);

static void     slot_update_preedit_caret              (ServerInstanceBase    *si,
                                                        int                    caret);
static void     slot_update_preedit_string             (ServerInstanceBase    *si,
                                                        const WideString      &str,
                                                        const AttributeList   &attrs);
static void     slot_update_status_string              (ServerInstanceBase    *si,
                                                        const WideString      &str,
                                                        const AttributeList   &attrs);
static void     slot_update_aux_string                 (ServerInstanceBase    *si,
                                                        const WideString      &str,
                                                        const AttributeList   &attrs);
static void     slot_commit_string                     (ServerInstanceBase    *si,
                                                        const WideString      &str);
static void     slot_forward_keyevent                  (ServerInstanceBase    *si,
                                                        const KeyEvent        &key);
static void     slot_update_lookup_table               (ServerInstanceBase    *si,
                                                        const LookupTable     &table);

static void     slot_update_full_width_punct           (ServerInstanceBase    *si,
                                                        bool                   full);
static void     slot_update_full_width_letter          (ServerInstanceBase    *si,
                                                        bool                   full);

/* Local variables declaration */
static GType                 _gtk_type_im_context_scim   = 0;
static GObjectClass         *_parent_klass               = 0;

static KeyEventVector        _trigger_keys;
static KeyEventVector        _next_server_keys;
static KeyEventVector        _previous_server_keys;
static int                   _valid_key_mask;

static ConfigModule         *_config_module              = 0;
static ServerModule         *_server_modules             = 0;
static ConfigPointer         _config;
static BackEndPointer        _backend;

static ServerFactoryPointer  _default_server_factory;
static int                   _default_server_factory_idx = -1;

static String                _current_encoding;

static GtkIMContextSCIM     *_focused_ic                 = 0;

static GtkIMContextSCIM     *_im_contexts [SCIM_MAX_IM_CONTEXT_NUMBER];

static bool                  _scim_initialized           = false;

static String                _panel_address;
static int                   _panel_timeout;
static bool                  _panel_launched             = false;

static GSourceFuncs          _panel_source_funcs        =
{
    panel_source_prepare,
    panel_source_check,
    panel_source_dispatch,
    NULL
};

/* Function Implementations */

/* Public functions */
GtkIMContext *
gtk_im_context_scim_new (void)
{
    SCIM_DEBUG_FRONTEND(1) << "gtk_im_context_scim_new...\n";

    GtkIMContextSCIM *result;
    
    result = GTK_IM_CONTEXT_SCIM (g_object_new (GTK_TYPE_IM_CONTEXT_SCIM, NULL));

    return GTK_IM_CONTEXT (result);
}

void
gtk_im_context_scim_register_type (GTypeModule *type_module)
{
    static const GTypeInfo im_context_scim_info =
    {
        sizeof               (GtkIMContextSCIMClass),
        (GBaseInitFunc)       NULL,
        (GBaseFinalizeFunc)   NULL,
        (GClassInitFunc)      gtk_im_context_scim_class_init,
        NULL,                 /* class_finalize */
        NULL,                 /* class_data */
        sizeof               (GtkIMContextSCIM),
        0,
        (GtkObjectInitFunc)  gtk_im_context_scim_init,
    };

    SCIM_DEBUG_FRONTEND(1) << "gtk_im_context_scim_register_type...\n";

    if (!_gtk_type_im_context_scim) {
        _gtk_type_im_context_scim = 
            g_type_module_register_type (type_module,
                                     GTK_TYPE_IM_CONTEXT,
                                     "GtkIMContextSCIM",
                                     &im_context_scim_info,
                                     (GTypeFlags) 0);
    }
}

void
gtk_im_context_scim_shutdown (void)
{
    SCIM_DEBUG_FRONTEND(1) << "gtk_im_context_scim_shutdown...\n";

    finalize ();
}

gchar *
gtk_im_context_scim_get_supported_locales ()
{
    SCIM_DEBUG_FRONTEND(1) << "gtk_im_context_scim_get_supported_locales...\n";

    return g_strdup (_backend->get_locales ().c_str ());
}

/* Private functions */
static void
gtk_im_context_scim_class_init (GtkIMContextSCIMClass *klass,
                                gpointer              *klass_data)
{
    SCIM_DEBUG_FRONTEND(1) << "gtk_im_context_scim_class_init...\n";

    GtkIMContextClass *im_context_klass = GTK_IM_CONTEXT_CLASS (klass);
    GObjectClass      *gobject_klass    = G_OBJECT_CLASS (klass);

    _parent_klass = (GObjectClass *) g_type_class_peek_parent (klass);

    im_context_klass->set_client_window   = gtk_im_context_scim_set_client_window;
    im_context_klass->filter_keypress     = gtk_im_context_scim_filter_keypress;
    im_context_klass->reset               = gtk_im_context_scim_reset;
    im_context_klass->get_preedit_string  = gtk_im_context_scim_get_preedit_string;
    im_context_klass->focus_in            = gtk_im_context_scim_focus_in;
    im_context_klass->focus_out           = gtk_im_context_scim_focus_out;
    im_context_klass->set_cursor_location = gtk_im_context_scim_set_cursor_location;
    im_context_klass->set_use_preedit     = gtk_im_context_scim_set_use_preedit;
    gobject_klass->finalize               = gtk_im_context_scim_finalize;

    if (!_scim_initialized) {
        initialize ();
        _scim_initialized = true;
    }
}

static void
gtk_im_context_scim_init (GtkIMContextSCIM      *context_scim,
                          GtkIMContextSCIMClass *klass)
{
    SCIM_DEBUG_FRONTEND(1) << "gtk_im_context_scim_init...\n";

    int   icid = -1;
    const gchar *charset;

    context_scim->impl = NULL;

    for (int i=0; i<SCIM_MAX_IM_CONTEXT_NUMBER; i++) {
        if (_im_contexts [i] == NULL) {
            icid = i;
            break;
        }
    }

    if (icid < 0)
        return;

    context_scim->impl = new GtkIMContextSCIMImpl;

    g_get_charset (&charset);

    if (charset)
        context_scim->impl->charset = String (charset);
    else
        context_scim->impl->charset = String ("UTF-8");

    if (!open_default_server_factory (context_scim, icid)) {
        delete context_scim->impl;
        context_scim->impl = 0;
        return;
    }

    context_scim->impl->client_window = NULL;
    context_scim->impl->preedit_caret = 0;
    context_scim->impl->cursor_x = 0;
    context_scim->impl->cursor_y = 0;
    context_scim->impl->is_on = FALSE;
    context_scim->impl->use_preedit = TRUE;
    context_scim->impl->panel_source_id = 0;

    _im_contexts [icid] = context_scim;

    panel_connect_server (context_scim);
}

static void
gtk_im_context_scim_finalize (GObject *obj)
{
    SCIM_DEBUG_FRONTEND(1) << "gtk_im_context_scim_finalize...\n";

    GtkIMContextSCIM *context_scim = GTK_IM_CONTEXT_SCIM (obj);
    int icid;

    if (context_scim->impl) {
        icid = context_scim->impl->si->get_id ();
        if (!context_scim->impl->si.null ())
            context_scim->impl->si.reset ();
        if (!context_scim->impl->factory.null ())
            context_scim->impl->factory.reset ();
        if (context_scim->impl->panel.is_connected ())
            context_scim->impl->panel.close ();
        if (context_scim->impl->panel_source_id)
            g_source_remove (context_scim->impl->panel_source_id);

        delete context_scim->impl;
        _im_contexts [icid] = NULL;
    }
}

static void
gtk_im_context_scim_set_client_window (GtkIMContext *context,
                                       GdkWindow    *client_window)
{
    SCIM_DEBUG_FRONTEND(1) << "gtk_im_context_scim_set_client_window...\n";

    GtkIMContextSCIM *context_scim = GTK_IM_CONTEXT_SCIM (context);

    if (context_scim && context_scim->impl) {
        context_scim->impl->client_window = client_window;
    }
}

static gboolean
gtk_im_context_scim_filter_keypress (GtkIMContext *context,
                                     GdkEventKey  *event)
{
//    SCIM_DEBUG_FRONTEND(1) << "gtk_im_context_scim_filter_keypress...\n";

    gboolean ret = TRUE;

    GtkIMContextSCIM *context_scim = GTK_IM_CONTEXT_SCIM (context);

    if (!context_scim || !context_scim->impl ||
        (event->type != GDK_KEY_PRESS && event->type != GDK_KEY_RELEASE))
        return FALSE;
 
    KeyEvent key = keyevent_gdk_to_scim (*event);

    key.mask &= _valid_key_mask;

    String keystr;

    scim_key_to_string (keystr, key);

    panel_prepare_transaction (context_scim);

    if (match_key_event (_trigger_keys, key)) {
        if (!context_scim->impl->is_on) {
            context_scim->impl->is_on = TRUE;
            set_focus_ic (context_scim);
        } else { 
            context_scim->impl->is_on = FALSE;
            set_focus_ic (context_scim);
            if (context_scim->impl->use_preedit)
                g_signal_emit_by_name(context_scim, "preedit_changed");
        }
        ret = TRUE;
    } else {
        if (_focused_ic != context_scim && context_scim->impl->is_on)
            set_focus_ic (context_scim);

        if (_focused_ic && _focused_ic->impl->is_on) {
            if (match_key_event (_next_server_keys, key)) {
                open_next_server_factory (context_scim);
                set_focus_ic (context_scim);
                ret = TRUE;
            } else if (match_key_event (_previous_server_keys, key)) {
                open_previous_server_factory (context_scim);
                set_focus_ic (context_scim);
                ret = TRUE;
            } else if (!_focused_ic->impl->si->process_key_event (key)) {
                ret = commit_key_event (_focused_ic, *event);
            } else {
                ret = TRUE;
            }
        } else {
            ret = commit_key_event (_focused_ic, *event);
        }
    }

    panel_send_request (context_scim);

    return ret;
}

static void
gtk_im_context_scim_reset (GtkIMContext *context)
{
    SCIM_DEBUG_FRONTEND(1) << "gtk_im_context_scim_reset...\n";

    GtkIMContextSCIM *context_scim = GTK_IM_CONTEXT_SCIM (context);

    if (context_scim && context_scim->impl && !context_scim->impl->si.null () &&
        context_scim == _focused_ic) {
        panel_prepare_transaction (context_scim);
        context_scim->impl->si->reset ();
        panel_send_request (context_scim);
    }
}

static void
gtk_im_context_scim_focus_in (GtkIMContext *context)
{
    SCIM_DEBUG_FRONTEND(1) << "gtk_im_context_scim_focus_in...\n";

    GtkIMContextSCIM *context_scim = GTK_IM_CONTEXT_SCIM (context);

    if (context_scim && context_scim->impl && !context_scim->impl->si.null ()) {
        panel_prepare_transaction (context_scim);
        set_focus_ic (context_scim);
        panel_send_request (context_scim);
    }
}

static void
gtk_im_context_scim_focus_out (GtkIMContext *context)
{
    SCIM_DEBUG_FRONTEND(1) << "gtk_im_context_scim_focus_out...\n";
    GtkIMContextSCIM *context_scim = GTK_IM_CONTEXT_SCIM (context);

    if (context_scim && context_scim->impl && !context_scim->impl->si.null () &&
        context_scim == _focused_ic) {
        panel_prepare_transaction (context_scim);
        context_scim->impl->si->focus_out ();
        panel_req_turn_off_panel (context_scim);
        panel_req_focus_out (context_scim);
        panel_send_request (context_scim);
        _focused_ic = 0;
    }
}

static void
gtk_im_context_scim_set_cursor_location (GtkIMContext *context,
                                         GdkRectangle *area)
{
//    SCIM_DEBUG_FRONTEND(1) << "gtk_im_context_scim_set_cursor_location...\n";

    gint x, y;
    GtkIMContextSCIM *context_scim = GTK_IM_CONTEXT_SCIM (context);
 
    if (context_scim && context_scim->impl && context_scim->impl->client_window &&
        context_scim == _focused_ic) {
        gdk_window_get_origin(context_scim->impl->client_window, &x, &y);
        if (context_scim->impl->cursor_x != x + area->x + area->width ||
            context_scim->impl->cursor_y != y + area->y + area->height + 8) {
            context_scim->impl->cursor_x = x + area->x + area->width;
            context_scim->impl->cursor_y = y + area->y + area->height + 8;

            panel_prepare_transaction (context_scim);
            panel_req_update_spot_location (context_scim);
            panel_send_request (context_scim);
        } 
    }
}

static void
gtk_im_context_scim_set_use_preedit (GtkIMContext *context,
                                     gboolean      use_preedit)
{
    SCIM_DEBUG_FRONTEND(1) << "gtk_im_context_scim_set_use_preedit...\n";

    GtkIMContextSCIM *context_scim = GTK_IM_CONTEXT_SCIM (context);

    if (context_scim && context_scim->impl) {
        context_scim->impl->use_preedit = use_preedit;
        if (context_scim == _focused_ic) {
            panel_prepare_transaction (context_scim);
            slot_show_preedit_string (context_scim->impl->si);
            panel_send_request (context_scim);
        }
    }
}

static void
gtk_im_context_scim_get_preedit_string (GtkIMContext   *context,
                                        gchar         **str,
                                        PangoAttrList **attrs,
                                        gint           *cursor_pos)
{
    SCIM_DEBUG_FRONTEND(1) << "gtk_im_context_scim_get_preedit_string...\n";

    GtkIMContextSCIM *context_scim = GTK_IM_CONTEXT_SCIM (context);

    if (context_scim && context_scim->impl) {
        String mbs = utf8_wcstombs (context_scim->impl->preedit_string);

        if (str) {
            if (mbs.length () && context_scim->impl->is_on)
                *str = g_strdup (mbs.c_str ());
            else
                *str = g_strdup ("");
        }

        if (cursor_pos) {
            if (context_scim->impl->is_on)
                *cursor_pos = context_scim->impl->preedit_caret;
            else
                *cursor_pos = 0;
        }

        if (attrs) {
            *attrs = pango_attr_list_new ();

            if (mbs.length () && context_scim->impl->is_on) {
                guint start_index, end_index;
                guint wlen = context_scim->impl->preedit_string.length ();

                PangoAttribute *attr;
                AttributeList::const_iterator i;

                attr = pango_attr_underline_new (PANGO_UNDERLINE_SINGLE);
                attr->start_index = 0;
                attr->end_index = mbs.length ();
                pango_attr_list_insert (*attrs, attr);

                for (i = context_scim->impl->preedit_attrlist.begin ();
                     i != context_scim->impl->preedit_attrlist.end (); ++i) {
                    start_index = i->get_start ();
                    end_index = i->get_end ();

                    if (end_index <= wlen && start_index < end_index) {
                        start_index = g_utf8_offset_to_pointer (mbs.c_str (), i->get_start ()) - mbs.c_str ();
                        end_index = g_utf8_offset_to_pointer (mbs.c_str (), i->get_end ()) - mbs.c_str ();

                        switch (i->get_type ()) {
                            case scim::SCIM_ATTR_UNDERLINE:
                                attr = pango_attr_underline_new (PANGO_UNDERLINE_DOUBLE);
                                attr->start_index = start_index;
                                attr->end_index = end_index;
                                pango_attr_list_insert (*attrs, attr);
                                break;
                            case scim::SCIM_ATTR_REVERSE:
                                attr = pango_attr_foreground_new (65535,65535,65535);
                                attr->start_index = start_index;
                                attr->end_index = end_index;
                                pango_attr_list_insert (*attrs, attr);
 
                                attr = pango_attr_background_new (0,0,0);
                                attr->start_index = start_index;
                                attr->end_index = end_index;
                                pango_attr_list_insert (*attrs, attr);
                                break;
                            case scim::SCIM_ATTR_HIGHLIGHT:
                                attr = pango_attr_foreground_new (65535,65535,65535);
                                attr->start_index = start_index;
                                attr->end_index = end_index;
                                pango_attr_list_insert (*attrs, attr);
 
                                attr = pango_attr_background_new (0,0,0);
                                attr->start_index = start_index;
                                attr->end_index = end_index;
                                pango_attr_list_insert (*attrs, attr);
                                break;
                        }
                    }
                }
            }
        }
    }
}

static void
launch_panel ()
{
    String panel_program;
    String panel_program_path;

    if (!_config.null ())
        panel_program = _config->read (String (SCIM_CONFIG_GTKIMMODULE_PANEL_PROGRAM),
                                       String (SCIM_PANEL_PROGRAM));

    if (!panel_program.length ())
        panel_program = String (SCIM_PANEL_PROGRAM);

    panel_program_path = String (SCIM_BINDIR) +
                         String (SCIM_PATH_DELIM_STRING) +
                         panel_program;

    if (access (panel_program_path.c_str (), X_OK) != 0)
        return;

    int pid;

    pid = fork ();

    if (pid) {
        waitpid (pid, NULL, 0);
        return;
    } else {
#if HAVE_DAEMON
        daemon (0, 0);
#else
        pid = fork ();
        if (pid) exit (0);
        else {
            pid = fork ();
            if (pid) exit (0);
        }
#endif

        char * arg0 = const_cast<char *> (panel_program.c_str ());

        char * argv [] = {arg0, "-d", 0};

        execv (panel_program_path.c_str (), argv);
        exit (-1);
    }
}

static bool
panel_connect_server (GtkIMContextSCIM *ic)
{
    if (!ic || !ic->impl) return false;

    SocketAddress address (_panel_address);
    bool ret = true;

    if (!ic->impl->panel.connect (address)) {
        ret = false;
        launch_panel ();
        for (int i=0; i<500; ++i) {
            usleep (10000);
            if (ic->impl->panel.connect (address)) {
                ret = true;
                break;
            }
        }
    }

    if (ret) {
        SCIM_DEBUG_FRONTEND(2) << " Attach panel source...\n";
        GSCIMPanelSource *source =
            (GSCIMPanelSource*) g_source_new (&_panel_source_funcs,
                                               sizeof (GSCIMPanelSource));

        source->context = ic;
        source->fd.fd = ic->impl->panel.get_id ();
        source->fd.events = G_IO_IN;
        g_source_add_poll ((GSource *)source, &(source->fd));
        g_source_set_can_recurse ((GSource *)source, FALSE);

        if (ic->impl->panel_source_id)
            g_source_remove (ic->impl->panel_source_id);

        ic->impl->panel_source_id = g_source_attach ((GSource *)source, NULL);

        g_source_unref ((GSource *)source);
    }

    return ret;
}

static bool
panel_prepare_transaction (GtkIMContextSCIM *ic)
{
    ic->impl->send.clear ();
    ic->impl->send.put_command (SCIM_TRANS_CMD_REQUEST);
    ic->impl->send.put_data ((uint32) 0); // context id
    return true;
}

static bool
panel_send_request (GtkIMContextSCIM *ic)
{
    if (ic->impl->panel.is_connected () &&
        ic->impl->send.get_data_type () != SCIM_TRANS_DATA_UNKNOWN)
        return ic->impl->send.write_to_socket (ic->impl->panel, 0x4d494353);
    return false;
}

static bool
panel_receive_reply (GtkIMContextSCIM *ic)
{
    SocketTransaction receive;
    if (ic->impl->panel.is_connected () &&
        receive.read_from_socket (ic->impl->panel, _panel_timeout)) {
        int cmd;
        uint32 context;

        if (receive.get_data_type () == SCIM_TRANS_DATA_COMMAND &&
            receive.get_command (cmd) && cmd == SCIM_TRANS_CMD_REPLY &&
            receive.get_data_type () == SCIM_TRANS_DATA_UINT32 &&
            receive.get_data (context) && context == 0) {
            while (receive.get_data_type () == SCIM_TRANS_DATA_COMMAND &&
                   receive.get_command (cmd)) {
                switch (cmd) {
                    case SCIM_TRANS_CMD_UPDATE_LOOKUP_TABLE_PAGE_SIZE:
                        {
                            uint32 size;
                            if (receive.get_data_type () == SCIM_TRANS_DATA_UINT32 &&
                                receive.get_data (size)) {
                                ic->impl->si->update_lookup_table_page_size (size);
                            }
                        }
                        break;
                    case SCIM_TRANS_CMD_TOGGLE_FULL_WIDTH_PUNCTUATION:
                        {
                            if (ic->impl->is_on)
                                ic->impl->si->toggle_full_width_punctuation ();
                        }
                        break;
                    case SCIM_TRANS_CMD_TOGGLE_FULL_WIDTH_LETTER:
                        {
                            if (ic->impl->is_on)
                                ic->impl->si->toggle_full_width_letter ();
                        }
                        break;
                    case SCIM_TRANS_CMD_TOGGLE_INPUT_STATUS:
                        {
                            if (ic->impl->is_on)
                                ic->impl->si->toggle_input_status ();
                        }
                        break;
                    case SCIM_TRANS_CMD_MOVE_PREEDIT_CARET:
                        {
                            uint32 caret;
                            if (receive.get_data_type () == SCIM_TRANS_DATA_UINT32 &&
                                receive.get_data (caret)) {
                                ic->impl->si->move_preedit_caret (caret);
                            }
                        }
                        break;
                    case SCIM_TRANS_CMD_SELECT_LOOKUP_TABLE:
                        {
                            uint32 item;
                            if (receive.get_data_type () == SCIM_TRANS_DATA_UINT32 &&
                                receive.get_data (item)) {
                                ic->impl->si->select_lookup_table (item);
                            }
                        }
                        break;
                    case SCIM_TRANS_CMD_PROCESS_KEYEVENT:
                        {
                            KeyEvent key;
                            if (receive.get_data_type () == SCIM_TRANS_DATA_KEYEVENT &&
                                receive.get_data (key)) {
                                ic->impl->si->process_key_event (key);
                            }
                        }
                        break;
                    case SCIM_TRANS_CMD_COMMIT_STRING:
                        {
                            String str;
                            if (receive.get_data_type () == SCIM_TRANS_DATA_STRING &&
                                receive.get_data (str)) {
                                g_signal_emit_by_name (ic, "commit", str.c_str ());
                            }
                        }
                        break;
                    case SCIM_TRANS_CMD_FORWARD_KEYEVENT:
                        {
                            KeyEvent key;
                            if (receive.get_data_type () == SCIM_TRANS_DATA_KEYEVENT &&
                                receive.get_data (key)) {
                                /* TODO */
                            }
                        }
                        break;
                    case SCIM_TRANS_CMD_PANEL_REQUEST_HELP:
                        {
                            panel_req_show_help (ic);
                        }
                        break;
                    case SCIM_TRANS_CMD_PANEL_REQUEST_SERVER_MENU:
                        {
                            panel_req_show_server_menu (ic);
                        }
                        break;
                    case SCIM_TRANS_CMD_PANEL_CHANGE_SERVER_FACTORY:
                        {
                            String uuid;
                            if (receive.get_data_type () == SCIM_TRANS_DATA_STRING &&
                                receive.get_data (uuid)) {
                                if (!uuid.length () && ic->impl->is_on) {
                                    ic->impl->is_on = FALSE;
                                    set_focus_ic (ic);
                                    if (ic->impl->use_preedit)
                                        g_signal_emit_by_name(ic, "preedit_changed");
                                } else if (uuid.length ()) {
                                    ic->impl->is_on = TRUE;
                                    open_specific_server_factory (ic, uuid);
                                    set_focus_ic (ic);
                                }
                            }
                        }
                    default:
                        break;
                }
            }
        }
        return true;
    }
    return false;
}

static void
panel_req_turn_on_panel (GtkIMContextSCIM *ic)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_PANEL_TURN_ON);
}

static void
panel_req_turn_off_panel (GtkIMContextSCIM *ic)
{
    panel_req_update_full_width_punct (ic, false);
    panel_req_update_full_width_letter (ic, false);
    panel_req_update_status_string (ic, utf8_mbstowcs (_("En")), AttributeList ());
    ic->impl->send.put_command (SCIM_TRANS_CMD_PANEL_TURN_OFF);
}

static void
panel_req_update_display (GtkIMContextSCIM *ic)
{
#if GDK_MULTIHEAD_SAFE
    if (ic->impl->client_window) {
        GdkDisplay *display = gdk_drawable_get_display (GDK_DRAWABLE (ic->impl->client_window));
        if (display) {
            String name (gdk_display_get_name (display));
            if (name.length ()) {
                ic->impl->send.put_command (SCIM_TRANS_CMD_PANEL_UPDATE_DISPLAY);
                ic->impl->send.put_data (name);
            }
        }
    }
#endif
}

static void
panel_req_update_screen (GtkIMContextSCIM *ic)
{
#if GDK_MULTIHEAD_SAFE
    if (ic->impl->client_window) {
        GdkScreen *screen = gdk_drawable_get_screen (GDK_DRAWABLE (ic->impl->client_window));
        if (screen) {
            uint32 number = gdk_screen_get_number (screen);
            ic->impl->send.put_command (SCIM_TRANS_CMD_PANEL_UPDATE_SCREEN);
            ic->impl->send.put_data (number);
        }
    }
#endif
}

static void
panel_req_show_help (GtkIMContextSCIM *ic)
{
    String help = get_help_info (ic);
    ic->impl->send.put_command (SCIM_TRANS_CMD_PANEL_SHOW_HELP);
    ic->impl->send.put_data (help);
}

static void
panel_req_show_server_menu (GtkIMContextSCIM *ic)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_PANEL_SHOW_SERVER_MENU);

    for (size_t i = 0; i < _backend->number_of_servers (); ++ i) {
        ServerFactoryPointer server = _backend->get_server_factory (i);
        if (!server.null () && server->validate_encoding (ic->impl->charset)) {
            ic->impl->send.put_data (server->get_uuid ());
            ic->impl->send.put_data (utf8_wcstombs (server->get_name ()));
            ic->impl->send.put_data (server->get_icon_file ());
        }
    }
}

static void
panel_req_update_server_info (GtkIMContextSCIM *ic)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_PANEL_UPDATE_SERVER_INFO);
    if (ic->impl->is_on) {
        ic->impl->send.put_data (utf8_wcstombs (ic->impl->factory->get_name ()));
        ic->impl->send.put_data (ic->impl->factory->get_icon_file ());
    } else {
        ic->impl->send.put_data (String (_("English")));
        ic->impl->send.put_data (String (SCIM_KEYBOARD_ICON_FILE));
    }
}

static void
panel_req_focus_in (GtkIMContextSCIM *ic)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_FOCUS_IN);
}

static void
panel_req_focus_out (GtkIMContextSCIM *ic)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_FOCUS_OUT);
}

static void
panel_req_update_spot_location (GtkIMContextSCIM *ic)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_PANEL_UPDATE_SPOT_LOCATION);
    ic->impl->send.put_data ((uint32) ic->impl->cursor_x);
    ic->impl->send.put_data ((uint32) ic->impl->cursor_y);
}


static void
panel_req_show_preedit_string (GtkIMContextSCIM *ic)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_SHOW_PREEDIT_STRING);
}

static void
panel_req_hide_preedit_string (GtkIMContextSCIM *ic)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_HIDE_PREEDIT_STRING);
}

static void
panel_req_update_preedit_string (GtkIMContextSCIM    *ic,
                                  const WideString    &str,
                                  const AttributeList &attrs)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_UPDATE_PREEDIT_STRING);
    ic->impl->send.put_data (utf8_wcstombs (str));
    ic->impl->send.put_data (attrs);
}

static void
panel_req_update_preedit_caret (GtkIMContextSCIM *ic,
                                 int              caret)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_UPDATE_PREEDIT_CARET);
    ic->impl->send.put_data ((uint32) caret);
}

static void
panel_req_show_aux_string (GtkIMContextSCIM *ic)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_SHOW_AUX_STRING);
}

static void
panel_req_hide_aux_string (GtkIMContextSCIM *ic)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_HIDE_AUX_STRING);
}

static void
panel_req_update_aux_string (GtkIMContextSCIM    *ic,
                              const WideString    &str,
                              const AttributeList &attrs)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_UPDATE_AUX_STRING);
    ic->impl->send.put_data (utf8_wcstombs (str));
    ic->impl->send.put_data (attrs);
}

static void
panel_req_show_status_string (GtkIMContextSCIM *ic)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_SHOW_STATUS_STRING);
}

static void
panel_req_hide_status_string (GtkIMContextSCIM *ic)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_HIDE_STATUS_STRING);
}

static void
panel_req_update_status_string (GtkIMContextSCIM    *ic,
                                 const WideString    &str,
                                 const AttributeList &attrs)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_UPDATE_STATUS_STRING);
    ic->impl->send.put_data (utf8_wcstombs (str));
    ic->impl->send.put_data (attrs);
}

static void
panel_req_show_lookup_table (GtkIMContextSCIM *ic)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_SHOW_LOOKUP_TABLE);
}

static void
panel_req_hide_lookup_table (GtkIMContextSCIM *ic)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_HIDE_LOOKUP_TABLE);
}

static void
panel_req_update_lookup_table (GtkIMContextSCIM  *ic,
                                const LookupTable &table)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_UPDATE_LOOKUP_TABLE);
    ic->impl->send.put_data (table);
}

static void
panel_req_update_full_width_punct (GtkIMContextSCIM *ic,
                                    bool             full)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_UPDATE_FULL_WIDTH_PUNCTUATION);
    ic->impl->send.put_data ((uint32) full);
}

static void
panel_req_update_full_width_letter (GtkIMContextSCIM *ic,
                                     bool             full)
{
    ic->impl->send.put_command (SCIM_TRANS_CMD_UPDATE_FULL_WIDTH_LETTER);
    ic->impl->send.put_data ((uint32) full);
}

static gboolean
panel_source_prepare (GSource *source,
                       gint    *timeout)
{
    *timeout = -1;

    return FALSE;
}

static gboolean
panel_source_check (GSource *source)
{
    GSCIMPanelSource *panel_source = (GSCIMPanelSource *) source;

    if (panel_source && (panel_source->fd.revents & G_IO_IN))
        return TRUE;
    return FALSE;
}

static gboolean
panel_source_dispatch (GSource     *source,
                        GSourceFunc  callback,
                        gpointer     user_data)
{
    SCIM_DEBUG_FRONTEND(1) << "panel_source_dispatch..\n";

    GSCIMPanelSource *panel_source = (GSCIMPanelSource *) source;

    if (panel_source &&
        panel_source->context &&
        panel_source->context->impl &&
        panel_source->context->impl->panel.is_connected ()) {

        GtkIMContextSCIM *context = panel_source->context;

        if (!panel_check_connection (context))
            if (!panel_connect_server (context))
                return FALSE;

        panel_prepare_transaction (context);
        if (panel_receive_reply (context)) {
            SCIM_DEBUG_FRONTEND(2) << " Reply readed, doing...\n";
            panel_send_request (context);
        }
        return TRUE;
    }
    return FALSE;
}

static bool
panel_check_connection (GtkIMContextSCIM *ic)
{
    unsigned char buf [sizeof(uint32)];

    int nbytes = 0;
 
    if (ic && ic->impl && ic->impl->panel.is_connected ())
        nbytes = ic->impl->panel.read_with_timeout (buf, sizeof(uint32), _panel_timeout);

    if (nbytes == sizeof (uint32))
        return true;

    return false;
}

static void
set_focus_ic (GtkIMContextSCIM *ic)
{
    if (!ic || !ic->impl || ic->impl->si.null ())
        return;

    if (_focused_ic && _focused_ic->impl && !_focused_ic->impl->si.null () &&
        _focused_ic != ic)
        _focused_ic->impl->si->focus_out ();

    _focused_ic = ic;

    panel_req_focus_in (ic);

    panel_req_update_display (ic);
    panel_req_update_screen (ic);
    panel_req_update_spot_location (ic);
    panel_req_update_server_info (ic);

    if (ic->impl->is_on) {
        panel_req_turn_on_panel (ic);
        ic->impl->si->focus_in ();
    } else {
        panel_req_turn_off_panel (ic);
        ic->impl->si->focus_out ();
    }
}

static bool
commit_key_event (GtkIMContextSCIM  *ic,
                  const GdkEventKey &event)
{
    KeyEvent key = keyevent_gdk_to_scim (event);

    if (!key.is_control_down () && !key.is_alt_down () && key.is_key_press ()) {
        guint32 u;
        gchar utf8[16];
        u = gdk_keyval_to_unicode(event.keyval);

        if (u) {
            utf8[g_unichar_to_utf8(u, utf8)] = '\0';
            g_signal_emit_by_name(ic, "commit", utf8);
            return true;
        }
    }
    return false;
}

static bool
match_key_event (const KeyEventVector &keys,
                 const KeyEvent       &key)
{
    KeyEventVector::const_iterator kit; 

    for (kit = keys.begin (); kit != keys.end (); ++kit) {
        if (key.code == kit->code &&
            (key.mask & kit->mask) == kit->mask &&
			(key.mask & SCIM_KEY_ReleaseMask) == (kit->mask & SCIM_KEY_ReleaseMask))
            return true;
    }
    return false;
}

static KeyEvent
keyevent_gdk_to_scim (const GdkEventKey &gdkevent)
{
    return KeyEvent (gdkevent.keyval,
                     (gdkevent.type == GDK_KEY_PRESS)?
                     (gdkevent.state & ~ SCIM_KEY_ReleaseMask):
                     (gdkevent.state | SCIM_KEY_ReleaseMask));
}

static void
initialize (void)
{
    ServerFactoryPointer  server;
    CommonBackEnd        *p_backend;

    std::vector<String>   server_list;
    std::vector<String>   config_list;

    String                def_config;

    int                   i;

    // Get debug info
    {
        char *str;
        unsigned int verbose = 0;
        std::vector<String>  debug_mask_list;

        str = getenv ("GTK_IM_SCIM_DEBUG_VERBOSE");

        if (str != NULL)
            verbose = atoi (str);
        DebugOutput::set_verbose_level (verbose);

        str = getenv ("GTK_IM_SCIM_DEBUG_MASK");

        if (str != NULL) {
            scim_split_string_list (debug_mask_list, String (str), ',');
    		if (debug_mask_list.size ()) {
                DebugOutput::disable_debug (SCIM_DEBUG_AllMask);
		    	for (i=0; i<debug_mask_list.size (); i++)
			    	DebugOutput::enable_debug_by_name (debug_mask_list [i]);
    		}
        }
    }

    //get modules list
    scim_get_server_module_list (server_list);
    scim_get_config_module_list (config_list);

    //Use socket config module as default if available.
    if (config_list.size ()) {
        def_config = String ("socket");
        char * def_config_str = getenv ("GTK_IM_SCIM_CONFIG_MODULE");
        if (def_config_str != NULL &&
            std::find (config_list.begin (), config_list.end (), def_config_str)
            != config_list.end ()) {
            def_config = String (def_config_str);
        }
        if (std::find (config_list.begin (), config_list.end (), def_config)
            == config_list.end ())
            def_config = config_list [0];
    }

    //Use only socket server module if available.
    if (server_list.size ()) {
        char *server_list_str = getenv ("GTK_IM_SCIM_SERVER_MODULES");
        if (server_list_str != NULL) {
            scim_split_string_list (server_list, server_list_str, ',');
        } else if (std::find (server_list.begin (),
                              server_list.end (),
                              String ("socket")) != server_list.end ()) {
            server_list.clear ();
            server_list.push_back (String ("socket"));
        }
    }

    //load config module
    SCIM_DEBUG_FRONTEND(1) << "Loading Config module: " << def_config << "...\n";
    _config_module = new ConfigModule (def_config);

    //create config instance
    if (_config_module != NULL && _config_module->valid ())
        _config = _config_module->create_config ("scim");
    else {
        SCIM_DEBUG_FRONTEND(1) << "Config module cannot be loaded, using dummy Config.\n";

        if (_config_module) delete _config_module;
        _config_module = NULL;

        _config = new DummyConfig ();
    }

    // create backend
    p_backend = new CommonBackEnd ();

    if (!p_backend) {
        fprintf (stderr, "GTK IM Module SCIM: Cannot create BackEnd Object!\n");
        exit (-1);
    }

    _backend = p_backend;

    _server_modules = new ServerModule [server_list.size ()];

    //load Server modules
    for (i=0; i<server_list.size (); i++) {
        SCIM_DEBUG_FRONTEND (1) << "Loading Server module: " << server_list [i] << " ...\n";
        _server_modules [i].load (server_list[i], _config);
        if (_server_modules [i].valid ()) {
            for (unsigned int j=0; j < _server_modules [i].number_of_servers (); j++) {
                server = _server_modules [i].create_factory (j);
                if (!server.null ()) {
                    p_backend->add_server_factory (server);
                    SCIM_DEBUG_FRONTEND(2) << "  Loading Server Factory " << j << " ... : OK\n";
                } else {
                    SCIM_DEBUG_FRONTEND(2) << "  Loading Server Factory " << j << " ... : Faile\n";
                }
            }
        }
    }

    if (!p_backend->number_of_servers ()) {
        server = new DummyServerFactory ();
        p_backend->add_server_factory (server);
    }
 
    //Read settings.
    _panel_address = SCIM_PANEL_DEFAULT_SOCKET_ADDRESS;
    _panel_timeout = 500;

    _default_server_factory     = p_backend->get_server_factory (0);
    _default_server_factory_idx = 0;

    if (!_config.null ()) {
        scim_string_to_key_list (_trigger_keys,
                        _config->read (String (SCIM_CONFIG_FRONTEND_KEYS_TRIGGER),
                                       String ("Control+space")));

        scim_string_to_key_list (_next_server_keys, 
                        _config->read (String (SCIM_CONFIG_FRONTEND_KEYS_NEXT_SERVER),
                                       String ("Control+Alt+Down,Control+Shift_R,Control+Shift_L")));

        scim_string_to_key_list (_previous_server_keys, 
                        _config->read (String (SCIM_CONFIG_FRONTEND_KEYS_PREVIOUS_SERVER),
                                       String ("Control+Alt+Up,Shift+Control_R,Shift+Control_L")));

        KeyEvent key;
        scim_string_to_key (key,
                        _config->read (String (SCIM_CONFIG_FRONTEND_KEYS_VALID_KEY_MASK),
                                       String ("Shift+Control+Alt+Lock")));

        _valid_key_mask = (key.mask > 0)?(key.mask):0xFFFF;
        _valid_key_mask |= SCIM_KEY_ReleaseMask;

        _panel_address = _config->read (String (SCIM_CONFIG_PANEL_SOCKET_ADDRESS),
                                        _panel_address);
        _panel_timeout = _config->read (String (SCIM_CONFIG_PANEL_SOCKET_TIMEOUT),
                                        _panel_timeout);

        String locale     = scim_get_current_locale ();
        String def_sfuuid = _config->read (String (SCIM_CONFIG_DEFAULT_SERVER_FACTORY) + String ("/") + locale,
                                           String (""));

        for (int i=0; i<(int)_backend->number_of_servers (); ++i) {
            ServerFactoryPointer server = _backend->get_server_factory (i);
            if (!server.null () && server->get_uuid () == def_sfuuid) {
                _default_server_factory     = server;
                _default_server_factory_idx = i;
                break;
            }
        }
    }

    // Override the config vaule if env is set.
    const char *env = getenv ("SCIM_PANEL_SOCKET_ADDRESS");
    if (env) {
        _panel_address = String (env);
    }

    env = getenv ("SCIM_SOCKET_TIMEOUT");
    if (env) {
        _panel_timeout = atoi (env);
    } else {
        env = getenv ("SCIM_PANEL_SOCKET_TIMEOUT");
        if (env)
        _panel_timeout = atoi (env);
    }

    if (_panel_address == "default")
        _panel_address = SCIM_CONFIG_PANEL_SOCKET_ADDRESS;

    // No timeout is ridiculous, set it to unlimited.
    if (_panel_timeout == 0)
        _panel_timeout = -1;

    if (_trigger_keys.size () == 0) {
        _trigger_keys.push_back (
            KeyEvent (SCIM_KEY_space, SCIM_KEY_ControlMask));
    }

    if (_next_server_keys.size () == 0) {
        _next_server_keys.push_back (
            KeyEvent (SCIM_KEY_Shift_R, SCIM_KEY_ControlMask));
        _next_server_keys.push_back (
            KeyEvent (SCIM_KEY_Shift_L, SCIM_KEY_ControlMask));
        _next_server_keys.push_back (
            KeyEvent (SCIM_KEY_Down, SCIM_KEY_ControlMask + SCIM_KEY_AltMask));
    }

    if (_previous_server_keys.size () == 0) {
        _next_server_keys.push_back (
            KeyEvent (SCIM_KEY_Control_R, SCIM_KEY_ShiftMask));
        _next_server_keys.push_back (
            KeyEvent (SCIM_KEY_Control_L, SCIM_KEY_ShiftMask));
        _previous_server_keys.push_back (
            KeyEvent (SCIM_KEY_Up, SCIM_KEY_ControlMask + SCIM_KEY_AltMask));
    }

    if (_valid_key_mask == 0) {
        KeyEvent key;
        scim_string_to_key (key, String ("Shift+Control+Alt+Lock+KeyRelease"));
        _valid_key_mask = key.mask;
    }

    for (i=0; i<SCIM_MAX_IM_CONTEXT_NUMBER; i++)
        _im_contexts [i] = NULL;
}

static void
finalize (void)
{
    int i;

    SCIM_DEBUG_FRONTEND(1) << "Finalizing GTK2 SCIM IMModule...\n";

    if (!_config.null () && !_default_server_factory.null ()) {
        String locale = scim_get_current_locale ();
        _config->write (String (SCIM_CONFIG_DEFAULT_SERVER_FACTORY) + String ("/") + locale,
                        _default_server_factory->get_uuid ());
    }

    SCIM_DEBUG_FRONTEND(2) << " Deleting all IM Contexts...\n";
    for (i=0; i<SCIM_MAX_IM_CONTEXT_NUMBER; i++) {
        if (_im_contexts [i] != NULL)
            g_object_unref (_im_contexts [i]);
    }

    SCIM_DEBUG_FRONTEND(2) << " Releasing default Server Factory...\n";
    _default_server_factory.reset ();

    SCIM_DEBUG_FRONTEND(2) << " Releasing BackEnd...\n";
    _backend.reset ();

    SCIM_DEBUG_FRONTEND(2) << " Releasing Config...\n";
    _config.reset ();

    SCIM_DEBUG_FRONTEND(2) << " Deleting _server_modules []...\n";
    delete [] _server_modules;

    if (_config_module) {
        SCIM_DEBUG_FRONTEND(2) << " Deleting _config_module...\n";
        delete _config_module;
    }
}

static void
attach_server_instance (GtkIMContextSCIM *context_scim)
{
    if (context_scim == NULL || context_scim->impl == NULL || context_scim->impl->si.null ())
        return;

    context_scim->impl->si->signal_connect_show_preedit_string (
        slot (slot_show_preedit_string));
    context_scim->impl->si->signal_connect_show_status_string (
        slot (slot_show_status_string));
    context_scim->impl->si->signal_connect_show_aux_string (
        slot (slot_show_aux_string));
    context_scim->impl->si->signal_connect_show_lookup_table (
        slot (slot_show_lookup_table));

    context_scim->impl->si->signal_connect_hide_preedit_string (
        slot (slot_hide_preedit_string));
    context_scim->impl->si->signal_connect_hide_status_string (
        slot (slot_hide_status_string));
    context_scim->impl->si->signal_connect_hide_aux_string (
        slot (slot_hide_aux_string));
    context_scim->impl->si->signal_connect_hide_lookup_table (
        slot (slot_hide_lookup_table));

    context_scim->impl->si->signal_connect_update_preedit_caret (
        slot (slot_update_preedit_caret));
    context_scim->impl->si->signal_connect_update_preedit_string (
        slot (slot_update_preedit_string));
    context_scim->impl->si->signal_connect_update_status_string (
        slot (slot_update_status_string));
    context_scim->impl->si->signal_connect_update_aux_string (
        slot (slot_update_aux_string));
    context_scim->impl->si->signal_connect_update_lookup_table (
        slot (slot_update_lookup_table));

    context_scim->impl->si->signal_connect_commit_string (
        slot (slot_commit_string));

    context_scim->impl->si->signal_connect_forward_keyevent (
        slot (slot_forward_keyevent));

    context_scim->impl->si->signal_connect_update_full_width_punctuation (
        slot (slot_update_full_width_punct));

    context_scim->impl->si->signal_connect_update_full_width_letter (
        slot (slot_update_full_width_letter));

}

static bool
open_default_server_factory (GtkIMContextSCIM *ic,
                             int               icid)
{
    if (!_default_server_factory.null () &&
         _default_server_factory->validate_encoding (ic->impl->charset)) {
        ic->impl->si =
            _default_server_factory->create_server_instance (ic->impl->charset, icid);
        if (!ic->impl->si.null ()) {
            ic->impl->factory = _default_server_factory;
            attach_server_instance (ic);
            return true;
        }
    }

    for (int i=0; i<(int)_backend->number_of_servers (); ++i) {
        ServerFactoryPointer server = _backend->get_server_factory (i);
        if (!server.null () && server->validate_encoding (ic->impl->charset)) {
            ic->impl->si = server->create_server_instance (ic->impl->charset, icid);
            if (!ic->impl->si.null ()) {
                _default_server_factory     = server;
                _default_server_factory_idx = i;
                ic->impl->factory = server;
                attach_server_instance (ic);
                return true;
            }
        }
    }

    return false;
}

static void
open_next_server_factory (GtkIMContextSCIM *ic)
{
    for (int i=_default_server_factory_idx+1;
         i<(int)_backend->number_of_servers (); ++i) {
        ServerFactoryPointer server = _backend->get_server_factory (i);
        if (!server.null () && server->validate_encoding (ic->impl->charset)) {
            ServerInstancePointer si =
                server->create_server_instance (ic->impl->charset, ic->impl->si->get_id ());
            if (!si.null ()) {
                _default_server_factory     = server;
                _default_server_factory_idx = i;
                ic->impl->factory = server;
                ic->impl->si = si;
                attach_server_instance (ic);
                return;
            }
        }
    }

    for (int i=0;i<=_default_server_factory_idx; ++i) {
        ServerFactoryPointer server = _backend->get_server_factory (i);
        if (!server.null () && server->validate_encoding (ic->impl->charset)) {
            ServerInstancePointer si = 
                server->create_server_instance (ic->impl->charset, ic->impl->si->get_id ());
            if (!si.null ()) {
                _default_server_factory     = server;
                _default_server_factory_idx = i;
                ic->impl->factory = server;
                ic->impl->si = si;
                attach_server_instance (ic);
                return;
            }
        }
    }
}

static void
open_previous_server_factory (GtkIMContextSCIM *ic)
{
    for (int i=_default_server_factory_idx - 1; i >= 0; --i) {
        ServerFactoryPointer server = _backend->get_server_factory (i);
        if (!server.null () && server->validate_encoding (ic->impl->charset)) {
            ServerInstancePointer si = 
                server->create_server_instance (ic->impl->charset, ic->impl->si->get_id ());
            if (!si.null ()) {
                _default_server_factory     = server;
                _default_server_factory_idx = i;
                ic->impl->factory = server;
                ic->impl->si = si;
                attach_server_instance (ic);
                return;
            }
        }
    }

    for (int i=(int)_backend->number_of_servers ();
         i>=_default_server_factory_idx; --i) {
        ServerFactoryPointer server = _backend->get_server_factory (i);
        if (!server.null () && server->validate_encoding (ic->impl->charset)) {
            ServerInstancePointer si = 
                server->create_server_instance (ic->impl->charset, ic->impl->si->get_id ());
            if (!si.null ()) {
                _default_server_factory     = server;
                _default_server_factory_idx = i;
                ic->impl->factory = server;
                ic->impl->si = si;
                attach_server_instance (ic);
                return;
            }
        }
    }
}

static void
open_specific_server_factory (GtkIMContextSCIM *ic,
                              const String     &uuid)
{
    if (ic->impl->factory->get_uuid () == uuid)
        return;

    for (int i=0;i<=(int)_backend->number_of_servers (); ++i) {
        ServerFactoryPointer server = _backend->get_server_factory (i);

        if (!server.null () && server->get_uuid () == uuid &&
            server->validate_encoding (ic->impl->charset)) {
            ServerInstancePointer si = 
                server->create_server_instance (ic->impl->charset, ic->impl->si->get_id ());
            if (!si.null ()) {
                _default_server_factory     = server;
                _default_server_factory_idx = i;
                ic->impl->factory = server;
                ic->impl->si = si;
                attach_server_instance (ic);
                return;
            }
        }
    }
}

static String
get_help_info (GtkIMContextSCIM *ic)
{
    String help;
    String tmp;

    help =  String (_("Smart Common Input Method platform ")) +
            String (SCIM_VERSION) +
            String (_("\n(C) 2002-2004 James Su <suzhe@tsinghua.org.cn>\n\n"
                          "Hot keys:\n\n  "));

    scim_key_list_to_string (tmp, _trigger_keys);
    help += tmp + String (_(":\n    open/close the input server.\n\n  "));
    
    scim_key_list_to_string (tmp, _next_server_keys);
    help += tmp + String (_(":\n    switch to the next input method.\n\n  "));

    scim_key_list_to_string (tmp, _previous_server_keys);
    help += tmp + String (_(":\n    switch to the previous input method.\n\n\n"));

    if (ic && ic->impl && !ic->impl->factory.null ()) {
        help += utf8_wcstombs (ic->impl->factory->get_name ());
        help += String (_(":\n\n"));

        help += utf8_wcstombs (ic->impl->factory->get_authors ());
        help += String (_("\n\n"));

        help += utf8_wcstombs (ic->impl->factory->get_help ());
        help += String (_("\n\n"));

        help += utf8_wcstombs (ic->impl->factory->get_credits ());
    }
    return help;
}

// Implementation of slot functions
static void
slot_show_preedit_string (ServerInstanceBase * si)
{
    SCIM_DEBUG_FRONTEND(1) << "slot_show_preedit_string...\n";

    if (si && si->get_id () >= 0 && si->get_id () < SCIM_MAX_IM_CONTEXT_NUMBER) {
        GtkIMContextSCIM *context_scim = _im_contexts [si->get_id ()];
        if (context_scim == _focused_ic && context_scim->impl) {
            if (context_scim->impl->use_preedit) {
                g_signal_emit_by_name(context_scim, "preedit_changed");
            } else {
                panel_req_show_preedit_string (context_scim);
            }
        }
    }
}

static void
slot_show_status_string (ServerInstanceBase * si)
{
    SCIM_DEBUG_FRONTEND(1) << "slot_show_status_string...\n";

    if (si && si->get_id () >= 0 && si->get_id () < SCIM_MAX_IM_CONTEXT_NUMBER) {
        GtkIMContextSCIM *context_scim = _im_contexts [si->get_id ()];
        if (context_scim == _focused_ic && context_scim->impl) {
            panel_req_show_status_string (context_scim);
        }
    }
}

static void 
slot_show_aux_string (ServerInstanceBase * si)
{
    SCIM_DEBUG_FRONTEND(1) << "slot_show_aux_string...\n";
    if (si && si->get_id () >= 0 && si->get_id () < SCIM_MAX_IM_CONTEXT_NUMBER) {
        GtkIMContextSCIM *context_scim = _im_contexts [si->get_id ()];
        if (context_scim == _focused_ic && context_scim->impl) {
            panel_req_show_aux_string (context_scim);
        }
    }
}

static void 
slot_show_lookup_table (ServerInstanceBase * si)
{
    SCIM_DEBUG_FRONTEND(1) << "slot_show_lookup_table...\n";
    if (si && si->get_id () >= 0 && si->get_id () < SCIM_MAX_IM_CONTEXT_NUMBER) {
        GtkIMContextSCIM *context_scim = _im_contexts [si->get_id ()];
        if (context_scim == _focused_ic && context_scim->impl) {
            panel_req_show_lookup_table (context_scim);
        }
    }
}

static void 
slot_hide_preedit_string (ServerInstanceBase * si)
{
    SCIM_DEBUG_FRONTEND(1) << "slot_hide_preedit_string...\n";
    if (si && si->get_id () >= 0 && si->get_id () < SCIM_MAX_IM_CONTEXT_NUMBER) {
        GtkIMContextSCIM *context_scim = _im_contexts [si->get_id ()];
        if (context_scim == _focused_ic && context_scim->impl) {
            context_scim->impl->preedit_string = WideString ();
            context_scim->impl->preedit_caret = 0;
            context_scim->impl->preedit_attrlist.clear ();
            if (context_scim->impl->use_preedit) {
                g_signal_emit_by_name(context_scim, "preedit_changed");
            } else {
                panel_req_hide_preedit_string (context_scim);
            }
        }
    }
}

static void 
slot_hide_status_string (ServerInstanceBase * si)
{
    SCIM_DEBUG_FRONTEND(1) << "slot_hide_status_string...\n";
    if (si && si->get_id () >= 0 && si->get_id () < SCIM_MAX_IM_CONTEXT_NUMBER) {
        GtkIMContextSCIM *context_scim = _im_contexts [si->get_id ()];
        if (context_scim == _focused_ic && context_scim->impl) {
            panel_req_hide_status_string (context_scim);
        }
    }
}

static void 
slot_hide_aux_string (ServerInstanceBase * si)
{
    SCIM_DEBUG_FRONTEND(1) << "slot_hide_aux_string...\n";
    if (si && si->get_id () >= 0 && si->get_id () < SCIM_MAX_IM_CONTEXT_NUMBER) {
        GtkIMContextSCIM *context_scim = _im_contexts [si->get_id ()];
        if (context_scim == _focused_ic && context_scim->impl) {
            panel_req_hide_aux_string (context_scim);
        }
    }
}

static void 
slot_hide_lookup_table (ServerInstanceBase * si)
{
    SCIM_DEBUG_FRONTEND(1) << "slot_hide_lookup_table...\n";
    if (si && si->get_id () >= 0 && si->get_id () < SCIM_MAX_IM_CONTEXT_NUMBER) {
        GtkIMContextSCIM *context_scim = _im_contexts [si->get_id ()];
        if (context_scim == _focused_ic && context_scim->impl) {
            panel_req_hide_lookup_table (context_scim);
        }
    }
}

static void 
slot_update_preedit_caret (ServerInstanceBase * si, int caret)
{
    SCIM_DEBUG_FRONTEND(1) << "slot_update_preedit_caret...\n";
    if (si && si->get_id () >= 0 && si->get_id () < SCIM_MAX_IM_CONTEXT_NUMBER) {
        GtkIMContextSCIM *context_scim = _im_contexts [si->get_id ()];
        if (context_scim == _focused_ic && context_scim->impl) {
            context_scim->impl->preedit_caret = caret;
            if (context_scim->impl->use_preedit) {
                g_signal_emit_by_name(context_scim, "preedit_changed");
            } else {
                panel_req_update_preedit_caret (context_scim, caret);
            }
        }
    }
}

static void 
slot_update_preedit_string (ServerInstanceBase * si,
                            const WideString & str,
                            const AttributeList & attrs)
{
    SCIM_DEBUG_FRONTEND(1) << "slot_update_preedit_string...\n";
    if (si && si->get_id () >= 0 && si->get_id () < SCIM_MAX_IM_CONTEXT_NUMBER) {
        GtkIMContextSCIM *context_scim = _im_contexts [si->get_id ()];
        if (context_scim == _focused_ic && context_scim->impl) {
            context_scim->impl->preedit_string   = str;
            context_scim->impl->preedit_attrlist = attrs;
            if (context_scim->impl->use_preedit) {
                g_signal_emit_by_name(context_scim, "preedit_changed");
            } else {
                panel_req_update_preedit_string (context_scim, str, attrs);
            }
        }
    }
}

static void 
slot_update_status_string (ServerInstanceBase * si,
                           const WideString & str,
                           const AttributeList & attrs)
{
    SCIM_DEBUG_FRONTEND(1) << "slot_update_status_string...\n";
    if (si && si->get_id () >= 0 && si->get_id () < SCIM_MAX_IM_CONTEXT_NUMBER) {
        GtkIMContextSCIM *context_scim = _im_contexts [si->get_id ()];
        if (context_scim == _focused_ic && context_scim->impl) {
            panel_req_update_status_string (context_scim, str, attrs);
        }
    }
}

static void 
slot_update_aux_string (ServerInstanceBase * si,
                        const WideString & str,
                        const AttributeList & attrs)
{
    SCIM_DEBUG_FRONTEND(1) << "slot_update_aux_string...\n";
    if (si && si->get_id () >= 0 && si->get_id () < SCIM_MAX_IM_CONTEXT_NUMBER) {
        GtkIMContextSCIM *context_scim = _im_contexts [si->get_id ()];
        if (context_scim == _focused_ic && context_scim->impl) {
            panel_req_update_aux_string (context_scim, str, attrs);
        }
    }
}

static void 
slot_commit_string (ServerInstanceBase * si,
                    const WideString & str)
{
    SCIM_DEBUG_FRONTEND(1) << "slot_commit_string...\n";
    if (si && si->get_id () >= 0 && si->get_id () < SCIM_MAX_IM_CONTEXT_NUMBER) {
        GtkIMContextSCIM *context_scim = _im_contexts [si->get_id ()];
        if (context_scim == _focused_ic && context_scim->impl) {
            g_signal_emit_by_name (context_scim, "commit", utf8_wcstombs (str).c_str ());
        }
    }
}

static void 
slot_forward_keyevent (ServerInstanceBase * si,
                       const KeyEvent & key)
{
}

static void 
slot_update_lookup_table (ServerInstanceBase * si,
                          const LookupTable & table)
{
    SCIM_DEBUG_FRONTEND(1) << "slot_update_lookup_table...\n";
    if (si && si->get_id () >= 0 && si->get_id () < SCIM_MAX_IM_CONTEXT_NUMBER) {
        GtkIMContextSCIM *context_scim = _im_contexts [si->get_id ()];
        if (context_scim == _focused_ic && context_scim->impl) {
            panel_req_update_lookup_table (context_scim, table);
        }
    }
}

static void 
slot_update_full_width_punct (ServerInstanceBase * si, bool full)
{
    SCIM_DEBUG_FRONTEND(1) << "slot_update_full_width_punct...\n";
    if (si && si->get_id () >= 0 && si->get_id () < SCIM_MAX_IM_CONTEXT_NUMBER) {
        GtkIMContextSCIM *context_scim = _im_contexts [si->get_id ()];
        if (context_scim == _focused_ic && context_scim->impl) {
            panel_req_update_full_width_punct (context_scim, full);
        }
    }
}

static void 
slot_update_full_width_letter (ServerInstanceBase * si, bool full)
{
    SCIM_DEBUG_FRONTEND(1) << "slot_update_full_width_letter...\n";
    if (si && si->get_id () >= 0 && si->get_id () < SCIM_MAX_IM_CONTEXT_NUMBER) {
        GtkIMContextSCIM *context_scim = _im_contexts [si->get_id ()];
        if (context_scim == _focused_ic && context_scim->impl) {
            panel_req_update_full_width_letter (context_scim, full);
        }
    }
}

/*
vi:ts=4:expandtab:nowrap
*/
