/** @file scim_socket_config.cpp
 * implementation of SocketConfig class.
 */

/*
 * Smart Common Input Method
 * 
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_socket_config.cpp,v 1.6 2004/04/12 21:46:42 liuspider Exp $
 */

#define Uses_SCIM_CONFIG_BASE
#define Uses_SCIM_SOCKET
#define Uses_SCIM_SOCKET_TRANSACTION
#define Uses_C_STDIO
#define Uses_C_STDLIB

#include "scim_private.h"
#include "scim.h"
#include "scim_socket_config.h"

#define scim_module_init socket_LTX_scim_module_init
#define scim_module_exit socket_LTX_scim_module_exit
#define scim_config_module_init socket_LTX_scim_config_module_init
#define scim_config_module_create_config socket_LTX_scim_config_module_create_config

extern "C" {
    void scim_module_init (void)
    {
        SCIM_DEBUG_CONFIG(1) << "Initializing Socket Config module...\n";
    }

    void scim_module_exit (void)
    {
        SCIM_DEBUG_CONFIG(1) << "Exiting Socket Config module...\n";
    }

    void scim_config_module_init ()
    {
        SCIM_DEBUG_CONFIG(1) << "Initializing Socket Config module (more)...\n";
    }

    ConfigPointer scim_config_module_create_config (const String &app_name)
    {
        SCIM_DEBUG_CONFIG(1) << "Creating a Socket Config instance...\n";
        return new SocketConfig (app_name);
    }
}

SocketConfig::SocketConfig (const String& app_name)
    : ConfigBase (app_name),
      m_socket_timeout (500),
      m_connected (false)
{
    SCIM_DEBUG_SERVER(2) << " Construct SocketConfig object.\n";

    String address;

    const char *env = getenv ("SCIM_SOCKET_ADDRESS");
    if (env) {
        address = String (env);
    } else {
        env = getenv ("SCIM_CONFIG_SOCKET_ADDRESS");
        if (env)
            address = String (env);
        else
            address = String ("local:/tmp/scim-socket-frontend");
    }

    env = getenv ("SCIM_SOCKET_TIMEOUT");
    if (env) {
        m_socket_timeout = atoi (env);
    } else {
        env = getenv ("SCIM_CONFIG_SOCKET_TIMEOUT");
        if (env)
            m_socket_timeout = atoi (env);
    }

    SocketAddress socket_address (address);

    // Connect to SocketFrontEnd.
    if (!m_socket_client.connect (socket_address)) {
        SCIM_DEBUG_SERVER(2) << " Cannot connect to SocketFrontEnd (" << address << ").\n";
        return;
    }

    // Init the connection, and get ServerFactory list.
    SocketTransaction trans;

    trans.put_command (SCIM_TRANS_CMD_REQUEST);
    trans.put_command (SCIM_TRANS_CMD_OPEN_CONNECTION);
    trans.write_to_socket (m_socket_client);

    int cmd;
    if (trans.read_from_socket (m_socket_client, m_socket_timeout) &&
        trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_REPLY &&
        trans.get_data (m_socket_magic_key)) {
        trans.clear ();
        trans.put_command (SCIM_TRANS_CMD_REPLY);
        trans.put_command (SCIM_TRANS_CMD_OK);
        trans.write_to_socket (m_socket_client);

        SCIM_DEBUG_SERVER(2) << " Connected to SocketFrontEnd (" << address
                << ") MagicKey (" << m_socket_magic_key << ").\n";

        m_connected = true;
    }
}

SocketConfig::~SocketConfig ()
{
}

bool
SocketConfig::valid () const
{
    return ConfigBase::valid() && m_connected;
}

// String
bool
SocketConfig::read (const String& key, String *pStr) const
{
    if (!valid () || !pStr || key.empty()) return false;

    SocketTransaction trans;
    int cmd;

    init_transaction (trans);
    trans.put_command (SCIM_TRANS_CMD_GET_CONFIG_STRING);
    trans.put_data (key);

    if (trans.write_to_socket (m_socket_client) &&
        trans.read_from_socket (m_socket_client, m_socket_timeout)) {
        if (trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_REPLY &&
            trans.get_data (*pStr) &&
            trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_OK) 
            return true;
    }

    *pStr = String ("");
    return false;
}

// int
bool
SocketConfig::read (const String& key, int *pl) const
{
    if (!valid () || !pl || key.empty()) return false;
    
    SocketTransaction trans;
    int cmd;

    init_transaction (trans);
    trans.put_command (SCIM_TRANS_CMD_GET_CONFIG_INT);
    trans.put_data (key);

    if (trans.write_to_socket (m_socket_client) &&
        trans.read_from_socket (m_socket_client, m_socket_timeout)) {
        uint32 val;
        if (trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_REPLY &&
            trans.get_data (val) &&
            trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_OK) {
            *pl = val;
            return true;
        }
    }

    *pl = 0;
    return false;
}

// double
bool
SocketConfig::read (const String& key, double* val) const
{
    if (!valid () || !val || key.empty()) return false;

    SocketTransaction trans;
    int cmd;

    init_transaction (trans);
    trans.put_command (SCIM_TRANS_CMD_GET_CONFIG_DOUBLE);
    trans.put_data (key);

    if (trans.write_to_socket (m_socket_client) &&
        trans.read_from_socket (m_socket_client, m_socket_timeout)) {
        String str;
        if (trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_REPLY &&
            trans.get_data (str) &&
            trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_OK) {
            sscanf (str.c_str (), "%lE", val);
            return true;
        }
    }

    *val = 0;
    return false;
}

// bool
bool
SocketConfig::read (const String& key, bool* val) const
{
    if (!valid () || !val || key.empty()) return false;
    
    SocketTransaction trans;
    int cmd;

    init_transaction (trans);
    trans.put_command (SCIM_TRANS_CMD_GET_CONFIG_BOOL);
    trans.put_data (key);

    if (trans.write_to_socket (m_socket_client) &&
        trans.read_from_socket (m_socket_client, m_socket_timeout)) {
        uint32 tmp;
        if (trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_REPLY &&
            trans.get_data (tmp) &&
            trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_OK) {
            *val = (bool)tmp;
            return true;
        }
    }

    *val = false;
    return false;
}

//String list
bool
SocketConfig::read (const String& key, std::vector <String>* val) const
{
    if (!valid () || !val || key.empty()) return false;
    
    val->clear ();

    SocketTransaction trans;
    int cmd;

    init_transaction (trans);
    trans.put_command (SCIM_TRANS_CMD_GET_CONFIG_VECTOR_STRING);
    trans.put_data (key);

    if (trans.write_to_socket (m_socket_client) &&
        trans.read_from_socket (m_socket_client, m_socket_timeout)) {
        if (trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_REPLY &&
            trans.get_data (*val) &&
            trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_OK) {
            return true;
        }
    }

    return false;
}

//int list
bool
SocketConfig::read (const String& key, std::vector <int>* val) const
{
    if (!valid () || !val || key.empty()) return false;
    
    val->clear();

    SocketTransaction trans;
    int cmd;

    init_transaction (trans);
    trans.put_command (SCIM_TRANS_CMD_GET_CONFIG_VECTOR_INT);
    trans.put_data (key);

    if (trans.write_to_socket (m_socket_client) &&
        trans.read_from_socket (m_socket_client, m_socket_timeout)) {
        std::vector<uint32> vec;
        if (trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_REPLY &&
            trans.get_data (vec) &&
            trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_OK) {
            for (uint32 i=0; i<vec.size (); ++i)
                val->push_back ((int) vec[i]);
            return true;
        }
    }

    return false;
}

// write the value (return true on success)
bool
SocketConfig::write (const String& key, const String& value)
{
    if (!valid () || key.empty()) return false;

    SocketTransaction trans;
    int cmd;

    init_transaction (trans);
    trans.put_command (SCIM_TRANS_CMD_SET_CONFIG_STRING);
    trans.put_data (key);
    trans.put_data (value);

    if (trans.write_to_socket (m_socket_client) &&
        trans.read_from_socket (m_socket_client, m_socket_timeout) &&
        trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_OK)
        return true;
    return false;
}

bool
SocketConfig::write (const String& key, int value)
{
    if (!valid () || key.empty()) return false;

    SocketTransaction trans;
    int cmd;

    init_transaction (trans);
    trans.put_command (SCIM_TRANS_CMD_SET_CONFIG_INT);
    trans.put_data (key);
    trans.put_data ((uint32)value);

    if (trans.write_to_socket (m_socket_client) &&
        trans.read_from_socket (m_socket_client, m_socket_timeout) &&
        trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_OK)
        return true;
    return false;
}

bool
SocketConfig::write (const String& key, double value)
{
    if (!valid () || key.empty()) return false;
    
    char buf [256];
    snprintf (buf, 255, "%lE", value);

    SocketTransaction trans;
    int cmd;

    init_transaction (trans);
    trans.put_command (SCIM_TRANS_CMD_SET_CONFIG_DOUBLE);
    trans.put_data (key);
    trans.put_data (String (buf));

    if (trans.write_to_socket (m_socket_client) &&
        trans.read_from_socket (m_socket_client, m_socket_timeout) &&
        trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_OK)
        return true;
    return false;
}

bool
SocketConfig::write (const String& key, bool value)
{
    if (!valid () || key.empty()) return false;

    SocketTransaction trans;
    int cmd;

    init_transaction (trans);
    trans.put_command (SCIM_TRANS_CMD_SET_CONFIG_BOOL);
    trans.put_data (key);
    trans.put_data ((uint32)value);

    if (trans.write_to_socket (m_socket_client) &&
        trans.read_from_socket (m_socket_client, m_socket_timeout) &&
        trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_OK)
        return true;
    return false;
}

bool
SocketConfig::write (const String& key, const std::vector <String>& value)
{
    if (!valid () || key.empty()) return false;

    SocketTransaction trans;
    int cmd;

    init_transaction (trans);
    trans.put_command (SCIM_TRANS_CMD_SET_CONFIG_VECTOR_STRING);
    trans.put_data (key);
    trans.put_data (value);

    if (trans.write_to_socket (m_socket_client) &&
        trans.read_from_socket (m_socket_client, m_socket_timeout) &&
        trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_OK)
        return true;
    return false;
}

bool
SocketConfig::write (const String& key, const std::vector <int>& value)
{
    if (!valid () || key.empty()) return false;

    std::vector <uint32> vec;

    for (uint32 i=0; i<value.size (); ++i)
        vec.push_back ((uint32) value [i]);

    SocketTransaction trans;
    int cmd;

    init_transaction (trans);
    trans.put_command (SCIM_TRANS_CMD_SET_CONFIG_VECTOR_INT);
    trans.put_data (key);
    trans.put_data (vec);

    if (trans.write_to_socket (m_socket_client) &&
        trans.read_from_socket (m_socket_client, m_socket_timeout) &&
        trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_OK)
        return true;
    return false;
}


// permanently writes all changes
bool
SocketConfig::flush()
{
    if (!valid ()) return false;

    SocketTransaction trans;
    int cmd;

    init_transaction (trans);
    trans.put_command (SCIM_TRANS_CMD_FLUSH_CONFIG);

    if (trans.write_to_socket (m_socket_client) &&
        trans.read_from_socket (m_socket_client, m_socket_timeout) &&
        trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_OK)
        return true;
    return false;
}

// delete entries
bool
SocketConfig::erase (const String& key)
{
    if (!valid ()) return false;

    SocketTransaction trans;
    int cmd;

    init_transaction (trans);
    trans.put_command (SCIM_TRANS_CMD_ERASE_CONFIG);
    trans.put_data (key);

    if (trans.write_to_socket (m_socket_client) &&
        trans.read_from_socket (m_socket_client, m_socket_timeout) &&
        trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_OK)
        return true;
    return false;
}

void
SocketConfig::init_transaction (SocketTransaction &trans) const
{
    trans.clear ();
    trans.put_command (SCIM_TRANS_CMD_REQUEST);
    trans.put_data (m_socket_magic_key);
}

/*
vi:ts=4:nowrap:ai:expandtab
*/
