/**
 * @file scim_socket_frontend.h
 * @brief definition of SocketFrontEnd related classes.
 */

/* 
 * Smart Common Input Method
 * 
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_socket_frontend.h,v 1.8 2003/12/31 05:09:24 suzhe Exp $
 */

#if !defined (__SCIM_SOCKET_FRONTEND_H)
#define __SCIM_SOCKET_FRONTEND_H

using namespace scim;

class SocketFrontEnd : public FrontEndBase
{
    /**
     * ::first = socket id, ::second = server instance id.
     */
    typedef std::vector <std::pair <int, int> > SocketServerInstanceRepository;

    /**
     * ::first = socket id, ::second = auth key.
     */
    typedef std::vector <std::pair <int, int> > SocketClientRepository;

    ConfigPointer     m_config;

    SocketServer      m_socket_server;

    SocketTransaction m_send_transaction;
    SocketTransaction m_receive_transaction;

    SocketServerInstanceRepository m_socket_si_repository;

    SocketClientRepository         m_socket_client_repository;

    bool   m_config_readonly;

    int    m_socket_timeout;

    int    m_current_server_instance;

    String m_socket_address;

public:
    SocketFrontEnd (const BackEndPointer &backend,
                    const ConfigPointer  &config);

    virtual ~SocketFrontEnd ();

protected:
    virtual void show_preedit_string (int id);
    virtual void show_status_string  (int id);
    virtual void show_aux_string     (int id);
    virtual void show_lookup_table   (int id);

    virtual void hide_preedit_string (int id);
    virtual void hide_status_string  (int id);
    virtual void hide_aux_string     (int id);
    virtual void hide_lookup_table   (int id);

    virtual void update_preedit_caret  (int id, int caret);
    virtual void update_preedit_string (int id, const WideString & str, const AttributeList & attrs);
    virtual void update_status_string  (int id, const WideString & str, const AttributeList & attrs);
    virtual void update_aux_string     (int id, const WideString & str, const AttributeList & attrs);
    virtual void commit_string         (int id, const WideString & str);
    virtual void forward_keyevent      (int id, const KeyEvent & key);
    virtual void update_lookup_table   (int id, const LookupTable & table);

    virtual void update_full_width_punctuation (int id, bool full);
    virtual void update_full_width_letter      (int id, bool full);

public:
    virtual void init (int argc, char **argv);
    virtual void run ();

private:
    int generate_key () const;

    void add_client (int client_id, int key);
    void delete_client (int client_id);

    bool validate_client (int client_id, int key) const;

    bool check_client_connection (const Socket &client) const;

    void socket_accept_callback    (SocketServer *server, const Socket &client);
    void socket_receive_callback   (SocketServer *server, const Socket &client);
    void socket_exception_callback (SocketServer *server, const Socket &client);

    void socket_open_connection (SocketServer *server, const Socket &client);

    //client_id is client's socket id
    void socket_get_server_factory_list     (int client_id);
    void socket_get_server_factory_name     (int client_id);
    void socket_get_server_factory_authors  (int client_id);
    void socket_get_server_factory_credits  (int client_id);
    void socket_get_server_factory_help     (int client_id);
    void socket_get_server_factory_locales  (int client_id);
    void socket_get_server_factory_icon_file(int client_id);

    void socket_new_server_instance         (int client_id);
    void socket_delete_server_instance      (int client_id);
    void socket_delete_all_server_instances (int client_id);

    void socket_process_key_event           (int client_id);
    void socket_move_preedit_caret          (int client_id);
    void socket_select_lookup_table         (int client_id);
    void socket_update_lookup_table_page_size (int client_id);
    void socket_reset_server_instance       (int client_id);
    void socket_focus_in_server_instance    (int client_id);
    void socket_focus_out_server_instance   (int client_id);
    void socket_toggle_full_width_punctuation (int client_id);
    void socket_toggle_full_width_letter    (int client_id);
    void socket_toggle_input_status         (int client_id);

    void socket_flush_config                (int client_id);
    void socket_erase_config                (int client_id);
    void socket_get_config_string           (int client_id);
    void socket_set_config_string           (int client_id);
    void socket_get_config_int              (int client_id);
    void socket_set_config_int              (int client_id);
    void socket_get_config_bool             (int client_id);
    void socket_set_config_bool             (int client_id);
    void socket_get_config_double           (int client_id);
    void socket_set_config_double           (int client_id);
    void socket_get_config_vector_string    (int client_id);
    void socket_set_config_vector_string    (int client_id);
    void socket_get_config_vector_int       (int client_id);
    void socket_set_config_vector_int       (int client_id);

    void socket_load_file                   (int client_id);
};

#endif

/*
vi:ts=4:nowrap:ai:expandtab
*/
