/** @file scim_rawcode_server.cpp
 * implementation of class RawCodeServerInstance.
 */

/*
 * Smart Common Input Method
 * 
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_rawcode_server.cpp,v 1.23 2004/04/13 03:44:43 suzhe Exp $
 *
 */

#define Uses_SCIM_SERVER
#define Uses_SCIM_ICONV
#define Uses_SCIM_CONFIG_BASE
#define Uses_SCIM_CONFIG_PATH
#include "scim_private.h"
#include "scim.h"
#include "scim_rawcode_server.h"

#define scim_module_init rawcode_LTX_scim_module_init
#define scim_module_exit rawcode_LTX_scim_module_exit
#define scim_server_module_init rawcode_LTX_scim_server_module_init
#define scim_server_module_create_factory rawcode_LTX_scim_server_module_create_factory

#define SCIM_CONFIG_SERVER_RAWCODE_LOCALE    "/Server/RawCode/Locale"

#define SCIM_RAWCODE_ICON_FILE  (SCIM_ICONDIR "/rawcode.png")

using namespace scim;

static Pointer <RawCodeServerFactory> _scim_rawcode_server_factory;
static ConfigPointer _scim_config;

static const char * _DEFAULT_LOCALES = N_(
    "zh_CN.UTF-8,zh_CN.GB18030,zh_CN.GBK,zh_CN.GB2312,zh_CN,"
    "zh_TW.UTF-8,zh_TW.Big5,zh_TW,zh_HK.UTF-8,zh_HK,"
    "ja_JP.eucJP,ja_JP.ujis,ja_JP,ja,"
    "ko_KR.eucKR,ko_KR");

extern "C" {
    void scim_module_init (void)
    {
    }

    void scim_module_exit (void)
    {
        _scim_rawcode_server_factory.reset ();
        _scim_config.reset ();
    }

    unsigned int scim_server_module_init (const ConfigPointer &config)
    {
        _scim_config = config;
        return 1;
    }

    ServerFactoryPointer scim_server_module_create_factory (unsigned int server)
    {
        String locale;

        if (server >= 1) return NULL;

        if (!_scim_config.null ()) {
            locale = _scim_config->read (
                        String (SCIM_CONFIG_SERVER_RAWCODE_LOCALE),
                        String ("default"));
        } else {
            locale = String ("default");
        }

        if (_scim_rawcode_server_factory.null ()) {
            _scim_rawcode_server_factory =
                new RawCodeServerFactory (utf8_mbstowcs (String (_("RAW CODE"))), locale);
        }
        return _scim_rawcode_server_factory;
    }
}

// implementation of RawCodeServer
RawCodeServerFactory::RawCodeServerFactory ()
{
    m_name = utf8_mbstowcs (_("RAW CODE"));
    set_locales (String (_(_DEFAULT_LOCALES)));
}

RawCodeServerFactory::RawCodeServerFactory (const WideString& name,
                                            const String& locales)
{
    // do not allow too long name
    if (name.length () <= 8)
        m_name = name;
    else
        m_name.assign (name.begin (), name.begin () + 8);

    if (locales == String ("default"))
        set_locales (String (_(_DEFAULT_LOCALES)));
    else
        set_locales (locales);
}

RawCodeServerFactory::~RawCodeServerFactory ()
{
}

WideString
RawCodeServerFactory::get_name () const
{
    return m_name;
}

WideString
RawCodeServerFactory::get_authors () const
{
    return utf8_mbstowcs (String (
                _("(C) 2002-2004 James Su <suzhe@tsinghua.org.cn>")));
}

WideString
RawCodeServerFactory::get_credits () const
{
    return WideString ();
}

WideString
RawCodeServerFactory::get_help () const
{
    return utf8_mbstowcs (String (_(
                            "Hot Keys:\n\n"
                            "  Control+u:\n"
                            "    switch between Multibyte encoding and Unicode.\n\n"
                            "  Control+comma:\n"
                            "    switch between full/half width punctuation mode.\n\n"
                            "  Shift+space:\n"
                            "    switch between full/half width letter mode.\n\n"
                            "  Esc:\n"
                            "    reset the input method.\n")));
}

String
RawCodeServerFactory::get_uuid () const
{
    return String ("6e029d75-ef65-42a8-848e-332e63d70f9c");
}

String
RawCodeServerFactory::get_icon_file () const
{
    return String (SCIM_RAWCODE_ICON_FILE);
}

ServerInstancePointer
RawCodeServerFactory::create_server_instance (const String& encoding, int id)
{
    return new RawCodeServerInstance (this, encoding, id);
}

int
RawCodeServerFactory::get_maxlen (const String &encoding)
{
    std::vector <String> locales;

    scim_split_string_list (locales, get_locales ());

    for (unsigned int i=0; i<locales.size (); ++i)
        if (scim_get_locale_encoding (locales [i]) == encoding)
            return scim_get_locale_maxlen (locales [i]);
    return 1;
}

// implementation of RawCodeServerInstance
RawCodeServerInstance::RawCodeServerInstance (RawCodeServerFactory *factory,
                                              const String& encoding,
                                              int id)
    : ServerInstanceBase (factory, encoding, id),
      m_factory (factory),
      m_full_width_punctuation (false),
      m_full_width_letter (false),
      m_unicode (false),
      m_forward (false),
      m_focused (false),
      m_max_preedit_len (1),
      m_iconv (encoding)
{
    if (factory)
        m_max_preedit_len = factory->get_maxlen (encoding) * 2;
}

RawCodeServerInstance::~RawCodeServerInstance ()
{
}

bool
RawCodeServerInstance::process_key_event (const KeyEvent& key)
{
    if (!m_focused) return false;
    
    // capture the state switch key events
    if ( ((key.code == SCIM_KEY_Alt_L || key.code == SCIM_KEY_Alt_R) &&
            key.is_shift_down ()) ||
         ((key.code == SCIM_KEY_Shift_L || key.code == SCIM_KEY_Shift_R) && 
            (key.is_alt_down () || key.is_control_down ())) &&
         key.is_key_press ()) {
        m_forward = !m_forward;
        refresh_status_string ();
        reset ();
        return true;
    }

    if (key.is_key_release ()) return true;

    // toggle full width punctuation mode
    if (key.code == SCIM_KEY_comma && key.is_control_down ()) {
        toggle_full_width_punctuation ();
        return true;
    }

    // toggle full width letter mode
    if (key.code == SCIM_KEY_space && key.is_shift_down ()) {
        toggle_full_width_letter ();
        return true;
    }

    if (!m_forward) {
        // switch between unicode and native code mode
        if ((key.code == SCIM_KEY_u || key.code == SCIM_KEY_U) &&
             key.is_control_down ()) {
            m_unicode = !m_unicode;
            refresh_status_string ();
            reset ();
            return true;
        }

        //reset key
        if (key.code == SCIM_KEY_Escape && key.mask == 0) {
            reset ();
            return true;
        }

        //delete key
        if (key.code == SCIM_KEY_BackSpace && key.mask == 0 &&
            m_preedit_string.size () != 0) {
            m_preedit_string.erase (m_preedit_string.length () - 1, 1);
            update_preedit_string (m_preedit_string);
            update_preedit_caret (m_preedit_string.length ());
            process_preedit_string ();
            return true;
        }

        // valid keys
        if (((key.code >= SCIM_KEY_0 && key.code <= SCIM_KEY_9) ||
             (key.code >= SCIM_KEY_A && key.code <= SCIM_KEY_F) ||
             (key.code >= SCIM_KEY_a && key.code <= SCIM_KEY_f)) &&
            (key.mask == 0 || key.is_shift_down ()) &&
            m_preedit_string.length () < m_max_preedit_len) {

            if (m_preedit_string.length () == 0)
                show_preedit_string ();

            ucs4_t ascii = (ucs4_t) tolower (key.get_ascii_code ());
            m_preedit_string.push_back (ascii);
            update_preedit_string (m_preedit_string);
            update_preedit_caret (m_preedit_string.length ());
            process_preedit_string ();
            return true;
        }

        //page up key.
        if (key.code == SCIM_KEY_comma && key.mask == 0 &&
            m_preedit_string.length () && m_lookup_table.number_of_entries ()) {
            m_lookup_table.page_up ();
            if (create_lookup_table (m_lookup_table.get_current_page_start ()) > 0)
                update_lookup_table (m_lookup_table);
            else
                hide_lookup_table ();
        }

        //page down key.
        if (key.code == SCIM_KEY_period && key.mask == 0 &&
            m_preedit_string.length () && m_lookup_table.number_of_entries ()) {
            m_lookup_table.page_down ();
            if (create_lookup_table (m_lookup_table.get_current_page_start ()) > 0)
                update_lookup_table (m_lookup_table);
            else
                hide_lookup_table ();
        }
    }

    //other keys is not allowed when preediting
    if (m_preedit_string.length ())
        return true;

    ucs4_t ascii = key.get_ascii_code ();

    if ((ispunct (ascii) && m_full_width_punctuation) ||
        ((isalnum (ascii) || ascii == 0x20) && m_full_width_letter)) {
        WideString str;
        str.push_back (scim_wchar_to_full_width (ascii));
        commit_string (str);
        return true;
    }

    return false;
}

void
RawCodeServerInstance::select_lookup_table (unsigned int item)
{
    KeyEvent key = m_lookup_table.get_page_index_key (item);
    process_key_event (key);
}

void
RawCodeServerInstance::update_lookup_table_page_size (unsigned int page_size)
{
    if (page_size > 0)
        m_lookup_table.set_page_size (page_size);
}

void
RawCodeServerInstance::move_preedit_caret (unsigned int /*pos*/)
{
}

void
RawCodeServerInstance::reset ()
{
    m_preedit_string = WideString ();

    if (m_unicode) m_max_preedit_len = 4;
    else if (m_factory)
        m_max_preedit_len = m_factory->get_maxlen (get_encoding ()) * 2;

    m_iconv.set_encoding (get_encoding ());
    m_lookup_table.clear ();

    hide_lookup_table ();
    hide_preedit_string ();
}

void
RawCodeServerInstance::focus_in ()
{
    m_focused = true;
    update_full_width_punctuation (m_full_width_punctuation);
    update_full_width_letter (m_full_width_letter);
    refresh_status_string ();
    show_status_string ();

    if (m_preedit_string.length ()) {
        update_preedit_string (m_preedit_string);
        update_preedit_caret (m_preedit_string.length ());
        show_preedit_string ();
        if (m_lookup_table.number_of_entries ()) {
            update_lookup_table (m_lookup_table);
            show_lookup_table ();
        }
    }
}

void
RawCodeServerInstance::focus_out ()
{
    m_focused = false;
}

void
RawCodeServerInstance::toggle_full_width_punctuation ()
{
    m_full_width_punctuation = !m_full_width_punctuation;
    update_full_width_punctuation (m_full_width_punctuation);
}

void
RawCodeServerInstance::toggle_full_width_letter ()
{
    m_full_width_letter = !m_full_width_letter;
    update_full_width_letter (m_full_width_letter);
}

void
RawCodeServerInstance::toggle_input_status ()
{
    m_forward = !m_forward;
    refresh_status_string ();
}

void
RawCodeServerInstance::refresh_status_string ()
{
    static ucs4_t status_en [] = {L'E',L'n',0};
    static ucs4_t status_ucs [] = {L'U',L'n',L'i',L'c',L'o',L'd',L'e',0};

    if (m_focused) {
        if (m_forward)
            update_status_string (WideString (status_en));
        else if (m_unicode)
            update_status_string (WideString (status_ucs));
        else {
            update_status_string (utf8_mbstowcs (get_encoding ()));
        }
    }
}

void
RawCodeServerInstance::process_preedit_string ()
{
    if (m_preedit_string.length () == 0) {
        hide_preedit_string ();
        hide_lookup_table ();
        return;
    }

    if (m_unicode) {
        if (m_preedit_string.length () == 3 &&
            create_lookup_table () > 0) {
            update_lookup_table (m_lookup_table);
        } else if (m_preedit_string.length () == 4) {
            WideString str;
            ucs4_t code = get_unicode_value (m_preedit_string);

            // If code is valid under current encoding,
            // then commit it.
            if (m_iconv.test_convert (&code, 1)) {
                str.push_back (code);
                commit_string (str);
                m_preedit_string = WideString ();
                m_lookup_table.clear ();

                update_preedit_string (m_preedit_string);
                update_preedit_caret (0);
                hide_preedit_string ();
            }
        } else if (m_lookup_table.number_of_entries ()){
            m_lookup_table.clear ();
        }
    } else {
        String code = get_multibyte_string (m_preedit_string);
        WideString str;

        // convert ok, then commit.
        if (m_iconv.convert (str, code) && str.length () > 0 && str [0] >= 128) {
            commit_string (str);
            m_preedit_string = WideString ();
            m_lookup_table.clear ();

            update_preedit_string (m_preedit_string);
            update_preedit_caret (0);
            hide_preedit_string ();
        } else if (create_lookup_table () > 0) {
            update_lookup_table (m_lookup_table);
        }
    }

    if (m_lookup_table.number_of_entries ())
        show_lookup_table ();
    else
        hide_lookup_table ();
}

inline static int ascii_to_hex (int ascii)
{
    if (ascii >= '0' && ascii <= '9')
        return ascii - '0';
    else if (ascii >= 'a' && ascii <= 'f')
        return ascii - 'a' + 10;
    else if (ascii >= 'A' && ascii <= 'F')
        return ascii - 'A' + 10;
    return 0;
}
inline static int hex_to_ascii (int hex)
{
    hex %= 16;

    if (hex >= 0 && hex <= 9)
        return hex + '0';

    return hex - 10 + 'a';
}

String
RawCodeServerInstance::get_multibyte_string (const WideString & preedit)
{
    String str;
    char ch = 0;

    if (preedit.length () == 0)
        return str;

    for (unsigned int i=0; i<preedit.length (); ++i) {
        if (i % 2 == 0)
            ch = (char) ascii_to_hex ((int) preedit [i]) & 0x0f;
        else {
            ch = (ch << 4) | ((char) ascii_to_hex ((int) preedit [i]) & 0x0f);
            str.push_back (ch);
            ch = 0;
        }
    }

    if (ch != 0)
        str.push_back (ch);

    return str;
}

ucs4_t
RawCodeServerInstance::get_unicode_value (const WideString &preedit)
{
    ucs4_t code = 0;
    for (unsigned int i=0; i<preedit.length (); ++i) {
        code = (code << 4) | (ascii_to_hex ((int) preedit [i]) & 0x0f);
    }
    return code;
}

int
RawCodeServerInstance::create_lookup_table (int start)
{
    std::vector<KeyEvent> keys;
    String mbs_code;
    ucs4_t ucs_code;
    WideString trail;
    WideString str;

    m_lookup_table.clear ();

    trail.push_back (0);

    for (int i=start; i<16; ++i) {
        trail [0] = (ucs4_t) hex_to_ascii (i);
        if (m_unicode) {
            ucs_code = get_unicode_value (m_preedit_string + trail);
            if (m_iconv.test_convert (&ucs_code, 1)) {
                keys.push_back (KeyEvent((int) trail [0], 0));
                m_lookup_table.append_entry (ucs_code);
            }
        } else {
            mbs_code = get_multibyte_string (m_preedit_string + trail);
            if (m_iconv.convert (str, mbs_code) && str.length () > 0 && str [0] >= 128) {
                keys.push_back (KeyEvent((int) trail [0], 0));
                m_lookup_table.append_entry (str [0]);
            }
        }
    }

    m_lookup_table.set_page_size (keys.size ());
    m_lookup_table.set_page_index_keys (keys);

    m_lookup_table.set_page_updown_keys (KeyEvent (SCIM_KEY_comma, 0),
                                         KeyEvent (SCIM_KEY_period, 0));
    return keys.size ();
}

/*
vi:ts=4:nowrap:ai:expandtab
*/
