/** @file scim_socket_server_setup.cpp
 * implementation of Setup Module of SocketServer.
 */

/*
 * Smart Common Input Method
 * 
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_socket_server_setup.cpp,v 1.4 2004/03/08 04:21:35 suzhe Exp $
 *
 */

#define Uses_SCIM_CONFIG_BASE

#include <iostream>

#include <gtk/gtk.h>

#include "scim_private.h"
#include "scim.h"

using namespace scim;

#define scim_module_init socket_server_setup_LTX_scim_module_init
#define scim_module_exit socket_server_setup_LTX_scim_module_exit

#define scim_setup_module_create_ui       socket_server_setup_LTX_scim_setup_module_create_ui
#define scim_setup_module_get_category    socket_server_setup_LTX_scim_setup_module_get_category
#define scim_setup_module_get_name        socket_server_setup_LTX_scim_setup_module_get_name
#define scim_setup_module_get_description socket_server_setup_LTX_scim_setup_module_get_description
#define scim_setup_module_load_config     socket_server_setup_LTX_scim_setup_module_load_config
#define scim_setup_module_save_config     socket_server_setup_LTX_scim_setup_module_save_config
#define scim_setup_module_query_changed   socket_server_setup_LTX_scim_setup_module_query_changed

#define SCIM_CONFIG_SERVER_SOCKET_TIMEOUT "/Server/Socket/Timeout"
#define SCIM_CONFIG_SERVER_SOCKET_ADDRESS "/Server/Socket/Address"

static GtkWidget * create_setup_window ();
static void        load_config (const ConfigPointer &config);
static void        save_config (const ConfigPointer &config);
static bool        query_changed ();

// Module Interface.
extern "C" {
    void scim_module_init (void)
    {
    }

    void scim_module_exit (void)
    {
    }

    GtkWidget * scim_setup_module_create_ui (void)
    {
        return create_setup_window ();
    }

    String scim_setup_module_get_category (void)
    {
        return String ("Server");
    }

    String scim_setup_module_get_name (void)
    {
        return String (_("Socket"));
    }

    String scim_setup_module_get_description (void)
    {
        return String (_("A Server Module which uses the input mothod service provided by a Socket FrontEnd."));
    }

    void scim_setup_module_load_config (const ConfigPointer &config)
    {
        load_config (config);
    }

    void scim_setup_module_save_config (const ConfigPointer &config)
    {
        save_config (config);
    }

    bool scim_setup_module_query_changed ()
    {
        return query_changed ();
    }
} // extern "C"

// Internal data declaration.
static String      __config_socket_address         = "default";
static int         __config_socket_timeout         = -1;

static bool        __have_changed                  = false;

static GtkWidget * __widget_socket_address         = 0;
static GtkWidget * __widget_socket_timeout         = 0;

static GtkTooltips * __widget_tooltips             = 0;

// Declarations of internal functions.
static void
on_socket_address_changed (GtkEditable *editable,
                           gpointer user_data);

static void
on_socket_timeout_value_changed (GtkSpinButton   *spinbutton,
                                 gpointer         user_data);

static void
setup_widget_value ();

//Implementations of functions.
GtkWidget *
create_setup_window ()
{
    static GtkWidget *window = 0;

    if (!window) {
        GtkWidget *frame;
        GtkWidget *table;
        GtkWidget *label;

        __widget_tooltips = gtk_tooltips_new ();

        // Create the setup block.
        frame = gtk_frame_new (NULL);
        gtk_widget_show (frame);
        gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_OUT);

        table = gtk_table_new (2,2, FALSE);
        gtk_widget_show (table);
        gtk_container_add (GTK_CONTAINER (frame), table);
        gtk_container_set_border_width (GTK_CONTAINER (table), 4);

        label = gtk_label_new (NULL);
        gtk_label_set_text_with_mnemonic (GTK_LABEL (label), _("_Connect Address:"));
        gtk_widget_show (label);
        gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
                          (GtkAttachOptions) (GTK_FILL),
                          (GtkAttachOptions) (GTK_FILL), 0, 8);
        gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
        gtk_misc_set_padding (GTK_MISC (label), 4, 0);

        __widget_socket_address = gtk_entry_new ();
        gtk_widget_show (__widget_socket_address);
        gtk_table_attach (GTK_TABLE (table), __widget_socket_address, 1, 2, 0, 1,
                          (GtkAttachOptions) (GTK_FILL|GTK_EXPAND),
                          (GtkAttachOptions) (GTK_FILL), 0, 8);
        gtk_label_set_mnemonic_widget (GTK_LABEL (label), __widget_socket_address);

        label = gtk_label_new (NULL);
        gtk_label_set_text_with_mnemonic (GTK_LABEL (label), _("Connect _Timeout:"));
        gtk_widget_show (label);
        gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
                          (GtkAttachOptions) (GTK_FILL),
                          (GtkAttachOptions) (GTK_FILL), 0, 8);
        gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
        gtk_misc_set_padding (GTK_MISC (label), 4, 0);

        __widget_socket_timeout = gtk_spin_button_new_with_range (-1, 2000, 1);
        gtk_widget_show (__widget_socket_timeout);
        gtk_table_attach (GTK_TABLE (table), __widget_socket_timeout, 1, 2, 1, 2,
                          (GtkAttachOptions) (GTK_FILL|GTK_EXPAND),
                          (GtkAttachOptions) (GTK_FILL), 0, 8);
        gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (__widget_socket_timeout), TRUE);
        gtk_spin_button_set_snap_to_ticks (GTK_SPIN_BUTTON (__widget_socket_timeout), TRUE);
        gtk_spin_button_set_digits (GTK_SPIN_BUTTON (__widget_socket_timeout), 0);
        gtk_label_set_mnemonic_widget (GTK_LABEL (label), __widget_socket_timeout);

        // Connect all signals.
        g_signal_connect ((gpointer) __widget_socket_address, "changed",
                          G_CALLBACK (on_socket_address_changed),
                          NULL);
        g_signal_connect ((gpointer) __widget_socket_timeout, "value_changed",
                          G_CALLBACK (on_socket_timeout_value_changed),
                          NULL);

        // Set all tooltips.
        gtk_tooltips_set_tip (__widget_tooltips, __widget_socket_address,
                                _("The address of the Socket FrontEnd to be connected. "
                                  "Must begin with local: or inet:. "
                                  "For example:\n"
                                  "  local:/tmp/scim-socket-frontend\n"
                                  "  inet:localhost:12345"), NULL);

        gtk_tooltips_set_tip (__widget_tooltips, __widget_socket_timeout,
                                _("The timeout value of the connection, "
                                  "in milliseconds. "
                                  "-1 means unlimited."), NULL);

        window = frame;

        setup_widget_value ();
    }

    return window;
}

void
setup_widget_value ()
{
    if (__widget_socket_address) {
        gtk_entry_set_text (
            GTK_ENTRY (__widget_socket_address),
            __config_socket_address.c_str ());
    }

    if (__widget_socket_timeout) {
        gtk_spin_button_set_value (
            GTK_SPIN_BUTTON (__widget_socket_timeout),
            __config_socket_timeout);
    }
}

void
load_config (const ConfigPointer &config)
{
    if (!config.null ()) {
        __config_socket_address =
            config->read (String (SCIM_CONFIG_SERVER_SOCKET_ADDRESS),
                          __config_socket_address);
        __config_socket_timeout =
            config->read (String (SCIM_CONFIG_SERVER_SOCKET_TIMEOUT),
                          __config_socket_timeout);

        setup_widget_value ();

        __have_changed = false;
    }
}

void
save_config (const ConfigPointer &config)
{
    if (!config.null ()) {
        __config_socket_address = String (gtk_entry_get_text (GTK_ENTRY (__widget_socket_address)));

        if (__config_socket_address != "default" &&
            __config_socket_address.substr (0, 6) != "local:" &&
            __config_socket_address.substr (0, 5) != "inet:") {

            GtkWidget * dialog = gtk_message_dialog_new (
                                        NULL,
                                        GTK_DIALOG_MODAL,
                                        GTK_MESSAGE_WARNING,
                                        GTK_BUTTONS_OK,
                                        _("Connect address must begin with "
                                          "\"local:\" or \"inet:\"."));

            gtk_dialog_run (GTK_DIALOG (dialog));

            gtk_widget_destroy (dialog);

            __config_socket_address = "default";

            setup_widget_value ();
        }

        config->write (String (SCIM_CONFIG_SERVER_SOCKET_ADDRESS),
                        __config_socket_address);
        config->write (String (SCIM_CONFIG_SERVER_SOCKET_TIMEOUT),
                        __config_socket_timeout);

        __have_changed = false;
    }
}

bool
query_changed ()
{
    return __have_changed;
}

void
on_socket_address_changed (GtkEditable *editable,
                           gpointer user_data)
{
    __have_changed = true;
}

void
on_socket_timeout_value_changed (GtkSpinButton   *spinbutton,
                                 gpointer         user_data)
{
    __config_socket_timeout = gtk_spin_button_get_value_as_int (spinbutton);
    __have_changed = true;
}


/*
vi:ts=4:nowrap:expandtab
*/
