/** @file scim_backend.h
 *  @brief scim::BackEndBase Interface.
 *
 *  Provide an abstract interface class to
 *  manage a set of ServerFactory instances.
 */

/* 
 * Smart Common Input Method
 * 
 * Copyright (c) 2004 James Su <suzhe@turbolinux.com.cn>
 * Copyright (c) 2003 James Su <suzhe@turbolinux.com.cn>
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_backend.h,v 1.13 2004/02/06 07:53:15 suzhe Exp $
 */

#ifndef __SCIM_BACKEND_H
#define __SCIM_BACKEND_H

namespace scim {

/**
 * @brief An exception class to hold BackEnd related errors.
 *
 * scim::BackEndBase and its derived classes must throw
 * scim::BackEndError object when error.
 */
class BackEndError: public Exception
{
public:
    BackEndError (const String& what_arg)
        : Exception (String("scim::BackEnd: ") + what_arg) { }
};

/**
 * @brief The interface class to manage a set of ServerFactory instances.
 *
 * This is mainly a helper interface class used by scim::FrontEndBase.
 * Its responsibility is to hold a set of ServerFactory instances
 * and manage the locales list supported by them. 
 * 
 * Most developer should just use the default implementation
 * scim::CommonBackEnd.
 */
class BackEndBase : public ReferencedObject
{
public:
    /**
     * @brief Virtual destructor
     */
    virtual ~BackEndBase ();

    /**
     * @brief Get a list of all locales supported by all FrontEnds.
     * @return A comma separated locales list.
     */
    virtual String get_locales () const = 0;

    /**
     * @brief Get the total number of ServerFactory instances held by this BackEnd.
     * @return The total number of ServerFactory instances in this BackEnd.
     */
    virtual unsigned int number_of_servers () const = 0;

    /**
     * @brief Get a ServerFactory instance by its index.
     * @return The ServerFactoryPointer according to the index.
     */
    virtual ServerFactoryPointer get_server_factory (unsigned int index) = 0;
};

/**
 * @typedef typedef Pointer <BackEndBase> BackEndPointer;
 *
 * A smart pointer for scim::BackEndBase and its derived classes.
 */
typedef Pointer <BackEndBase> BackEndPointer;

/**
 * @brief The default implementation of scim::BackEndBase interface.
 */
class CommonBackEnd : public BackEndBase
{
    typedef std::vector<ServerFactoryPointer> ServerFactoryPool;

    ServerFactoryPool m_factories;

    String m_supported_unicode_locales;

public:
    virtual String get_locales () const;
    virtual unsigned int number_of_servers () const;
    virtual ServerFactoryPointer get_server_factory (unsigned int index);

public:
    /**
     * @brief Add a ServerFactory instance into this BackEnd.
     * @param factory - the smart pointer of the ServerFactory instance.
     */
    bool add_server_factory (const ServerFactoryPointer &factory);

    /**
     * @brief Set the list of unicode locales which should be supported
     *        among the other locales.
     * @param locales - the comma separated list of unicode locales should be supported.
     */
    void set_supported_unicode_locales (const String &locales);

    /**
     * @brief Get the list of supported unicode locales which set by
     *        set_supported_unicode_locales
     * @return The comma separated list of supported unicode locales.
     */
    String get_supported_unicode_locales () const;
};

} // namespace scim

#endif //__SCIM_BACKEND_H

/*
vi:ts=4:nowrap:ai:expandtab
*/
