/**
 * @file scim_event.h
 * @brief Defines the scim::KeyEvent class and related enums, functions.
 */

/* 
 * Smart Common Input Method
 * 
 * Copyright (c) 2004 James Su <suzhe@turbolinux.com.cn>
 * Copyright (c) 2003 James Su <suzhe@turbolinux.com.cn>
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_event.h,v 1.15 2004/02/12 09:40:11 suzhe Exp $
 */

#ifndef __SCIM_EVENT_H
#define __SCIM_EVENT_H

namespace scim {

/**
 * @addtogroup Helper
 * @{
 */

/**
 * @brief Enum values of all valid key masks.
 *
 * The key masks indicate which modifier keys
 * is pressed down during the keyboard hit.
 *
 * The special SCIM_KEY_ReleaseMask indicates
 * the key release event.
 */
enum KeyMask
{
    SCIM_KEY_NullMask     = 0,        /**< Key press event without modifier key. */
    SCIM_KEY_ShiftMask   = (1<<0),    /**< The shift key is pressed down */
    SCIM_KEY_LockMask    = (1<<1),    /**< The lock key is pressed down */
    SCIM_KEY_CapsLockMask= (1<<1),    /**< The caps lock key is pressed down */
    SCIM_KEY_ControlMask = (1<<2),    /**< The ctrl key is pressed down */
    SCIM_KEY_AltMask     = (1<<3),    /**< The alt key is pressed down */
    SCIM_KEY_Mod1Mask    = (1<<3),    /**< The mod1 key is pressed down */
    SCIM_KEY_Mod2Mask    = (1<<4),    /**< The mod2 key is pressed down */
    SCIM_KEY_NumLockMask = (1<<4),    /**< The num lock key is pressed down */
    SCIM_KEY_Mod3Mask    = (1<<5),    /**< The mod3 key is pressed down */
    SCIM_KEY_Mod4Mask    = (1<<6),    /**< The mod4 key is pressed down */
    SCIM_KEY_Mod5Mask    = (1<<7),    /**< The mod5 key is pressed down */
    SCIM_KEY_ScrollLockMask = (1<<7), /**< The scroll lock key is pressed down */
    SCIM_KEY_ReleaseMask = (1<<30)    /**< It's a key release event */
};

/**
 * @brief Enum values of all valid key codes.
 */
enum KeyCode
{
    SCIM_KEY_NullKey     = 0,

    /* modifier keys */
    SCIM_KEY_Shift_L     = 0xFFE1,    /* Left shift */
    SCIM_KEY_Shift_R     = 0xFFE2,    /* Right shift */
    SCIM_KEY_Control_L   = 0xFFE3,    /* Left control */
    SCIM_KEY_Control_R   = 0xFFE4,    /* Right control */
    SCIM_KEY_Caps_Lock   = 0xFFE5,    /* Caps lock */
    SCIM_KEY_Shift_Lock  = 0xFFE6,    /* Shift lock */

    SCIM_KEY_Meta_L      = 0xFFE7,    /* Left meta */
    SCIM_KEY_Meta_R      = 0xFFE8,    /* Right meta */
    SCIM_KEY_Alt_L       = 0xFFE9,    /* Left alt */
    SCIM_KEY_Alt_R       = 0xFFEA,    /* Right alt */
    SCIM_KEY_Super_L     = 0xFFEB,    /* Left super */
    SCIM_KEY_Super_R     = 0xFFEC,    /* Right super */
    SCIM_KEY_Hyper_L     = 0xFFED,    /* Left hyper */
    SCIM_KEY_Hyper_R     = 0xFFEE,    /* Right hyper */

    /* function keys */
    SCIM_KEY_BackSpace   = 0xFF08,    /* back space, back char */
    SCIM_KEY_Tab         = 0xFF09,
    SCIM_KEY_Linefeed    = 0xFF0A,    /* Linefeed, LF */
    SCIM_KEY_Clear       = 0xFF0B,
    SCIM_KEY_Return      = 0xFF0D,    /* Return, enter */
    SCIM_KEY_Pause       = 0xFF13,    /* Pause, hold */
    SCIM_KEY_Scroll_Lock = 0xFF14,
    SCIM_KEY_Sys_Req     = 0xFF15,
    SCIM_KEY_Escape      = 0xFF1B,
    SCIM_KEY_Delete      = 0xFFFF,    /* Delete, rubout */

    /* International & multi-key character composition */
    SCIM_KEY_Multi_key   = 0xFF20,    /* Multi-key character compose */
    SCIM_KEY_Codeinput   = 0xFF37,
    SCIM_KEY_SingleCandidate   = 0xFF3C,
    SCIM_KEY_MultipleCandidate = 0xFF3D,
    SCIM_KEY_PreviousCandidate = 0xFF3E,

    /* Japanese keyboard support */
    SCIM_KEY_Kanji       = 0xFF21,    /* Kanji, Kanji convert */
    SCIM_KEY_Muhenkan    = 0xFF22,    /* Cancel Conversion */
    SCIM_KEY_Henkan_Mode = 0xFF23,    /* Start/Stop Conversion */
    SCIM_KEY_Henkan      = 0xFF23,    /* Alias for Henkan_Mode */
    SCIM_KEY_Romaji      = 0xFF24,    /* to Romaji */
    SCIM_KEY_Hiragana    = 0xFF25,    /* to Hiragana */
    SCIM_KEY_Katakana    = 0xFF26,    /* to Katakana */
    SCIM_KEY_Hiragana_Katakana = 0xFF27, /* Hiragana/Katakana toggle */
    SCIM_KEY_Zenkaku     = 0xFF28,    /* to Zenkaku */
    SCIM_KEY_Hankaku     = 0xFF29,    /* to Hankaku */
    SCIM_KEY_Zenkaku_Hankaku   = 0xFF2A, /* Zenkaku/Hankaku toggle */
    SCIM_KEY_Touroku     = 0xFF2B,    /* Add to Dictionary */
    SCIM_KEY_Massyo      = 0xFF2C,    /* Delete from Dictionary */
    SCIM_KEY_Kana_Lock   = 0xFF2D,    /* Kana Lock */
    SCIM_KEY_Kana_Shift  = 0xFF2E,    /* Kana Shift */
    SCIM_KEY_Eisu_Shift  = 0xFF2F,    /* Alphanumeric Shift */
    SCIM_KEY_Eisu_toggle = 0xFF30,    /* Alphanumeric toggle */
    SCIM_KEY_Kanji_Bangou = 0xFF37,   /* Codeinput */
    SCIM_KEY_Zen_Koho    = 0xFF3D,    /* Multiple/All Candidate(s) */
    SCIM_KEY_Mae_Koho    = 0xFF3E,    /* Previous Candidate */

    /* Cursor control & motion */
    SCIM_KEY_Home        = 0xFF50,
    SCIM_KEY_Left        = 0xFF51,    /* Move left, left arrow */
    SCIM_KEY_Up          = 0xFF52,    /* Move up, up arrow */
    SCIM_KEY_Right       = 0xFF53,    /* Move right, right arrow */
    SCIM_KEY_Down        = 0xFF54,    /* Move down, down arrow */
    SCIM_KEY_Prior       = 0xFF55,    /* Prior, previous */
    SCIM_KEY_Page_Up     = 0xFF55,
    SCIM_KEY_Next        = 0xFF56,    /* Next */
    SCIM_KEY_Page_Down   = 0xFF56,
    SCIM_KEY_End         = 0xFF57,    /* EOL */
    SCIM_KEY_Begin       = 0xFF58,    /* BOL */

    /* Misc Functions */
    SCIM_KEY_Select      = 0xFF60,    /* Select, mark */
    SCIM_KEY_Print       = 0xFF61,
    SCIM_KEY_Execute     = 0xFF62,    /* Execute, run, do */
    SCIM_KEY_Insert      = 0xFF63,    /* Insert, insert here */
    SCIM_KEY_Undo        = 0xFF65,    /* Undo, oops */
    SCIM_KEY_Redo        = 0xFF66,    /* redo, again */
    SCIM_KEY_Menu        = 0xFF67,
    SCIM_KEY_Find        = 0xFF68,    /* Find, search */
    SCIM_KEY_Cancel      = 0xFF69,    /* Cancel, stop, abort, exit */
    SCIM_KEY_Help        = 0xFF6A,    /* Help */
    SCIM_KEY_Break       = 0xFF6B,
    SCIM_KEY_Mode_switch = 0xFF7E,    /* Character set switch */
    SCIM_KEY_Num_Lock    = 0xFF7F,

    /* keypad */
    SCIM_KEY_KP_Space    = 0xFF80,    /* space */
    SCIM_KEY_KP_Tab      = 0xFF89,
    SCIM_KEY_KP_Enter    = 0xFF8D,    /* enter */
    SCIM_KEY_KP_F1       = 0xFF91,    /* PF1, KP_A, ... */
    SCIM_KEY_KP_F2       = 0xFF92,
    SCIM_KEY_KP_F3       = 0xFF93,
    SCIM_KEY_KP_F4       = 0xFF94,
    SCIM_KEY_KP_Home     = 0xFF95,
    SCIM_KEY_KP_Left     = 0xFF96,
    SCIM_KEY_KP_Up       = 0xFF97,
    SCIM_KEY_KP_Right    = 0xFF98,
    SCIM_KEY_KP_Down     = 0xFF99,
    SCIM_KEY_KP_Prior    = 0xFF9A,
    SCIM_KEY_KP_Page_Up  = 0xFF9A,
    SCIM_KEY_KP_Next     = 0xFF9B,
    SCIM_KEY_KP_Page_Down= 0xFF9B,
    SCIM_KEY_KP_End      = 0xFF9C,
    SCIM_KEY_KP_Begin    = 0xFF9D,
    SCIM_KEY_KP_Insert   = 0xFF9E,
    SCIM_KEY_KP_Delete   = 0xFF9F,
    SCIM_KEY_KP_Equal    = 0xFFBD,   /* equals */
    SCIM_KEY_KP_Multiply = 0xFFAA,
    SCIM_KEY_KP_Add      = 0xFFAB,
    SCIM_KEY_KP_Separator= 0xFFAC,   /* separator, often comma */
    SCIM_KEY_KP_Subtract = 0xFFAD,
    SCIM_KEY_KP_Decimal  = 0xFFAE,
    SCIM_KEY_KP_Divide   = 0xFFAF,

    SCIM_KEY_KP_0        = 0xFFB0,
    SCIM_KEY_KP_1        = 0xFFB1,
    SCIM_KEY_KP_2        = 0xFFB2,
    SCIM_KEY_KP_3        = 0xFFB3,
    SCIM_KEY_KP_4        = 0xFFB4,
    SCIM_KEY_KP_5        = 0xFFB5,
    SCIM_KEY_KP_6        = 0xFFB6,
    SCIM_KEY_KP_7        = 0xFFB7,
    SCIM_KEY_KP_8        = 0xFFB8,
    SCIM_KEY_KP_9        = 0xFFB9,

    /* Auxilliary Functions */
    SCIM_KEY_F1          = 0xFFBE,
    SCIM_KEY_F2          = 0xFFBF,
    SCIM_KEY_F3          = 0xFFC0,
    SCIM_KEY_F4          = 0xFFC1,
    SCIM_KEY_F5          = 0xFFC2,
    SCIM_KEY_F6          = 0xFFC3,
    SCIM_KEY_F7          = 0xFFC4,
    SCIM_KEY_F8          = 0xFFC5,
    SCIM_KEY_F9          = 0xFFC6,
    SCIM_KEY_F10         = 0xFFC7,
    SCIM_KEY_F11         = 0xFFC8,
    SCIM_KEY_F12         = 0xFFC9,

    /* Latin 1 */
    SCIM_KEY_space       = 0x020,
    SCIM_KEY_exclam      = 0x021,
    SCIM_KEY_quotedbl    = 0x022,
    SCIM_KEY_numbersign  = 0x023,
    SCIM_KEY_dollar      = 0x024,
    SCIM_KEY_percent     = 0x025,
    SCIM_KEY_ampersand   = 0x026,
    SCIM_KEY_apostrophe  = 0x027,
    SCIM_KEY_quoteright  = 0x027,    /* deprecated */
    SCIM_KEY_parenleft   = 0x028,
    SCIM_KEY_parenright  = 0x029,
    SCIM_KEY_asterisk    = 0x02a,
    SCIM_KEY_plus        = 0x02b,
    SCIM_KEY_comma       = 0x02c,
    SCIM_KEY_minus       = 0x02d,
    SCIM_KEY_period      = 0x02e,
    SCIM_KEY_slash       = 0x02f,
    SCIM_KEY_0           = 0x030,
    SCIM_KEY_1           = 0x031,
    SCIM_KEY_2           = 0x032,
    SCIM_KEY_3           = 0x033,
    SCIM_KEY_4           = 0x034,
    SCIM_KEY_5           = 0x035,
    SCIM_KEY_6           = 0x036,
    SCIM_KEY_7           = 0x037,
    SCIM_KEY_8           = 0x038,
    SCIM_KEY_9           = 0x039,
    SCIM_KEY_colon       = 0x03a,
    SCIM_KEY_semicolon   = 0x03b,
    SCIM_KEY_less        = 0x03c,
    SCIM_KEY_equal       = 0x03d,
    SCIM_KEY_greater     = 0x03e,
    SCIM_KEY_question    = 0x03f,
    SCIM_KEY_at          = 0x040,
    SCIM_KEY_A           = 0x041,
    SCIM_KEY_B           = 0x042,
    SCIM_KEY_C           = 0x043,
    SCIM_KEY_D           = 0x044,
    SCIM_KEY_E           = 0x045,
    SCIM_KEY_F           = 0x046,
    SCIM_KEY_G           = 0x047,
    SCIM_KEY_H           = 0x048,
    SCIM_KEY_I           = 0x049,
    SCIM_KEY_J           = 0x04a,
    SCIM_KEY_K           = 0x04b,
    SCIM_KEY_L           = 0x04c,
    SCIM_KEY_M           = 0x04d,
    SCIM_KEY_N           = 0x04e,
    SCIM_KEY_O           = 0x04f,
    SCIM_KEY_P           = 0x050,
    SCIM_KEY_Q           = 0x051,
    SCIM_KEY_R           = 0x052,
    SCIM_KEY_S           = 0x053,
    SCIM_KEY_T           = 0x054,
    SCIM_KEY_U           = 0x055,
    SCIM_KEY_V           = 0x056,
    SCIM_KEY_W           = 0x057,
    SCIM_KEY_X           = 0x058,
    SCIM_KEY_Y           = 0x059,
    SCIM_KEY_Z           = 0x05a,
    SCIM_KEY_bracketleft = 0x05b,
    SCIM_KEY_backslash   = 0x05c,
    SCIM_KEY_bracketright= 0x05d,
    SCIM_KEY_asciicircum = 0x05e,
    SCIM_KEY_underscore  = 0x05f,
    SCIM_KEY_grave       = 0x060,
    SCIM_KEY_a           = 0x061,
    SCIM_KEY_b           = 0x062,
    SCIM_KEY_c           = 0x063,
    SCIM_KEY_d           = 0x064,
    SCIM_KEY_e           = 0x065,
    SCIM_KEY_f           = 0x066,
    SCIM_KEY_g           = 0x067,
    SCIM_KEY_h           = 0x068,
    SCIM_KEY_i           = 0x069,
    SCIM_KEY_j           = 0x06a,
    SCIM_KEY_k           = 0x06b,
    SCIM_KEY_l           = 0x06c,
    SCIM_KEY_m           = 0x06d,
    SCIM_KEY_n           = 0x06e,
    SCIM_KEY_o           = 0x06f,
    SCIM_KEY_p           = 0x070,
    SCIM_KEY_q           = 0x071,
    SCIM_KEY_r           = 0x072,
    SCIM_KEY_s           = 0x073,
    SCIM_KEY_t           = 0x074,
    SCIM_KEY_u           = 0x075,
    SCIM_KEY_v           = 0x076,
    SCIM_KEY_w           = 0x077,
    SCIM_KEY_x           = 0x078,
    SCIM_KEY_y           = 0x079,
    SCIM_KEY_z           = 0x07a,
    SCIM_KEY_braceleft   = 0x07b,
    SCIM_KEY_bar         = 0x07c,
    SCIM_KEY_braceright  = 0x07d,
    SCIM_KEY_asciitilde  = 0x07e
};

/**
 * @brief The class to store a keyboard event.
 *
 * A keyboard event contains a key code and a set of key masks.
 * The key masks indicate which modifier keys are pressed down and
 * if it's a key release event.
 */
struct KeyEvent
{
    int code;    /**< key code */
    int mask;    /**< modifier keys' mask */

    /**
     * @brief Constructor.
     * @param c - the key code.
     * @param m - the key masks.
     */
    KeyEvent (int c = 0, int m = 0)
        : code (c), mask (m) { }

    /**
     * @brief Check if this KeyEvent is empty.
     * @return true if this is a empty event.
     */
    bool empty () const { return mask == 0 && code == 0; }

    /**
     * @brief Get the ascii code of this key event.
     *
     * Not all key events have ascii codes.
     *
     * @return the ascii code of this key event.
     *         Zero means no ascii code.
     */
    char get_ascii_code () const;

    /**
     * @brief Check if the shift key is pressed down.
     */
    bool is_shift_down () const { return (mask & SCIM_KEY_ShiftMask) != 0; }

    /**
     * @brief Check if the lock key is pressed down.
     */
    bool is_lock_down () const { return (mask & SCIM_KEY_LockMask) != 0; }

    /**
     * @brief Check if the ctrl key is pressed down.
     */
    bool is_control_down () const { return (mask & SCIM_KEY_ControlMask) != 0; }

    /**
     * @brief Check if the alt key is pressed down.
     */
    bool is_alt_down () const { return is_mod1_down (); }

    /**
     * @brief Check if the mod1 key is pressed down.
     */
    bool is_mod1_down () const { return (mask & SCIM_KEY_Mod1Mask) != 0; }

    /**
     * @brief Check if the mod2 key is pressed down.
     */
    bool is_mod2_down () const { return (mask & SCIM_KEY_Mod2Mask) != 0; }

    /**
     * @brief Check if the mod3 key is pressed down.
     */
    bool is_mod3_down () const { return (mask & SCIM_KEY_Mod3Mask) != 0; }

    /**
     * @brief Check if the mod4 key is pressed down.
     */
    bool is_mod4_down () const { return (mask & SCIM_KEY_Mod4Mask) != 0; }

    /**
     * @brief Check if the mod5 key is pressed down.
     */
    bool is_mod5_down () const { return (mask & SCIM_KEY_Mod5Mask) != 0; }

    /**
     * @brief Check if the num lock key is pressed down.
     */
    bool is_num_lock_down () const { return (mask & SCIM_KEY_NumLockMask) != 0; }

    /**
     * @brief Check if the caps lock key is pressed down.
     */
    bool is_caps_lock_down () const { return (mask & SCIM_KEY_CapsLockMask) != 0; }

    /**
     * @brief Check if the scroll lock key is pressed down.
     */
    bool is_scroll_lock_down () const { return (mask & SCIM_KEY_ScrollLockMask) != 0; }

    /**
     * @brief Check if it's a key press event.
     */
    bool is_key_press () const { return (mask & SCIM_KEY_ReleaseMask) == 0; }

    /**
     * @brief Check if it's a key release event.
     */
    bool is_key_release () const { return (mask & SCIM_KEY_ReleaseMask) != 0; }

    /**
     * @brief Compare two key events.
     * @return true if they are equal.
     */
    bool operator == (const KeyEvent & key) const {
        return code == key.code && mask == key.mask;
    }

    /**
     * @brief Compare two key events.
     *
     * This operator is mainly for sorting.
     * 
     * @return true if the first is smaller.
     */
    bool operator < (const KeyEvent & key) const {
        return code < key.code || (code == key.code && mask < key.mask);
    }
};

/**
 * @typedef typedef std::vector<KeyEvent> KeyEventList
 * @brief The container to store a set of KeyEvent objects.
 *
 * You should use the STL container methods to manipulate its objects.
 */
typedef std::vector<KeyEvent> KeyEventList;

/**
 * @brief Convert a key event to a string.
 * @param str - the result string will be stored here.
 * @param key - the KeyEvent to be converted.
 * @return true if success.
 */
bool scim_key_to_string (String &str, const KeyEvent & key);

/**
 * @brief Convert a string to a KeyEvent.
 * @param key - the result KeyEvent will be stored here.
 * @param str - the string to be converted.
 * @return true if success.
 */
bool scim_string_to_key (KeyEvent &key, const String & str);

/**
 * @brief Convert a set of KeyEvents to a string.
 * @param str - the result string will be stored here.
 * @param keylist - the keys to be converted.
 * @return true if success.
 */
bool scim_key_list_to_string (String &str, const KeyEventList & keylist);

/**
 * @brief Covnert a string to a set of KeyEvents.
 * @param keylist - the result KeyEvents will be stored here.
 * @param str - the string to be converted.
 * @return true if success.
 */
bool scim_string_to_key_list (KeyEventList &keylist, const String &str);

/** @} */

} // namespace scim

#endif //__SCIM_EVENT_H

/*
vi:ts=4:nowrap:ai:expandtab
*/

