/*
 * Smart Common Input Method
 * 
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_lookup_table.cpp,v 1.19 2003/11/15 09:11:37 suzhe Exp $
 *
 */

#define Uses_SCIM_LOOKUP_TABLE
#include "scim_private.h"
#include "scim.h"

namespace scim {

struct LookupTable::LookupTableImpl
{
    std::vector<int>      m_page_history;
    int                   m_page_size;
    int                   m_current_page_start;
    int                   m_cursor_pos;
    bool                  m_cursor_visible;

    std::vector<KeyEvent> m_page_index_keys;
    KeyEvent              m_page_up_key;
    KeyEvent              m_page_down_key;

    LookupTableImpl (int page_size)
        : m_page_size (page_size),
          m_current_page_start (0),
          m_cursor_pos (0),
          m_cursor_visible (false) {
        if (m_page_size <= 0 || m_page_size > SCIM_LOOKUP_TABLE_MAX_PAGESIZE)
            m_page_size = SCIM_LOOKUP_TABLE_MAX_PAGESIZE;
    }
};


//implementation of class LookupTable
LookupTable::LookupTable (int                          page_size,
                          const KeyEvent&              page_up_key,
                          const KeyEvent&              page_down_key,
                          const std::vector<KeyEvent> &page_index_keys)
    : m_impl (new LookupTableImpl (page_size))
{
    set_page_updown_keys (page_up_key, page_down_key);
    set_page_index_keys (page_index_keys);
}

LookupTable::~LookupTable ()
{
    delete m_impl;
}

void
LookupTable::set_page_index_keys (const std::vector<KeyEvent> &page_index_keys)
{
    if (page_index_keys.size ()) {
        m_impl->m_page_index_keys = page_index_keys;
    } else {
        for (int i=0; i<9; ++i)
            m_impl->m_page_index_keys.push_back (KeyEvent (SCIM_KEY_1 + i, 0));

        m_impl->m_page_index_keys.push_back (KeyEvent (SCIM_KEY_0, 0));
    }
}

void
LookupTable::set_page_updown_keys (const KeyEvent &page_up,
                                   const KeyEvent &page_down)
{
    if (!page_up.empty ()) m_impl->m_page_up_key = page_up;
    else m_impl->m_page_up_key = KeyEvent (SCIM_KEY_comma, 0);
    if (!page_down.empty ()) m_impl->m_page_down_key = page_down;
    else m_impl->m_page_down_key = KeyEvent (SCIM_KEY_period, 0);
}

void
LookupTable::set_page_size (int page_size)
{
    if (page_size > 0 && page_size <= SCIM_LOOKUP_TABLE_MAX_PAGESIZE) {
        m_impl->m_page_size = page_size;
        if (m_impl->m_cursor_pos >= m_impl->m_current_page_start + get_current_page_size ())
            m_impl->m_cursor_pos = m_impl->m_current_page_start + get_current_page_size () - 1;
        if (m_impl->m_cursor_pos < 0)
            m_impl->m_cursor_pos = 0;
    }
}

int
LookupTable::get_page_size () const
{
    return m_impl->m_page_size;
}

int
LookupTable::get_current_page_start () const
{
    return m_impl->m_current_page_start;
}

int
LookupTable::get_cursor_pos() const
{
    return m_impl->m_cursor_pos;
}

int
LookupTable::get_cursor_pos_in_page () const
{
    return m_impl->m_cursor_pos - m_impl->m_current_page_start;
}

int
LookupTable::get_current_page_size () const
{
    return std::min ((uint32) m_impl->m_page_size, (uint32)(number_of_entries () - m_impl->m_current_page_start));
}

bool
LookupTable::page_up ()
{
    if (m_impl->m_current_page_start <= 0)
        return false;

    if (m_impl->m_page_history.size ()) {
        m_impl->m_page_size = m_impl->m_page_history.back ();
        m_impl->m_page_history.pop_back ();
    }

    if (m_impl->m_current_page_start >= m_impl->m_page_size)
        m_impl->m_current_page_start -= m_impl->m_page_size;
    else
        m_impl->m_current_page_start = 0;

    if (m_impl->m_cursor_pos >= m_impl->m_page_size)
        m_impl->m_cursor_pos -= m_impl->m_page_size;
    else
        m_impl->m_cursor_pos = 0;

    if (m_impl->m_cursor_pos < m_impl->m_current_page_start)
        m_impl->m_cursor_pos = m_impl->m_current_page_start;
    else if (m_impl->m_cursor_pos >= m_impl->m_current_page_start + get_current_page_size ())
        m_impl->m_cursor_pos = m_impl->m_current_page_start + get_current_page_size () - 1;

    return true;
}

bool
LookupTable::page_down ()
{
    if (m_impl->m_current_page_start + m_impl->m_page_size >= number_of_entries ())
        return false;

    m_impl->m_current_page_start += m_impl->m_page_size;
    m_impl->m_page_history.push_back (m_impl->m_page_size);

    m_impl->m_cursor_pos += m_impl->m_page_size;
    
    if (m_impl->m_cursor_pos < m_impl->m_current_page_start)
        m_impl->m_cursor_pos = m_impl->m_current_page_start;
    else if (m_impl->m_cursor_pos >= m_impl->m_current_page_start + get_current_page_size ())
        m_impl->m_cursor_pos = m_impl->m_current_page_start + get_current_page_size () - 1;

    return true;
}

bool
LookupTable::cursor_up ()
{
    if (m_impl->m_cursor_pos <= 0) return false;

    if (!m_impl->m_cursor_visible)
        m_impl->m_cursor_visible = true;

    m_impl->m_cursor_pos --;
    
    if (m_impl->m_cursor_pos < m_impl->m_current_page_start) {
        page_up ();
        m_impl->m_cursor_pos = m_impl->m_current_page_start + get_current_page_size () - 1;
    }
    return true;
}

bool
LookupTable::cursor_down ()
{
    if (m_impl->m_cursor_pos + 1 >= number_of_entries ())
        return false;

    if (!m_impl->m_cursor_visible)
        m_impl->m_cursor_visible = true;

    ++m_impl->m_cursor_pos;
    
    if (m_impl->m_cursor_pos >= m_impl->m_current_page_start + get_current_page_size ()) {
        page_down ();
        m_impl->m_cursor_pos = m_impl->m_current_page_start;
    }
    return true;
}

void
LookupTable::show_cursor (bool show)
{
    m_impl->m_cursor_visible = show;
}

bool
LookupTable::is_cursor_visible () const
{
    return m_impl->m_cursor_visible;
}

KeyEvent
LookupTable::get_page_index_key (int page_index) const
{
    if (page_index >= 0 && page_index < get_current_page_size ())
        return m_impl->m_page_index_keys [page_index];
    return KeyEvent ();
}

KeyEvent
LookupTable::get_page_up_key () const
{
    return m_impl->m_page_up_key;
}

KeyEvent
LookupTable::get_page_down_key () const
{
    return m_impl->m_page_down_key;
}

WideString
LookupTable::get_content_in_page (int page_index) const
{
    if (page_index >= 0 && page_index < get_current_page_size ())
        return get_content (page_index + m_impl->m_current_page_start);
    return WideString ();
}

void
LookupTable::clear ()
{
    m_impl->m_current_page_start = 0;
    m_impl->m_cursor_pos = 0;
    std::vector <int> ().swap (m_impl->m_page_history);
}

//implementation of CommonLookupTable
CommonLookupTable::CommonLookupTable (int page_size)
    : LookupTable (page_size, KeyEvent (), KeyEvent (), std::vector<KeyEvent> ())
{
}

CommonLookupTable::CommonLookupTable (int                          page_size,
                                      const std::vector<KeyEvent> &page_index_keys)
    : LookupTable (page_size, KeyEvent (), KeyEvent (), page_index_keys)
{
}

CommonLookupTable::CommonLookupTable (int                          page_size,
                                      const KeyEvent&              page_up_key,
                                      const KeyEvent&              page_down_key,
                                      const std::vector<KeyEvent> &page_index_keys)
    : LookupTable (page_size, page_up_key, page_down_key, page_index_keys)
{
}

uint32
CommonLookupTable::number_of_entries () const
{
    return m_index.size ();
}

bool
CommonLookupTable::append_entry (const WideString & entry)
{
    if (entry.length () == 0)
        return false;

    m_index.push_back (m_buffer.size ());
    m_buffer.insert (m_buffer.end (), entry.begin (), entry.end ());

    return true;
}

bool
CommonLookupTable::append_entry (ucs4_t entry)
{
    if (entry == 0)
        return false;

    m_index.push_back (m_buffer.size ());
    m_buffer.push_back (entry);

    return true;
}

WideString
CommonLookupTable::get_content (int index) const
{
    if (index < 0 || index >= number_of_entries ())
        return WideString ();

    std::vector <ucs4_t>::const_iterator start, end;

    start = m_buffer.begin () + m_index [index];

    if (index < number_of_entries () - 1)
        end = m_buffer.begin () + m_index [index+1];
    else
        end = m_buffer.end ();

    return WideString (start, end);
}

void
CommonLookupTable::clear ()
{
    LookupTable::clear ();
    std::vector <ucs4_t> ().swap (m_buffer);
    std::vector <uint32> ().swap (m_index);
}

} // namespace scim

/*
vi:ts=4:nowrap:ai:expandtab
*/
