/** @file scim_lookup_table.h
 * @brief definition of LookupTable classes.
 */

/*
 * Smart Common Input Method
 * 
 * Copyright (c) 2004 James Su <suzhe@turbolinux.com.cn>
 * Copyright (c) 2003 James Su <suzhe@turbolinux.com.cn>
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_lookup_table.h,v 1.19 2004/03/01 03:47:34 suzhe Exp $
 */

#ifndef __SCIM_LOOKUP_TABLE_H
#define __SCIM_LOOKUP_TABLE_H

namespace scim {
/**
 * @addtogroup Helper
 * @{
 */

#define SCIM_LOOKUP_TABLE_MAX_PAGESIZE  16

/**
 * @brief The base class of lookup table.
 *
 * This is abstract class and cannot store data.
 * Server should use its derivation class.
 * This class is the interface that uses within FrontEnd class.
 */
class LookupTable
{
    class LookupTableImpl;

    LookupTableImpl * m_impl;

    LookupTable (const LookupTable &);
    const LookupTable & operator= (const LookupTable &);

public:
    /**
     * @brief Constructor
     * @param page_size - the maximum page size, can be set by set_page_size() later.
     * @param page_up_key - the key event to page up.
     * @param page_down_key - the key event to page down.
     * @param page_index_keys - the key events to select items in current page directly.
     */
    LookupTable (int                          page_size,
                 const KeyEvent&              page_up_key,
                 const KeyEvent&              page_down_key,
                 const std::vector<KeyEvent> &page_index_keys);

    /**
     * @brief Virtual destructor.
     */
    virtual ~LookupTable ();

    /**
     * @brief Set the key events of page indexes.
     * @param page_index_keys - the key events to select items in current page directly.
     */
    void set_page_index_keys (const std::vector<KeyEvent> &page_index_keys);

    /**
     * @brief Set the key events of page up and down.
     * @param page_up - the key event to page up.
     * @param page_down - the key event to page down.
     */
    void set_page_updown_keys (const KeyEvent &page_up,
                               const KeyEvent &page_down);

    /**
     * @brief Set the maximum page size.
     * @param page_size - the max page size of the table.
     */
    void set_page_size (int page_size); 

    /**
     * @brief Get the maximum page size.
     * @return the max page size of the table.
     */
    int get_page_size () const;

    /**
     * @brief Get current page size,
     * @return the page size of current page.It can be less than the max page size.
     */
    int get_current_page_size () const;

    /**
     * @brief Get the start index of current page.
     * @return the start item index of current page, starting from 0.
     */
    int get_current_page_start () const;

    /**
     * @brief Get current cursor position.
     * @return the cursor position in the table, starting from 0.
     */
    int get_cursor_pos () const;

    /**
     * @brief Get the cursor position in current page.
     * @return the cursor position in current page,
     *         equals to get_cursor_pos () - get_current_page_start ().
     */
    int get_cursor_pos_in_page () const;

    /**
     * @brief Flip to the previous page.
     * @return true if success, false if it's already in the first page.
     */
    bool page_up ();

    /**
     * @brief Flip to the next page.
     * @return true if success, false if it's already in the last page.
     */
    bool page_down ();

    /**
     * @brief Move cursor position to the previous entry.
     * @return true if success, false if it's already at the first entry.
     */
    bool cursor_up ();

    /**
     * @brief Move cursor position to the next entry.
     * @return true if success. false if it's already at the last entry.
     */
    bool cursor_down ();

    /**
     * @brief Set the cursor visibility.
     * @param show - true to show the cursor.
     */
    void show_cursor (bool show=true);

    /**
     * @brief Check if the cursor is visible.
     * @return true if the cursor should be shown.
     */
    bool is_cursor_visible () const;
    
    /**
     * @brief Get the key event of a page index.
     * @param page_index - the page index, 0 to (max page size - 1).
     * @return the corresponding key event of page_index. 
     */
    KeyEvent get_page_index_key (int page_index) const;

    /**
     * @brief Get the key event of page up.
     * @return the page up key event.
     */
    KeyEvent get_page_up_key () const;

    /**
     * @brief Get key event of page down.
     * @return the page down key event.
     */
    KeyEvent get_page_down_key () const;

    /**
     * @brief Get the content of an entry in current page.
     *
     * @param page_index - the index in current page.
     * @return the content.
     * 
     * @sa get_content
     */
    WideString get_content_in_page (int page_index) const;

public:
    /**
     * @name Pure Virtual functions.
     * These functions should be implemented in derivation classes.
     *
     * @{
     */

    /**
     * @brief Get the content of an entry.
     * @param index - the index in the lookup table.
     * @return the content.
     */
    virtual WideString get_content (int index) const = 0;

    /**
     * @brief Return the number of entries in this table.
     * @return the number of entries currently in this table.
     */
    virtual uint32 number_of_entries () const = 0;

    /**
     * @brief Clear the table.
     */
    virtual void clear () = 0;

    /**
     * @}
     */
};


/**
 * @brief A common lookup table class.
 *
 * This class implements the LookupTable interface in a common way.
 *
 */
class CommonLookupTable : public LookupTable
{
    std::vector <ucs4_t> m_buffer;
    std::vector <uint32> m_index;

public:
    CommonLookupTable (int                          page_size,
                       const KeyEvent&              page_up_key,
                       const KeyEvent&              page_down_key,
                       const std::vector<KeyEvent> &page_index_keys);

    CommonLookupTable (int                          page_size,
                       const std::vector<KeyEvent> &page_index_keys);

    CommonLookupTable (int page_size = 10);

    virtual WideString get_content (int index) const;

    virtual uint32 number_of_entries () const;

    virtual void clear ();

public:
    /**
     * @brief Append a string into the table.
     * @param entry - an entry to be added into the table.
     * @return true if success.
     */
    bool append_entry (const WideString& entry);

    /**
     * @brief Append a single char into the table.
     * @param entry - an entry to be added into the table.
     * @return true if success.
     */
    bool append_entry (ucs4_t entry);
};

/** @} */

} // namespace scim

#endif //__SCIM_LOOKUP_TABLE_H

/*
vi:ts=4:nowrap:ai:expandtab
*/
