/** @file scim_server.cpp
 *  @brief Implementation of class ServerInstanceBase.
 */

/*
 * Smart Common Input Method
 * 
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_server.cpp,v 1.25 2003/12/31 05:09:25 suzhe Exp $
 *
 */

#define Uses_SCIM_SERVER
#include "scim_private.h"
#include "scim.h"

#define SCIM_KEYBOARD_ICON_FILE            (SCIM_ICONDIR "/keyboard.png")

namespace scim {

ServerFactoryBase::~ServerFactoryBase ()
{
}

bool
ServerFactoryBase::validate_encoding (const String& encoding) const
{
    if (encoding == "UTF-8")
        return true;

    for (unsigned int i=0; i<m_encoding_list.size (); ++i)
        if (m_encoding_list [i] == encoding)
            return true;

    return false;
}

bool
ServerFactoryBase::validate_locale (const String& locale) const
{
    for (unsigned int i=0; i<m_locale_list.size (); ++i)
        if (m_locale_list [i] == locale)
            return true;

    if (scim_get_locale_encoding (locale) == "UTF-8")
        return true;

    return false;
}

String
ServerFactoryBase::get_encodings () const
{
    return scim_combine_string_list (m_encoding_list);
}

String
ServerFactoryBase::get_locales () const
{
    return scim_combine_string_list (m_locale_list);
}

String
ServerFactoryBase::get_default_encoding () const
{
    if (m_encoding_list.size ())
        return m_encoding_list [0];
    return String ("UTF-8");
}

String
ServerFactoryBase::get_default_locale () const
{
    if (m_locale_list.size ())
        return m_locale_list [0];
    return String ("");
}

void
ServerFactoryBase::set_locales (const String& locales)
{
    m_locale_list.clear ();
    m_encoding_list.clear ();

    if (locales.size () == 0) return;

    String locale;
    std::vector <String> lc_list;

    scim_split_string_list (lc_list, locales);

    for (unsigned int i=0; i<lc_list.size (); ++i) {
        locale = scim_validate_locale (lc_list [i]);
        if (locale.length ()) {
            m_locale_list.push_back (locale);
            m_encoding_list.push_back (scim_get_locale_encoding (locale));
        }
    }
}

ServerInstanceBase::ServerInstanceBase (ServerFactoryBase *factory,
                                        const String& encoding,
                                        int id)
    : m_factory (factory),
      m_encoding (encoding),
      m_id (id)
{
    if (!m_factory.null ()) {
        if (!m_factory->validate_encoding (encoding)) {
            m_encoding = m_factory->get_default_encoding ();
        }
    } else {
        m_encoding = String ("UTF-8");
    }
}

ServerInstanceBase::~ServerInstanceBase ()
{
}

bool
ServerInstanceBase::set_encoding (const String& encoding)
{
    if (!m_factory.null () && m_factory->validate_encoding (encoding)) {
        m_encoding = encoding;
        reset ();
        return true;
    }
    return false;
}

String
ServerInstanceBase::get_encoding () const
{
    return m_encoding;
}

int
ServerInstanceBase::get_id () const
{
    return m_id;
}

String
ServerInstanceBase::get_factory_uuid () const
{
    if (!m_factory.null ())
        return m_factory->get_uuid ();

    return String ();
}

Connection
ServerInstanceBase::signal_connect_show_preedit_string (ServerSlotVoid *slot)
{
    return m_signal_show_preedit_string.connect (slot);
}

Connection
ServerInstanceBase::signal_connect_show_status_string (ServerSlotVoid *slot)
{
    return m_signal_show_status_string.connect (slot);
}

Connection
ServerInstanceBase::signal_connect_show_aux_string (ServerSlotVoid *slot)
{
    return m_signal_show_aux_string.connect (slot);
}

Connection
ServerInstanceBase::signal_connect_show_lookup_table (ServerSlotVoid *slot)
{
    return m_signal_show_lookup_table.connect (slot);
}

Connection
ServerInstanceBase::signal_connect_hide_preedit_string (ServerSlotVoid *slot)
{
    return m_signal_hide_preedit_string.connect (slot);
}

Connection
ServerInstanceBase::signal_connect_hide_status_string (ServerSlotVoid *slot)
{
    return m_signal_hide_status_string.connect (slot);
}

Connection
ServerInstanceBase::signal_connect_hide_aux_string (ServerSlotVoid *slot)
{
    return m_signal_hide_aux_string.connect (slot);
}

Connection
ServerInstanceBase::signal_connect_hide_lookup_table (ServerSlotVoid *slot)
{
    return m_signal_hide_lookup_table.connect (slot);
}

Connection
ServerInstanceBase::signal_connect_update_preedit_caret (ServerSlotInt *slot)
{
    return m_signal_update_preedit_caret.connect (slot);
}

Connection
ServerInstanceBase::signal_connect_update_preedit_string (ServerSlotWideStringAttributeList *slot)
{
    return m_signal_update_preedit_string.connect (slot);
}

Connection
ServerInstanceBase::signal_connect_update_status_string (ServerSlotWideStringAttributeList *slot)
{
    return m_signal_update_status_string.connect (slot);
}

Connection
ServerInstanceBase::signal_connect_update_aux_string (ServerSlotWideStringAttributeList *slot)
{
    return m_signal_update_aux_string.connect (slot);
}

Connection
ServerInstanceBase::signal_connect_update_lookup_table (ServerSlotLookupTable *slot)
{
    return m_signal_update_lookup_table.connect (slot);
}

Connection
ServerInstanceBase::signal_connect_commit_string (ServerSlotWideString *slot)
{
    return m_signal_commit_string.connect (slot);
}

Connection
ServerInstanceBase::signal_connect_forward_keyevent (ServerSlotKeyEvent *slot)
{
    return m_signal_forward_keyevent.connect (slot);
}

Connection
ServerInstanceBase::signal_connect_update_full_width_punctuation (ServerSlotBool *slot)
{
    return m_signal_update_full_width_punctuation.connect (slot);
}

Connection
ServerInstanceBase::signal_connect_update_full_width_letter (ServerSlotBool *slot)
{
    return m_signal_update_full_width_letter.connect (slot);
}


// implementation of DummyServer
DummyServerFactory::DummyServerFactory ()
{
    set_locales ("C");
}

DummyServerFactory::~DummyServerFactory ()
{
}

WideString
DummyServerFactory::get_name () const
{
    return utf8_mbstowcs (_("English"));
}

WideString
DummyServerFactory::get_authors () const
{
    return WideString ();
}

WideString
DummyServerFactory::get_credits () const
{
    return WideString ();
}

WideString
DummyServerFactory::get_help () const
{
    return WideString ();
}

String
DummyServerFactory::get_uuid () const
{
    return String ("0148bcec-850d-45f2-ba95-a493bb31492e");
}

String
DummyServerFactory::get_icon_file () const
{
    return String (SCIM_KEYBOARD_ICON_FILE);
}

bool
DummyServerFactory::validate_encoding (const String& encoding) const
{
    return true;
}

bool
DummyServerFactory::validate_locale (const String& locale) const
{
    return true;
}

ServerInstancePointer
DummyServerFactory::create_server_instance (const String& encoding, int id)
{
    return new DummyServerInstance (this, encoding, id);
}

DummyServerInstance::DummyServerInstance (DummyServerFactory *factory,
                                        const String& encoding,
                                        int id)
    : ServerInstanceBase (factory, encoding, id)
{
}

DummyServerInstance::~DummyServerInstance ()
{
}

bool
DummyServerInstance::process_key_event (const KeyEvent& key)
{
    return false;
}

void
DummyServerInstance::select_lookup_table (unsigned int /*item*/)
{
}

void
DummyServerInstance::move_preedit_caret (unsigned int /*pos*/)
{
}

void
DummyServerInstance::reset ()
{
}

void
DummyServerInstance::focus_in ()
{
    update_full_width_punctuation (false);
    update_full_width_letter (false);

    show_status_string ();
    update_status_string (utf8_mbstowcs (_("En")));
}

void
DummyServerInstance::focus_out ()
{
}

void
DummyServerInstance::toggle_full_width_punctuation ()
{
}

void
DummyServerInstance::toggle_full_width_letter ()
{
}

void
DummyServerInstance::toggle_input_status ()
{
}

void
DummyServerInstance::update_lookup_table_page_size (unsigned int /*page_size*/)
{
}

} // namespace scim

/*
vi:ts=4:nowrap:ai:expandtab
*/
