/**
 * @file scim_server.h
 * @brief Defines scim::ServerFactoryBase and scim::ServerInstanceBase interfaces.
 *
 * scim::ServerFactoryBase and scim::ServerInstanceBase are the most important
 * part of SCIM platform.
 *
 * These interfaces are for writing input method server modules.
 */

/* 
 * Smart Common Input Method
 * 
 * Copyright (c) 2004 James Su <suzhe@turbolinux.com.cn>
 * Copyright (c) 2003 James Su <suzhe@turbolinux.com.cn>
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_server.h,v 1.27 2004/02/12 09:40:12 suzhe Exp $
 */

#ifndef __SCIM_SERVER_H
#define __SCIM_SERVER_H

namespace scim {
/**
 * @addtogroup Server
 * The base classes for input method server modules.
 * @{
 */

/**
 * @brief An exception class to hold Server related errors.
 *
 * scim::ServerBase and its derived classes must throw
 * scim::ServerError object when error.
 */
class ServerError: public Exception
{
public:
    ServerError (const String& what_arg)
        : Exception (String("scim::Server: ") + what_arg) { }
};

class ServerFactoryBase;
class ServerInstanceBase;

/**
 * @typedef typedef Pointer <ServerFactoryBase> ServerFactoryPointer;
 *
 * A smart pointer for scim::ServerFactoryBase and its derived classes.
 */
typedef Pointer <ServerFactoryBase>  ServerFactoryPointer;

/**
 * @typedef typedef Pointer <ServerInstanceBase> ServerInstancePointer;
 *
 * A smart pointer for scim::ServerInstanceBase and its derived classes.
 */
typedef Pointer <ServerInstanceBase> ServerInstancePointer;

typedef Slot1<void, ServerInstanceBase*>
        ServerSlotVoid;

typedef Slot2<void, ServerInstanceBase*,int>
        ServerSlotInt;

typedef Slot2<void, ServerInstanceBase*,bool>
        ServerSlotBool;

typedef Slot2<void, ServerInstanceBase*,const WideString&>
        ServerSlotWideString;

typedef Slot2<void, ServerInstanceBase*,const KeyEvent&>
        ServerSlotKeyEvent;

typedef Slot2<void, ServerInstanceBase*,const LookupTable&>
        ServerSlotLookupTable;

typedef Slot3<void, ServerInstanceBase*,const WideString&,const AttributeList&>
        ServerSlotWideStringAttributeList;


typedef Signal1<void, ServerInstanceBase*>
        ServerSignalVoid;

typedef Signal2<void, ServerInstanceBase*,int>
        ServerSignalInt;

typedef Signal2<void, ServerInstanceBase*,bool>
        ServerSignalBool;

typedef Signal2<void, ServerInstanceBase*,const WideString&>
        ServerSignalWideString;

typedef Signal2<void, ServerInstanceBase*,const KeyEvent&>
        ServerSignalKeyEvent;

typedef Signal2<void, ServerInstanceBase*,const LookupTable&>
        ServerSignalLookupTable;

typedef Signal3<void, ServerInstanceBase*,const WideString&,const AttributeList&>
        ServerSignalWideStringAttributeList;

/**
 * @brief The base class of the real input methods' ServerFactory classes.
 *
 * Each input method should implement a class derived from scim::ServerFactoryBase,
 * which takes charge of holding shared data, creating ServerInstances etc.
 */
class ServerFactoryBase : public ReferencedObject
{
    std::vector<String> m_encoding_list;
    std::vector<String> m_locale_list;

public:
    /**
     * @brief Virtual destructor.
     */
    virtual ~ServerFactoryBase ();

    /**
     * @name Pure virtual members.
     *
     * These member functions must be implemented in derived classes.
     *
     * @{
     */

    /**
     * @brief Get the name of this input method server.
     *
     * This name should be a localized string.
     *
     * @return A WideString containing the name.
     */
    virtual WideString  get_name () const = 0;

    /**
     * @brief Get the authors of this input method server.
     *
     * This string should be a localized string.
     *
     * @return A WideString containing a list of the authors' name.
     */
    virtual WideString  get_authors () const = 0;

    /**
     * @brief Get the credits information of this input method server.
     *
     * This string should be a localized string.
     *
     * @return A WideString containing the credits information.
     */
    virtual WideString  get_credits () const = 0;

    /**
     * @brief Get the help information of this input method server.
     *
     * This string should be a localized string.
     *
     * @return A WideString containing the help information.
     */
    virtual WideString  get_help () const = 0;

    /**
     * @brief Get the UUID of this input method server.
     *
     * Each input method server has an unique UUID to
     * distinguish itself from other servers.
     *
     * You may use uuidgen command shipped with e2fsprogs package to generate this UUID.
     *
     * @return A String containing an unique UUID.
     */
    virtual String      get_uuid () const = 0;

    /**
     * @brief Get the icon file path of this input method server.
     *
     * @return A String containing the icon file path on the local filesystem.
     */
    virtual String      get_icon_file () const = 0;

    /**
     * @brief Create a new ServerInstance object.
     *
     * This method creates a new scim::ServerInstanceBase object with given encoding and id.
     *
     * @param encoding - the working encoding.
     * @param id - the instance id, should be unique.
     * @return A smart pointer points to this new ServerInstance object.
     */
    virtual ServerInstancePointer create_server_instance (const String& encoding, int id = -1) = 0;
    /**
     * @}
     */

    /**
     * @brief Check if a encoding is supported by this ServerFactory.
     *
     * The default implementation of this virtual function validates the
     * encoding against the locale list set by method set_locales.
     * 
     * It should be enough in most case.
     *
     * @param encoding - the encoding name to be checked.
     * @return true if the encoding is supported, otherwise false.
     */
    virtual bool validate_encoding (const String& encoding) const; 

    /**
     * @brief Check if a locale is supported by this ServerFactory.
     *
     * The default implementation of this virtual function validates the
     * locale against the locale list set by method set_locales.
     * 
     * It should be enough in most case.
     *
     * @param locale - the locale name to be checked.
     * @return true if the locale is supported, otherwise false.
     */
    virtual bool validate_locale (const String& locale) const;

    /**
     * @brief Get a list of all supported encodings, separated by comma.
     *
     * @return A comma separated encoding list.
     */
    String get_encodings () const;

    /**
     * @brief Get a list of all supported locales, separated by comma.
     * 
     * @return A comma separated locale list.
     */
    String get_locales () const;

    /**
     * @brief Get the default encoding of this input method server.
     *
     * The default encoding is the first locale's encoding in the locale list,
     * which is set by method set_locales.
     *
     * @return The default encoding name.
     */
    String get_default_encoding () const;

    /**
     * @brief Get the default locale of this input method server.
     *
     * The default locale is the first locale in the locale list,
     * which is set by method set_locales.
     *
     * @return The default locale name.
     */
    String get_default_locale () const;

protected:
    /**
     * @brief Set the locales supported by this input method server.
     *
     * This method should be called within the constructors of the derived classes.
     *
     * @param locales - a comma separated list containing all valid locales
     *                  should be supported by this input method server.
     *                  The first locale is the default one.
     */
    void set_locales (const String &locales);
};

/**
 * @brief The base class of the real input methods' ServerInstance classes.
 * 
 * Each input method should implement a class derived from scim::ServerInstanceBase,
 * which takes charge of recording Input Context status and processing user input events.
 */
class ServerInstanceBase : public ReferencedObject
{
    ServerFactoryPointer         m_factory;
    String                       m_encoding;

    ServerSignalVoid             m_signal_show_preedit_string;
    ServerSignalVoid             m_signal_show_status_string;
    ServerSignalVoid             m_signal_show_aux_string;
    ServerSignalVoid             m_signal_show_lookup_table;

    ServerSignalVoid             m_signal_hide_preedit_string;
    ServerSignalVoid             m_signal_hide_status_string;
    ServerSignalVoid             m_signal_hide_aux_string;
    ServerSignalVoid             m_signal_hide_lookup_table;

    ServerSignalInt              m_signal_update_preedit_caret;
    ServerSignalWideStringAttributeList m_signal_update_preedit_string;
    ServerSignalWideStringAttributeList m_signal_update_status_string;
    ServerSignalWideStringAttributeList m_signal_update_aux_string;
    ServerSignalWideString       m_signal_commit_string;
    ServerSignalLookupTable      m_signal_update_lookup_table;
    ServerSignalBool             m_signal_update_full_width_punctuation; 
    ServerSignalBool             m_signal_update_full_width_letter; 

    ServerSignalKeyEvent         m_signal_forward_keyevent;

    int m_id;

public:
    /**
     * @brief Constructor.
     *
     * @param factory - the factory which creates this instance.
     * @param encoding - the working encoding.
     * @param id - the unique id of this instance.
     */
    ServerInstanceBase (ServerFactoryBase *factory,
                        const String &encoding,
                        int id = -1);

    /**
     * @brief Virtual destructor.
     */
    virtual ~ServerInstanceBase ();

    /**
     * @brief Set the working encoding for this instance.
     *
     * One server instance can only support one encoding at the same time.
     * This encoding must be supported by the ServerFactory as well.
     * 
     * @return true if the encoding is supported, otherwise false.
     */
    bool set_encoding (const String& encoding);

    /**
     * @brief Get the working encoding of this instance.
     *
     * @return The current working encoding.
     */
    String get_encoding () const;

    /**
     * @brief Get the unique id of this instance.
     *
     * @return The id of this instance.
     */
    int get_id () const;

    /**
     * @brief Get the UUID of the server factory.
     *
     * @return The UUID string of the server factory.
     */
    String get_factory_uuid () const;

public:
    /**
     * @name Signal connection functions.
     *
     * These functions are used by FrontEnds to connect their corresponding slots to
     * this ServerInstance's signals.
     *
     * @{
     */
    Connection signal_connect_show_preedit_string   (ServerSlotVoid *slot);
    Connection signal_connect_show_status_string    (ServerSlotVoid *slot);
    Connection signal_connect_show_aux_string       (ServerSlotVoid *slot);
    Connection signal_connect_show_lookup_table     (ServerSlotVoid *slot);
    Connection signal_connect_hide_preedit_string   (ServerSlotVoid *slot);
    Connection signal_connect_hide_status_string    (ServerSlotVoid *slot);
    Connection signal_connect_hide_aux_string       (ServerSlotVoid *slot);
    Connection signal_connect_hide_lookup_table     (ServerSlotVoid *slot);
    Connection signal_connect_update_preedit_caret  (ServerSlotInt *slot);
    Connection signal_connect_update_preedit_string (ServerSlotWideStringAttributeList *slot);
    Connection signal_connect_update_status_string  (ServerSlotWideStringAttributeList *slot);
    Connection signal_connect_update_aux_string     (ServerSlotWideStringAttributeList *slot);
    Connection signal_connect_update_lookup_table   (ServerSlotLookupTable *slot);
    Connection signal_connect_commit_string         (ServerSlotWideString *slot);
    Connection signal_connect_forward_keyevent      (ServerSlotKeyEvent *slot);
    Connection signal_connect_update_full_width_punctuation (ServerSlotBool *slot);
    Connection signal_connect_update_full_width_letter      (ServerSlotBool *slot);
    /** @} */

public:
    /**
     * @name Action functions.
     *
     * These functions will be called by FrontEnds to send events to
     * this ServerInstance.
     *
     * @{
     */

    /**
     * @brief Process a key event.
     *
     * @param key - the key event to be processed.
     * @return true if the event is processed, otherwise the event
     *         is not processed and should be forward to client application.
     */
    virtual bool process_key_event (const KeyEvent& key) = 0;

    /**
     * @brief Move the preedit caret in the preedit string.
     *
     * @param pos - the new position that user requested.
     */
    virtual void move_preedit_caret (unsigned int pos) = 0;

    /**
     * @brief Select the item in current lookup table.
     *
     * @param item - the item to be selected in current page.
     */
    virtual void select_lookup_table (unsigned int item) = 0;

    /**
     * @brief Update the page size of current lookup table.
     *
     * In the next time, the lookup table should page down by
     * this size.
     *
     * @param page_size - the new size of current page.
     */
    virtual void update_lookup_table_page_size (unsigned int page_size) = 0;

    /**
     * @brief Reset this server instance.
     *
     * All status of this server instance should be reset,
     * including the working encoding.
     */
    virtual void reset () = 0;

    /**
     * @brief Focus in this server instance.
     *
     * This function should update/show/hide the status area,
     * preedit area and lookup table, and update the
     * full width punctuation/letter state.
     */
    virtual void focus_in () = 0;

    /**
     * @brief Focus out this server instance.
     */
    virtual void focus_out () = 0;

    /**
     * @brief Toggle full/half width punctuation state.
     */
    virtual void toggle_full_width_punctuation () = 0;

    /**
     * @brief Toggle full/half width letter state.
     */
    virtual void toggle_full_width_letter () = 0;

    /**
     * @brief Toggle the current input status of this server instance.
     *
     * This action will be called when the status button
     * is pressed by user. It indicates that user want to
     * switch among the input modes, eg. Chinese/English etc.
     */
    virtual void toggle_input_status () = 0;

    /** @} */

protected:
    /**
     * @name Signal activation functions
     * 
     * These functions should be called by derived classes
     * to fire the corresponding signals. The FrontEnd
     * connected to those signals will receive and process them.
     *
     * @{
     */

    /**
     * @brief Show the preedit string area.
     *
     * The preedit string should be updated by calling
     * update_preedit_string before or right after this call.
     */
    void show_preedit_string () {
        m_signal_show_preedit_string (this);
    }

    /**
     * @brief Show the status string area.
     *
     * The status string should be updated by calling
     * update_status_string before or right after this call.
     */
    void show_status_string () {
        m_signal_show_status_string (this);
    }
    
    /**
     * @brief Show the aux string area.
     *
     * The aux string should be updated by calling
     * update_aux_string before or right after this call.
     *
     * The aux string can contain any additional information whatever
     * the input method server want.
     */
    void show_aux_string () {
        m_signal_show_aux_string (this);
    }
    
    /**
     * @brief Show the lookup table area.
     *
     * The lookup table should be updated by calling
     * update_lookup_table before or right after this call.
     */
    void show_lookup_table () {
        m_signal_show_lookup_table (this);
    }

    /**
     * @brief Hide the preedit string area.
     */
    void hide_preedit_string () {
        m_signal_hide_preedit_string (this);
    }

    /**
     * @brief Hide the status string area.
     */
    void hide_status_string () {
        m_signal_hide_status_string (this);
    }
 
    /**
     * @brief Hide the aux string area.
     */
    void hide_aux_string () {
        m_signal_hide_aux_string (this);
    }

    /**
     * @brief Hide the lookup table area.
     */
    void hide_lookup_table () {
        m_signal_hide_lookup_table (this);
    }

    /**
     * @brief Update the preedit caret position in the preedit string.
     *
     * @param caret - the new position of the preedit caret.
     */
    void update_preedit_caret (int caret) {
        m_signal_update_preedit_caret (this,caret);
    }

    /**
     * @brief Update the content of the preedit string,
     * 
     * @param str - the string content
     * @param attrs - the string attributes
     */
    void update_preedit_string (const WideString& str,
                                const AttributeList& attrs = AttributeList ()) {
        m_signal_update_preedit_string (this,str,attrs);
    }

    /**
     * @brief Update the content of the status string,
     * 
     * @param str - the string content
     * @param attrs - the string attributes
     */
    void update_status_string  (const WideString& str,
                                const AttributeList& attrs = AttributeList ()) {
        m_signal_update_status_string (this,str,attrs);
    }

    /**
     * @brief Update the content of the aux string,
     * 
     * @param str - the string content
     * @param attrs - the string attribute
     */
    void update_aux_string (const WideString& str,
                            const AttributeList& attrs = AttributeList ()) {
        m_signal_update_aux_string (this,str,attrs);
    }

    /**
     * @brief Update the content of the lookup table,
     *
     * FrontEnd may reduce the page size of the table
     * according to screen resolution. If the page size
     * is changed, FrontEnd will inform this server instance
     * by calling update_lookup_table_page_size method.
     *
     * @param table - the new LookupTable
     */
    void update_lookup_table (const LookupTable& table) {
        m_signal_update_lookup_table (this,table);
    }

    /**
     * @brief Commit a string to the client application.
     *
     * The preedit string should be hid before calling
     * this method. Otherwise the clients which use
     * OnTheSpot input mode will flicker annoyingly.
     *
     * @param str - the string to be committed.
     */
    void commit_string (const WideString& str) {
        m_signal_commit_string (this,str);
    }

    /**
     * @brief Forward a key event to the client application.
     *
     * @param key - the key event to be forwarded.
     */
    void forward_keyevent (const KeyEvent& key) {
        m_signal_forward_keyevent (this,key);
    }

    /**
     * @brief Update the punctuation input mode indicator.
     *
     * @param full - true for full width mode, false for half width mode.
     */
    void update_full_width_punctuation (bool full) {
        m_signal_update_full_width_punctuation (this,full);
    }

    /**
     * @brief Update the letter input mode indicator.
     *
     * @param full - true for full width mode, false for half width mode.
     */
    void update_full_width_letter (bool full) {
        m_signal_update_full_width_letter (this,full);
    }

    /** @} */
};

class DummyServerFactory : public ServerFactoryBase
{
public:
    DummyServerFactory ();
    virtual ~DummyServerFactory ();

    virtual WideString  get_name () const;
    virtual WideString  get_authors () const;
    virtual WideString  get_credits () const;
    virtual WideString  get_help () const;

    virtual String      get_uuid () const;

    virtual String      get_icon_file () const;

    virtual bool validate_encoding (const String& encoding) const;
    virtual bool validate_locale (const String& locale) const;

    virtual ServerInstancePointer create_server_instance (const String& encoding, int id = -1);
};

class DummyServerInstance : public ServerInstanceBase
{
public:
    DummyServerInstance (DummyServerFactory *factory,
                         const String& encoding,
                         int id = -1);

    virtual ~DummyServerInstance ();

    virtual bool process_key_event (const KeyEvent& key);
    virtual void move_preedit_caret (unsigned int pos);
    virtual void select_lookup_table (unsigned int item);
    virtual void update_lookup_table_page_size (unsigned int page_size);
    virtual void reset ();
    virtual void focus_in ();
    virtual void focus_out ();
    virtual void toggle_full_width_punctuation ();
    virtual void toggle_full_width_letter ();
    virtual void toggle_input_status ();
};

/**  @} */

} // namespace scim

#endif //__SCIM_SERVER_H

/*
vi:ts=4:nowrap:ai:expandtab
*/

