/***************************************************************************
 *   copyright           : (C) 2002 by Hendrik Sattler                     *
 *   mail                : post@hendrik-sattler.de                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef _GNU_SOURCE
# define _GNU_SOURCE
#endif
#ifndef _REENTRANT
# define _REENTRANT
#endif

#include <charsets.h>
#include <helper.h>
#include <smspdu.h>
#include <timeincl.h>
#include "smscoding.h"
#include "smsudh.h"

#include <stdlib.h>
#include <string.h>
#include <ctype.h>

void sms_statusreport_match (struct sms** list) {
  struct sms** outgoing;
  struct sms** reports;
  unsigned int l = 0;
  unsigned int o = 0;
  unsigned int r = 0;
  unsigned int rmax = 0;
  struct sms_pdu_data* oc;
  struct sms_pdu_data* rc;

  //sort from list into outgoing and reports
  while (list[l] != NULL) ++l;
  outgoing = mem_alloc((l+1)*sizeof(*outgoing),0);
  reports = mem_alloc((l+1)*sizeof(*reports),0);
  l = 0;
  while (list[l] != NULL) {
    switch (list[l]->decoded->pdu->options.type) {
    case SMS_TYPE_SUBMIT:
      outgoing[o++] = list[l++];
      break;
    case SMS_TYPE_STATUS_REPORT:
      if (list[l]->decoded->pdu->options.sr == 0) {
	//SMS-STATUS-REPORT is answer to a SMS-SUBMIT
	// == 1 would be answer to a SMS-COMMAND
	reports[rmax++] = list[l++];
      }
      break;

    default: ++l; break;
    }
  }
  outgoing[o] = NULL;
  reports[rmax] = NULL;

  //now trying to match all reports to an outgoing message
  for (o = 0; outgoing[o] != NULL; ++o) {
    oc = outgoing[o]->decoded->pdu;
    for (r = 0; r < rmax; ++r) {
      if (reports[r] != NULL) {
	rc = reports[r]->decoded->pdu;
	if (oc->mref == rc->mref &&
	    struct_sms_number_compare(&oc->address,&rc->address)) {
	  //there may be more than one status report per message...
	  if (oc->ud[oc->partnum].ack != NULL) {
	    //...so match the latest one to link with the final status
	    if (oc->ud[oc->partnum].ack->t.value < rc->status->t.value) {
	      oc->ud[oc->partnum].ack->message_parent = NULL;
	      oc->ud[oc->partnum].ack = rc->status;
	      oc->ud[oc->partnum].ack->message_parent = &(oc->ud[oc->partnum]);
	    }
	  } else {
	    oc->ud[oc->partnum].ack = rc->status;
	    oc->ud[oc->partnum].ack->message_parent = &(oc->ud[oc->partnum]);
	  }
	  //disable all previously matched reports in the list
	  reports[r] = NULL;
	}
      }
    }
  }

  mem_realloc(outgoing,0);
  mem_realloc(reports,0);
}

int sms_multipart_merge (struct sms** list) {
  struct sms** multipart; //for multipart matches
  struct sms** results; //for the final list, copied back to list
  struct sms** nondel; //matched messages that need special deletion
  unsigned int l = 0;
  unsigned int m = 0;
  unsigned int r = 0;
  unsigned int n = 0;
  struct sms_pdu_data* c1;
  struct sms_pdu_data* c2;

  if (list == NULL || list[0] == NULL) {
    return 0;
  }

  while (list[l] != NULL) ++l;
  multipart = mem_alloc((l+1)*sizeof(*multipart),0);
  multipart[0] = NULL;
  results = mem_alloc((l+1)*sizeof(*results),0); 
  results[0] = NULL;
  nondel = mem_alloc((l+1)*sizeof(*nondel),0);  
  nondel[0] = NULL;

  for (l = 0; list[l] != NULL; ++l) {
    if (list[l]->decoded != NULL &&
	list[l]->decoded->pdu->parts > 1) {
      c1 = list[l]->decoded->pdu;
      for (m = 0; multipart[m] != NULL; ++m) {
	c2 = multipart[m]->decoded->pdu;
	//criteria for matching the parts
	if (c1->options.type == c2->options.type &&
	    c1->multipart_id == c2->multipart_id &&
	    c1->parts == c2->parts &&
	    struct_sms_number_compare(&c1->address,&c2->address)) {
	  if (c1->partnum == c2->partnum ||
	      (multipart[m]->encoded[c1->partnum] != NULL)) {
	    /* We now have two concatenated short messages
	     * from the same sender with the same id.
	     * In this case, a clean match is not possible,
	     * thus abort the whole process.
	     */
	    //reverting all previous matches
	    for (m = 0; multipart[m] != NULL; ++m) {
	      c2 = multipart[m]->decoded->pdu;
	      for (n = 0; n < c2->parts; ++n) {
		if (c2->partnum != n) {
		  struct_sms_pdu_ud_delete(&c2->ud[n]);
		  struct_sms_pdu_ud_init(&c2->ud[n]);
		  multipart[m]->encoded[n] = NULL;		  
		}
	      }
	    }
	    mem_realloc(multipart,0);
	    mem_realloc(results,0);
	    mem_realloc(nondel,0);
	    fprintf(stderr,"Warning: Duplicated multipart messages id from the same sender detected.\n");
	    return -1;
	  } else {
	    c2->ud[c1->partnum] = c1->ud[c1->partnum];
	    multipart[m]->encoded[c1->partnum] = list[l]->encoded[c1->partnum];
	    nondel[n++] = list[l];
	    nondel[n] = NULL;
	    break;
	  }
	}
      }
      if (multipart[m] == NULL) {
	multipart[m++] = list[l];
	multipart[m] = NULL;
	results[r++] = list[l];
	results[r] = NULL;
      }
    } else {
      results[r++] = list[l];
      results[r] = NULL;
    }
  }

  mem_realloc(multipart,0);
  /* Copy back the results
   */
  for (l = 0; list[l] != NULL; ++l) {
    list[l] = NULL;
  }
  for (r = 0; results[r] != NULL; ++r) {
    list[r] = results[r];
  }
  mem_realloc(results,0);
  /* for nondel list, make sure that encoded messages,
   * message header and text do not get deleted
   */
  for (n = 0; nondel[n] != NULL; ++n) {
    nondel[n]->encoded = NULL;
    nondel[n]->decoded->pdu->ud = NULL;
    struct_sms_delete(nondel[n]);
    mem_realloc(nondel[n],0);
  }
  mem_realloc(nondel,0);
  return 1;
}


enum sms_pdu_type get_sms_pdu_type (enum sms_direction d, uint8_t type) {
  switch (d) {
  case SMS_INCOMING:
    switch (type&3) {
    case 0: return SMS_TYPE_DELIVER;
    case 1: return SMS_TYPE_SUBMIT_REPORT;
    case 2: return SMS_TYPE_STATUS_REPORT;
    default: return SMS_TYPE_DELIVER;
    }
  case SMS_OUTGOING:
    switch (type&3) {
    case 0: return SMS_TYPE_DELIVER_REPORT;
    case 1: return SMS_TYPE_SUBMIT;
    case 2: return SMS_TYPE_COMMAND;
    default: return SMS_TYPE_SUBMIT;
    }
  }
  //never reaches this point
  return SMS_TYPE_DELIVER;
}

void sms_pdu_decode_options (enum sms_direction d, uint8_t type,
			  struct sms_pdu_options* p)
{
  if (p == NULL) return;

  p->type = get_sms_pdu_type(d,type);
  if (type&0x04) {
    if (p->type == SMS_TYPE_DELIVER ||
	p->type == SMS_TYPE_STATUS_REPORT) {
      p->mms = 1;
    } else if (p->type == SMS_TYPE_SUBMIT) {
      p->rd = 1;
    }
  }
  if ((p->type == SMS_TYPE_DELIVER ||
       p->type == SMS_TYPE_SUBMIT ||
       p->type == SMS_TYPE_COMMAND ||
       p->type == SMS_TYPE_STATUS_REPORT) &&
       type&0x20) {
    p->sr = 1;
  }
  if (type&0x40) {
    p->udh_present = 1;
  }
  if ((p->type == SMS_TYPE_DELIVER ||
       p->type == SMS_TYPE_SUBMIT) &&
      type&0x80) {
    p->rp = 1;
  }
}

void sms_pdu_dcs_decode (uint8_t dcs, struct sms_pdu_dcs* p) {
  if (p == NULL) return;
  p->dcs_present = 1;
  if ((dcs&0x8c) == 0x00 || //0xxx 00xx
      (dcs&0xe0) == 0xb0 || //110x xxxx
      (dcs&0xf4) == 0xf0  //1111 x0xx
      /*((dcs&0xf0) >= 0x80 && (dcs&0xf0) <= 0xb0)*/ //reserved
      ) {
    p->encoding = SMS_CHARSET_GSM;
  } else if ((dcs&0x8c) == 0x08 || //0xxx 10xx
	     (dcs&0xf0) == 0xe0    //1110 xxxx
	     ) {
    p->encoding = SMS_CHARSET_UCS2;
  } else {
    p->encoding = SMS_CHARSET_8BIT;
  }

  if ((dcs&0x80) == 0) {
    if (dcs&0x40) p->autodel = 1;
    if (dcs&0x20) p->compressed = 1;
    if (dcs&0x08) {
      p->options = SMS_DCS_OPT_CLASS;
      p->class = dcs&3;
    }
  } else {
    if (dcs >= 0xF0) {
      p->options = SMS_DCS_OPT_CLASS;
      p->class = dcs&3;
    } else if (dcs >= 0xC0) {
      p->options = SMS_DCS_OPT_IND;
      if ((dcs&0x30) == 0) {
	p->indgroup = SMS_DCS_IND_DISCARD;
      } else {
	p->indgroup = SMS_DCS_IND_STORE;
      }
      if (dcs&0x08) {
	p->indsense = 1;
      } else {
	p->indsense = 0;
      }
      switch(dcs&0x03) {
      case 0: p->indtype = SMS_DCS_IND_VOICE; break;
      case 1: p->indtype = SMS_DCS_IND_FAX; break;
      case 2: p->indtype = SMS_DCS_IND_EMAIL; break;
      case 3: p->indtype = SMS_DCS_IND_OTHER; break;
      }
    }
  }
}

/* returns the number of character to advance in pdu
 */
unsigned int sms_pdu_address_decode (char* pdu, struct sms_number* n)
{
  unsigned int len;
  unsigned int i;
  
  len = hexstr2int(pdu,2);
  len += len%2;
  struct_sms_number_init(n);
  n->type = hexstr2int(pdu+2,2);
  if (len) {
    switch (n->type&SMS_NUMBER_TYPEMASK) {
    case SMS_NUMTYPE_TEXT:
      //those are octets, not semi-octets
      n->text = sms_data_gsm_decode(pdu+4,(len*4)/7,0);
      break;
    default:
      for (i=0;i<len;i+=2) {
	n->digits[i] = pdu[4+i+1];
	n->digits[i+1] = pdu[4+i];
      }
      if (n->digits[i-1] == 'F') {
	memset(n->digits+i-1,0,1);
      }
    }
  }
  return 2+2+len;
}

unsigned int sms_pdu_scts_decode (char* pdu, struct sms_pdu_time* t)
{
  char* temp;
  struct tm sct_tm;

  t->format = SMS_PDU_TIME_ABSOLUTE;
  temp = mem_alloc(18,1);
  sprintf(temp,"%c%c/%c%c/%c%c,%c%c:%c%c:%c%c",
	  pdu[1],pdu[0],pdu[3],pdu[2],pdu[5],pdu[4],
	  pdu[7],pdu[6],pdu[9],pdu[8],pdu[11],pdu[10]);
  memset(&sct_tm,0,sizeof(sct_tm));
  if (strptime(temp,"%y/%m/%d,%H:%M:%S",&sct_tm) != NULL) {
    /* Not set by strptime(); tells mktime() to determine whether
     * daylight saving time is in effect
     */
    sct_tm.tm_isdst = -1;
    /* work-around for lazy *BSD implementation of strptime()
     * that does not set tm_yday in struct tm
     */
    tzset();
    t->value = mktime(&sct_tm);
    localtime_r(&t->value,&sct_tm);
    
    /* mktime() does not work correctly for this, so we use the following from
     * http://www.opengroup.org/onlinepubs/007904975/basedefs/xbd_chap04.html#tag_04_14
     * (slightly modified)
     */
    t->value = sct_tm.tm_sec + sct_tm.tm_min*60 + sct_tm.tm_hour*3600;
    t->value += 86400 * (sct_tm.tm_yday
			 + (sct_tm.tm_year-70)*365
			 + (sct_tm.tm_year-69)/4
			 - (sct_tm.tm_year-1)/100
			 + (sct_tm.tm_year+299)/400);
    
    //read real timezone offset
    temp = mem_realloc(temp,3);
    memset(temp,0,3);
    temp[0] = pdu[13];
    temp[1] = pdu[12];
    if ((atoi(temp)%80)/4 <= 12) {
      if (atoi(temp) >= 80) { //apply timezone offset
	t->value += (atoi(temp)%80)*15*60;
      } else {
	t->value -= (atoi(temp)%80)*15*60;
      }
    }
  } else { //fallback for strptime failure
    t->format = SMS_PDU_TIME_NONE;
  }
  mem_realloc(temp,0);
  return 14;
}

void sms_pdu_userdata_decode (char* pdu, struct sms_pdu_data* sms)
{
  uint8_t len = hexstr2int(pdu,2);
  uint8_t udhsize = 0;
  
  pdu += 2;
  if (len) {
    //assume that only one part is present until we can look at the header
    sms->parts = 1;
    sms->partnum = 0;
    //fill all headers (also calls sms_udh_multipart_mark())
    if (sms->options.udh_present) {
      udhsize = hexstr2int(pdu,2);
    } else {
      udhsize = 0;
    }
    sms_udh_fill(sms,pdu);
    //decode text
    sms->ud[sms->partnum].text = sms_data_decode(sms->scheme.encoding,
						 pdu,len,
						 sms->options.udh_present);
  }
}

struct sms_pdu_data* sms_pdu_decode_statusreport (char* pdu) {
  struct sms_pdu_data* sms;
  uint8_t temp = 0;

  sms = mem_alloc(sizeof(*sms),0);
  struct_sms_pdu_data_init(sms);

  sms_pdu_decode_options(SMS_INCOMING,hexstr2int(pdu,2)&0xFF,&sms->options);
  pdu += 2;

  /* From ETSI TS 123.040 v5.3.0 R5 9.2.2.3:
   * Where the SMS-STATUS-REPORT is the result of an
   * SMS-COMMAND and the TP-Command-Type was an Enquiry,
   * the TP-MR returned in the SMS-STATUS-REPORT shall be the
   * TP-MN which was sent in the SMS-COMMAND (i.e. the TP-MR
   * of the previously submitted SM to which the Enquiry refers).
   */
  sms->mref = hexstr2int(pdu,2)&0xFF;
  pdu += 2;

  pdu += sms_pdu_address_decode(pdu,&sms->address);  

  pdu += sms_pdu_scts_decode(pdu,&sms->timedata);

  sms->status = mem_alloc(sizeof(*sms->status),0);
  struct_sms_pdu_message_status_init(sms->status);
  sms->status->status_parent = sms;
  pdu += sms_pdu_scts_decode(pdu,&sms->status->t);
  //fixme: decode status->s with more detail
  sms->status->s = hexstr2int(pdu,2)&0xFF;
  pdu += 2;

  /* From ETSI TS 123.040 v5.3.0 R5 9.2.2.3:
   * parameter indicator:
   * Mandatory if any of the optional parameters following
   * TP-PI is present, otherwise optional.
   */
  if (strlen(pdu) > 0) {
    temp = hexstr2int(pdu,2)&0xFF;
    if (temp != 0xff) { //Siemens S55 now fills the rest with 0xff
      pdu += 2;
      if (temp&0x01) { //PID present
	sms->pid = hexstr2int(pdu,2)&0xFF;
	pdu += 2;
      }
      if (temp&0x02) { //DCS present
	sms_pdu_dcs_decode(hexstr2int(pdu,2)&0xFF,&sms->scheme);
      pdu += 2;
      }
      if (temp&0x04) { //user data present
	sms_pdu_userdata_decode(pdu,sms);
      }
    }
  }
  return sms;
}

struct sms_pdu_data* sms_pdu_decode_deliver (char* pdu)
{
  struct sms_pdu_data* sms;

  sms = mem_alloc(sizeof(*sms),0);
  struct_sms_pdu_data_init(sms);

  sms_pdu_decode_options(SMS_INCOMING,hexstr2int(pdu,2)&0xFF,&sms->options);
  pdu += 2;

  pdu += sms_pdu_address_decode(pdu,&sms->address);

  sms->pid = hexstr2int(pdu,2)&0xFF;
  pdu += 2;

  sms_pdu_dcs_decode(hexstr2int(pdu,2)&0xFF,&sms->scheme);
  pdu += 2;

  pdu += sms_pdu_scts_decode(pdu,&sms->timedata);

  sms_pdu_userdata_decode(pdu,sms);

  return sms;
}

struct sms_pdu_data* sms_pdu_decode_submit (char* pdu)
{
  struct sms_pdu_data* sms;
  int i;
  char* temp;
  uint8_t type = 0;

  temp = NULL;

  sms = mem_alloc(sizeof(*sms),0);
  struct_sms_pdu_data_init(sms);

  type = hexstr2int(pdu,2)&0xFF;
  sms_pdu_decode_options(SMS_OUTGOING,type,&sms->options);
  pdu += 2;

  sms->mref = hexstr2int(pdu,2);
  pdu += 2;

  pdu += sms_pdu_address_decode(pdu,&sms->address);

  sms->pid = hexstr2int(pdu,2)&0xFF;
  pdu += 2;

  sms_pdu_dcs_decode(hexstr2int(pdu,2)&0xFF,&sms->scheme);
  pdu += 2;

  switch (type&0x18) {
  case 0x00: //VP not present
    sms->timedata.format = SMS_PDU_TIME_NONE;
    break;
  case 0x08: //VP enhanced
    i = hexstr2int(pdu,2)&0xFF; //reading octet 1
    if (i&0x80) { //extension bit
      //not supported
      sms->timedata.format = SMS_PDU_TIME_NONE;
    }
    //ignore bit 6 (single shot SM)
    //bits 5,4 and 3 are reserved
    switch (i&0x07) { //bits 2, 1 and 0
    default: //reserved values
    case 0x00:
      sms->timedata.format = SMS_PDU_TIME_NONE;
      break;
    case 0x01: //value in seconds (1 octet)
      sms->timedata.format = SMS_PDU_TIME_RELATIVE;
      sms->timedata.value = hexstr2int(pdu+2,2)&0xFF;
      if (sms->timedata.value == 0) {
	sms->timedata.format = SMS_PDU_TIME_NONE;
      }
      break;
    case 0x02: //real relative value
      sms->timedata.format = SMS_PDU_TIME_RELATIVE;
      i = hexstr2int(pdu,2)&0xFF;
      if (0 <= i && i <= 143) { //5 minutes
	sms->timedata.value = (i+1)*5*60;
      } else if (144 <= i && i <= 167) { //30 minutes
	sms->timedata.value = (i-143+24)*30*60;
      } else if (168 <= i && i <= 196) {//1 day
	sms->timedata.value = (i-166)*(3600*24);
      } else if (197 <= i && i <= 255) {//1 week
	sms->timedata.value = (i-192)*(3600*24*7);
      }
      break;
    case 0x03: // value as HH:MM:SS
      sms->timedata.format = SMS_PDU_TIME_RELATIVE;
      temp = mem_alloc(9,1);
      sprintf(temp,"%c%c:%c%c:%c%c",
	      pdu[1],pdu[0],pdu[3],pdu[2],pdu[5],pdu[4]);
      sms->timedata.value = atoi(temp)*3600 + atoi(temp+3)*60 + atoi(temp+6);
      break;
    }
    pdu += 14;
    break;
  case 0x10: //VP relative
    sms->timedata.format = SMS_PDU_TIME_RELATIVE;
    i = hexstr2int(pdu,2)&0xFF;
    if (0 <= i && i <= 143) { //5 minutes
      sms->timedata.value = (i+1)*5*60;
    } else if (144 <= i && i <= 167) { //30 minutes
      sms->timedata.value = (i-143+24)*30*60;
    } else if (168 <= i && i <= 196) {//1 day
      sms->timedata.value = (i-166)*(3600*24);
    } else if (197 <= i && i <= 255) {//1 week
      sms->timedata.value = (i-192)*(3600*24*7);
    }
    pdu += 2;
    break;
  case 0x18: //VP absolute
    pdu += sms_pdu_scts_decode(pdu,&sms->timedata);
    break;
  }

  sms_pdu_userdata_decode(pdu,sms);

  return sms;
}

struct sms_tpdu_data* sms_pdu_decode (enum sms_direction type,
				      struct sms_slot_data* sms)
{
  struct sms_tpdu_data* decoded;
  char* pdu;
  uint8_t sca_len = 0;
  int i;

  decoded = mem_alloc(sizeof(*decoded),0);
  struct_sms_tpdu_data_init(decoded);
  pdu = NULL;

  //extracting SMSC
  sca_len = hexstr2int(sms->tpdu,2);
  if (sca_len > 0) {
    sca_len *= 2;
    decoded->sca.type = hexstr2int(sms->tpdu+2,2);
    for (i=0;i<sca_len-2;i+=2) {
      decoded->sca.digits[i] = sms->tpdu[4+i+1];
      decoded->sca.digits[i+1] = sms->tpdu[4+i];
    }
    if (decoded->sca.digits[i-1] == 'F') {
      memset(decoded->sca.digits+i-1,0,1);
    }
  }
  pdu = str_dup(sms->tpdu+sca_len+2);

  //checking if PDUtype is supported and can thus be decoded
  switch (get_sms_pdu_type(type,hexstr2int(pdu,2))) {
  case SMS_TYPE_DELIVER:
    decoded->pdu = sms_pdu_decode_deliver(pdu);
    break;
  case SMS_TYPE_SUBMIT:
    decoded->pdu = sms_pdu_decode_submit(pdu);
    break;
  case SMS_TYPE_STATUS_REPORT:
    decoded->pdu = sms_pdu_decode_statusreport(pdu);
    break;
  case SMS_TYPE_SUBMIT_REPORT:
    fprintf(stderr,"Unsupported pdu type: %s\n","SMS-SUBMIT-REPORT");
    struct_sms_tpdu_data_delete(decoded);
    decoded = mem_realloc(decoded,0);
    break;
  case SMS_TYPE_DELIVER_REPORT:
    fprintf(stderr,"Unsupported pdu type: %s\n","SMS-DELIVER-REPORT");
    struct_sms_tpdu_data_delete(decoded);
    decoded = mem_realloc(decoded,0);
    break;
  case SMS_TYPE_COMMAND:
    fprintf(stderr,"Unsupported pdu type: %s\n","SMS-COMMAND");
    struct_sms_tpdu_data_delete(decoded);
    decoded = mem_realloc(decoded,0);
    break;
  }  
  mem_realloc(pdu,0);
  return decoded;
}
