/*
 *  $Id: spscan.c,v 1.1 2003/06/02 16:47:24 ajung Exp $
 *
 * SCTP implementation according to RFC 2960.
 * Copyright (C) 2000 by Siemens AG, Munich, Germany.
 *
 * Realized in co-operation between Siemens AG
 * and University of Essen, Institute of Computer Networking Technology.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * There are two mailinglists available at http://www.sctp.de which should be
 * used for any discussion related to this implementation.
 *
 * Contact: discussion@sctp.de
 *
 * spscan.c  - an Sctp Port SCANner
 * SCTP is a connection oriented protocol. This simple port scanner tries to establish
 * associations with a range of SCTP ports of a peer endpoint. By default, it tries to
 * setup associations with ports 1 through 10000.
 * It will do a full association establishment and teardown with listening ports.
 * At the end, it will report all listening ports.
 */



#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>

#include <sctp_wrapper.h>

#define MAXIMUM_NUMBER_OF_LOCAL_ADDRESSES    10
#define MAXIMUM_NUMBER_OF_IN_STREAMS         10
#define MAXIMUM_NUMBER_OF_OUT_STREAMS        10
#define MAXIMUM_NUMBER_OF_PORTS              0xFFFF


typedef struct port_INFO {
    short portNum;
    unsigned int assocId;
    int associateDone;
    int resultReceived;
    int portOpen;
}port_info;

typedef struct port_DATA {
    port_info * scan_ports;
    int port_count;
    int resultsReceived;
}port_data;


static port_data *ports = NULL;

static unsigned char  localAddressList[MAXIMUM_NUMBER_OF_LOCAL_ADDRESSES][SCTP_MAX_IP_LEN];
static unsigned short noOfLocalAddresses = 0;
static unsigned char destinationAddress[SCTP_MAX_IP_LEN];
static unsigned int terminationTimer = 0;

static int verbose                  = 0;
static int vverbose                 = 0;
static int unknownCommand           = 0;
static int portsSelected            = 0;
static int hasDestinationAddress    = 0;
static int rateSet                  = 0;
static int sendRate                 = 100;
static int initCounter              = 0;
static unsigned int deleteAssoc[1000];
static int  numOfDeletedAssocs      = 0;


/*
 * this function has been copied and modified from nmap. Long live the GPL !!! :-)
 */
port_data* getpts(char *origexpr) {
    unsigned char portTable[65536];
    int portwarning = 0; /* have we warned idiot about dup ports yet? */
    int rangestart = -2343242, rangeend = -9324423;
    char *current_range;
    char *endptr;
    int i;
    int portcount = 0;
    port_data *sps;

    bzero(portTable, sizeof(portTable));

    current_range = origexpr;

    do {
        while(isspace((int) *current_range))
            current_range++; /* I don't know why I should allow spaces here, but I will */
            /* we don't check for other protocols, since we do SCTP only */
        if (*current_range == '-') {
            rangestart = 1;
        }
        else if (isdigit((int) *current_range)) {
            rangestart = strtol(current_range, &endptr, 10);
            if (rangestart <= 0 || rangestart > 65535) {
                printf("Ports to be scanned must be between 1 and 65535 inclusive !\n");
                exit (-1);
            }
            current_range = endptr;
            while(isspace((int) *current_range)) current_range++;
        } else {
            printf("Error #485: Your port specifications are illegal. \n");
            printf("Example of proper form: \"-100,200-1024,T:3000-4000,U:60000-\"\n");
            exit(-1);
        }
        /* Now I have a rangestart, time to go after rangeend */
        if (!*current_range || *current_range == ',') {
            /* Single port specification */
            rangeend = rangestart;
        } else if (*current_range == '-') {
            current_range++;
            if (!*current_range || *current_range == ',') {
                /* Ended with a -, meaning up until the last possible port */
                rangeend = 65535;
            } else if (isdigit((int) *current_range)) {
               rangeend = strtol(current_range, &endptr, 10);
                if (rangeend <= 0 || rangeend > 65535) {
                    printf("Ports to be scanned must be between 0 and 65535 inclusive.\n");
                    exit(-1);
                }
                current_range = endptr;
            } else {
                printf("Error #486: Your port specifications are illegal. \n");
                printf("Example of proper form: \"-100,200-1024,T:3000-4000,U:60000-\"\n");
                exit(-1);
            }
        } else {
            printf("Error #486: Your port specifications are illegal. \n");
            printf("Example of proper form: \"-100,200-1024,T:3000-4000,U:60000-\"\n");
            exit(-1);
        }

        /* Now I have a rangestart and a rangeend, so I can add these ports */
        while(rangestart <= rangeend) {
            if (portTable[rangestart] == 1) {
                if (!portwarning) {
                    printf("WARNING:  Duplicate port number(s) specified !!!\n");
                    printf("Are you alert enough to be using a port scanner ? Have some coffee or Jolt(tm).\n");
                    portwarning++;
                }
            } else {
                portcount++;
                portTable[rangestart] = 1;
            }
            rangestart++;
        }

        /* Find the next range */
        while(isspace((int) *current_range)) current_range++;
        if (*current_range && *current_range != ',') {
            printf("Error #488: Your port specifications are illegal. \n");
            printf("Example of proper form: \"-100,200-1024,T:3000-4000,U:60000-\"\n");
            exit(-1);
        }
        if (*current_range == ',') current_range++;

    } while(current_range && *current_range);

    if (portcount == 0) {
        printf("No ports specified -- If you really don't want to scan any ports.....\n");
        exit(-1);
    }
    sps = (port_data *) malloc(sizeof(port_data));
    if (!sps) return NULL;
    sps->scan_ports = (port_info *)malloc(portcount * sizeof(port_info));
    if (!sps->scan_ports) return NULL;
    sps->port_count = portcount;
    sps->resultsReceived = 0;
    portcount = 0;
    for (i=1; i< 65535; i++) {
        if(portTable[i] == 1) {
            sps->scan_ports[portcount].portNum = i;
            sps->scan_ports[portcount].assocId = 0;
            sps->scan_ports[portcount].associateDone = 0;
            sps->scan_ports[portcount].resultReceived = 0;
            sps->scan_ports[portcount++].portOpen = 0;
        }
    }
    return sps;
}

port_data* getDefaultPorts(void)
{
    port_data* sps;
    int portcount = 10000, i;
    sps = (port_data *) malloc(sizeof(port_data));
    if (!sps) return NULL;
    sps->scan_ports = (port_info *)malloc(portcount * sizeof(port_info));
    if (!sps->scan_ports) return NULL;
    sps->port_count = portcount;
    sps->resultsReceived = 0;
    portcount = 0;
    for (i=1; i< 10000; i++) {
        sps->scan_ports[portcount].portNum = i;
        sps->scan_ports[portcount].assocId = 0;
        sps->scan_ports[portcount].associateDone = 0;
        sps->scan_ports[portcount].resultReceived = 0;
        sps->scan_ports[portcount++].portOpen = 0;
    }
    return sps;
}

void printUsage(void)
{
    printf("Usage:    spscan -d D.E.S.T [options]\n");
    printf("options:\n");

    printf("-d X.Y.Z.A              destination IP address\n");
    printf("-p 10-140,995,2960      port ranges to scan (default: 1 to 10000\n");
    printf("-s                      source address\n");
    printf("-r rate                 send maximum of Rate INITs per second (default 100 per sec)");
    printf("-v                      verbose mode\n");
    printf("-V                      very verbose mode\n");  
    printf("-w                      receiver Window\n");    
}

void getArgs(int argc, char **argv)
{
    int c;
    extern char *optarg;
    extern int optind;

    while ((c = getopt(argc, argv, "hp:r:s:vVd:")) != -1)
    {
        switch (c) {
        case 'h':
            printUsage();
            exit(0);
        case 'p':
            if (portsSelected == 0) {
                ports = getpts(optarg);
                if (ports == NULL) {
                    printf("Port Selection Failed !!\n");
                    exit(-1);
                }
                portsSelected = 1;
            } else {
               printf("Only one port specification allowed !!\n");
               exit(-1);
            }
            break;
        case 'r':
            sendRate = atoi(optarg);
            rateSet = 1;
            break;
        case 'd':
            if (strlen(optarg) < SCTP_MAX_IP_LEN) {
                strcpy(destinationAddress, optarg);
            }
            hasDestinationAddress = 1;
            break;
        case 's':
            if ((noOfLocalAddresses < MAXIMUM_NUMBER_OF_LOCAL_ADDRESSES) &&
                (strlen(optarg) < SCTP_MAX_IP_LEN  )) {
                strcpy(localAddressList[noOfLocalAddresses], optarg);
                noOfLocalAddresses++;
            }; 
            break;
        case 'v':
            verbose = 1;
            break;
        case 'V':
            verbose = 1;
            vverbose = 1;
            break;
        default:
            unknownCommand = 1;
            break;
        }
    }
}


void
resetCounterTimerRunoffFunction(unsigned int timerID, void *parameter1, void *parameter2)
{
    int count;            

    for (count = 0; count < numOfDeletedAssocs; count++) {
        SCTP_deleteAssociation(deleteAssoc[count]);
        deleteAssoc[count] = 0;
    }
    numOfDeletedAssocs = 0;
    /* just globally reset init Counter, and restarts itself */
    initCounter = 0;
    SCTP_startTimer(1000, resetCounterTimerRunoffFunction, NULL, NULL);
}


void
printStatisticsFunction(unsigned int timerID, void *parameter1, void *parameter2)
{
    int numPortsOpen = 0, numPortsClosed = 0, count;

    for (count = 0; count < ports->port_count; count++) {
        if (ports->scan_ports[count].resultReceived == 1) {
            if (ports->scan_ports[count].portOpen == 1) {
                numPortsOpen++;
            } else {
                numPortsClosed++;
            }
        } 
    }    
    /* reset terminationTimer, that way we will not be called again */
    terminationTimer = 0;
    /* print stats: number of ports scanned */
    printf("===============================================================\n");
    printf("%d ports were scanned on %s\n", ports->port_count, destinationAddress);
    printf("%d ports answered in the scan time\n", ports->resultsReceived);
    /* print stats: number of open/closed ports  */          
    printf("%d ports were OPEN  <-> %d ports are reported CLOSED \n", numPortsOpen, numPortsClosed);
    /* print stats: list of open ports, all other were closed  */
    printf("---------------------------------------------------------------\n");
    printf("Open Ports:\n");
    for (count = 0; count < ports->port_count; count++) {
        if (ports->scan_ports[count].portOpen == 1) printf("%u, ",ports->scan_ports[count].portNum);
    }
    printf("\nAll scanned Ports not reported here were closed\n");
    printf("===============================================================\n");
    /* TODO: check whether host was up at all (i.e. no answer after 30 secs or so) */
    /* clean up/abort all association still active */
    exit(0);
}

void checkArgs(void)
{
    int abortProgram;
    int printUsageInfo;
    
    abortProgram = 0;
    printUsageInfo = 0;
    
    if (noOfLocalAddresses == 0) {
#ifdef HAVE_IPV6
        strcpy(localAddressList[noOfLocalAddresses], "::0");
#else
        strcpy(localAddressList[noOfLocalAddresses], "0.0.0.0");
#endif
        noOfLocalAddresses++;
    }
    if (hasDestinationAddress==0) {
        printf("Error:   An destination address must be specified.\n");
        abortProgram = 1;
        printUsageInfo = 1;
    }

    if (portsSelected == 0) {
        ports = getDefaultPorts();
        portsSelected = 1;
    }
    if (unknownCommand ==1) {
         printf("Error:   Unkown options in command.\n");
         printUsageInfo = 1;
    }
    
    if (printUsageInfo == 1)
        printUsage();
    if (abortProgram == 1)
        exit(-1);
}


void dataArriveNotif(unsigned int assocID, unsigned int streamID, unsigned int len,
                     unsigned short streamSN, unsigned int TSN,  unsigned int protoID,
                     unsigned int unordered, void* ulpDataPtr)
{
    unsigned char chunk[8192];
    int length;
    unsigned short ssn;
    unsigned int tsn;

    if (vverbose) {  
      fprintf(stdout, "%-8x: Data arrived (%u bytes; TSN = %u, SSN = %u, SID= %u, %s)\n",
                      assocID, len, TSN, streamSN, streamID, (unordered==SCTP_ORDERED_DELIVERY)?"ordered":"unordered");
      fflush(stdout);
    }
    /* read it */
    if (len > 8190) {
        fprintf(stderr, "Message received. That is strange...but this message is way too long, too !\n");
        exit(-2);
    }
    length = 8192;
    SCTP_receive(assocID, streamID, chunk, &length, &ssn, &tsn, SCTP_MSG_DEFAULT);
    if (vverbose) {  
        fprintf(stdout, "%-8x: Data read (%u bytes; TSN = %u, SSN = %u, SID= %u, %s)\n",
                          assocID, len, TSN, ssn, streamID, (unordered==SCTP_ORDERED_DELIVERY)?"ordered":"unordered");
        fflush(stdout);
    }
    /* and forget it */
}

void sendFailureNotif(unsigned int assocID,
                      unsigned char *unsent_data, unsigned int dataLength, unsigned int *context, void* dummy)
{
  if (verbose) {  
    fprintf(stdout, "%-8x: Send failure\n", assocID);
    fflush(stdout);
  }
}

void networkStatusChangeNotif(unsigned int assocID, unsigned int destAddrIndex, int newState, void* ulpDataPtr)
{
    if (verbose) {  
        fprintf(stdout, "%-8x: Network status change: path %u is now %s\n", 
                assocID, destAddrIndex, ((newState == SCTP_PATH_OK) ? "ACTIVE" : "INACTIVE"));
        fflush(stdout);
    }
}

void* communicationUpNotif(unsigned int assocID, int status,
                           unsigned int noOfDestinations,
                           unsigned short noOfInStreams, unsigned short noOfOutStreams,
                           int associationSupportsPRSCTP, int adLayerIndLen, void* adLayerInd, void* dummy)
{   
    port_info* pInfo = NULL;
    
    if (verbose) {  
        fprintf(stdout, "%-8x: Communication up (%u paths)\n", assocID, noOfDestinations);
        fflush(stdout);
    }
    /* do the housekeeping */
    pInfo = (port_info*) dummy;
    pInfo->portOpen = 1;
    pInfo->resultReceived = 1;
    ports->resultsReceived++;

    /* abort assoc */
    SCTP_abort(assocID);

    if (ports->resultsReceived == ports->port_count) {
        if (terminationTimer != 0) {
            sctp_stopTimer(terminationTimer);
            printStatisticsFunction(0, NULL, NULL);
        }
    }

    return NULL;
}

void communicationLostNotif(unsigned int assocID, int status, void* ulpDataPtr)
{
    port_info* pInfo = NULL;
        
    if (verbose) {
        fprintf(stdout, "%-8x: Communication lost (status %d)\n", assocID, status);
        fflush(stdout);
    }
    /* do the housekeeping */
    pInfo = (port_info*) ulpDataPtr;
    if (pInfo->resultReceived == 0) {
        /* probably a peer reply to our scan, or timeout */
        pInfo->resultReceived = 1;
        pInfo->portOpen = 0;
        ports->resultsReceived++;
    }
    deleteAssoc[numOfDeletedAssocs++] = assocID;

    if (ports->resultsReceived == ports->port_count) {
        if (terminationTimer != 0) {
            sctp_stopTimer(terminationTimer);
            printStatisticsFunction(0, NULL, NULL);
        }
    }
}

void communicationErrorNotif(unsigned int assocID, int status, void* dummy)
{
    if (verbose) {  
        fprintf(stdout, "%-8x: Communication error (status %d)\n", assocID, status);
        fflush(stdout);
    }
}

void restartNotif(unsigned int assocID, void* ulpDataPtr)
{    
    if (verbose) {  
        fprintf(stdout, "%-8x: Restart\n", assocID);
        fflush(stdout);
    }
}

void shutdownCompleteNotif(unsigned int assocID, void* ulpDataPtr)
{
    if (verbose) {
        fprintf(stdout, "%-8x: Shutdown complete\n", assocID);
        fflush(stdout);
    }
    SCTP_deleteAssociation(assocID);
}

void shutdownReceivedNotif(unsigned int assocID, void* ulpDataPtr)
{
    if (verbose) {  
        fprintf(stdout, "%-8x: Shutdown received\n", assocID);
        fflush(stdout);
    }
}


int main(int argc, char **argv)
{
    SCTP_ulpCallbacks spscanUlp;
    SCTP_LibraryParameters params;
    /* SCTP_InstanceParameters instanceParameters; */

    int sctpInstance;
    int totalNumberSent = 0;

    /* initialize the discard_ulp variable */
    spscanUlp.dataArriveNotif           = &dataArriveNotif;
    spscanUlp.sendFailureNotif          = &sendFailureNotif;
    spscanUlp.networkStatusChangeNotif  = &networkStatusChangeNotif;
    spscanUlp.communicationUpNotif      = &communicationUpNotif;
    spscanUlp.communicationLostNotif    = &communicationLostNotif;
    spscanUlp.communicationErrorNotif   = &communicationErrorNotif;
    spscanUlp.restartNotif              = &restartNotif;
    spscanUlp.shutdownCompleteNotif     = &shutdownCompleteNotif;
    spscanUlp.peerShutdownReceivedNotif = &shutdownReceivedNotif;

    /* handle all command line options */
    getArgs(argc, argv);
    checkArgs();

    bzero(deleteAssoc, sizeof(deleteAssoc));

    SCTP_initLibrary();
    SCTP_getLibraryParameters(&params);
    /*
        params.sendOotbAborts = 1;
        params.supportPRSCTP = 1;
        params.checksumAlgorithm = SCTP_CHECKSUM_ALGORITHM_ADLER32;
     */
    params.checksumAlgorithm = SCTP_CHECKSUM_ALGORITHM_CRC32C;
    SCTP_setLibraryParameters(&params);

    /* set up the instance that will spawn a number of associating clients */
    sctpInstance = SCTP_registerInstance(0,
                                         MAXIMUM_NUMBER_OF_IN_STREAMS,
                                         MAXIMUM_NUMBER_OF_OUT_STREAMS,
                                         noOfLocalAddresses, localAddressList,
                                         spscanUlp);

    SCTP_startTimer(1000, &resetCounterTimerRunoffFunction, NULL, NULL);
    
    while (totalNumberSent < ports->port_count) {
        while (initCounter < sendRate) {
            ports->scan_ports[totalNumberSent].assocId = SCTP_associate(sctpInstance,
                                                                        MAXIMUM_NUMBER_OF_OUT_STREAMS,
                                                                        destinationAddress,
                                                                        ports->scan_ports[totalNumberSent].portNum,
                                                                        &ports->scan_ports[totalNumberSent]);
            ports->scan_ports[totalNumberSent].associateDone = 1;
            initCounter++;
            totalNumberSent++;
        }
        SCTP_eventLoop();
    }
    /* wait 30 more seconds and then print the results                      */
    /* maybe terminate this timer, if all answers have been received before */
    terminationTimer =  SCTP_startTimer(30000, &printStatisticsFunction, NULL, NULL);

    while (1) {
        SCTP_eventLoop();
    }
    /* this will never be reached here */
    exit(0);
}




