/* ScummVM - Scumm Interpreter
 * Copyright (C) 2002-2005 The ScummVM project
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * $Header: /cvsroot/scummvm/scummvm/scumm/util.cpp,v 2.14.2.1 2005/10/18 02:11:25 sev Exp $
 *
 */

#include "scumm/util.h"
#include "common/util.h"
#include "common/md5.h"

using Common::File;

namespace Scumm {

#pragma mark -
#pragma mark --- ScummFile ---
#pragma mark -

ScummFile::ScummFile() : _encbyte(0), _subFileStart(0), _subFileLen(0) {
}

void ScummFile::setEnc(byte value) {
	_encbyte = value;
}

void ScummFile::setSubfileRange(uint32 start, uint32 len) {
	// TODO: Add sanity checks
	const uint32 fileSize = File::size();
	assert(start <= fileSize);
	assert(start + len <= fileSize);
	_subFileStart = start;
	_subFileLen = len;
	seek(0, SEEK_SET);
}

void ScummFile::resetSubfile() {
	_subFileStart = 0;
	_subFileLen = 0;
	seek(0, SEEK_SET);
}

bool ScummFile::open(const char *filename, AccessMode mode) {
	if (File::open(filename, mode)) {
		resetSubfile();
		return true;
	} else {
		return false;
	}
}

bool ScummFile::openSubFile(const char *filename) {
	assert(isOpen());

	// Disable the XOR encryption and reset any current subfile range
	setEnc(0);
	resetSubfile();

	// Read in the filename table and look for the specified file

	unsigned long file_off, file_len;
	char file_name[0x20+1];
	unsigned long i;

	// Get the length of the data file to use for consistency checks
	const uint32 data_file_len = size();

	// Read offset and length to the file records */
	const uint32 file_record_off = readUint32BE();
	const uint32 file_record_len = readUint32BE();

	// Do a quick check to make sure the offset and length are good
	if (file_record_off + file_record_len > data_file_len) {
		return false;
	}

	// Do a little consistancy check on file_record_length
	if (file_record_len % 0x28) {
		return false;
	}

	// Scan through the files
	for (i = 0; i < file_record_len; i += 0x28) {
		// read a file record
		seek(file_record_off + i, SEEK_SET);
		file_off = readUint32BE();
		file_len = readUint32BE();
		read(file_name, 0x20);
		file_name[0x20] = 0;

		assert(file_name[0]);
		//debug(7, "  extracting \'%s\'", file_name);

		// Consistency check. make sure the file data is in the file
		if (file_off + file_len > data_file_len) {
			return false;
		}

		if (scumm_stricmp(file_name, filename) == 0) {
			// We got a match!
			setSubfileRange(file_off, file_len);
			return true;
		}
	}

	return false;
}


bool ScummFile::eof() {
	return _subFileLen ? (pos() >= _subFileLen) : File::eof();
}

uint32 ScummFile::pos() {
	return File::pos() - _subFileStart;
}

uint32 ScummFile::size() {
	return _subFileLen ? _subFileLen : File::size();
}

void ScummFile::seek(int32 offs, int whence) {
	if (_subFileLen) {
		// Constrain the seek to the subfile
		switch (whence) {
		case SEEK_END:
			offs = _subFileStart + _subFileLen - offs;
			break;
		case SEEK_SET:
			offs += _subFileStart;
			break;
		case SEEK_CUR:
			offs += File::pos();
			break;
		}
		assert((int32)_subFileStart <= offs && offs <= (int32)(_subFileStart + _subFileLen));
		whence = SEEK_SET;
	}
	File::seek(offs, whence);
}

uint32 ScummFile::read(void *dataPtr, uint32 dataSize) {
	uint32 realLen;

	if (_subFileLen) {
		// Limit the amount we read by the subfile boundaries.
		const uint32 curPos = pos();
		assert(_subFileLen >= curPos);
		uint32 newPos = curPos + dataSize;
		if (newPos > _subFileLen) {
			dataSize = _subFileLen - curPos;
			_ioFailed = true;
		}
	}

	realLen = File::read(dataPtr, dataSize);


	// If an encryption byte was specified, XOR the data we just read by it.
	// This simple kind of "encryption" was used by some of the older SCUMM
	// games.
	if (_encbyte) {
		byte *p = (byte *)dataPtr;
		byte *end = p + realLen;
		while (p < end)
			*p++ ^= _encbyte;
	}

	return realLen;
}

uint32 ScummFile::write(const void *, uint32) {
	error("ScummFile does not support writing!");
	return 0;
}

#pragma mark -
#pragma mark --- Utilities ---
#pragma mark -

void checkRange(int max, int min, int no, const char *str) {
	if (no < min || no > max) {
		char buf[256];
		snprintf(buf, sizeof(buf), str, no);
		error("Value %d is out of bounds (%d,%d) (%s)", no, min, max, buf);
	}
}

/**
 * Convert an old style direction to a new style one (angle),
 */
int newDirToOldDir(int dir) {
	if (dir >= 71 && dir <= 109)
		return 1;
	if (dir >= 109 && dir <= 251)
		return 2;
	if (dir >= 251 && dir <= 289)
		return 0;
	return 3;
}

/**
 * Convert an new style (angle) direction to an old style one.
 */
int oldDirToNewDir(int dir) {
	assert(0 <= dir && dir <= 3);
	const int new_dir_table[4] = { 270, 90, 180, 0 };
	return new_dir_table[dir];
}

/**
 * Convert an angle to a simple direction.
 */
int toSimpleDir(int dirType, int dir) {
	if (dirType) {
		const int16 directions[] = { 22,  72, 107, 157, 202, 252, 287, 337 };
		for (int i = 0; i < 7; i++)
			if (dir >= directions[i] && dir <= directions[i+1])
				return i+1;
	} else {
		const int16 directions[] = { 71, 109, 251, 289 };
		for (int i = 0; i < 3; i++)
			if (dir >= directions[i] && dir <= directions[i+1])
				return i+1;
	}

	return 0;
}

/**
 * Convert a simple direction to an angle.
 */
int fromSimpleDir(int dirType, int dir) {
	if (dirType)
		return dir * 45;
	else
		return dir * 90;
}

/**
 * Normalize the given angle - that means, ensure it is positive, and
 * change it to the closest multiple of 45 degree by abusing toSimpleDir.
 */
int normalizeAngle(int angle) {
	int temp;

	temp = (angle + 360) % 360;

	return toSimpleDir(1, temp) * 45;
}

const char *tag2str(uint32 tag) {
	static char str[5];
	str[0] = (char)(tag >> 24);
	str[1] = (char)(tag >> 16);
	str[2] = (char)(tag >> 8);
	str[3] = (char)tag;
	str[4] = '\0';
	return str;
}

#pragma mark -
#pragma mark --- ScummNESFile ---
#pragma mark -

enum ResType {
	NES_UNKNOWN,
	NES_GLOBDATA,
	NES_ROOM,
	NES_SCRIPT,
	NES_SOUND,
	NES_COSTUME,
	NES_ROOMGFX,
	NES_COSTUMEGFX,
	NES_SPRPALS,
	NES_SPRDESC,
	NES_SPRLENS,
	NES_SPROFFS,
	NES_SPRDATA,
	NES_CHARSET,
	NES_PREPLIST
};

struct ScummNESFile::Resource {
	uint32 offset[kROMsetNum];
	uint16 length[kROMsetNum];
	ResType type;
};

ScummNESFile::ScummNESFile() : _stream(0), _buf(0), _ROMset(kROMsetNum) {
}

uint32 ScummNESFile::write(const void *, uint32) {
	error("ScummNESFile does not support writing!");
	return 0;
}

void ScummNESFile::setEnc(byte enc) {
	_stream->setEnc(enc);
}

#ifdef PALMOS_68K
static ScummNESFile::Resource *res_roomgfx;
static ScummNESFile::Resource *res_costumegfx;
static ScummNESFile::Resource *res_rooms;
static ScummNESFile::Resource *res_scripts;
static ScummNESFile::Resource *res_sounds;
static ScummNESFile::Resource *res_costumes;
static ScummNESFile::Resource *res_sprpals;
static ScummNESFile::Resource *res_sprdesc;
static ScummNESFile::Resource *res_sprlens;
static ScummNESFile::Resource *res_sproffs;
static ScummNESFile::Resource *res_sprdata;

#else
static ScummNESFile::Resource res_roomgfx[40] = {
	{ {0x04001,0x04001,0x04001,0x04001}, {0x03C9,0x03B9,0x03F0,0x0426}, NES_ROOMGFX },
	{ {0x043CA,0x043BA,0x043F1,0x04427}, {0x069E,0x069E,0x069E,0x069E}, NES_ROOMGFX },
	{ {0x04A68,0x04A58,0x04A8F,0x04AC5}, {0x0327,0x0327,0x0327,0x0327}, NES_ROOMGFX },
	{ {0x04D8F,0x04D7F,0x04DB6,0x04DEC}, {0x053B,0x053B,0x053B,0x053B}, NES_ROOMGFX },
	{ {0x052CA,0x052BA,0x052F1,0x05327}, {0x06BE,0x06BE,0x06BE,0x06BE}, NES_ROOMGFX },
	{ {0x05988,0x05978,0x059AF,0x059E5}, {0x0682,0x0682,0x0682,0x0682}, NES_ROOMGFX },
	{ {0x0600A,0x05FFA,0x06031,0x06067}, {0x0778,0x0778,0x0778,0x0778}, NES_ROOMGFX },
	{ {0x06782,0x06772,0x067A9,0x067DF}, {0x0517,0x0517,0x0517,0x0517}, NES_ROOMGFX },
	{ {0x06C99,0x06C89,0x06CC0,0x06CF6}, {0x07FB,0x07FB,0x07FB,0x07FB}, NES_ROOMGFX },
	{ {0x07494,0x07484,0x074BB,0x074F1}, {0x07BE,0x07BE,0x07BE,0x07BE}, NES_ROOMGFX },
	{ {0x08001,0x08001,0x08001,0x08001}, {0x07A5,0x07A5,0x07A5,0x07A5}, NES_ROOMGFX },
	{ {0x087A6,0x087A6,0x087A6,0x087A6}, {0x06DD,0x06DD,0x06DD,0x06DD}, NES_ROOMGFX },
	{ {0x08E83,0x08E83,0x08E83,0x08E83}, {0x04EA,0x04EA,0x04EA,0x04EA}, NES_ROOMGFX },
	{ {0x0936D,0x0936D,0x0936D,0x0936D}, {0x0846,0x0846,0x07E2,0x07E2}, NES_ROOMGFX },
	{ {0x09BB3,0x09BB3,0x09B4F,0x09B4F}, {0x08C8,0x08C8,0x0791,0x0791}, NES_ROOMGFX },
	{ {0x0A47B,0x0A47B,0x0A2E0,0x0A2E0}, {0x0844,0x0844,0x07B5,0x07B5}, NES_ROOMGFX },
	{ {0x0ACBF,0x0ACBF,0x0AA95,0x0AA95}, {0x0515,0x0515,0x0515,0x0515}, NES_ROOMGFX },
	{ {0x0B1D4,0x0B1D4,0x0AFAA,0x0AFAA}, {0x0799,0x0799,0x0799,0x0799}, NES_ROOMGFX },
	{ {0x0B96D,0x0B96D,0x0B743,0x0B743}, {0x04BB,0x04BB,0x04BF,0x04BB}, NES_ROOMGFX },
	{ {0x07C52,0x07C42,0x0BC02,0x0BBFE}, {0x0319,0x0319,0x0319,0x0319}, NES_ROOMGFX },
	{ {0x0C001,0x0C001,0x0C001,0x0C001}, {0x0464,0x0464,0x0464,0x0464}, NES_ROOMGFX },
	{ {0x0C465,0x0C465,0x0C465,0x0C465}, {0x076D,0x076D,0x072C,0x072C}, NES_ROOMGFX },
	{ {0x0CBD2,0x0CBD2,0x0CB91,0x0CB91}, {0x0827,0x0827,0x0827,0x0827}, NES_ROOMGFX },
	{ {0x0D3F9,0x0D3F9,0x0D3B8,0x0D3B8}, {0x0515,0x0515,0x0515,0x0515}, NES_ROOMGFX },
	{ {0x0D90E,0x0D90E,0x0D8CD,0x0D8CD}, {0x064E,0x064E,0x064E,0x064E}, NES_ROOMGFX },
	{ {0x0DF5C,0x0DF5C,0x0DF1B,0x0DF1B}, {0x0775,0x0775,0x0775,0x0775}, NES_ROOMGFX },
	{ {0x0E6D1,0x0E6D1,0x0E690,0x0E690}, {0x06DD,0x06DD,0x06DD,0x06DD}, NES_ROOMGFX },
	{ {0x0EDAE,0x0EDAE,0x0ED6D,0x0ED6D}, {0x0376,0x0376,0x0376,0x0376}, NES_ROOMGFX },
	{ {0x0F124,0x0F124,0x0F0E3,0x0F0E3}, {0x05F7,0x05F7,0x05F7,0x05F7}, NES_ROOMGFX },
	{ {0x0F71B,0x0F71B,0x0F6DA,0x0F6DA}, {0x0787,0x0787,0x0791,0x0787}, NES_ROOMGFX },
	{ {0x10001,0x10001,0x07C79,0x10001}, {0x02D6,0x02D6,0x02D6,0x02D6}, NES_ROOMGFX },
	{ {0x102D7,0x102D7,0x10001,0x102D7}, {0x06A3,0x06A3,0x06A3,0x06A3}, NES_ROOMGFX },
	{ {0x1097A,0x1097A,0x106A4,0x1097A}, {0x099F,0x099F,0x0921,0x0921}, NES_ROOMGFX },
	{ {0x11319,0x11319,0x10FC5,0x1129B}, {0x0361,0x0361,0x0361,0x0361}, NES_ROOMGFX },
	{ {0x1167A,0x1167A,0x11326,0x115FC}, {0x0489,0x0489,0x0489,0x0489}, NES_ROOMGFX },
	{ {0x11B03,0x11B03,0x117AF,0x11A85}, {0x0437,0x0437,0x0437,0x0437}, NES_ROOMGFX },
	{ {0x11F3A,0x11F3A,0x11BE6,0x11EBC}, {0x084D,0x084D,0x084F,0x070D}, NES_ROOMGFX },
	{ {0x0BE28,0x12787,0x12435,0x07CAF}, {0x0199,0x0199,0x0199,0x0199}, NES_ROOMGFX },
	{ {0x12787,0x12920,0x125CE,0x125C9}, {0x09A7,0x09A7,0x0947,0x0947}, NES_ROOMGFX },
	{ {0x1312E,0x132C7,0x12F15,0x12F10}, {0x037A,0x037A,0x037A,0x037A}, NES_ROOMGFX }
};

static ScummNESFile::Resource res_costumegfx[2] = {
	{ {0x30001,0x30001,0x2EFE1,0x30001}, {0x0EB8,0x0EB8,0x0EB8,0x0EB8}, NES_COSTUMEGFX },
	{ {0x2F9F1,0x2F9F1,0x30001,0x2F608}, {0x0340,0x0340,0x0340,0x0340}, NES_COSTUMEGFX }
};

static ScummNESFile::Resource res_rooms[55] = {
	{ {0x00000,0x00000,0x00000,0x00000}, {0x0000,0x0000,0x0000,0x0000}, NES_ROOM },
	{ {0x14001,0x14001,0x14001,0x14001}, {0x0D0C,0x0D0C,0x0D12,0x0D76}, NES_ROOM },
	{ {0x134A8,0x13641,0x1328F,0x1328A}, {0x04B3,0x04B3,0x04B3,0x04C6}, NES_ROOM },
	{ {0x15397,0x15397,0x15367,0x15451}, {0x0849,0x0849,0x0859,0x0885}, NES_ROOM },
	{ {0x15C68,0x15C68,0x13742,0x13750}, {0x0685,0x0685,0x0694,0x0693}, NES_ROOM },
	{ {0x16381,0x16381,0x15C45,0x15D68}, {0x0715,0x0715,0x0707,0x0709}, NES_ROOM },
	{ {0x1395B,0x16CE8,0x1658F,0x166D4}, {0x04E7,0x04E7,0x04E0,0x0528}, NES_ROOM },
	{ {0x16CE8,0x18001,0x16A6F,0x16BFC}, {0x0AC0,0x0ABF,0x0AC8,0x0ACC}, NES_ROOM },
	{ {0x18001,0x171CF,0x18001,0x18001}, {0x06BA,0x06BA,0x06C7,0x06E2}, NES_ROOM },
	{ {0x17AED,0x13AF4,0x1789C,0x17A63}, {0x03CB,0x03D2,0x03EA,0x03E5}, NES_ROOM },
	{ {0x18BE7,0x18E1A,0x18C09,0x18C3B}, {0x0663,0x0663,0x0649,0x066A}, NES_ROOM },
	{ {0x192A6,0x194D9,0x192AE,0x19301}, {0x0580,0x04A9,0x04AB,0x049E}, NES_ROOM },
	{ {0x19A44,0x19BA0,0x19982,0x199C8}, {0x0443,0x0443,0x0447,0x044B}, NES_ROOM },
	{ {0x1A106,0x1A262,0x1A04D,0x1A0B1}, {0x0563,0x047C,0x047E,0x0478}, NES_ROOM },
	{ {0x1A669,0x1A6DE,0x1A4CB,0x1A529}, {0x0446,0x0446,0x0444,0x043F}, NES_ROOM },
	{ {0x1AAAF,0x1AB24,0x1A90F,0x1A968}, {0x03A7,0x03A7,0x03B9,0x03C8}, NES_ROOM },
	{ {0x1AE56,0x1AECB,0x1ACC8,0x1AD30}, {0x07E3,0x07E3,0x07E9,0x086F}, NES_ROOM },
	{ {0x1B699,0x1B70E,0x1B511,0x1B5FF}, {0x0692,0x0692,0x06A4,0x069B}, NES_ROOM },
	{ {0x1C001,0x1C001,0x1C001,0x1C001}, {0x0B49,0x0ACA,0x0B1A,0x0AA9}, NES_ROOM },
	{ {0x1CD09,0x1CC8A,0x1CCFD,0x1CC97}, {0x04C6,0x04C6,0x0486,0x049E}, NES_ROOM },
	{ {0x1D4C2,0x1D443,0x1D482,0x1D42C}, {0x0568,0x0568,0x0579,0x05A8}, NES_ROOM },
	{ {0x1DF6C,0x1DEED,0x1DF61,0x1DF71}, {0x0514,0x0514,0x051E,0x054E}, NES_ROOM },
	{ {0x1E8FA,0x1E87B,0x1E8EC,0x1E9D1}, {0x05CC,0x05CC,0x05CF,0x0606}, NES_ROOM },
	{ {0x1EF83,0x1EF04,0x1EF73,0x1F0A2}, {0x0389,0x0389,0x0398,0x039A}, NES_ROOM },
	{ {0x1F5E4,0x1F565,0x1F5F0,0x1F74E}, {0x0723,0x0723,0x071A,0x071C}, NES_ROOM },
	{ {0x20001,0x20001,0x20001,0x20001}, {0x049A,0x049A,0x049C,0x04B5}, NES_ROOM },
	{ {0x20511,0x20511,0x2051E,0x2052E}, {0x04F8,0x04F8,0x051E,0x04FF}, NES_ROOM },
	{ {0x21666,0x21666,0x21725,0x2172E}, {0x05CB,0x05D5,0x05D5,0x05DB}, NES_ROOM },
	{ {0x21DD6,0x21DE0,0x21EA5,0x21EAD}, {0x046B,0x046B,0x047F,0x0489}, NES_ROOM },
	{ {0x222F0,0x222FA,0x223D1,0x223E1}, {0x0460,0x0460,0x0460,0x0465}, NES_ROOM },
	{ {0x227B6,0x227C0,0x22897,0x228AC}, {0x0909,0x0909,0x090D,0x0957}, NES_ROOM },
	{ {0x24001,0x24001,0x24001,0x24001}, {0x0366,0x0366,0x0378,0x037E}, NES_ROOM },
	{ {0x23BDF,0x247DB,0x247C9,0x2481A}, {0x03CA,0x03CA,0x03CA,0x03CA}, NES_ROOM },
	{ {0x247DB,0x24BA5,0x24B93,0x24BE4}, {0x050D,0x050D,0x050D,0x050D}, NES_ROOM },
	{ {0x25ACF,0x23BE9,0x25267,0x252C0}, {0x0346,0x0346,0x0346,0x0346}, NES_ROOM },
	{ {0x1BDBD,0x17DB5,0x17CD0,0x1BD30}, {0x01CA,0x01CA,0x01CA,0x01CA}, NES_ROOM },
	{ {0x25E15,0x25E99,0x255AD,0x25606}, {0x0457,0x0457,0x0453,0x046D}, NES_ROOM },
	{ {0x2626C,0x262F0,0x25A00,0x25A73}, {0x0547,0x0547,0x053E,0x055A}, NES_ROOM },
	{ {0x267B3,0x26837,0x25F3E,0x25FCD}, {0x064A,0x064A,0x0647,0x0654}, NES_ROOM },
	{ {0x1FD72,0x1FCF3,0x1BC49,0x26C98}, {0x024B,0x024B,0x024B,0x024B}, NES_ROOM },
	{ {0x2739A,0x2741E,0x26B58,0x26EE3}, {0x01FA,0x01FA,0x01FA,0x01FA}, NES_ROOM },
	{ {0x2766D,0x276F1,0x26E27,0x271DD}, {0x0219,0x0219,0x0217,0x0217}, NES_ROOM },
	{ {0x28001,0x28001,0x27345,0x27713}, {0x02F4,0x02F4,0x02F4,0x02F4}, NES_ROOM },
	{ {0x284D6,0x284D6,0x27829,0x28001}, {0x045C,0x045C,0x045C,0x045C}, NES_ROOM },
	{ {0x289A3,0x289A3,0x28001,0x284CE}, {0x09CF,0x09CF,0x098A,0x0975}, NES_ROOM },
	{ {0x293C6,0x293C6,0x289DF,0x28E97}, {0x05A0,0x05A0,0x05A1,0x05E6}, NES_ROOM },
	{ {0x27B65,0x27BE9,0x2A442,0x27C3A}, {0x0201,0x0201,0x0201,0x0201}, NES_ROOM },
	{ {0x2ADD1,0x2ADE3,0x2A6E9,0x2A9D6}, {0x0325,0x0325,0x0325,0x0325}, NES_ROOM },
	{ {0x2B339,0x2B34B,0x1FD75,0x2AF88}, {0x01FC,0x01FC,0x01FC,0x01FC}, NES_ROOM },
	{ {0x2B535,0x2B547,0x2AC64,0x2B184}, {0x02A9,0x02A9,0x02A9,0x02A9}, NES_ROOM },
	{ {0x2B7DE,0x2B7F0,0x2AF0D,0x2B42D}, {0x02DE,0x02DE,0x02D1,0x02DF}, NES_ROOM },
	{ {0x2C001,0x2C001,0x2B2E6,0x2B818}, {0x03CE,0x03CE,0x03CC,0x03EC}, NES_ROOM },
	{ {0x2BBC0,0x2BBD2,0x23D61,0x2BD67}, {0x0205,0x0205,0x0205,0x0209}, NES_ROOM },
	{ {0x2C53A,0x2C53A,0x2B818,0x2C001}, {0x0170,0x0170,0x0168,0x0168}, NES_ROOM },
	{ {0x13E42,0x2BDD7,0x27CF6,0x2C4BF}, {0x0169,0x0169,0x0169,0x0169}, NES_ROOM }
};

static ScummNESFile::Resource res_scripts[179] = {
	{ {0x00000,0x00000,0x00000,0x00000}, {0x0000,0x0000,0x0000,0x0000}, NES_SCRIPT }, // 0
	{ {0x29966,0x29966,0x28F80,0x2947D}, {0x044D,0x044D,0x043B,0x0480}, NES_SCRIPT },
	{ {0x29DB3,0x29DB3,0x293BB,0x298FD}, {0x0207,0x0207,0x0209,0x0226}, NES_SCRIPT },
	{ {0x29FBA,0x29FBA,0x295C4,0x29B23}, {0x009F,0x009F,0x00AB,0x0092}, NES_SCRIPT },
	{ {0x2A059,0x2A059,0x2966F,0x29BB5}, {0x03F4,0x03F4,0x03FD,0x040C}, NES_SCRIPT },
	{ {0x2A44D,0x2A44D,0x29A6C,0x29FC1}, {0x01A1,0x01A1,0x01A1,0x01A1}, NES_SCRIPT },
	{ {0x00000,0x00000,0x00000,0x00000}, {0x0000,0x0000,0x0000,0x0000}, NES_SCRIPT },
	{ {0x2A5EE,0x2A5EE,0x29C0D,0x2A162}, {0x004A,0x005C,0x005C,0x005C}, NES_SCRIPT },
	{ {0x00000,0x00000,0x00000,0x00000}, {0x0000,0x0000,0x0000,0x0000}, NES_SCRIPT },
	{ {0x2A638,0x2A64A,0x29C69,0x2A1BE}, {0x0005,0x0005,0x0005,0x0005}, NES_SCRIPT },
	{ {0x2C6AA,0x2C6AA,0x2B980,0x2C169}, {0x000D,0x000D,0x000D,0x000D}, NES_SCRIPT }, // 10
	{ {0x2C6B7,0x2C6B7,0x2B98D,0x2C176}, {0x000D,0x000D,0x000D,0x000D}, NES_SCRIPT },
	{ {0x186BB,0x17889,0x186C8,0x186E3}, {0x0040,0x0040,0x0040,0x0040}, NES_SCRIPT },
	{ {0x186FB,0x178C9,0x18708,0x18723}, {0x0016,0x0016,0x0016,0x0016}, NES_SCRIPT },
	{ {0x1B639,0x1B6AE,0x1B4B1,0x1B59F}, {0x0046,0x0046,0x0046,0x0046}, NES_SCRIPT },
	{ {0x1EEC6,0x1EE47,0x1EEBB,0x1EFD7}, {0x00BD,0x00BD,0x00B8,0x00CB}, NES_SCRIPT },
	{ {0x21C31,0x21C3B,0x21CFA,0x21D09}, {0x0055,0x0055,0x005C,0x0054}, NES_SCRIPT },
	{ {0x177A8,0x18AC0,0x17537,0x176C8}, {0x0027,0x0027,0x0027,0x0027}, NES_SCRIPT },
	{ {0x1FD07,0x1FC88,0x1FD0A,0x1FE6A}, {0x0027,0x0027,0x0027,0x0027}, NES_SCRIPT },
	{ {0x1FD2E,0x1FCAF,0x1FD31,0x1FE91}, {0x0027,0x0027,0x0027,0x0027}, NES_SCRIPT },
	{ {0x1BD2B,0x1BDA0,0x1BBB5,0x1BC9A}, {0x0022,0x0022,0x0022,0x0022}, NES_SCRIPT }, // 20
	{ {0x15BE0,0x15BE0,0x15BC0,0x15CD6}, {0x0088,0x0088,0x0085,0x0092}, NES_SCRIPT },
	{ {0x22241,0x2224B,0x22324,0x22336}, {0x0020,0x0020,0x001E,0x001C}, NES_SCRIPT },
	{ {0x22261,0x2226B,0x22342,0x22352}, {0x008F,0x008F,0x008F,0x008F}, NES_SCRIPT },
	{ {0x1924A,0x1947D,0x19252,0x192A5}, {0x002B,0x002B,0x002B,0x002B}, NES_SCRIPT },
	{ {0x1CB4A,0x1CACB,0x1CB1B,0x1CAAA}, {0x0061,0x0061,0x006D,0x0069}, NES_SCRIPT },
	{ {0x1CBAB,0x1CB2C,0x1CB88,0x1CB13}, {0x003C,0x003C,0x004C,0x0054}, NES_SCRIPT },
	{ {0x1CBE7,0x1CB68,0x1CBD4,0x1CB67}, {0x0042,0x0042,0x0044,0x0048}, NES_SCRIPT },
	{ {0x1CC29,0x1CBAA,0x1CC18,0x1CBAF}, {0x004F,0x004F,0x0053,0x0058}, NES_SCRIPT },
	{ {0x2049B,0x2049B,0x2049D,0x204B6}, {0x0076,0x0076,0x0081,0x0078}, NES_SCRIPT },
	{ {0x16A96,0x16A96,0x1634C,0x16471}, {0x0035,0x0035,0x0035,0x0035}, NES_SCRIPT }, // 30
	{ {0x16ACB,0x16ACB,0x16381,0x164A6}, {0x001C,0x001C,0x001C,0x001C}, NES_SCRIPT },
	{ {0x16AE7,0x16AE7,0x1639D,0x164C2}, {0x0014,0x0014,0x0014,0x0014}, NES_SCRIPT },
	{ {0x16AFB,0x16AFB,0x163B1,0x164D6}, {0x001C,0x001C,0x001C,0x001C}, NES_SCRIPT },
	{ {0x16B17,0x16B17,0x163CD,0x164F2}, {0x0027,0x0027,0x0027,0x0027}, NES_SCRIPT },
	{ {0x16B3E,0x16B3E,0x163F4,0x16519}, {0x01AA,0x01AA,0x019B,0x01BB}, NES_SCRIPT },
	{ {0x1D1CF,0x1D150,0x1D183,0x1D135}, {0x0096,0x0096,0x0094,0x008D}, NES_SCRIPT },
	{ {0x1D265,0x1D1E6,0x1D217,0x1D1C2}, {0x010E,0x010E,0x0117,0x0119}, NES_SCRIPT },
	{ {0x1D373,0x1D2F4,0x1D32E,0x1D2DB}, {0x001C,0x001C,0x001C,0x001C}, NES_SCRIPT },
	{ {0x1D38F,0x1D310,0x1D34A,0x1D2F7}, {0x0056,0x0056,0x0056,0x0056}, NES_SCRIPT },
	{ {0x1D3E5,0x1D366,0x1D3A0,0x1D34D}, {0x0072,0x0072,0x0072,0x0072}, NES_SCRIPT }, // 40
	{ {0x1E480,0x1E401,0x1E47F,0x1E4BF}, {0x0028,0x0028,0x0028,0x0028}, NES_SCRIPT },
	{ {0x1E4A8,0x1E429,0x1E4A7,0x1E4E7}, {0x017D,0x017D,0x0175,0x01E0}, NES_SCRIPT },
	{ {0x1E625,0x1E5A6,0x1E61C,0x1E6C7}, {0x0229,0x0229,0x022B,0x0241}, NES_SCRIPT },
	{ {0x28932,0x28932,0x27C85,0x2845D}, {0x0071,0x0071,0x0071,0x0071}, NES_SCRIPT },
	{ {0x17EB8,0x13EC6,0x17C86,0x17E48}, {0x004D,0x004D,0x004A,0x004C}, NES_SCRIPT },
	{ {0x162ED,0x162ED,0x13DD6,0x13DE3}, {0x0039,0x0039,0x0039,0x0039}, NES_SCRIPT },
	{ {0x18711,0x178DF,0x1871E,0x18739}, {0x028B,0x028B,0x0270,0x0296}, NES_SCRIPT },
	{ {0x1899C,0x17B6A,0x1898E,0x189CF}, {0x00BB,0x00BB,0x00C0,0x00C2}, NES_SCRIPT },
	{ {0x18A57,0x17C25,0x18A4E,0x18A91}, {0x018B,0x018B,0x01B6,0x01A5}, NES_SCRIPT },
	{ {0x00000,0x00000,0x00000,0x00000}, {0x0000,0x0000,0x0000,0x0000}, NES_SCRIPT }, // 50
	{ {0x19E87,0x19FE3,0x19DC9,0x19E13}, {0x00ED,0x00ED,0x00EE,0x00F3}, NES_SCRIPT },
	{ {0x21C86,0x21C90,0x21D56,0x21D5D}, {0x00F6,0x00F6,0x00F5,0x00F6}, NES_SCRIPT },
	{ {0x1E84E,0x1E7CF,0x1E847,0x1E908}, {0x009B,0x009B,0x0094,0x00B8}, NES_SCRIPT },
	{ {0x21D7C,0x21D86,0x21E4B,0x21E53}, {0x0047,0x0047,0x0047,0x0047}, NES_SCRIPT },
	{ {0x2C6C4,0x2C6C4,0x2B99A,0x2C183}, {0x004D,0x004D,0x004D,0x004D}, NES_SCRIPT },
	{ {0x16326,0x16326,0x13E0F,0x13E1C}, {0x0024,0x0024,0x0024,0x0024}, NES_SCRIPT },
	{ {0x14D0D,0x14D0D,0x14D13,0x14D77}, {0x0014,0x0014,0x0014,0x0014}, NES_SCRIPT },
	{ {0x177CF,0x18AE7,0x1755E,0x176EF}, {0x0059,0x0059,0x0054,0x0059}, NES_SCRIPT },
	{ {0x17828,0x18B40,0x175B2,0x17748}, {0x0109,0x011E,0x011A,0x013F}, NES_SCRIPT },
	{ {0x17931,0x18C5E,0x176CC,0x17887}, {0x0009,0x0009,0x0009,0x0009}, NES_SCRIPT }, // 60
	{ {0x14D21,0x14D21,0x14D27,0x14D8B}, {0x01B6,0x01B6,0x01B9,0x01D4}, NES_SCRIPT },
	{ {0x2B0F6,0x2B108,0x2AA0E,0x2ACFB}, {0x0243,0x0243,0x0256,0x028D}, NES_SCRIPT },
	{ {0x230BF,0x230C9,0x231A4,0x23203}, {0x067F,0x067F,0x06D2,0x0779}, NES_SCRIPT },
	{ {0x2C711,0x2C711,0x2B9E7,0x2C1D0}, {0x001C,0x001C,0x001D,0x001B}, NES_SCRIPT },
	{ {0x2C72D,0x2C72D,0x2BA04,0x2C1EB}, {0x001A,0x001A,0x0016,0x001F}, NES_SCRIPT },
	{ {0x2C747,0x2C747,0x2BA1A,0x2C20A}, {0x0021,0x0021,0x002D,0x0024}, NES_SCRIPT },
	{ {0x2C768,0x2C768,0x2BA47,0x2C22E}, {0x0024,0x0024,0x0027,0x0019}, NES_SCRIPT },
	{ {0x2C78C,0x2C78C,0x2BA6E,0x2C247}, {0x0017,0x0017,0x0016,0x0018}, NES_SCRIPT },
	{ {0x2C7A3,0x2C7A3,0x2BA84,0x2C25F}, {0x0017,0x0017,0x0014,0x001D}, NES_SCRIPT },
	{ {0x2C7BA,0x2C7BA,0x2BA98,0x2C27C}, {0x0014,0x0014,0x0015,0x0016}, NES_SCRIPT }, // 70
	{ {0x2C7CE,0x2C7CE,0x2BAAD,0x2C292}, {0x0024,0x0024,0x0029,0x0027}, NES_SCRIPT },
	{ {0x00000,0x00000,0x00000,0x00000}, {0x0000,0x0000,0x0000,0x0000}, NES_SCRIPT },
	{ {0x2C7F2,0x2C7F2,0x2BAD6,0x2C2B9}, {0x0011,0x0011,0x0010,0x0011}, NES_SCRIPT },
	{ {0x1793A,0x18C67,0x176D5,0x17890}, {0x009D,0x009D,0x00A2,0x00AA}, NES_SCRIPT },
	{ {0x22750,0x2275A,0x22831,0x22846}, {0x0066,0x0066,0x0066,0x0066}, NES_SCRIPT },
	{ {0x14ED7,0x14ED7,0x14EE0,0x14F5F}, {0x0075,0x0075,0x0077,0x0083}, NES_SCRIPT },
	{ {0x1F30C,0x1F28D,0x1F30B,0x1F43C}, {0x0120,0x0120,0x011A,0x013A}, NES_SCRIPT },
	{ {0x1FD55,0x1FCD6,0x1FD58,0x1FEB8}, {0x001D,0x001D,0x001D,0x001D}, NES_SCRIPT },
	{ {0x1F42C,0x1F3AD,0x1F425,0x1F576}, {0x008F,0x008F,0x0095,0x0098}, NES_SCRIPT },
	{ {0x1F4BB,0x1F43C,0x1F4BA,0x1F60E}, {0x0097,0x0097,0x009E,0x009B}, NES_SCRIPT }, // 80
	{ {0x179D7,0x18D04,0x17777,0x1793A}, {0x006A,0x006A,0x006F,0x006E}, NES_SCRIPT },
	{ {0x17A41,0x18D6E,0x177E6,0x179A8}, {0x0030,0x0030,0x002F,0x0033}, NES_SCRIPT },
	{ {0x1F552,0x1F4D3,0x1F558,0x1F6A9}, {0x0092,0x0092,0x0098,0x00A5}, NES_SCRIPT },
	{ {0x2C803,0x2C803,0x2BAE6,0x2C2CA}, {0x00CC,0x00CC,0x00C4,0x00BA}, NES_SCRIPT },
	{ {0x2C8CF,0x2C8CF,0x2BBAA,0x2C384}, {0x00BA,0x00BA,0x00AE,0x00AC}, NES_SCRIPT },
	{ {0x2C989,0x2C989,0x2BC58,0x2C430}, {0x0088,0x0088,0x0088,0x008F}, NES_SCRIPT },
	{ {0x20A09,0x20A09,0x20A3C,0x20A2D}, {0x01B0,0x01B0,0x01BB,0x01BE}, NES_SCRIPT },
	{ {0x20BB9,0x20BB9,0x20BF7,0x20BEB}, {0x0168,0x0168,0x0197,0x0158}, NES_SCRIPT },
	{ {0x20D21,0x20D21,0x20D8E,0x20D43}, {0x006C,0x006C,0x006E,0x0079}, NES_SCRIPT },
	{ {0x20D8D,0x20D8D,0x20DFC,0x20DBC}, {0x0037,0x0037,0x0028,0x002B}, NES_SCRIPT }, // 90
	{ {0x20DC4,0x20DC4,0x20E24,0x20DE7}, {0x00E4,0x00E4,0x00EA,0x00E8}, NES_SCRIPT },
	{ {0x20EA8,0x20EA8,0x20F0E,0x20ECF}, {0x0045,0x0045,0x0049,0x004A}, NES_SCRIPT },
	{ {0x20EED,0x20EED,0x20F57,0x20F19}, {0x00E1,0x00E1,0x00E7,0x0110}, NES_SCRIPT },
	{ {0x20FCE,0x20FCE,0x2103E,0x21029}, {0x00F6,0x00F6,0x010C,0x0136}, NES_SCRIPT },
	{ {0x210C4,0x210C4,0x2114A,0x2115F}, {0x0141,0x0141,0x0151,0x0152}, NES_SCRIPT },
	{ {0x21205,0x21205,0x2129B,0x212B1}, {0x0183,0x0183,0x01B0,0x01B3}, NES_SCRIPT },
	{ {0x21388,0x21388,0x2144B,0x21464}, {0x0034,0x0034,0x0034,0x0032}, NES_SCRIPT },
	{ {0x213BC,0x213BC,0x2147F,0x21496}, {0x00A9,0x00A9,0x00A9,0x00A9}, NES_SCRIPT },
	{ {0x24367,0x24367,0x24379,0x2437F}, {0x011B,0x011B,0x010E,0x0133}, NES_SCRIPT },
	{ {0x1BD4D,0x1BDC2,0x1BBD7,0x1BCBC}, {0x0070,0x0070,0x0072,0x0074}, NES_SCRIPT }, // 100
	{ {0x1CC78,0x1CBF9,0x1CC6B,0x1CC07}, {0x0091,0x0091,0x0092,0x0090}, NES_SCRIPT },
	{ {0x29372,0x29372,0x2898B,0x28E43}, {0x0054,0x0054,0x0054,0x0054}, NES_SCRIPT },
	{ {0x19F74,0x1A0D0,0x19EB7,0x19F06}, {0x00CE,0x00CE,0x00D3,0x00DB}, NES_SCRIPT },
	{ {0x1A042,0x1A19E,0x19F8A,0x19FE1}, {0x0077,0x0077,0x0077,0x0080}, NES_SCRIPT },
	{ {0x14F4C,0x14F4C,0x14F57,0x14FE2}, {0x0057,0x0057,0x0057,0x0057}, NES_SCRIPT },
	{ {0x27886,0x2790A,0x2703E,0x273F4}, {0x02DF,0x02DF,0x0307,0x031F}, NES_SCRIPT },
	{ {0x1DA2A,0x1D9AB,0x1D9FB,0x1D9D4}, {0x0219,0x0219,0x024F,0x0238}, NES_SCRIPT },
	{ {0x1DC43,0x1DBC4,0x1DC4A,0x1DC0C}, {0x00F9,0x00F9,0x00E4,0x00FE}, NES_SCRIPT },
	{ {0x1DD3C,0x1DCBD,0x1DD2E,0x1DD0A}, {0x0056,0x0056,0x0059,0x005A}, NES_SCRIPT },
	{ {0x1DD92,0x1DD13,0x1DD87,0x1DD64}, {0x01C2,0x01C2,0x01C2,0x01F5}, NES_SCRIPT }, // 110
	{ {0x14FA3,0x14FA3,0x14FAE,0x15039}, {0x004D,0x004D,0x004D,0x004D}, NES_SCRIPT },
	{ {0x27594,0x27618,0x26D52,0x270DD}, {0x00D9,0x00D9,0x00D5,0x0100}, NES_SCRIPT },
	{ {0x21DC3,0x21DCD,0x21E92,0x21E9A}, {0x0013,0x0013,0x0013,0x0013}, NES_SCRIPT },
	{ {0x2A63D,0x2A64F,0x29C6E,0x2A1C3}, {0x00F0,0x00F0,0x00F0,0x00F0}, NES_SCRIPT },
	{ {0x24482,0x24482,0x24487,0x244B2}, {0x00E7,0x00E7,0x00E0,0x00E4}, NES_SCRIPT },
	{ {0x21465,0x21465,0x21528,0x2153F}, {0x00F2,0x00F2,0x00F2,0x00EC}, NES_SCRIPT },
	{ {0x24569,0x24569,0x24567,0x24596}, {0x002B,0x002B,0x0023,0x0033}, NES_SCRIPT },
	{ {0x2C3CF,0x2C3CF,0x2B6B2,0x2BC04}, {0x010F,0x010F,0x010B,0x0108}, NES_SCRIPT },
	{ {0x24594,0x24594,0x2458A,0x245C9}, {0x00AA,0x00AA,0x00A1,0x009F}, NES_SCRIPT },
	{ {0x24CE8,0x250B2,0x250A0,0x250F1}, {0x0DAB,0x0DAB,0x018B,0x0193}, NES_SCRIPT }, // 120
	{ {0x1B67F,0x1B6F4,0x1B4F7,0x1B5E5}, {0x000D,0x000D,0x000D,0x000D}, NES_SCRIPT },
	{ {0x1B68C,0x1B701,0x1B504,0x1B5F2}, {0x000D,0x000D,0x000D,0x000D}, NES_SCRIPT },
	{ {0x2373E,0x23748,0x23876,0x2397C}, {0x017C,0x017C,0x018E,0x0199}, NES_SCRIPT },
	{ {0x282F5,0x282F5,0x27639,0x27A07}, {0x01E1,0x01E1,0x01F0,0x0233}, NES_SCRIPT },
	{ {0x238BA,0x238C4,0x23A04,0x23B15}, {0x0153,0x0153,0x017B,0x0171}, NES_SCRIPT },
	{ {0x23A0D,0x23A17,0x23B7F,0x23C86}, {0x019C,0x019C,0x01AC,0x01BC}, NES_SCRIPT },
	{ {0x23BA9,0x23BB3,0x23D2B,0x23E42}, {0x0016,0x0016,0x0016,0x0016}, NES_SCRIPT },
	{ {0x2C4DE,0x2C4DE,0x2B7BD,0x2BD0C}, {0x005C,0x005C,0x005B,0x005B}, NES_SCRIPT },
	{ {0x23BBF,0x23BC9,0x23D41,0x23E58}, {0x0020,0x0020,0x0020,0x0020}, NES_SCRIPT },
	{ {0x27D66,0x27DEA,0x2A643,0x27E3B}, {0x00A5,0x00A5,0x00A6,0x00B9}, NES_SCRIPT }, // 130
	{ {0x2A72D,0x2A73F,0x29D5E,0x2A2B3}, {0x034D,0x034D,0x0399,0x03D3}, NES_SCRIPT },
	{ {0x14FF0,0x14FF0,0x14FFB,0x15086}, {0x00E3,0x00E3,0x00D2,0x00E4}, NES_SCRIPT },
	{ {0x2BABC,0x2BACE,0x2B1DE,0x2B70C}, {0x005F,0x005F,0x0063,0x0067}, NES_SCRIPT },
	{ {0x00000,0x00000,0x00000,0x00000}, {0x0000,0x0000,0x0000,0x0000}, NES_SCRIPT },
	{ {0x25A93,0x25E5D,0x2522B,0x25284}, {0x003C,0x003C,0x003C,0x003C}, NES_SCRIPT },
	{ {0x1E8E9,0x1E86A,0x1E8DB,0x1E9C0}, {0x0011,0x0011,0x0011,0x0011}, NES_SCRIPT },
	{ {0x1634A,0x1634A,0x13E33,0x13E40}, {0x0018,0x0018,0x0018,0x0018}, NES_SCRIPT },
	{ {0x26DFD,0x26E81,0x26585,0x26621}, {0x001F,0x001F,0x001F,0x001F}, NES_SCRIPT },
	{ {0x26E1C,0x26EA0,0x265A4,0x26640}, {0x0054,0x0054,0x0054,0x0054}, NES_SCRIPT },
	{ {0x26E70,0x26EF4,0x265F8,0x26694}, {0x0149,0x0149,0x017D,0x0173}, NES_SCRIPT }, // 140
	{ {0x26FB9,0x2703D,0x26775,0x26807}, {0x004B,0x004B,0x004B,0x004B}, NES_SCRIPT },
	{ {0x27004,0x27088,0x267C0,0x26852}, {0x017D,0x017D,0x0165,0x0190}, NES_SCRIPT },
	{ {0x27181,0x27205,0x26925,0x269E2}, {0x0027,0x0027,0x0027,0x0027}, NES_SCRIPT },
	{ {0x271A8,0x2722C,0x2694C,0x26A09}, {0x0041,0x0041,0x0041,0x0041}, NES_SCRIPT },
	{ {0x271E9,0x2726D,0x2698D,0x26A4A}, {0x01B1,0x01B1,0x01CB,0x024E}, NES_SCRIPT },
	{ {0x16362,0x16362,0x13E4B,0x13E58}, {0x001F,0x001F,0x001F,0x001F}, NES_SCRIPT },
	{ {0x2463E,0x2463E,0x2462B,0x24668}, {0x002A,0x002A,0x002A,0x002A}, NES_SCRIPT },
	{ {0x150D3,0x150D3,0x150CD,0x1516A}, {0x019E,0x019E,0x0187,0x01C9}, NES_SCRIPT },
	{ {0x19275,0x194A8,0x1927D,0x192D0}, {0x0031,0x0031,0x0031,0x0031}, NES_SCRIPT },
	{ {0x17A71,0x18D9E,0x17815,0x179DB}, {0x007C,0x007C,0x0087,0x0088}, NES_SCRIPT }, // 150
	{ {0x21557,0x21557,0x2161A,0x2162B}, {0x00DC,0x00DC,0x00D8,0x00D0}, NES_SCRIPT },
	{ {0x1D457,0x1D3D8,0x1D412,0x1D3BF}, {0x0018,0x0018,0x0018,0x0018}, NES_SCRIPT },
	{ {0x1D46F,0x1D3F0,0x1D42A,0x1D3D7}, {0x0053,0x0053,0x0058,0x0055}, NES_SCRIPT },
	{ {0x18BE2,0x17DB0,0x18C04,0x18C36}, {0x0005,0x0005,0x0005,0x0005}, NES_SCRIPT },
	{ {0x15271,0x15271,0x15254,0x15333}, {0x011B,0x011B,0x0108,0x0113}, NES_SCRIPT },
	{ {0x1538C,0x1538C,0x1535C,0x15446}, {0x000B,0x000B,0x000B,0x000B}, NES_SCRIPT },
	{ {0x24668,0x24668,0x24655,0x24692}, {0x0138,0x0138,0x0139,0x014D}, NES_SCRIPT },
	{ {0x247A0,0x247A0,0x2478E,0x247DF}, {0x0014,0x0014,0x0014,0x0014}, NES_SCRIPT },
	{ {0x1DF54,0x1DED5,0x1DF49,0x1DF59}, {0x0018,0x0018,0x0018,0x0018}, NES_SCRIPT },
	{ {0x247B4,0x247B4,0x247A2,0x247F3}, {0x0027,0x0027,0x0027,0x0027}, NES_SCRIPT }, // 160
	{ {0x1A0B9,0x1A215,0x1A001,0x1A061}, {0x004D,0x004D,0x004C,0x0050}, NES_SCRIPT },
	{ {0x00000,0x00000,0x00000,0x00000}, {0x0000,0x0000,0x0000,0x0000}, NES_SCRIPT },
	{ {0x2BB1B,0x2BB2D,0x2B241,0x2B773}, {0x00A5,0x00A5,0x00A5,0x00A5}, NES_SCRIPT },
	{ {0x2AA7A,0x2AA8C,0x2A0F7,0x2A686}, {0x00C1,0x00C1,0x00B5,0x00BA}, NES_SCRIPT },
	{ {0x2AB3B,0x2AB4D,0x2A1AC,0x2A740}, {0x0140,0x0140,0x0140,0x0140}, NES_SCRIPT },
	{ {0x19826,0x19982,0x19759,0x1979F}, {0x00BF,0x00BF,0x00CA,0x00CA}, NES_SCRIPT },
	{ {0x198E5,0x19A41,0x19823,0x19869}, {0x014D,0x014D,0x014D,0x014D}, NES_SCRIPT },
	{ {0x19A32,0x19B8E,0x19970,0x199B6}, {0x0012,0x0012,0x0012,0x0012}, NES_SCRIPT },
	{ {0x2AC7B,0x2AC8D,0x2A2EC,0x2A880}, {0x0005,0x0005,0x0005,0x0005}, NES_SCRIPT },
	{ {0x2AC80,0x2AC92,0x2A2F1,0x2A885}, {0x0005,0x0005,0x0005,0x0005}, NES_SCRIPT }, // 170
	{ {0x2AC85,0x2AC97,0x2A2F6,0x2A88A}, {0x0005,0x0005,0x0005,0x0005}, NES_SCRIPT },
	{ {0x2AC8A,0x2AC9C,0x2A2FB,0x2A88F}, {0x0005,0x0005,0x0005,0x0005}, NES_SCRIPT },
	{ {0x2AC8F,0x2ACA1,0x2A300,0x2A894}, {0x0005,0x0005,0x0005,0x0005}, NES_SCRIPT },
	{ {0x21633,0x21633,0x216F2,0x216FB}, {0x0033,0x0033,0x0033,0x0033}, NES_SCRIPT },
	{ {0x2AC94,0x2ACA6,0x2A305,0x2A899}, {0x0005,0x0005,0x0005,0x0005}, NES_SCRIPT },
	{ {0x00000,0x00000,0x00000,0x00000}, {0x0000,0x0000,0x0000,0x0000}, NES_SCRIPT },
	{ {0x2AC99,0x2ACAB,0x2A30A,0x2A89E}, {0x009C,0x009C,0x009C,0x009C}, NES_SCRIPT },
	{ {0x2AD35,0x2AD47,0x2A3A6,0x2A93A}, {0x009C,0x009C,0x009C,0x009C}, NES_SCRIPT }
};

static ScummNESFile::Resource res_sounds[82] = {
	{ {0x0FFE8,0x0BF54,0x0BF58,0x07F74}, {0x000A,0x000A,0x000A,0x000A}, NES_SOUND },
	{ {0x30ECA,0x30ECA,0x30352,0x30ECA}, {0x0832,0x0832,0x0832,0x0832}, NES_SOUND },
	{ {0x30ECA,0x30ECA,0x30352,0x30ECA}, {0x0832,0x0832,0x0832,0x0832}, NES_SOUND },
	{ {0x30ECA,0x30ECA,0x30352,0x30ECA}, {0x0832,0x0832,0x0832,0x0832}, NES_SOUND },
	{ {0x30ECA,0x30ECA,0x30352,0x30ECA}, {0x0832,0x0832,0x0832,0x0832}, NES_SOUND },
	{ {0x30ECA,0x30ECA,0x30352,0x30ECA}, {0x0832,0x0832,0x0832,0x0832}, NES_SOUND },
	{ {0x17FCA,0x0BF5E,0x0BF62,0x0BF6C}, {0x0011,0x0011,0x0011,0x0011}, NES_SOUND },
	{ {0x27E0B,0x27ECB,0x27E5F,0x1BEFA}, {0x0073,0x0073,0x0073,0x0073}, NES_SOUND },
	{ {0x17FDB,0x0BF6F,0x17F5A,0x17F10}, {0x0011,0x0011,0x0011,0x0011}, NES_SOUND },
	{ {0x17FEC,0x0FF5D,0x17F6B,0x17F21}, {0x0011,0x0011,0x0011,0x0011}, NES_SOUND },
	{ {0x27E7E,0x316FC,0x27ED2,0x1FED5}, {0x0056,0x0056,0x0056,0x0056}, NES_SOUND },
	{ {0x27ED4,0x13F4E,0x1BF55,0x17F32}, {0x001F,0x001F,0x001F,0x001F}, NES_SOUND },
	{ {0x23FEE,0x0FF6E,0x23F66,0x17F51}, {0x0011,0x0011,0x0011,0x0011}, NES_SOUND },
	{ {0x0FFF2,0x13F6D,0x0BF73,0x0FF76}, {0x000A,0x000A,0x000A,0x000A}, NES_SOUND },
	{ {0x27EF3,0x1BF47,0x1BF74,0x17F62}, {0x000A,0x000A,0x000A,0x000A}, NES_SOUND },
	{ {0x27EFD,0x1BF51,0x27F28,0x1FF2B}, {0x0019,0x0019,0x0019,0x0019}, NES_SOUND },
	{ {0x27F16,0x31752,0x2BF0A,0x23E78}, {0x004B,0x004B,0x004B,0x004B}, NES_SOUND },
	{ {0x27F61,0x1BF6A,0x1FF71,0x17F6C}, {0x000A,0x000A,0x000A,0x000A}, NES_SOUND },
	{ {0x27F6B,0x27F3E,0x27F41,0x1BF6D}, {0x000F,0x000F,0x000F,0x000F}, NES_SOUND },
	{ {0x27F7A,0x27F4D,0x27F50,0x1FF44}, {0x001D,0x001D,0x001D,0x001D}, NES_SOUND },
	{ {0x27F97,0x3179D,0x2FEAA,0x23EC3}, {0x0045,0x0045,0x0045,0x0045}, NES_SOUND },
	{ {0x27FDC,0x27F6A,0x27F6D,0x1FF61}, {0x000F,0x000F,0x000F,0x000F}, NES_SOUND },
	{ {0x2FD42,0x2BF40,0x2BF55,0x23F08}, {0x001B,0x001B,0x001B,0x001B}, NES_SOUND },
	{ {0x2FD5D,0x317E2,0x2FEEF,0x23F23}, {0x0033,0x0033,0x0033,0x0033}, NES_SOUND },
	{ {0x27FEB,0x2BF5B,0x2FF22,0x23F56}, {0x0011,0x0011,0x0011,0x0011}, NES_SOUND },
	{ {0x2BFEF,0x2BF6C,0x2BF70,0x1FF70}, {0x000F,0x000F,0x000F,0x000F}, NES_SOUND },
	{ {0x2FD90,0x31815,0x30B84,0x27EF4}, {0x0075,0x0075,0x0075,0x0075}, NES_SOUND },
	{ {0x2FE05,0x2FF6C,0x2FF33,0x23F67}, {0x0014,0x0014,0x0014,0x0014}, NES_SOUND },
	{ {0x0FFE8,0x0BF54,0x0BF58,0x07F74}, {0x000A,0x000A,0x000A,0x000A}, NES_SOUND },
	{ {0x2FE19,0x3188A,0x30BF9,0x2FB83}, {0x00FF,0x00FF,0x00FF,0x00FF}, NES_SOUND },
	{ {0x2FF18,0x31989,0x2FF47,0x27F69}, {0x000F,0x000F,0x000F,0x000F}, NES_SOUND },
	{ {0x2FF27,0x31998,0x2FF56,0x2BF70}, {0x000F,0x000F,0x000F,0x000F}, NES_SOUND },
	{ {0x2FF36,0x319A7,0x30CF8,0x2FC82}, {0x0092,0x0092,0x0092,0x0092}, NES_SOUND },
	{ {0x2FF36,0x319A7,0x30CF8,0x2FC82}, {0x0092,0x0092,0x0092,0x0092}, NES_SOUND },
	{ {0x2FFC8,0x31A39,0x30D8A,0x2FD14}, {0x002D,0x002D,0x002D,0x002D}, NES_SOUND },
	{ {0x316FC,0x31A66,0x30DB7,0x2FD41}, {0x00F8,0x00F8,0x00F8,0x00F8}, NES_SOUND },
	{ {0x317F4,0x31B5E,0x2FF65,0x2FE39}, {0x0016,0x0016,0x0016,0x0016}, NES_SOUND },
	{ {0x3180A,0x31B74,0x30EAF,0x2FE4F}, {0x0011,0x0011,0x0011,0x0011}, NES_SOUND },
	{ {0x3181B,0x31B85,0x30EC0,0x2FE60}, {0x004B,0x004B,0x004B,0x004B}, NES_SOUND },
	{ {0x31866,0x31BD0,0x30F0B,0x2FEAB}, {0x0011,0x0011,0x0011,0x0011}, NES_SOUND },
	{ {0x31877,0x31BE1,0x30F1C,0x2FEBC}, {0x003B,0x003B,0x003B,0x003B}, NES_SOUND },
	{ {0x318B2,0x31C1C,0x30F57,0x316FC}, {0x008A,0x008A,0x008A,0x008A}, NES_SOUND },
	{ {0x3193C,0x31CA6,0x30FE1,0x2FEF7}, {0x0011,0x0011,0x0011,0x0011}, NES_SOUND },
	{ {0x3194D,0x31CB7,0x30FF2,0x2FF08}, {0x000F,0x000F,0x000F,0x000F}, NES_SOUND },
	{ {0x3195C,0x31CC6,0x31001,0x31786}, {0x00A2,0x00A2,0x00A2,0x00A2}, NES_SOUND },
	{ {0x319FE,0x31D68,0x310A3,0x31828}, {0x00D3,0x00D3,0x00D3,0x00D3}, NES_SOUND },
	{ {0x31AD1,0x31E3B,0x31176,0x318FB}, {0x0097,0x0097,0x0097,0x0097}, NES_SOUND },
	{ {0x2BFEF,0x2BF6C,0x2BF70,0x1FF70}, {0x000F,0x000F,0x000F,0x000F}, NES_SOUND },
	{ {0x3195C,0x31CC6,0x31001,0x31786}, {0x00A2,0x00A2,0x00A2,0x00A2}, NES_SOUND },
	{ {0x31B68,0x31ED2,0x3120D,0x31992}, {0x05D1,0x05D1,0x05D1,0x05D1}, NES_SOUND },
	{ {0x31B68,0x31ED2,0x3120D,0x31992}, {0x05D1,0x05D1,0x05D1,0x05D1}, NES_SOUND },
	{ {0x32139,0x324A3,0x317DE,0x2FF17}, {0x0011,0x0011,0x0011,0x0011}, NES_SOUND },
	{ {0x0FFE8,0x0BF54,0x0BF58,0x07F74}, {0x000A,0x000A,0x000A,0x000A}, NES_SOUND },
	{ {0x2FD90,0x31815,0x30B84,0x27EF4}, {0x0075,0x0075,0x0075,0x0075}, NES_SOUND },
	{ {0x27ED4,0x13F4E,0x1BF55,0x17F32}, {0x001F,0x001F,0x001F,0x001F}, NES_SOUND },
	{ {0x3214A,0x324B4,0x317EF,0x31F63}, {0x098E,0x098E,0x098E,0x098E}, NES_SOUND },
	{ {0x3181B,0x31B85,0x30EC0,0x2FE60}, {0x004B,0x004B,0x004B,0x004B}, NES_SOUND },
	{ {0x32AD8,0x32E42,0x3217D,0x2FF28}, {0x0011,0x0011,0x0011,0x0011}, NES_SOUND },
	{ {0x30ECA,0x30ECA,0x30352,0x30ECA}, {0x0832,0x0832,0x0832,0x0832}, NES_SOUND },
	{ {0x32AE9,0x32E53,0x3218E,0x2FF39}, {0x000F,0x000F,0x000F,0x000F}, NES_SOUND },
	{ {0x32AF8,0x32E62,0x3219D,0x2FF48}, {0x002F,0x002F,0x002F,0x002F}, NES_SOUND },
	{ {0x32B27,0x32E91,0x321CC,0x328F1}, {0x001D,0x001D,0x001D,0x001D}, NES_SOUND },
	{ {0x32B44,0x32EAE,0x321E9,0x3290E}, {0x0018,0x0018,0x0018,0x0018}, NES_SOUND },
	{ {0x32B5C,0x32EC6,0x32201,0x32926}, {0x0016,0x0016,0x0016,0x0016}, NES_SOUND },
	{ {0x32B72,0x32EDC,0x32217,0x3293C}, {0x001B,0x001B,0x001B,0x001B}, NES_SOUND },
	{ {0x32B8D,0x32EF7,0x32232,0x32957}, {0x0088,0x0088,0x0088,0x0088}, NES_SOUND },
	{ {0x32C15,0x32F7F,0x322BA,0x329DF}, {0x0065,0x0065,0x0065,0x0065}, NES_SOUND },
	{ {0x32C7A,0x32FE4,0x3231F,0x32A44}, {0x0065,0x0065,0x0065,0x0065}, NES_SOUND },
	{ {0x32CDF,0x33049,0x32384,0x32AA9}, {0x0073,0x0073,0x0073,0x0073}, NES_SOUND },
	{ {0x32D52,0x330BC,0x323F7,0x32B1C}, {0x00F9,0x00F9,0x00F9,0x00F9}, NES_SOUND },
	{ {0x32E4B,0x331B5,0x324F0,0x32C15}, {0x049E,0x049E,0x049E,0x049E}, NES_SOUND },
	{ {0x34001,0x34001,0x3298E,0x330B3}, {0x0EA8,0x0EA8,0x0EA8,0x0EA8}, NES_SOUND },
	{ {0x332E9,0x34EA9,0x34001,0x34001}, {0x0B18,0x0B18,0x0B18,0x0B18}, NES_SOUND },
	{ {0x34EA9,0x359C1,0x34B19,0x34B19}, {0x0B9C,0x0B9C,0x0B9C,0x0B9C}, NES_SOUND },
	{ {0x35A45,0x3655D,0x356B5,0x356B5}, {0x0C6B,0x0C6B,0x0C6B,0x0C6B}, NES_SOUND },
	{ {0x366B0,0x38001,0x36320,0x36320}, {0x0E56,0x0E56,0x0E56,0x0E56}, NES_SOUND },
	{ {0x38001,0x371C8,0x37176,0x37176}, {0x0C70,0x0C70,0x0C70,0x0C70}, NES_SOUND },
	{ {0x38C71,0x38E57,0x38001,0x38001}, {0x0DEC,0x0DEC,0x0DEC,0x0DEC}, NES_SOUND },
	{ {0x39A5D,0x39C43,0x38DED,0x38DED}, {0x0B77,0x0B77,0x0B77,0x0B77}, NES_SOUND },
	{ {0x37506,0x33653,0x33836,0x39964}, {0x042F,0x042F,0x042F,0x042F}, NES_SOUND },
	{ {0x3A5D4,0x3A7BA,0x39964,0x39D93}, {0x0AC5,0x0AC5,0x0AC5,0x0AC5}, NES_SOUND },
	{ {0x3B099,0x3B27F,0x3A429,0x3A858}, {0x0BE4,0x0BE4,0x0BE4,0x0BE4}, NES_SOUND }
};

static ScummNESFile::Resource res_costumes[25] = {
	{ {0x17F05,0x0BEFF,0x0FEF5,0x0BF17}, {0x0055,0x0055,0x0055,0x0055}, NES_COSTUME },
	{ {0x17F05,0x0BEFF,0x0FEF5,0x0BF17}, {0x0055,0x0055,0x0055,0x0055}, NES_COSTUME },
	{ {0x17F05,0x0BEFF,0x0FEF5,0x0BF17}, {0x0055,0x0055,0x0055,0x0055}, NES_COSTUME },
	{ {0x17F05,0x0BEFF,0x0FEF5,0x0BF17}, {0x0055,0x0055,0x0055,0x0055}, NES_COSTUME },
	{ {0x17F05,0x0BEFF,0x0FEF5,0x0BF17}, {0x0055,0x0055,0x0055,0x0055}, NES_COSTUME },
	{ {0x17F05,0x0BEFF,0x0FEF5,0x0BF17}, {0x0055,0x0055,0x0055,0x0055}, NES_COSTUME },
	{ {0x17F05,0x0BEFF,0x0FEF5,0x0BF17}, {0x0055,0x0055,0x0055,0x0055}, NES_COSTUME },
	{ {0x17F05,0x0BEFF,0x0FEF5,0x0BF17}, {0x0055,0x0055,0x0055,0x0055}, NES_COSTUME },
	{ {0x13FAB,0x0FEA2,0x17E9A,0x13E77}, {0x004B,0x004B,0x004B,0x004B}, NES_COSTUME },
	{ {0x17F05,0x0BEFF,0x0FEF5,0x0BF17}, {0x0055,0x0055,0x0055,0x0055}, NES_COSTUME },
	{ {0x17F05,0x0BEFF,0x0FEF5,0x0BF17}, {0x0055,0x0055,0x0055,0x0055}, NES_COSTUME },
	{ {0x17F05,0x0BEFF,0x0FEF5,0x0BF17}, {0x0055,0x0055,0x0055,0x0055}, NES_COSTUME },
	{ {0x17F5A,0x0FEED,0x0FF4A,0x07F3E}, {0x0036,0x0036,0x0036,0x0036}, NES_COSTUME },
	{ {0x17F90,0x0FF23,0x17EE5,0x13EC2}, {0x003A,0x003A,0x003A,0x003A}, NES_COSTUME },
	{ {0x17F90,0x0FF23,0x17EE5,0x13EC2}, {0x003A,0x003A,0x003A,0x003A}, NES_COSTUME },
	{ {0x17F05,0x0BEFF,0x0FEF5,0x0BF17}, {0x0055,0x0055,0x0055,0x0055}, NES_COSTUME },
	{ {0x1BF87,0x13F13,0x17F1F,0x13EFC}, {0x003B,0x003B,0x003B,0x003B}, NES_COSTUME },
	{ {0x17F05,0x0BEFF,0x0FEF5,0x0BF17}, {0x0055,0x0055,0x0055,0x0055}, NES_COSTUME },
	{ {0x23FA9,0x23F2F,0x1BE94,0x13F37}, {0x0045,0x0045,0x0045,0x0045}, NES_COSTUME },
	{ {0x1FFBD,0x1FF3E,0x1BED9,0x17E94}, {0x0040,0x0040,0x0040,0x0040}, NES_COSTUME },
	{ {0x1BFC2,0x27E8F,0x1BF19,0x17ED4}, {0x003C,0x003C,0x003C,0x003C}, NES_COSTUME },
	{ {0x17F90,0x0FF23,0x17EE5,0x13EC2}, {0x003A,0x003A,0x003A,0x003A}, NES_COSTUME },
	{ {0x17F90,0x0FF23,0x17EE5,0x13EC2}, {0x003A,0x003A,0x003A,0x003A}, NES_COSTUME },
	{ {0x17F05,0x0BEFF,0x0FEF5,0x0BF17}, {0x0055,0x0055,0x0055,0x0055}, NES_COSTUME },
	{ {0x13FAB,0x0FEA2,0x17E9A,0x13E77}, {0x004B,0x004B,0x004B,0x004B}, NES_COSTUME }
};
#endif

static ScummNESFile::Resource res_globdata =
	{ {0x2CA11,0x2CA11,0x2C001,0x2C628}, {0x0307,0x0307,0x0307,0x0307}, NES_GLOBDATA };

#ifndef PALMOS_68K
// sprite palette data
static ScummNESFile::Resource res_sprpals[2] = {
	{ {0x0BFC1,0x07F61,0x07F55,0x07ED8}, {0x0010,0x0010,0x0010,0x0010}, NES_SPRPALS },
	{ {0x0BFD1,0x0BEB2,0x07F65,0x07EE8}, {0x0010,0x0010,0x0010,0x0010}, NES_SPRPALS }
};

// associates costume IDs with sprite sets (indexes into SPRLENS/SPROFFS)
static ScummNESFile::Resource res_sprdesc[2] = {
	{ {0x0FFB7,0x0BEC2,0x0BF1B,0x07EF8}, {0x0031,0x0031,0x0031,0x0031}, NES_SPRDESC },
	{ {0x0BFE1,0x07F71,0x07F75,0x07F29}, {0x0009,0x0009,0x0009,0x0009}, NES_SPRDESC }
};

// number of sprites in each set (indicates length within SPRDATA)
static ScummNESFile::Resource res_sprlens[2] = {
	{ {0x0FEA2,0x1BE32,0x13E6A,0x0FE61}, {0x0115,0x0115,0x0115,0x0115}, NES_SPRLENS },
	{ {0x07FF5,0x07F5B,0x07F4F,0x07ED2}, {0x0006,0x0006,0x0006,0x0006}, NES_SPRLENS }
};

// offset of each sprite set (indexes into SPRDATA)
static ScummNESFile::Resource res_sproffs[2] = {
	{ {0x2BDC5,0x2FD42,0x2BCE0,0x2F959}, {0x022A,0x022A,0x022A,0x022A}, NES_SPROFFS },
	{ {0x0BFEA,0x0BEF3,0x0BF4C,0x07F32}, {0x000C,0x000C,0x000C,0x000C}, NES_SPROFFS }
};

// sprite data sets (packed NES sprite data)
static ScummNESFile::Resource res_sprdata[2] = {
	{ {0x2CE11,0x2CE11,0x2C401,0x2CA28}, {0x2BE0,0x2BE0,0x2BE0,0x2BE0}, NES_SPRDATA },
	{ {0x07F6B,0x0BE28,0x0FE6B,0x07E48}, {0x008A,0x008A,0x008A,0x008A}, NES_SPRDATA }
};
#endif

static ScummNESFile::Resource res_charset =
	{ {0x3F6EE,0x3F724,0x3F739,0x3F739}, {0x0090,0x0090,0x0090,0x0090}, NES_CHARSET };

static ScummNESFile::Resource res_preplist =
	{ {0x3FB5A,0x3FB90,0x3FBA9,0x3FBAF}, {0x000E,0x000E,0x000E,0x0010}, NES_PREPLIST };

uint16 write_byte(Common::WriteStream *out, byte val) {
	val ^= 0xFF;
	if (out != 0)
		out->writeByte(val);
	return 1;
}

uint16 write_word(Common::WriteStream *out, uint16 val) {
	val ^= 0xFFFF;
	if (out != 0)
		out->writeUint16LE(val);
	return 2;
}

byte ScummNESFile::fileReadByte() {
	byte b = 0;
	File::read(&b, 1);
	return b;
}

uint16 ScummNESFile::fileReadUint16LE() {
	uint16 a = fileReadByte();
	uint16 b = fileReadByte();
	return a | (b << 8);
}

uint32 ScummNESFile::resOffset(Resource *res) {
	return res->offset[_ROMset];
}

uint16 ScummNESFile::resLength(Resource *res) {
	return res->length[_ROMset];
}

uint16 ScummNESFile::extractResource(Common::WriteStream *output, Resource *res) {
	uint16 len, i, j;
	byte val;
	byte cnt;
	uint16 reslen = 0;

	if (res == NULL)
		error("extract_resource - no resource specified");

	if ((resOffset(res) == 0) && (resLength(res) == 0))
		return 0;	/* there are 8 scripts that are zero bytes long, so we should skip them */

	File::seek(resOffset(res),SEEK_SET);

	switch (res->type) {
	case NES_GLOBDATA:
		len = resLength(res);

		for (i = 0; i < len; i++)
			reslen += write_byte(output, fileReadByte());

		break;

	case NES_ROOMGFX:
	case NES_COSTUMEGFX:
		reslen += write_word(output, (uint16)(resLength(res) + 2));
		len = fileReadByte();
		reslen += write_byte(output, (byte)len);

		if (!len)
			len = 256;
		len = len << 4;

		for (i = 0; i < len;) {
			reslen += write_byte(output, cnt = fileReadByte());
			for (j = 0; j < (cnt & 0x7F); j++, i++)
				if ((cnt & 0x80) || (j == 0))
					reslen += write_byte(output, fileReadByte());
		}

		if (File::pos() - resOffset(res) != resLength(res))
			error("extract_resource - length mismatch while extracting graphics resource (was %04X, should be %04X)", File::pos() - resOffset(res), resLength(res));

		break;

	case NES_ROOM:
	case NES_SCRIPT:
		len = fileReadUint16LE();

		if (len != resLength(res))
			error("extract_resource - length mismatch while extracting room/script resource (was %04X, should be %04X)", len, resLength(res));

		File::seek(-2, SEEK_CUR);

		for (i = 0; i < len; i++)
			reslen += write_byte(output, fileReadByte());

		break;

	case NES_SOUND:
		len = resLength(res) + 2;
		val = fileReadByte();
		cnt = fileReadByte();

		if ((val == 2) && (cnt == 100)) {
			reslen += write_word(output, len);
			reslen += write_byte(output, val);
			reslen += write_byte(output, cnt);

			cnt = fileReadByte();
			reslen += write_byte(output, cnt);
			for (i = 0; i < cnt; i++)
				reslen += write_byte(output, fileReadByte());
			for (i = 0; i < cnt; i++)
				reslen += write_byte(output, fileReadByte());

			while (1) {
				reslen += write_byte(output, val = fileReadByte());
				if (val >= 0xFE)
					break;
			}
		} else if (((val == 0) || (val == 1) || (val == 4)) && (cnt == 10)) {
			reslen += write_word(output, len);
			reslen += write_byte(output, val);
			reslen += write_byte(output, cnt);
			while (1) {
				reslen += write_byte(output, val = fileReadByte());

				if (val >= 0xFE)
					break;

				if (val >= 0x10)
					reslen += write_byte(output, fileReadByte());
				else {
					reslen += write_byte(output, fileReadByte());
					reslen += write_byte(output, fileReadByte());
					reslen += write_byte(output, fileReadByte());
					reslen += write_byte(output, fileReadByte());
				}
			}
		} else
			error("extract_resource - unknown sound type %d/%d detected",val,cnt);

		if (File::pos() - resOffset(res) != resLength(res))
			error("extract_resource - length mismatch while extracting sound resource (was %04X, should be %04X)", File::pos() - resOffset(res), resLength(res));

		break;

	case NES_COSTUME:
	case NES_SPRPALS:
	case NES_SPRDESC:
	case NES_SPRLENS:
	case NES_SPROFFS:
	case NES_SPRDATA:
	case NES_CHARSET:
		len = resLength(res);
		reslen += write_word(output, (uint16)(len + 2));

		for (i = 0; i < len; i++)
			reslen += write_byte(output, fileReadByte());

		break;

	case NES_PREPLIST:
		len = resLength(res);
                reslen += write_word(output, 0x002A);

		reslen += write_byte(output, ' ');
		for (i = 1; i < 8; i++)
			reslen += write_byte(output, 0);

		for (j = 0; j < 4; j++)
		{
			reslen += write_byte(output,' ');
			for (i = 1; (val = fileReadByte()); i++)
				reslen += write_byte(output, val);
			for (; i < 8; i++)
				reslen += write_byte(output, 0);
		}
		break;

	default:
		error("extract_resource - unknown resource type %d specified!", res->type);
	}

	return reslen;
}

// based on structure of Classic PC Maniac Mansion LFL files
// (roomgfx resources are arranged in order, one per file,
// after the room blocks) */
static ScummNESFile::Resource *lfl_01[] = { &res_rooms[1], &res_roomgfx[1], &res_scripts[57], &res_scripts[61], &res_scripts[76], &res_scripts[105], &res_scripts[111], &res_sounds[5], &res_scripts[132], &res_scripts[148], &res_scripts[155], &res_scripts[156], &res_sounds[39], NULL };
static ScummNESFile::Resource *lfl_02[] = { &res_rooms[2], &res_roomgfx[2], NULL };
static ScummNESFile::Resource *lfl_03[] = { &res_rooms[3], &res_roomgfx[3], &res_scripts[21], &res_sounds[26], NULL };
static ScummNESFile::Resource *lfl_04[] = { &res_rooms[4], &res_roomgfx[4], &res_scripts[46], &res_scripts[56], &res_scripts[137], &res_scripts[146], &res_sounds[12], &res_sounds[11], &res_sounds[13], &res_sounds[42], NULL };
static ScummNESFile::Resource *lfl_05[] = { &res_rooms[5], &res_roomgfx[5], &res_scripts[30], &res_scripts[31], &res_scripts[32], &res_scripts[33], &res_scripts[34], &res_scripts[35], &res_sounds[22], &res_sounds[23], &res_sounds[24], &res_sounds[21], &res_sounds[46], NULL };
static ScummNESFile::Resource *lfl_06[] = { &res_rooms[6], &res_roomgfx[6], NULL };
static ScummNESFile::Resource *lfl_07[] = { &res_rooms[7], &res_roomgfx[7], &res_scripts[17], &res_scripts[58], &res_scripts[59], &res_scripts[60], &res_scripts[74], &res_scripts[81], &res_scripts[82], &res_scripts[150], &res_sounds[14], &res_sounds[15], &res_sounds[16], &res_sounds[17], NULL };
static ScummNESFile::Resource *lfl_08[] = { &res_rooms[8], &res_roomgfx[8], &res_scripts[7], &res_scripts[12], &res_scripts[13], &res_scripts[47], &res_scripts[48], &res_scripts[49], &res_scripts[154], &res_sounds[32], &res_sounds[33], &res_sounds[36], NULL };
static ScummNESFile::Resource *lfl_09[] = { &res_rooms[9], &res_roomgfx[9], &res_scripts[10], &res_scripts[11], &res_scripts[45], &res_scripts[55], &res_scripts[84], &res_scripts[85], &res_scripts[86], NULL };
static ScummNESFile::Resource *lfl_10[] = { &res_rooms[10], &res_roomgfx[10], &res_scripts[24], &res_scripts[149], &res_sounds[28], NULL };
static ScummNESFile::Resource *lfl_11[] = { &res_rooms[11], &res_roomgfx[11], &res_scripts[166], &res_scripts[167], &res_scripts[168], NULL };
static ScummNESFile::Resource *lfl_12[] = { &res_rooms[12], &res_roomgfx[12], &res_scripts[51], &res_scripts[103], &res_scripts[104], &res_scripts[161], &res_sounds[63], &res_costumes[14], NULL };
static ScummNESFile::Resource *lfl_13[] = { &res_rooms[13], &res_roomgfx[13], NULL };
static ScummNESFile::Resource *lfl_14[] = { &res_rooms[14], &res_roomgfx[14], NULL };
static ScummNESFile::Resource *lfl_15[] = { &res_rooms[15], &res_roomgfx[15], &res_sounds[27], NULL };
static ScummNESFile::Resource *lfl_16[] = { &res_rooms[16], &res_roomgfx[16], &res_scripts[14], &res_scripts[121], &res_scripts[122], &res_sounds[40], &res_sounds[64], &res_sounds[68], NULL };
static ScummNESFile::Resource *lfl_17[] = { &res_rooms[17], &res_roomgfx[17], &res_scripts[20], &res_scripts[100], &res_sounds[25], &res_sounds[44], &res_sounds[2], &res_sounds[50], &res_sounds[52], NULL };
static ScummNESFile::Resource *lfl_18[] = { &res_rooms[18], &res_roomgfx[18], &res_scripts[25], &res_scripts[26], &res_scripts[27], &res_scripts[28], &res_scripts[64], &res_scripts[65], &res_scripts[66], &res_scripts[67], &res_scripts[68], &res_scripts[69], &res_scripts[70], &res_scripts[71], &res_scripts[73], &res_scripts[101], &res_sounds[35], NULL };
static ScummNESFile::Resource *lfl_19[] = { &res_rooms[19], &res_roomgfx[19], &res_scripts[36], &res_scripts[37], &res_scripts[38], &res_scripts[39], &res_scripts[40], &res_scripts[152], &res_scripts[153], &res_costumes[10], NULL };
static ScummNESFile::Resource *lfl_20[] = { &res_rooms[20], &res_roomgfx[20], &res_scripts[107], &res_scripts[108], &res_scripts[109], &res_scripts[110], &res_scripts[159], NULL };
static ScummNESFile::Resource *lfl_21[] = { &res_rooms[21], &res_roomgfx[21], &res_scripts[41], &res_scripts[42], &res_scripts[43], &res_scripts[53], &res_scripts[136], &res_sounds[29], &res_sounds[20], &res_sounds[37], NULL };
static ScummNESFile::Resource *lfl_22[] = { &res_rooms[22], &res_roomgfx[22], &res_scripts[15], NULL };
static ScummNESFile::Resource *lfl_23[] = { &res_rooms[23], &res_roomgfx[23], &res_scripts[77], &res_scripts[79], &res_scripts[80], &res_scripts[83], &res_sounds[41], NULL };
static ScummNESFile::Resource *lfl_24[] = { &res_rooms[24], &res_roomgfx[24], &res_scripts[18], &res_scripts[19], &res_scripts[78], &res_sounds[7], &res_sounds[3], &res_sounds[18], &res_sounds[34], &res_costumes[12], NULL };
static ScummNESFile::Resource *lfl_25[] = { &res_rooms[25], &res_roomgfx[25], &res_scripts[29], &res_sounds[30], &res_sounds[31], NULL };
static ScummNESFile::Resource *lfl_26[] = { &res_rooms[26], &res_roomgfx[26], &res_scripts[87], &res_scripts[88], &res_scripts[89], &res_scripts[90], &res_scripts[91], &res_scripts[92], &res_scripts[93], &res_scripts[94], &res_scripts[95], &res_scripts[96], &res_scripts[97], &res_scripts[98], &res_scripts[116], &res_scripts[151], &res_scripts[174], &res_costumes[11], NULL };
static ScummNESFile::Resource *lfl_27[] = { &res_rooms[27], &res_roomgfx[27], &res_scripts[16], &res_scripts[52], &res_scripts[54], &res_scripts[113], &res_sounds[45], &res_costumes[19], NULL };
static ScummNESFile::Resource *lfl_28[] = { &res_rooms[28], &res_roomgfx[28], &res_scripts[22], &res_scripts[23], NULL };
static ScummNESFile::Resource *lfl_29[] = { &res_rooms[29], &res_roomgfx[29], &res_scripts[75], &res_sounds[43], NULL };
static ScummNESFile::Resource *lfl_30[] = { &res_rooms[30], &res_roomgfx[30], &res_scripts[63], &res_sounds[0], &res_scripts[123], &res_scripts[125], &res_scripts[126], &res_scripts[127], &res_scripts[129], &res_sounds[55], &res_sounds[59], &res_sounds[60], &res_costumes[8], NULL };
static ScummNESFile::Resource *lfl_31[] = { &res_rooms[31], &res_roomgfx[31], &res_scripts[99], &res_scripts[115], &res_scripts[117], &res_scripts[119], &res_scripts[147], &res_scripts[157], &res_scripts[158], &res_scripts[160], &res_costumes[13], &res_costumes[9], &res_costumes[23], &res_costumes[24], NULL };
static ScummNESFile::Resource *lfl_32[] = { &res_rooms[32], &res_roomgfx[32], &res_costumes[15], NULL };
static ScummNESFile::Resource *lfl_33[] = { &res_rooms[33], &res_roomgfx[33], &res_scripts[120], &res_scripts[135], &res_sounds[56], &res_sounds[57], &res_sounds[58], &res_sounds[1], &res_costumes[22], NULL };
static ScummNESFile::Resource *lfl_34[] = { &res_rooms[34], &res_roomgfx[34], NULL };
static ScummNESFile::Resource *lfl_35[] = { &res_rooms[35], &res_roomgfx[35], NULL };
static ScummNESFile::Resource *lfl_36[] = { &res_rooms[36], &res_roomgfx[36], &res_sounds[10], &res_sounds[4], NULL };
static ScummNESFile::Resource *lfl_37[] = { &res_rooms[37], &res_roomgfx[37], NULL };
static ScummNESFile::Resource *lfl_38[] = { &res_rooms[38], &res_roomgfx[38], &res_scripts[138], &res_scripts[139], &res_scripts[140], &res_scripts[141], &res_scripts[142], &res_scripts[143], &res_scripts[144], &res_scripts[145], NULL };
static ScummNESFile::Resource *lfl_39[] = { &res_rooms[39], &res_roomgfx[39], NULL };
static ScummNESFile::Resource *lfl_40[] = { &res_rooms[40], &res_roomgfx[0], &res_scripts[112], &res_costumes[17], NULL };
static ScummNESFile::Resource *lfl_41[] = { &res_rooms[41], &res_scripts[106], &res_sounds[47], &res_sounds[48], &res_sounds[53], &res_sounds[49], &res_sounds[51], NULL };
static ScummNESFile::Resource *lfl_42[] = { &res_rooms[42], &res_scripts[124], &res_costumes[18],  NULL };
static ScummNESFile::Resource *lfl_43[] = { &res_rooms[43], &res_scripts[44], &res_sounds[19], NULL };
static ScummNESFile::Resource *lfl_44[] = { &res_rooms[44], &res_scripts[102], &res_sounds[6], &res_sounds[38], &res_sounds[8], &res_sounds[9], &res_costumes[1], &res_costumes[2], &res_costumes[5], &res_costumes[6], &res_costumes[3], &res_costumes[4], &res_costumes[7], NULL };
static ScummNESFile::Resource *lfl_45[] = { &res_rooms[45], &res_scripts[1], &res_scripts[2], &res_scripts[3], &res_scripts[4], &res_scripts[5], &res_scripts[9], &res_scripts[114], &res_scripts[131], &res_scripts[164], &res_scripts[165], &res_scripts[169], &res_scripts[170], &res_scripts[171], &res_scripts[172], &res_scripts[173], &res_scripts[175], &res_sounds[54], NULL };
static ScummNESFile::Resource *lfl_46[] = { &res_rooms[46], &res_scripts[130], &res_sounds[65], &res_costumes[0], &res_costumes[21], NULL };
static ScummNESFile::Resource *lfl_47[] = { &res_rooms[47], &res_scripts[62], &res_sounds[69], NULL };
static ScummNESFile::Resource *lfl_48[] = { &res_rooms[48], NULL };
static ScummNESFile::Resource *lfl_49[] = { &res_rooms[49], NULL };
static ScummNESFile::Resource *lfl_50[] = { &res_rooms[50], &res_scripts[133], &res_scripts[163], NULL };
static ScummNESFile::Resource *lfl_51[] = { &res_rooms[51], &res_scripts[118], &res_scripts[128], &res_sounds[61], &res_sounds[62], &res_sounds[67], &res_sounds[66], &res_costumes[16], &res_costumes[20], NULL };
static ScummNESFile::Resource *lfl_52[] = { &res_rooms[52], NULL };
/*	remaining 'standard' resources (not used by any of the original LFL files) */
static ScummNESFile::Resource *lfl_53[] = { &res_rooms[53], &res_scripts[177], &res_scripts[178], &res_sounds[70], &res_sounds[71], &res_sounds[72], &res_sounds[73], &res_sounds[74], &res_sounds[75], &res_sounds[76], &res_sounds[77], &res_sounds[78], &res_sounds[79], &res_sounds[80], &res_sounds[81], NULL };
/*	all 'non-standard' resources (the costume-related stuff) */
static ScummNESFile::Resource *lfl_54[] = { &res_rooms[54], &res_sprdesc[0], &res_sprdesc[1], &res_sprlens[0], &res_sprlens[1], &res_sproffs[0], &res_sproffs[1], &res_sprdata[0], &res_sprdata[1], &res_costumegfx[0], &res_costumegfx[1], &res_sprpals[0], &res_sprpals[1], &res_charset, &res_preplist, NULL };

typedef	struct _lfl {
	int num;
	ScummNESFile::Resource **entries;
} t_lfl, *p_lfl;

t_lfl lfls[] = {
	{  1, lfl_01 },
	{  2, lfl_02 },
	{  3, lfl_03 },
	{  4, lfl_04 },
	{  5, lfl_05 },
	{  6, lfl_06 },
	{  7, lfl_07 },
	{  8, lfl_08 },
	{  9, lfl_09 },
	{ 10, lfl_10 },
	{ 11, lfl_11 },
	{ 12, lfl_12 },
	{ 13, lfl_13 },
	{ 14, lfl_14 },
	{ 15, lfl_15 },
	{ 16, lfl_16 },
	{ 17, lfl_17 },
	{ 18, lfl_18 },
	{ 19, lfl_19 },
	{ 20, lfl_20 },
	{ 21, lfl_21 },
	{ 22, lfl_22 },
	{ 23, lfl_23 },
	{ 24, lfl_24 },
	{ 25, lfl_25 },
	{ 26, lfl_26 },
	{ 27, lfl_27 },
	{ 28, lfl_28 },
	{ 29, lfl_29 },
	{ 30, lfl_30 },
	{ 31, lfl_31 },
	{ 32, lfl_32 },
	{ 33, lfl_33 },
	{ 34, lfl_34 },
	{ 35, lfl_35 },
	{ 36, lfl_36 },
	{ 37, lfl_37 },
	{ 38, lfl_38 },
	{ 39, lfl_39 },
	{ 40, lfl_40 },
	{ 41, lfl_41 },
	{ 42, lfl_42 },
	{ 43, lfl_43 },
	{ 44, lfl_44 },
	{ 45, lfl_45 },
	{ 46, lfl_46 },
	{ 47, lfl_47 },
	{ 48, lfl_48 },
	{ 49, lfl_49 },
	{ 50, lfl_50 },
	{ 51, lfl_51 },
	{ 52, lfl_52 },
	{ 53, lfl_53 },
	{ 54, lfl_54 },
	{ -1, NULL }
};

#pragma START_PACK_STRUCTS
struct _lfl_index {
	byte	room_lfl[55];
	uint16	room_addr[55];
	byte	costume_lfl[80];
	uint16	costume_addr[80];
	byte	script_lfl[200];
	uint16	script_addr[200];
	byte	sound_lfl[100];
	uint16	sound_addr[100];
} GCC_PACK lfl_index;
#pragma END_PACK_STRUCTS


bool ScummNESFile::generateResource(int res) {
	p_lfl lfl = &lfls[res - 1];
	int j;
	int bufsize = 2;

	for (j = 0; lfl->entries[j] != NULL; j++)
		bufsize += extractResource(0, lfl->entries[j]);

	free(_buf);
	_buf = (byte *)calloc(1, bufsize);

	Common::MemoryWriteStream out(_buf, bufsize);

	for (j = 0; lfl->entries[j] != NULL; j++) {
		Resource *entry = lfl->entries[j];
		extractResource(&out, entry);
	}
	write_byte(&out, 0xD1);
	write_byte(&out, 0xF5);

	if (_stream)
		delete _stream;

	_stream = new Common::MemoryReadStream(_buf, bufsize);

	return true;
}

bool ScummNESFile::generateIndex() {
	int i, j;

	for (i = 0; lfls[i].num != -1; i++) {
		p_lfl lfl = &lfls[i];
		uint16 respos = 0;

		for (j = 0; lfl->entries[j] != NULL; j++) {
			Resource *entry = lfl->entries[j];

			switch (entry->type) {
			case NES_ROOM:
				lfl_index.room_lfl[entry - res_rooms] = lfl->num;
				lfl_index.room_addr[entry - res_rooms] = TO_LE_16(respos);
				break;
			case NES_COSTUME:
				lfl_index.costume_lfl[entry - res_costumes] = lfl->num;
				lfl_index.costume_addr[entry - res_costumes] = TO_LE_16(respos);
				break;
			case NES_SPRDESC:
				lfl_index.costume_lfl[entry - res_sprdesc + 25] = lfl->num;
				lfl_index.costume_addr[entry - res_sprdesc + 25] = TO_LE_16(respos);
				break;
			case NES_SPRLENS:
				lfl_index.costume_lfl[entry - res_sprlens + 27] = lfl->num;
				lfl_index.costume_addr[entry - res_sprlens + 27] = TO_LE_16(respos);
				break;
			case NES_SPROFFS:
				lfl_index.costume_lfl[entry - res_sproffs + 29] = lfl->num;
				lfl_index.costume_addr[entry - res_sproffs + 29] = TO_LE_16(respos);
				break;
			case NES_SPRDATA:
				lfl_index.costume_lfl[entry - res_sprdata + 31] = lfl->num;
				lfl_index.costume_addr[entry - res_sprdata + 31] = TO_LE_16(respos);
				break;
			case NES_COSTUMEGFX:
				lfl_index.costume_lfl[entry - res_costumegfx + 33] = lfl->num;
				lfl_index.costume_addr[entry - res_costumegfx + 33] = TO_LE_16(respos);
				break;
			case NES_SPRPALS:
				lfl_index.costume_lfl[entry - res_sprpals + 35] = lfl->num;
				lfl_index.costume_addr[entry - res_sprpals + 35] = TO_LE_16(respos);
				break;
			case NES_ROOMGFX:
				lfl_index.costume_lfl[entry - res_roomgfx + 37] = lfl->num;
				lfl_index.costume_addr[entry - res_roomgfx + 37] = TO_LE_16(respos);
				break;
			case NES_SCRIPT:
				lfl_index.script_lfl[entry - res_scripts] = lfl->num;
				lfl_index.script_addr[entry - res_scripts] = TO_LE_16(respos);
				break;
			case NES_SOUND:
				lfl_index.sound_lfl[entry - res_sounds] = lfl->num;
				lfl_index.sound_addr[entry - res_sounds] = TO_LE_16(respos);
				break;
			case NES_CHARSET:
				lfl_index.costume_lfl[77] = lfl->num;
				lfl_index.costume_addr[77] = TO_LE_16(respos);
				break;
			case NES_PREPLIST:
				lfl_index.costume_lfl[78] = lfl->num;
				lfl_index.costume_addr[78] = TO_LE_16(respos);
				break;
			default:
				error("Unindexed entry found!");
				break;
			}
			respos += extractResource(0, entry);
		}
	}

	int bufsize = 2;

	bufsize += 775;
	bufsize += sizeof(lfl_index);

	free(_buf);
	_buf = (byte *)calloc(1, bufsize);

	Common::MemoryWriteStream out(_buf, bufsize);

	write_byte(&out, 0x43);
	write_byte(&out, 0x46);

	extractResource(&out, &res_globdata);

	for (i = 0; i < (int)sizeof(lfl_index); i++)
		write_byte(&out, ((byte *)&lfl_index)[i]);

	if (_stream)
		delete _stream;

	_stream = new Common::MemoryReadStream(_buf, bufsize);

	return true;
}

bool ScummNESFile::open(const char *filename, AccessMode mode) {
	uint8 md5sum[16];

	if (_ROMset == kROMsetNum) {
		if (Common::md5_file(filename, md5sum)) {
			char md5str[32+1];
			for (int j = 0; j < 16; j++) {
				sprintf(md5str + j*2, "%02x", (int)md5sum[j]);
			}

			if (!strcmp(md5str, "3905799e081b80a61d4460b7b733c206")) {
				_ROMset = kROMsetUSA;
				debug(1, "ROM contents verified as Maniac Mansion (USA)");
			} else if (!strcmp(md5str, "d8d07efcb88f396bee0b402b10c3b1c9")) {
				_ROMset = kROMsetEurope;
				debug(1, "ROM contents verified as Maniac Mansion (Europe)");
			} else if (!strcmp(md5str, "22d07d6c386c9c25aca5dac2a0c0d94b")) {
				_ROMset = kROMsetSweden;
				debug(1, "ROM contents verified as Maniac Mansion (Sweden)");
			} else if (!strcmp(md5str, "81bbfa181184cb494e7a81dcfa94fbd9")) {
				_ROMset = kROMsetFrance;
				debug(2, "ROM contents verified as Maniac Mansion (France)");
			} else {
				error("Unsupported Maniac Mansion ROM, md5: %s", md5str);
				return false;
			}
		} else {
			return false;
		}
	}

	if (File::open(filename, mode)) {
		if (_stream)
			delete _stream;
		_stream = 0;

		free(_buf);
		_buf = 0;

		return true;
	} else {
		return false;
	}
}

void ScummNESFile::close() {
	if (_stream)
		delete _stream;
	_stream = 0;

	free(_buf);
	_buf = 0;

	File::close();
}

bool ScummNESFile::openSubFile(const char *filename) {
	assert(isOpen());

	const char *ext = strrchr(filename, '.');
	char resNum[3];
	int res;

	// We always have file name in form of XX.lfl
	resNum[0] = ext[-2];
	resNum[1] = ext[-1];
	resNum[2] = 0;

	res = atoi(resNum);

	if (res == 0) {
		return generateIndex();
	} else {
		return generateResource(res);
	}
}

#pragma mark -
#pragma mark --- ScummC64File ---
#pragma mark -

static const int maniacResourcesPerFile[55] = {
	 0, 11,  1,  3,  9, 12,  1, 13, 10,  6,
	 4,  1,  7,  1,  1,  2,  7,  8, 19,  9,
	 6,  9,  2,  6,  8,  4, 16,  8,  3,  3,
	12, 12,  2,  8,  1,  1,  2,  1,  9,  1,
	 3,  7,  3,  3, 13,  5,  4,  3,  1,  1,
	 3, 10,  1,  0,  0
};

static const int zakResourcesPerFile[59] = {
	 0, 29, 12, 14, 13,  4,  4, 10,  7,  4,
	14, 19,  5,  4,  7,  6, 11,  9,  4,  4,
	 1,  3,  3,  5,  1,  9,  4, 10, 13,  6,
	 7, 10,  2,  6,  1, 11,  2,  5,  7,  1,
	 7,  1,  4,  2,  8,  6,  6,  6,  4, 13,
	 3,  1,  2,  1,  2,  1, 10,  1,  1
};


ScummC64File::ScummC64File(char *disk1, char *disk2, bool maniac) : _stream(0), _buf(0), _maniac(maniac) {
	_disk1 = disk1;
	_disk2 = disk2;

	_openedDisk = 0;

	if (maniac) {
		_numGlobalObjects = 256;
		_numRooms = 55;
		_numCostumes = 25;
		_numScripts = 160;
		_numSounds = 70;
		_resourcesPerFile = maniacResourcesPerFile;
	} else {
		_numGlobalObjects = 775;
		_numRooms = 59;
		_numCostumes = 38;
		_numScripts = 155;
		_numSounds = 127;
		_resourcesPerFile = zakResourcesPerFile;
	}
}

uint32 ScummC64File::write(const void *, uint32) {
	error("ScummC64File does not support writing!");
	return 0;
}

void ScummC64File::setEnc(byte enc) {
	_stream->setEnc(enc);
}

byte ScummC64File::fileReadByte() {
	byte b = 0;
	File::read(&b, 1);
	return b;
}

uint16 ScummC64File::fileReadUint16LE() {
	uint16 a = fileReadByte();
	uint16 b = fileReadByte();
	return a | (b << 8);
}

bool ScummC64File::openDisk(char num) {
	if (num == '1')
		num = 1;
	if (num == '2')
		num = 2;

	if (_openedDisk != num || !File::isOpen()) {
		if (File::isOpen())
			File::close();

		if (num == 1)
			File::open(_disk1.c_str());
		else if (num == 2)
			File::open(_disk2.c_str());
		else {
			error("ScummC64File::open(): wrong disk (%c)", num);
			return false;
		}

		_openedDisk = num;

		if (!File::isOpen()) {
			error("ScummC64File::open(): cannot open disk (%d)", num);
			return false;
		}
	}
	return true;
}

bool ScummC64File::open(const char *filename, AccessMode mode) {
	uint16 signature;

	// check signature
	openDisk(1);
	File::seek(0);

	signature = fileReadUint16LE();
	if (signature != 0x0A31) {
		error("ScummC64File::open(): signature not found in disk 1!");
		return false;
	}

	extractIndex(0); // Fill in resource arrays

	openDisk(2);
	File::seek(0);

	signature = fileReadUint16LE();
	if (signature != 0x0132)
		error("Error: signature not found in disk 2!\n");

	return true;
}


uint16 ScummC64File::extractIndex(Common::WriteStream *out) {
	int i;
	uint16 reslen = 0;

	openDisk(1);
	File::seek(0);

	// skip signature
	fileReadUint16LE();

	// write expected signature
	reslen += write_word(out, 0x0132);

	// copy object flags
	for (i = 0; i < _numGlobalObjects; i++)
		reslen += write_byte(out, fileReadByte());

	// copy room offsets
	for (i = 0; i < _numRooms; i++) {
		_roomDisks[i] = fileReadByte();
		reslen += write_byte(out, _roomDisks[i]);
	}
	for (i = 0; i < _numRooms; i++) {
		_roomSectors[i] = fileReadByte();
		reslen += write_byte(out, _roomSectors[i]);
		_roomTracks[i] = fileReadByte();
		reslen += write_byte(out, _roomTracks[i]);
	}
	for (i = 0; i < _numCostumes; i++)
		reslen += write_byte(out, fileReadByte());
	for (i = 0; i < _numCostumes; i++)
		reslen += write_word(out, fileReadUint16LE());

	for (i = 0; i < _numScripts; i++)
		reslen += write_byte(out, fileReadByte());
	for (i = 0; i < _numScripts; i++)
		reslen += write_word(out, fileReadUint16LE());

	for (i = 0; i < _numSounds; i++)
		reslen += write_byte(out, fileReadByte());
	for (i = 0; i < _numSounds; i++)
		reslen += write_word(out, fileReadUint16LE());

	return reslen;
}

bool ScummC64File::generateIndex() {
	int bufsize;

	bufsize = extractIndex(0);

	free(_buf);
	_buf = (byte *)calloc(1, bufsize);

	Common::MemoryWriteStream out(_buf, bufsize);

	extractIndex(&out);

	if (_stream)
		delete _stream;

	_stream = new Common::MemoryReadStream(_buf, bufsize);

	return true;
}

uint16 ScummC64File::extractResource(Common::WriteStream *out, int res) {
	const int SectorOffset[36] = {
		0,
		0, 21, 42, 63, 84, 105, 126, 147, 168, 189, 210, 231, 252, 273, 294, 315, 336,
		357, 376, 395, 414, 433, 452, 471,
		490, 508, 526, 544, 562, 580,
		598, 615, 632, 649, 666
	};
	int i;
	uint16 reslen = 0;

	openDisk(_roomDisks[res]);

	File::seek((SectorOffset[_roomTracks[res]] + _roomSectors[res]) * 256);

	for (i = 0; i < _resourcesPerFile[res]; i++) {
		uint16 len = fileReadUint16LE();
		reslen += write_word(out, len);

		for (len -= 2; len > 0; len--)
			reslen += write_byte(out, fileReadByte());
	}

	return reslen;
}

bool ScummC64File::generateResource(int res) {
	int bufsize;

	if (res >= _numRooms)
		return false;

	bufsize = extractResource(0, res);

	free(_buf);
	_buf = (byte *)calloc(1, bufsize);

	Common::MemoryWriteStream out(_buf, bufsize);

	extractResource(&out, res);

	if (_stream)
		delete _stream;

	_stream = new Common::MemoryReadStream(_buf, bufsize);

	return true;
}

void ScummC64File::close() {
	if (_stream)
		delete _stream;
	_stream = 0;

	free(_buf);
	_buf = 0;

	File::close();
}

bool ScummC64File::openSubFile(const char *filename) {
	assert(isOpen());

	const char *ext = strrchr(filename, '.');
	char resNum[3];
	int res;

	// We always have file name in form of XX.lfl
	resNum[0] = ext[-2];
	resNum[1] = ext[-1];
	resNum[2] = 0;

	res = atoi(resNum);

	if (res == 0) {
		return generateIndex();
	} else {
		return generateResource(res);
	}

	return true;
}

} // End of namespace Scumm
