#ifndef __SD_LIB_H__
#define __SD_LIB_H__

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <cstdio>
#include <glib.h>

#include <vector>
#include <functional>

#include "wisetypes.hpp"

const int MAX_FUZZY_DISTANCE= 3; // at most MAX_FUZZY_DISTANCE-1 differences allowed when find similar words

extern bool bIsPureEnglish(const char *str);
extern bool bContainRule(const char* sWord);

struct cacheItem {
  guint32 offset;
  gchar *data;
  //write code here to make it inline
  cacheItem() {data= NULL;}
  ~cacheItem() {g_free(data);}
};

const int WORDDATA_CACHE_NUM = 10;
const int INVALID_INDEX=-100;

struct dictData;

class DictBase {
public:
  DictBase();
  ~DictBase();
  gchar * GetWordData(guint32 idxitem_offset, guint32 idxitem_size);
protected:
  std::string sametypesequence;
  FILE *dictfile;
  dictData *dictdzfile;
private:
  cacheItem cache[WORDDATA_CACHE_NUM];
  gint cache_cur;	
};

//this structure contain all information about dictionary
struct DictInfo {
  std::string ifo_file_name;
  guint32 wordcount;
  std::string bookname;
  std::string author;
  std::string email;
  std::string website;
  std::string date;
  std::string description;
  guint32 index_file_size;
  std::string sametypesequence;
  bool load_from_ifo_file(const std::string& ifofilename);
};

typedef std::vector<DictInfo> DictInfoList;

//#define MEASURE_TIME

class Dict : public DictBase {
public:
  struct WordCoord {
    glong offset;
    glong size;
  };
  struct Character {
    gunichar ch;
    glong index;
    friend bool operator<(const Character& lh, const Character& rh) {
      return lh.ch<rh.ch;
    }
  };
  typedef std::vector<Character> CharList;
private:
  guint32 wordcount;
  std::string bookname;

  FILE *idxfile;
  union {
    gchar **wordlist;
    WordCoord *wordoffset;
  };

  union {
    gchar *idxdatabuffer;
    glong cur_wordindex;
  };

  gchar wordentry_buf[256]; // The length of "word_str" should be less than 256. See src/tools/DICTFILE_FORMAT.
  guint32 wordentry_offset;
  guint32 wordentry_size;
  CharList alphabet;
	
  bool load_ifofile(const char *ifofilename, gulong *idxfilesize);
  bool load_wordoffset_from_cache(const std::string& idxname, const std::string& cache_dir="");
  void save_wordoffset_to_cache(const std::string& idxname, const std::string& cache_dir="");
  void loadwordlist();
  void loadwordoffset();
public:
#ifdef MEASURE_TIME
  static double middle_lookup_time, measure_time;
#endif
  Dict();
  ~Dict();
  bool load(const char *ifofilename, const std::string& cache_dir="");
  inline glong length() { return wordcount; }
  inline const std::string& GetBookname() { return bookname; }
  bool Lookup(const char *sWord, glong& Index);
  bool LookupWithRule(GPatternSpec *pspec, glong *aIndex, int iBuffLen);
  gchar * GetWord(glong index);
  gchar * GetWordData(glong index);
};

//============================================================================
class Dicts {
public:
  struct FileHandler : public std::binary_function<const std::string&, bool, void> {
    virtual ~FileHandler(void) {}
    virtual void operator()(const std::string& filename, bool disable) = 0;
  };

  void (*progress_func)(void);//callback to make possible reaction of
  //application during search

  Dicts(void);
  ~Dicts();

  void Clear(void);

  static  void Load(const TStringsList& dicts_dir_list,
		    const TStringsList& order_list, 
		    const TStringsList& disable_list, 
		    FileHandler& file_handler);
  void Load(const TStringsList& dicts_dir_list, 
	    const TStringsList& order_list, 
	    const TStringsList& disable_list);

  Dict *operator[](int i) { return oLib[i]; }
  
  inline int total_libs(void ) const { return oLib.size(); }

  gchar *poGetCurrentWord(glong * iCurrent);
  gchar *poGetNextWord(const gchar *word, glong *iCurrent);
  gchar *poGetPreWord(glong *iCurrent);
  bool LookupSimilarWord(const gchar* sWord, glong& iWordIndex, int iLib);

  struct TSearchRes {
    int iDict;
    glong iWord;
    TSearchRes(int iDict_, glong iWord_=-1) : 
      iDict(iDict_), iWord(iWord_) {}
  };

  const char *GetWord(const TSearchRes& r) 
  { 
    return oLib[r.iDict]->GetWord(r.iWord); 
  }
  const char *GetData(const TSearchRes& r) 
  { 
    return oLib[r.iDict]->GetWordData(r.iWord); 
  }

  typedef std::vector<TSearchRes> TSearchResList;
  
  bool SimpleLookup(const char *sWord, TSearchResList& rl);
  bool LookupWithFuzzy(const gchar *sWord, TStringsList& rl, size_t res_max);
  bool LookupWithRule(const char *sWord, TStringsList& rl, size_t max_per_dict);
private:
  typedef std::vector<Dict *> TDictsList;
  TDictsList oLib; // word library.
  friend struct DictLoader;

  int iMaxFuzzyDistance;   
	
  static void LoadDir(const std::string& dirname,
		      const TStringsList& order_list,
		      const TStringsList& disable_list,
		      FileHandler& file_handler);

  bool SimpleLookupWord(const gchar* sWord, glong& iWordIndex, int iLib);  
};


#endif//lib.h
