#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>

#include "sfm.h"

#include "horizon.h"
#include "sarreality.h"

#include "obj.h"
#include "objutils.h"

#include "simcb.h"
#include "simutils.h"
#include "simsurface.h"
#include "simcontact.h"
#include "simop.h"
#include "simmanage.h"

#include "sar.h"


int SARSimUpdateScene(
        sar_core_struct *core_ptr, sar_scene_struct *scene
);
int SARSimUpdateSceneObjects(
        sar_core_struct *core_ptr, sar_scene_struct *scene
);


#define POW(x,y)        (((x) > 0.0f) ? pow(x,y) : 0.0f)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))

#define DEGTORAD(d)     ((d) * PI / 180.0)
#define RADTODEG(r)     ((r) * 180.0 / PI)


/*
 *	Updates the given scene structure's members (but does not update
 *	any objects).
 *
 *	Time of day will be updated on the scene, then along with the
 *	primary light position light_pos and color light_color.
 *
 *	Horizon texture will be (re)generated as need per a given interval
 *	(every 5 minutes).
 *
 *	Returns 0 on success and non-zero on error or memory pointer
 *	change.
 */
int SARSimUpdateScene(sar_core_struct *core_ptr, sar_scene_struct *scene)
{
	float scene_lumination_coeff, horizon_sat_max_coeff;
	int prev_tod_code, new_tod_code;
	sar_position_struct *pos;
	sar_color_struct *c;
	sar_scene_horizon_struct *horizon_ptr;
        const sar_option_struct *opt = &core_ptr->option;
	if(scene == NULL)
	    return(-1);

/* Convert hours into seconds. */
#define HTOS(h)	((h) * 3600.0)

	/* Get pointer to horizon structure. */
	horizon_ptr = &scene->horizon;

	/* Update time of day (in seconds) since midnight. */
	scene->tod += (float)lapsed_millitime * time_compression / 1000.0f;

	/* Sanitize time of day. */
	while(scene->tod >= HTOS(24.0))
	    scene->tod -= HTOS(24.0);
	while(scene->tod < HTOS(0.0))
	    scene->tod += HTOS(24.0);

	/* Record previous time of day code. */
	prev_tod_code = scene->tod_code;

	/* Set time of day code depending on new time of day (in seconds)
	 * updated above.
	 */
	if(scene->tod < HTOS(6.0))		/* Before 6:00 am. */
	    new_tod_code = SAR_SCENE_TOD_NIGHT;
	else if(scene->tod < HTOS(7.0))		/* Before 7:00 am. */
	    new_tod_code = SAR_SCENE_TOD_DAWN;
        else if(scene->tod < HTOS(17.0))	/* Before 5:00 pm. */
            new_tod_code = SAR_SCENE_TOD_DAY;
	else if(scene->tod < HTOS(18.0))	/* Before 6:00 pm. */
	    new_tod_code = SAR_SCENE_TOD_DUSK;
	else					/* After 6:00 pm. */
	    new_tod_code = SAR_SCENE_TOD_NIGHT;

	scene->tod_code = new_tod_code;		/* Set new tod_code. */


	/* Calculate global scene lumination coeff for time 4:00 to
	 * 20:00, where 0.0 is darkest and 1.0 is brightest. Darkest
	 * occures at 4:00 and 20:00, brightest is at 12:00.
	 *
	 * Also calculate horizon saturation extreme coefficient, the
	 * coefficient to determine the amount of dawn/dusk saturation
	 * on the horizon.
	 */

	/* First calculate from -1.0 to 1.0 linearly where -1.0 is
	 * at 4:00 and 1.0 is at 20:00.
	 */
	scene_lumination_coeff = (float)((scene->tod - HTOS(12.0)) / HTOS(8.0));
	if(scene_lumination_coeff < 0.0f)
	{
	    /* Before 12:00. */
	    float pow_coeff = (float)POW(-scene_lumination_coeff, 2);

            /* Horizon saturation coefficient, calculate from
             * scene lumination so far (with power of 2).   
             */
	    horizon_sat_max_coeff = (float)CLIP(
                1.0 - pow_coeff,
		0.0, 1.0
            );

	    /* Scene lumination; flip, change sign, and apply non-linear
	     * curvature (power of 5).
	     */
	    pow_coeff = (float)POW(-scene_lumination_coeff, 5);
	    scene_lumination_coeff = (float)CLIP(
		1.0 - pow_coeff,
		0.0, 1.0
	    );
	}
	else
	{
	    /* 12:00 or later. */
            float pow_coeff = (float)POW(scene_lumination_coeff, 2);

            /* Horizon saturation coefficient, calculate from
	     * scene lumination so far (with power of 2).
	     */
            horizon_sat_max_coeff = (float)CLIP(
                1.0 - pow_coeff, 0.0, 1.0
            );

	    /* Scene lumination; flip and apply non-linear
	     * curvature (power of 5).
	     */
	    pow_coeff = (float)POW(scene_lumination_coeff, 5);
	    scene_lumination_coeff = (float)CLIP(
		1.0 - pow_coeff, 0.0, 1.0
	    );
	}

	/* Move scene's global light position relative to the origin based
	 * on the time of day. So this position is really an offset that
	 * needs to be added to the actual camera's position.
	 *
	 * The scene's global light is in a orbital pattern depending on
	 * the time of day, to simulate it as the sun.
	 */
	pos = &scene->light_pos;	/* Scene global light position. */
	if(scene->tod > HTOS(12.0))
	{
	    float orbital_coeff;

	    /* Afternoon or later, calculate orbital position coefficient
	     * linearly from 12:00 (coeff 1.0) to 19:00 (coeff 0.0).
	     */
	    orbital_coeff = (float)(1.0 - MIN(
		(scene->tod - HTOS(12.0)) / HTOS(7.0),
		1.0
	    ));

	    /* Calculate longitude aligned orbital position of light
	     * relative to camera's xy plane.
	     */
	    pos->x = (float)(-cos(orbital_coeff * (0.5 * PI)));
	    pos->y = 0.0f;	/* Calculated later. */
	    pos->z = (float)(sin(orbital_coeff * (0.5 * PI)));
	}
	else
	{
	    float orbital_coeff;

	    /* Morning, calculate orbital position coefficient
             * linearly from 2:00 (coeff 0.0) to 12:00 (coeff 1.0).
             */
            orbital_coeff = (float)MAX(
                (scene->tod - HTOS(5.0)) / HTOS(7.0),
                0.0
            );

            /* Calculate longitude aligned orbital position of light    
             * relative to camera's xy plane.
             */
            pos->x = (float)(cos(orbital_coeff * (0.5 * PI)));
            pos->y = 0.0f;	/* Calculated later. */
            pos->z = (float)(sin(orbital_coeff * (0.5 * PI)));
	}
	/* Apply latitude offset to scene's global light position. */
/*	if(scene->planet_radius > 0.0) */
	if(1)
	{
/*
	    float lat_offset_deg = (float)RADTODEG(pos->y / scene->planet_radius) +
		scene->dms_y_offset;
 */
	    float lat_offset_deg = scene->dms_y_offset;
	    float lat_offset_rad, len;


	    /* Calculate latitude offset in radians. */
	    lat_offset_rad = (float)CLIP(
		DEGTORAD(lat_offset_deg),
		-0.5 * PI, 0.5 * PI
	    );
	    pos->y = (float)(pos->y + sin(-lat_offset_rad));
/*	    pos->z = MAX(pos->z - sin(-lat_offset_rad), 0.0); */

	    /* Make into unit length. */
	    len = (float)SFMHypot3(pos->x, pos->y, pos->z);
	    if(len > 0.0f)
	    {
		pos->x = pos->x / len;
                pos->y = pos->y / len;
                pos->z = pos->z / len;
	    }

            /* Update light position hint indicating if it is above or
             * below the horizon (foundation of z = 0.0).
             */
            scene->light_visibility_hint = (pos->z > 0.0f) ? 1 : 0;
	}

	/* Adjust scene's global light color. */
        c = &scene->light_color;	/* Scene global light color. */
	c->a = 1.0f;
	c->r = scene_lumination_coeff;
	c->g = scene_lumination_coeff;
	c->b = scene_lumination_coeff;


        /* Move scene moon position based on time of day. */
        pos = &scene->moon_pos;		/* Scene global moon position. */
        if(scene->tod > HTOS(12.0))
        {
            /* Before midnight, calculate orbital position coefficient
             * linearly from 17:00 (coeff 0.0) to 24:00 (coeff 1.0).
             */
            float orbital_coeff = (float)MAX(
                (scene->tod - HTOS(17.0)) / HTOS(7.0),
                0.0
            );

            /* Calculate longitude aligned orbital position of light
             * relative to camera's xy plane.
             */
            pos->x = (float)(cos(orbital_coeff * (0.5 * PI)));
            pos->y = 0.0f;	/* Calculated later. */
            pos->z = (float)(sin(orbital_coeff * (0.5 * PI)));
        }
        else
        {
            float orbital_coeff;

            /* After midnight, calculate orbital position coefficient
	     * linearly from 0:00 (coeff 1.0) to 7:00 (coeff 0.0).
	     */
            orbital_coeff = (float)(1.0 - MIN(
                (scene->tod - HTOS(0.0)) / HTOS(7.0),
                1.0
            ));

            /* Calculate longitude aligned orbital position of light
             * relative to camera's xy plane.
             */
            pos->x = (float)(-cos(orbital_coeff * (0.5 * PI)));
            pos->y = 0.0f;       /* Calculated later. */
            pos->z = (float)(sin(orbital_coeff * (0.5 * PI)));
        }
        if(1)
        {
            float lat_offset_deg = scene->dms_y_offset;
            float lat_offset_rad, len;


            /* Calculate latitude offset in radians. */
            lat_offset_rad = (float)CLIP(
                DEGTORAD(lat_offset_deg),
                -0.5 * PI, 0.5 * PI
            );
            pos->y = (float)(pos->y + sin(-lat_offset_rad));

            /* Make into unit length. */
            len = (float)SFMHypot3(pos->x, pos->y, pos->z);
            if(len > 0.0f)
            {
                pos->x = pos->x / len;
                pos->y = pos->y / len;
                pos->z = pos->z / len;
            }

            /* Update moon position hint indicating if it is above or
             * below the horizon (foundation of z = 0.0).
             */
            scene->moon_visibility_hint = (pos->z > 0.0) ? 1 : 0;
        }



	/* Regenerate horizon as needed every 5 minutes. */
	if(horizon_ptr->last_tod != (int)(scene->tod / (5 * 60)))
	{
	    int i, total, tex_num;
	    float rc, gc, bc;		/* Coefficient color change. */
	    float darken_coeff, midpoint = 0.88f;
	    sar_color_struct start_color, end_color;


	    if(opt->runtime_debug)
		printf(
 "SARSimUpdateScene(): Updating horizon textures.\n"
		);

	    /* Update last tod on horizon structure to the current
	     * to current time in 5 minute units so we know when to
	     * update the horizon again.
	     */
	    horizon_ptr->last_tod = (int)(scene->tod / (5 * 60));

/* Creates a new horizon texture on the horizon_ptr. */
#define DO_CREATE_TEXTURE	\
{ \
 tex_num = MAX(horizon_ptr->total_textures, 0); \
 horizon_ptr->total_textures = tex_num + 1; \
 horizon_ptr->texture = (v3d_texture_ref_struct **)realloc( \
  horizon_ptr->texture, \
  horizon_ptr->total_textures * sizeof(v3d_texture_ref_struct *) \
 ); \
 if(horizon_ptr->texture == NULL) \
 { \
  horizon_ptr->total_textures = 0; \
 } \
 else \
 { \
  horizon_ptr->texture[tex_num] = SARCreateHorizonTexture( \
   NULL, /* No name. */ \
   &start_color, &end_color, \
   32, midpoint \
  ); \
 } \
}

	    /* Adjust gradient horizon textures by recreating them.
	     * Do not add lumination/gamma to them, the light color
	     * will be multiplied during drawing.
	     */

	    for(i = 0; i < horizon_ptr->total_textures; i++)
		V3DTextureDestroy(horizon_ptr->texture[i]);
	    free(horizon_ptr->texture);
	    horizon_ptr->texture = NULL;
	    horizon_ptr->total_textures = 0;


	    /* Create textures, starting with most brightest. */
	    memcpy(
		&start_color, &scene->sky_nominal_color,
		sizeof(sar_color_struct)
	    );

	    /* Create 7 horizon textures from order of brightest to
	     * to darkest.
	     */
	    total = 7;
	    for(i = 0; i < total; i++)
	    {
		/* Calculate darken coefficient based on index of current
		 * horizon texture.
		 */
		darken_coeff = (float)i / (float)(total - 1);

		/* Calculate rgb blend values from 1.0 to 2.0. */
		rc = (float)MIN(
		    (scene->sky_brighten_color.r * (1.0 - darken_coeff)) +
		    (scene->sky_darken_color.r * darken_coeff),
		    1.0
		);
                gc = (float)MIN(
                    (scene->sky_brighten_color.g * (1.0 - darken_coeff)) +
                    (scene->sky_darken_color.g * darken_coeff),
                    1.0
                );
                bc = (float)MIN(
                    (scene->sky_brighten_color.b * (1.0 - darken_coeff)) +
                    (scene->sky_darken_color.b * darken_coeff),
                    1.0
                );

		/* Calculate end color. */
		end_color.a = 1.0f;
                end_color.r = (float)MIN(
		    (1.0 * horizon_sat_max_coeff) +
			(rc * (1.0 - horizon_sat_max_coeff)),
		    1.0
		);
                end_color.g = (float)MIN(
                    (1.0 * horizon_sat_max_coeff) +
			(gc * (1.0 - horizon_sat_max_coeff)), 
                    1.0
                );
                end_color.b = (float)MIN(
                    (1.0 * horizon_sat_max_coeff) +
			(bc * (1.0 - horizon_sat_max_coeff)),
                    1.0
                );

		DO_CREATE_TEXTURE
	    }
	}
#undef DO_CREATE_TEXTURE










#undef HTOS

	return(0);
}

/*
 *	First updates the timmings on the scene structure's SFMRealmStruct
 *	and then updates all objects with respect to the given scene and
 *	core structure.
 *
 *	Returns 0 on success and non-zero on error or memory pointer
 *	change.
 */
int SARSimUpdateSceneObjects(
	sar_core_struct *core_ptr, sar_scene_struct *scene
)
{
	int i, status;
	int *obj_total;
        sar_object_struct *obj_ptr, ***obj_pa;
	sar_object_aircraft_struct *obj_aircraft_ptr;
	sar_contact_bounds_struct *cb;
	SFMModelStruct *fdm;
	if(scene == NULL)
	    return(-1);

	/* Update timming on realm. */
	if(scene->realm != NULL)
	{
	    SFMSetTimming(scene->realm, lapsed_millitime);
	    SFMSetTimeCompression(scene->realm, time_compression);
	}


	/* Begin handling each object on core structure. */
        obj_pa = &core_ptr->object;
        obj_total = &core_ptr->total_objects;
        for(i = 0; i < (*obj_total); i++)
        {
            obj_ptr = (*obj_pa)[i];
            if(obj_ptr == NULL)
                continue;

	    status = 0;		/* Reset `SFM handled' value. */

	    /* SFM realm structure allocated? */
	    if(scene->realm != NULL)
	    {
		/* Handle by object type. */
		switch(obj_ptr->type)
		{
		  case SAR_OBJ_TYPE_AIRCRAFT:
		    obj_aircraft_ptr = SAR_OBJ_GET_AIRCRAFT(obj_ptr);
		    if(obj_aircraft_ptr == NULL)
			break;

                    fdm = obj_aircraft_ptr->fdm;
                    if(fdm == NULL)
			break;

		    /* Set object values to SFM structure. */
                    SARSimSetSFMValues(core_ptr, scene, obj_ptr);

                    /* Apply control positions if this is the
                     * player object.
                     */
                    if(obj_ptr == scene->player_obj_ptr)
		    {
                        SARSimApplyGCTL(core_ptr, obj_ptr);
		    }
		    else
		    {

 /* Otherwise apply AI? */

		    }

		    /* Set status to 1, noting that the SFM structure
		     * was handled (which we will actually do so just
		     * below).
		     */
		    status = 1;

                    /* Begin SFM updating. */
                    if(SFMForceApplyArtificial(scene->realm, fdm))
                        break;
/* Swapped, calculating natural forces after artifical gives us
 * more accurate center to ground height.
 */
                    if(SFMForceApplyNatural(scene->realm, fdm))
			break;
		    if(SFMForceApplyControl(scene->realm, fdm))
			break;

                    /* Get new object values from the just processed
		     * SFM.
		     */
                    SARSimGetSFMValues(scene, obj_ptr);
		    break;

 /* Add SFM handling support for other types of objects as needed here. */

		}
	    }
	    /* Check if object survived SFM handling, in other words is
	     * it still allocated?
	     */
	    if(status)
	    {
		/* SFM was handled on the object, so check if object
		 * is still allocated.
		 */
		if(SARObjIsAllocated(*obj_pa, *obj_total, i))
		    obj_ptr = (*obj_pa)[i];
		else
		    continue;
	    }

            /* Apply natural forces to object. */
            if(SARSimApplyNaturalForce(core_ptr, obj_ptr))
		continue;

            /* Apply artificial forces to object. */
            if(SARSimApplyArtificialForce(core_ptr, obj_ptr))
		continue;


            /* Get pointer to object's contact bounds structure (which
	     * may be NULL).
	     */
	    cb = obj_ptr->contact_bounds;

	    /* Can this object crash into other objects? */
            if(((cb != NULL) ? cb->crash_flags : 0) &
		SAR_CRASH_FLAG_CRASH_OTHER
	    )
            {
		/* Perform object to object crash contact check. */
                if(SARSimCrashContactCheck(core_ptr, obj_ptr) > -1)
                {
                    /* Crash contact into another object has occured,
		     * so we need to check if this object is still
                     * allocated.
                     */
                    if(SARObjIsAllocated(*obj_pa, *obj_total, i))
                        obj_ptr = (*obj_pa)[i];
                    else
                        continue;
                }
            }

            /* Rescue basket contact check (only if this object is a
	     * type that has a rescue basket). If a contact is made
	     * then a valid index is returned and the proper procedure
	     * will have been taken.
	     */
	    if(SARSimRescueBasketContactCheck(core_ptr, obj_ptr) > -1)
	    {
		/* Rescue basket contact has occured, need to check if
		 * this object is still allocated.
		 */
		if(SARObjIsAllocated(*obj_pa, *obj_total, i))
		    obj_ptr = (*obj_pa)[i];
		else
		    continue;
            }

            /* Check if the object is `too old', if it is then it
	     * will be deleted.
	     */
            if(SARSimDoMortality(core_ptr, obj_ptr))
	    {
                /* Object has been deleted, check if it is still
		 * allocated (which it probably won't be).
                 */
                if(SARObjIsAllocated(*obj_pa, *obj_total, i))
                    obj_ptr = (*obj_pa)[i];
		else
		    continue;
	    }

/* Add more object updates here. */

        }

	return(0);
}
