/*
 * $Id: securecgi.c,v 1.23 2003/07/28 13:30:50 d3xter Exp $
 *
 * SecureCGI
 * (c) 2001-2002 Krzysztof Bielawski
 * (c) 2001-2003 Piotr Roszatycki <dexter@debian.org>
 *
 * This software is under GPL
 */

#include "config.h"

#include <sys/types.h>
#include <sys/resource.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <errno.h>
#include <libgen.h>
#include <grp.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef HAVE_LIBCAP
#undef _POSIX_SOURCE
#include <sys/capability.h>
#endif

#include <unistd.h>

#include "base64.h"
#include "securecgi.h"


extern char **environ;


#ifdef UNSETENV
/* Environment variables to unset */
char *unsetenv_tab[] = {
    "BASH_ENV",
    "CDPATH",
    "ENV",
    "IFS",
    "REDIRECT_HTTP_AUTHORIZATION",
    "REDIRECT_SCRIPT_URI",
    "REDIRECT_STATUS__",
    "REDIRECT_UNIQUE_ID",
    "REDIRECT_URL",
    "SCGI_ERROR_DOCUMENT_404",
    "SCGI_ERROR_DOCUMENT_500",
    "SCGI_INTERPRETER",
    "SCGI_REMOTE_USER_MODE",
    "SCGI_RLIMIT_CORE",
    "SCGI_RLIMIT_CPU",
    "SCGI_RLIMIT_DATA",
    "SCGI_RLIMIT_FSIZE",
    "SCGI_RLIMIT_MEMLOCK",
    "SCGI_RLIMIT_NOFILE",
    "SCGI_RLIMIT_NPROC",
    "SCGI_RLIMIT_RSS",
    "SCGI_RLIMIT_STACK",
    "SCGI_RLIMIT_AS",
    "SCGI_PRIORITY",
    "SCGI_CAP",
    NULL
};

#else
/* Safe environment variables */
char *safeenv_tab[] =
{
    "AUTH_TYPE",
    "CONTENT_LENGTH",
    "CONTENT_TYPE",
    "DATE_GMT",
    "DATE_LOCAL",
    "DOCUMENT_NAME",
    "DOCUMENT_PATH_INFO",
    "DOCUMENT_ROOT",
    "DOCUMENT_URI",
    "FILEPATH_INFO",
    "GATEWAY_INTERFACE",
    "HOME",
    "LAST_MODIFIED",
    "PATH_INFO",
    "PATH_TRANSLATED",
    "QUERY_STRING",
    "QUERY_STRING_UNESCAPED",
    "REDIRECT_QUERY_STRING",
    "REDIRECT_STATUS",
    "REDIRECT_URL",
    "REMOTE_ADDR",
    "REMOTE_HOST",
    "REMOTE_IDENT",
    "REMOTE_PORT",
    "REMOTE_USER",
    "REQUEST_METHOD",
    "REQUEST_URI",
    "SCRIPT_FILENAME",
    "SCRIPT_NAME",
    "SCRIPT_URI",
    "SCRIPT_URL",
    "SERVER_ADMIN",
    "SERVER_NAME",
    "SERVER_ADDR",
    "SERVER_PORT",
    "SERVER_PROTOCOL",
    "SERVER_SOFTWARE",
    "UNIQUE_ID",
    "USER_NAME",
    "TZ",
    NULL
};
#endif


/*
 * Generate 404 error
 */
static void err404(const char *msg)
{
    char *tmp;

#ifdef NPH_MODE
    printf("HTTP/1.0 404 Not Found\n");
#endif
    if ((tmp = getenv("SCGI_ERROR_DOCUMENT_404")) != NULL && *tmp) {
	printf("Location: %s\n", tmp);
#ifndef NPH_MODE
    } else {
	printf("Status: 404 Not Found\n");
#endif
    }
    printf("Content-type: text/html\n"
	   "\n"
	   "<!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML 2.0//EN\">\n"
	   "<HTML><HEAD>\n"
	   "<TITLE>404 Not Found</TITLE>\n"
	   "</HEAD><BODY>\n"
	   "<H1>Not Found</H1>\n"
	   "<P>%s</P>\n"
	   "</BODY></HTML>\n", 
	   msg
    );
    exit(1);
}


/*
 * Generate 500 error
 */
static void err500(const char *msg)
{
    char *tmp;

#ifdef NPH_MODE
    printf("HTTP/1.0 500 Internal Server Error\nConnection: close\n");
#endif
    if ((tmp = getenv("SCGI_ERROR_DOCUMENT_500")) != NULL && *tmp) {
	printf("Location: %s\n", tmp);
#ifndef NPH_MODE
    } else {
	printf("Status: 500 Internal Server Error\n");
#endif
    }
    printf("Content-type: text/html\n"
	   "\n"
	   "<!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML 2.0//EN\">\n"
	   "<HTML><HEAD>\n"
	   "<TITLE>500 Internal Server Error</TITLE>\n"
	   "</HEAD><BODY>\n"
	   "<H1>Internal Server Error</H1>\n"
	   "<P>%s</P>\n"
	   "</BODY></HTML>\n", 
	   msg
    );
    exit(1);
}


/*
 * Generate 500 error with strerror
 */
static void perr500(const char *msg)
{
    char *tmp;

#ifdef NPH_MODE
    printf("HTTP/1.0 500 Internal Server Error\nConnection: close\n");
#endif
    if ((tmp = getenv("SCGI_ERROR_DOCUMENT_500")) != NULL && *tmp) {
	printf("Location: %s\n", tmp);
#ifndef NPH_MODE
    } else {
	printf("Status: 500 Internal Server Error\n");
#endif
    }
    printf("Content-type: text/html\n"
	   "\n"
	   "<!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML 2.0//EN\">\n"
	   "<HTML><HEAD>\n"
	   "<TITLE>500 Internal Server Error</TITLE>\n"
	   "</HEAD><BODY>\n"
	   "<H1>Internal Server Error</H1>\n"
	   "<P>%s: %s</P>\n"
	   "</BODY></HTML>\n", 
	   msg, strerror(errno)
    );
    exit(1);
}


/*
 * Rename variables
 */
static void rename_env(void)
{
    char **renameenv;
    char **ep;
    int ridx = 0;
    char *tmp, *tmp2;

    if ((renameenv = (char **) calloc(ENVBUF_SIZE, sizeof(char *))) == NULL) {
	err500("Out of memory");
    }

    for (ep = environ; *ep && ridx < ENVBUF_SIZE - 1; ep++) {
	if (!strncmp(*ep, "REDIRECT_", sizeof("REDIRECT_")-1) && 
	  strncmp(*ep, "REDIRECT_STATUS=", sizeof("REDIRECT_STATUS=")-1) && 
	  strncmp(*ep, "REDIRECT_URL=", sizeof("REDIRECT_URL=")-1)) {
	    if ((renameenv[ridx] = strdup(*ep+9)) == NULL) {
		perr500("strdup");
	    }
	    ridx++;
	} else {
	    tmp = strdup(*ep);
	    tmp2 = strchr(tmp, '=');
	    if (tmp2 == NULL) {
		err500("Bad environment variable");
	    }
	    *tmp2 = '\000';
	    tmp2 = malloc(strlen(tmp)+sizeof("REDIRECT_")+1);
	    strcpy(tmp2, "REDIRECT_");
	    strcat(tmp2, tmp);
	    /* Copy only if REDIRECT_* env doesn't exist */
	    if (getenv(tmp2) == NULL) {
	        renameenv[ridx] = *ep;
		ridx++;
	    }
	    free(tmp2);
	    free(tmp);
	}
    }

    renameenv[ridx] = NULL;

    environ = renameenv;
}


#ifndef UNSETENV
/*
 * Clean the environment
 */
static void clean_env(void)
{
    char pathbuf[512];
    char **cleanenv;
    char **ep;
    int cidx = 0;
    int idx;

    if ((cleanenv = (char **) calloc(ENVBUF_SIZE, sizeof(char *))) == NULL) {
	err500("Out of memory");
    }

    sprintf(pathbuf, "PATH=%s", SAFE_PATH);
    if ((cleanenv[cidx] = strdup(pathbuf)) == NULL) {
	perr500("strdup");
    }
    cidx++;

    for (ep = environ; *ep && cidx < ENVBUF_SIZE - 1; ep++) {
	if (!strncmp(*ep, "HTTP_", sizeof("HTTP_")-1)) {
	    cleanenv[cidx] = *ep;
	    cidx++;
	} else {
	    for (idx = 0; safeenv_tab[idx]; idx++) {
		if (!strncmp(*ep, safeenv_tab[idx],
			     strlen(safeenv_tab[idx]))) {
		    cleanenv[cidx] = *ep;
		    cidx++;
		    break;
		}
	    }
	}
    }

    cleanenv[cidx] = NULL;

    environ = cleanenv;
}
#endif


/*
 * SecureCGI main function
 */
int main(int argc, char *argv[])
{
#ifdef HAVE_LIBCAP
    cap_t cap_d;
#endif
    struct rlimit lim;
    uid_t uid;
    gid_t gid;
    int i, ret;
    int remote_user_mode;
    long l;
    char *tmp;
    char *script_path;
    char *dirc, *dname;
    char *capabilities;
    char *remote_user;
    char *auth, *username, *password;
    char *redirect_status;
    char *path_translated, *path_info, *interpreter;
    struct stat st;
    struct passwd *user;

    /* Check if wrapper has suid */
    if (geteuid() != 0) {
	err500("Not SUIDed");
    }

    /* Only WWW server can run wrapper */
    uid = getuid();
    if (uid != 0 && uid != SCGI_SERVER_UID) {
	err500("Invalid server user");
    }

    /* Rename env vars if run by Apache */
    redirect_status = getenv("REDIRECT_STATUS");
    if (redirect_status != NULL) {
	rename_env();
    }

    /* Get path to interpreter */
    interpreter = getenv("SCGI_INTERPRETER");

    /* Check if interpreter is valid */
    if (interpreter != NULL) {
	if (stat(interpreter, &st)) {
	    perr500("stat");
	}

	if (!(S_ISREG(st.st_mode))) {
	    err500("Interpreter is not a regular file");
	}
    }

    /* Get file name of script */
    path_translated = getenv("PATH_TRANSLATED");
    if (path_translated == NULL) {
	err500("Undefined PATH_TRANSLATED");
    }

    /* Duplicate path to further usage */        
    if ((script_path = strdup(path_translated)) == NULL) {
        perr500("strdup");
    }

    /* Get mode for changing user */
    if ((tmp = getenv("SCGI_REMOTE_USER_MODE")) != NULL && atoi(tmp) == 1) {
	remote_user_mode = 1;
    } else {
	remote_user_mode = 0;
    }

    /* Ignore PATH_TRANSLATED if SCGI_REMOTE_USER_MODE is set and interpreter is set */
    if (!remote_user_mode || interpreter == NULL) {
	/* Find real script path */
        ret = 1;
	if (*script_path == '/') {
	    for (; *script_path && (ret=stat(script_path, &st)); *strrchr(script_path, '/') = '\000');
	} else {
	    ret=stat(script_path, &st);
	}

	/* Stat returned error */
	if (ret != 0) {
	    err404("Script not found");
	}

	/* Stat the script file */
	if (!(S_ISREG(st.st_mode))) {
	    err404("Script not found");
	}
    }

    /* Set user based on passwd entry if SCGI_REMOTE_USER_MODE is set */
    if (remote_user_mode)  {
	remote_user = getenv("REMOTE_USER");
	if (remote_user == NULL) {
	    err500("Undefined REMOTE_USER");
	}

	if ((user = (struct passwd *) getpwnam(remote_user)) == NULL) {
	    err500("Unknown user");
	}
	
	uid = user->pw_uid;
	gid = user->pw_gid;
	
	if (uid == 0) {
	    err500("Invalid auth user");
	}

	if (gid == 0) {
	    err500("Invalid auth group");
	}
    } else {
	user = NULL;
	uid = st.st_uid;
	gid = st.st_gid;

        if (uid == 0) {
	    err500("Invalid script user");
	}

	if (gid == 0) {
	    err500("Invalid script group");
	}
    }

#define SETRLIMIT(name, resource) \
    if ((tmp = getenv(name)) != NULL) { \
	if (strcmp(tmp, "RLIM_INFINITY") == 0 || strcmp(tmp, "INF") == 0) { \
	    l = RLIM_INFINITY; \
	} else { \
	    l = atol(tmp); \
	} \
	lim.rlim_max = l; \
	lim.rlim_cur = l; \
	if (setrlimit(resource, &lim) != 0) { \
	    perr500(name); \
	} \
    }

    SETRLIMIT("SCGI_RLIMIT_CORE", RLIMIT_CORE);
    SETRLIMIT("SCGI_RLIMIT_CPU", RLIMIT_CPU);
    SETRLIMIT("SCGI_RLIMIT_DATA", RLIMIT_DATA);
    SETRLIMIT("SCGI_RLIMIT_FSIZE", RLIMIT_FSIZE);
    SETRLIMIT("SCGI_RLIMIT_MEMLOCK", RLIMIT_MEMLOCK);
    SETRLIMIT("SCGI_RLIMIT_NOFILE", RLIMIT_NOFILE);
    SETRLIMIT("SCGI_RLIMIT_NPROC", RLIMIT_NPROC);
    SETRLIMIT("SCGI_RLIMIT_RSS", RLIMIT_RSS);
    SETRLIMIT("SCGI_RLIMIT_STACK", RLIMIT_STACK);
    SETRLIMIT("SCGI_RLIMIT_AS", RLIMIT_AS);
    SETRLIMIT("SCGI_RLIMIT_LOCKS", RLIMIT_LOCKS);

#undef SETRLIMIT

    if ((tmp = getenv("SCGI_PRIORITY")) != NULL) {
	i = atoi(tmp);
	if (setpriority(PRIO_PROCESS, getpid(), i)) {
	    perr500("SCGI_PRIORITY");
	}
    }

#ifdef HAVE_LIBCAP
    /* Drop inherited capabilities */
    if ((capabilities = getenv("SCGI_CAP")) != NULL) {
	if ((tmp = malloc(strlen(capabilities) + sizeof("=ep +i"))) == NULL) {
	    err500("Out of memory");
	}
	if (strlen(capabilities) > 0) {
	    sprintf(tmp, "=ep %s+i", capabilities);
	} else {
	    sprintf(tmp, "=ep");
	}
	if ((cap_d = cap_from_text(tmp)) == NULL) {
	    perr500("cap_from_text");
	}
	if ((capsetp(getpid(), cap_d)) != 0) {
	    perr500("capsetp");
	}
	free(tmp);
    }
#endif

    /* Change personality */
    setgroups(0, NULL);
    setregid(gid, gid);
    setreuid(uid, uid);
    
    /* Get the passwd entry for new user */
    if (user == NULL) {
	user = (struct passwd *) getpwuid(uid);
    }

    /* Prepare environment */
    if (user != NULL) {
	setenv("USER", user->pw_name, 1);
	setenv("HOME", user->pw_dir, 1);
    }
    
    setenv("WRAPPER_PATH", argv[0], 1);
    setenv("SCRIPT_FILENAME", script_path, 1);

    path_info = getenv("PATH_INFO");
    if (path_info != NULL) {
	setenv("SCRIPT_NAME", path_info, 1);
    }

    /* Clean up environment */
#ifdef UNSETENV
    for (i = 0; unsetenv_tab[i] != NULL; i++) {
	unsetenv(unsetenv_tab[i]);
    }
#else
    clean_env();
#endif
    
    /* Change current work directory */
    if ((dirc = strdup(script_path)) == NULL) {
	perr500("strdup");
    }
    dname = dirname(dirc);
    if (chdir(dname)) {
	perr500("chdir");
    }
    free(dirc);

    /* HTTP authorization */
    if ((tmp = getenv("HTTP_AUTHORIZATION")) != NULL
	&& strncmp(tmp, "Basic ", sizeof("Basic ") - 1) == 0) {
	/* Cut off "Basic " */
	tmp += 6;			
	if ((auth = base64_decode(tmp)) == NULL) {
	    perr500("base64_decode");
	}
	username = (char *) strtok(auth, ":");
	password = (char *) strtok(NULL, ":");
	setenv("REMOTE_USER", username, 1);
	setenv("REMOTE_PASSWORD", password, 1);
	setenv("PHP_AUTH_USER", username, 1);
	setenv("PHP_AUTH_PW", password, 1);
	free(auth);
    }

    /* Execute real script */
    if (interpreter != NULL) {
	execl(interpreter, interpreter, script_path, NULL);
	perr500("execl");
    } else {
	execl(script_path, script_path, NULL);
	perr500("execl");
    }

    /* Not reached */
    return 0;
}
