/*
 * Copyright (C) 2005 Alex Murray <murray.alex@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include "sensors-applet-gconf.h"
#include "prefs-dialog.h"

void prefs_dialog_show(SensorsApplet *sensors_applet) {
	gtk_widget_show_all(GTK_WIDGET(sensors_applet->prefs_dialog->dialog));
	gtk_window_present(GTK_WINDOW(sensors_applet->prefs_dialog->dialog));
}

/* when a user closes the prefs-dialog we assume that applet is now
   setup, so store all values in gconf */
static void prefs_dialog_close(SensorsApplet *sensors_applet) {
	if (sensors_applet->sensors != NULL) {
		if (sensors_applet_gconf_save_sensors(sensors_applet))
			panel_applet_gconf_set_bool(sensors_applet->applet, IS_SETUP, TRUE, NULL);
	}
	gtk_widget_hide(GTK_WIDGET(sensors_applet->prefs_dialog->dialog));
}

static void prefs_dialog_fahrenheit_toggled(GtkToggleButton *button, SensorsApplet *sensors_applet) {
	gboolean value;
	g_object_get(button, "active", &value, NULL);
	panel_applet_gconf_set_bool(sensors_applet->applet, FAHRENHEIT, value, NULL);
	sensors_applet_update_display(sensors_applet);
}
		
static void prefs_dialog_show_labels_toggled(GtkToggleButton *button, SensorsApplet *sensors_applet) {
	gboolean value;
	g_object_get(button, "active", &value, NULL);
	panel_applet_gconf_set_bool(sensors_applet->applet, SHOW_LABELS, value, NULL);
	sensors_applet_pack_labels(sensors_applet);
}

static void prefs_dialog_labels_inline_toggled(GtkToggleButton *button, SensorsApplet *sensors_applet) {
	gboolean value;
	g_object_get(button, "active", &value, NULL);
	panel_applet_gconf_set_bool(sensors_applet->applet, LABELS_INLINE, value, NULL);
	sensors_applet_pack_labels(sensors_applet);
}

static void prefs_dialog_show_units_toggled(GtkToggleButton *button, SensorsApplet *sensors_applet) {
	gboolean value;
	g_object_get(button, "active", &value, NULL);
	panel_applet_gconf_set_bool(sensors_applet->applet, SHOW_UNITS, value, NULL);
	sensors_applet_update_display(sensors_applet);
}

static void prefs_dialog_num_sensors_changed(GtkSpinButton *button, SensorsApplet *sensors_applet) {
	gint value;
	value = gtk_spin_button_get_value_as_int(button);
	panel_applet_gconf_set_int(sensors_applet->applet, NUM_SENSORS, value, NULL);
	sensors_applet_pack_labels(sensors_applet);
}

static void prefs_dialog_timeout_changed(GtkSpinButton *button, SensorsApplet *sensors_applet) {
	gint value;
	value = gtk_spin_button_get_value_as_int(button) * 1000;
	panel_applet_gconf_set_int(sensors_applet->applet, TIMEOUT, value, NULL);
	/* remove old timeout, and add new one */
	g_source_remove(sensors_applet->timeout_id);
	sensors_applet->timeout_id = g_timeout_add(value, (GSourceFunc)sensors_applet_update_display, sensors_applet);
}

static void prefs_dialog_font_size_changed(GtkComboBox *font_size_combo_box,
					   SensorsApplet *sensors_applet) {
	panel_applet_gconf_set_int(sensors_applet->applet,
				   FONT_SIZE,
				   gtk_combo_box_get_active(font_size_combo_box),
				   NULL);
	
	sensors_applet_update_display(sensors_applet);
}

/* callbacks for the tree of sensors */
static void prefs_dialog_sensor_toggled(GtkCellRenderer *renderer, gchar *path_str, SensorsApplet *sensors_applet) {
	GtkTreeIter iter;
	GtkTreePath *path = gtk_tree_path_new_from_string(path_str);
	gboolean value;
	gint alarm_timeout_index;
	
	gtk_tree_model_get_iter(GTK_TREE_MODEL(sensors_applet->sensors), &iter, path);
	gtk_tree_model_get(GTK_TREE_MODEL(sensors_applet->sensors), 
			   &iter, 
			   ENABLE_COLUMN, &value, 
			   ALARM_TIMEOUT_INDEX_COLUMN, &alarm_timeout_index,
			   -1);
	
	if (value == TRUE && alarm_timeout_index != -1) {
		/* alarm is on but about to turn off sensor so need to
		   kill alarm also */
		if (!g_source_remove(alarm_timeout_index)) {
			g_printf("Error removing alarm source\n");
		}
		gtk_tree_store_set(sensors_applet->sensors,
				   &iter,
				   ALARM_TIMEOUT_INDEX_COLUMN, -1,
				   -1);
	}
	value = !value;
	gtk_tree_store_set(sensors_applet->sensors, &iter, ENABLE_COLUMN, value, -1);
	
	gtk_tree_path_free(path);

	/* need to update entire display if a sensor has been added /
	   removed */
	sensors_applet_update_display(sensors_applet);
}

static void prefs_dialog_sensor_name_changed(GtkCellRenderer *renderer, gchar *path_str, gchar *new_text, SensorsApplet *sensors_applet) {
	GtkTreeIter iter;
	GtkTreePath *path = gtk_tree_path_new_from_string(path_str);
	
	gtk_tree_model_get_iter(GTK_TREE_MODEL(sensors_applet->sensors), &iter, path);

	gtk_tree_store_set(sensors_applet->sensors, &iter, LABEL_COLUMN, new_text, -1);
	
	gtk_tree_path_free(path);

	sensors_applet_update_display(sensors_applet);
}

static void prefs_dialog_sensor_config_button_clicked(GtkButton *button, SensorsApplet *sensors_applet) {
	sensor_config_dialog_create(button, sensors_applet);
}

/* if a row is selected, make sure button is able to be clicked */
static void prefs_dialog_selection_changed(GtkTreeSelection *selection, SensorsApplet *sensors_applet) {
	if (gtk_tree_selection_get_selected(selection, NULL, NULL)) {
		gtk_widget_set_sensitive(GTK_WIDGET(sensors_applet->prefs_dialog->sensor_config_button), TRUE);
	} else {
		gtk_widget_set_sensitive(GTK_WIDGET(sensors_applet->prefs_dialog->sensor_config_button), FALSE);
	}
}

void prefs_dialog_init(PrefsDialog *prefs_dialog) {
	g_assert(prefs_dialog != NULL);
	g_assert(prefs_dialog->sensors_applet != NULL);
	SensorsApplet *sensors_applet = prefs_dialog->sensors_applet;

	prefs_dialog->dialog = GTK_DIALOG(gtk_dialog_new_with_buttons(_("Sensors Applet Preferences"),
										      NULL,
										      GTK_DIALOG_NO_SEPARATOR,
										      GTK_STOCK_CLOSE,
										      GTK_RESPONSE_CLOSE,
										      NULL));
	g_object_set(prefs_dialog->dialog, 
		     "border-width", 12,
		     "default-width", 350,
		     "default-height", 350,
		     NULL);
	
	gtk_box_set_homogeneous(GTK_BOX(prefs_dialog->dialog->vbox), FALSE);
	
	gtk_box_set_spacing(GTK_BOX(prefs_dialog->dialog->vbox), 5);
	
	gtk_dialog_set_default_response(prefs_dialog->dialog, GTK_RESPONSE_CLOSE);
	
	g_signal_connect_swapped(prefs_dialog->dialog,
				 "response", G_CALLBACK(prefs_dialog_close), 
				 prefs_dialog->sensors_applet);
	
	g_signal_connect_swapped(prefs_dialog->dialog,
				 "delete-event", G_CALLBACK(prefs_dialog_close), 
				 prefs_dialog->sensors_applet);
	
	/* if no SensorsList's have been created, this is because
	   we haven't been able to access any sensors */
	if (sensors_applet->sensors == NULL) {
		GtkWidget *label;
		label = gtk_label_new(_("No sensors found!"));
		gtk_box_pack_start_defaults(GTK_BOX(prefs_dialog->dialog->vbox), label);
		return;
	}
	
	/* Create checkbuttons and attach signal handlers */
	prefs_dialog->show_labels_checkbutton = g_object_new(GTK_TYPE_CHECK_BUTTON,
							     "use-underline", TRUE,
							     "label", _("Show _labels"),
							     "active", panel_applet_gconf_get_bool(prefs_dialog->sensors_applet->applet, SHOW_LABELS, NULL),
							     NULL);
	
	g_signal_connect(prefs_dialog->show_labels_checkbutton, "toggled", G_CALLBACK(prefs_dialog_show_labels_toggled), prefs_dialog->sensors_applet);
	
	prefs_dialog->labels_inline_checkbutton = g_object_new(GTK_TYPE_CHECK_BUTTON,
							       "use-underline", TRUE,
							       "label", _("Display labels _inline with sensor values"),
							       "active", panel_applet_gconf_get_bool(prefs_dialog->sensors_applet->applet, LABELS_INLINE, NULL),
							       NULL);
	
	g_signal_connect(prefs_dialog->labels_inline_checkbutton, "toggled", G_CALLBACK(prefs_dialog_labels_inline_toggled), prefs_dialog->sensors_applet);
	
	prefs_dialog->show_units_checkbutton = g_object_new(GTK_TYPE_CHECK_BUTTON,
							    "use-underline", TRUE,
							    "label", _("Display _units with sensor values"),
							    "active", panel_applet_gconf_get_bool(prefs_dialog->sensors_applet->applet, SHOW_UNITS, NULL),
							    NULL);
	
	g_signal_connect(prefs_dialog->show_units_checkbutton, "toggled", G_CALLBACK(prefs_dialog_show_units_toggled), prefs_dialog->sensors_applet);
	
	
	prefs_dialog->fahrenheit_checkbutton = g_object_new(GTK_TYPE_CHECK_BUTTON,
							    "use-underline", TRUE,
							    "label", _("Use _fahrenheit for temperature sensors"),
							    "active", panel_applet_gconf_get_bool(prefs_dialog->sensors_applet->applet, FAHRENHEIT, NULL),
							    NULL);
	
	g_signal_connect(prefs_dialog->fahrenheit_checkbutton, "toggled", G_CALLBACK(prefs_dialog_fahrenheit_toggled), prefs_dialog->sensors_applet);
	
	prefs_dialog->num_sensors_adjust = g_object_new(GTK_TYPE_ADJUSTMENT,
							"value", 1.0,
							"lower", 1.0,
							"upper", 10.0,
							"step-increment", 1.0,
							"page-increment", 1.0,
							"page-size", 0.0,
							NULL);
	
	prefs_dialog->num_sensors_spinbutton = g_object_new(GTK_TYPE_SPIN_BUTTON,
							    "adjustment", prefs_dialog->num_sensors_adjust,
							    "climb-rate", 1.0,
							    "digits", 0,
							    "value", (gdouble) panel_applet_gconf_get_int(prefs_dialog->sensors_applet->applet, NUM_SENSORS, NULL),
							    "width-chars", 4,
							    NULL);

	prefs_dialog->num_sensors_label = g_object_new(GTK_TYPE_LABEL,
						       "use-underline", TRUE,
						       "label", _("Number of sensors per _group"),
						       "mnemonic-widget", prefs_dialog->num_sensors_spinbutton,
						       NULL);
	
	gtk_label_set_justify(prefs_dialog->num_sensors_label, GTK_JUSTIFY_RIGHT);

	g_signal_connect(prefs_dialog->num_sensors_spinbutton, "value-changed", G_CALLBACK(prefs_dialog_num_sensors_changed), prefs_dialog->sensors_applet);
	
	prefs_dialog->globals_hbox[0] = g_object_new(GTK_TYPE_HBOX,
					     "homogeneous", FALSE,
					     "spacing", 0,
					     NULL);

	gtk_box_pack_end(prefs_dialog->globals_hbox[0], GTK_WIDGET(prefs_dialog->num_sensors_spinbutton), FALSE, FALSE, 0);
	gtk_box_pack_end(prefs_dialog->globals_hbox[0], GTK_WIDGET(prefs_dialog->num_sensors_label), FALSE, FALSE, 3);

	prefs_dialog->timeout_adjust = g_object_new(GTK_TYPE_ADJUSTMENT,
						    "value", 2.0,
						    "lower", 1.5,
						    "upper", 10.0,
						    "step-increment", 0.5,
						    "page-increment", 1.0,
						    "page-size", 0.0,
						    NULL);
	
	prefs_dialog->timeout_spinbutton = g_object_new(GTK_TYPE_SPIN_BUTTON,
							    "adjustment", prefs_dialog->timeout_adjust,
							    "climb-rate", 0.5,
							    "digits", 1,
							    "value", (gdouble) panel_applet_gconf_get_int(prefs_dialog->sensors_applet->applet, TIMEOUT, NULL) / 1000.0,
							    "width-chars", 4,
							    NULL);

	prefs_dialog->timeout_label = g_object_new(GTK_TYPE_LABEL,
						   "use-underline", TRUE,
						   "label", _("Refresh _delay (secs)"),
						   "mnemonic-widget", prefs_dialog->timeout_spinbutton,
						   NULL);
	gtk_label_set_justify(prefs_dialog->timeout_label, GTK_JUSTIFY_RIGHT);


	g_signal_connect(prefs_dialog->timeout_spinbutton, "value-changed", G_CALLBACK(prefs_dialog_timeout_changed), prefs_dialog->sensors_applet);
	

	prefs_dialog->globals_hbox[1] = g_object_new(GTK_TYPE_HBOX,
					     "homogeneous", FALSE,
					     "spacing", 0,
					     NULL);

	gtk_box_pack_end(prefs_dialog->globals_hbox[1], GTK_WIDGET(prefs_dialog->timeout_spinbutton), FALSE, FALSE, 0);
	gtk_box_pack_end(prefs_dialog->globals_hbox[1], GTK_WIDGET(prefs_dialog->timeout_label), FALSE, FALSE, 3);
	
	
	prefs_dialog->font_size_combo_box = GTK_COMBO_BOX(gtk_combo_box_new_text());
	gtk_combo_box_append_text(prefs_dialog->font_size_combo_box, _("largest"));
	gtk_combo_box_append_text(prefs_dialog->font_size_combo_box, _("larger"));
	gtk_combo_box_append_text(prefs_dialog->font_size_combo_box, _("large"));
	gtk_combo_box_append_text(prefs_dialog->font_size_combo_box, _("normal"));
	gtk_combo_box_append_text(prefs_dialog->font_size_combo_box, _("small"));
	gtk_combo_box_append_text(prefs_dialog->font_size_combo_box, _("smaller"));
	gtk_combo_box_append_text(prefs_dialog->font_size_combo_box, _("smallest"));
	
	
	gtk_combo_box_set_active(prefs_dialog->font_size_combo_box, panel_applet_gconf_get_int(prefs_dialog->sensors_applet->applet, FONT_SIZE, NULL));

	prefs_dialog->font_size_label = g_object_new(GTK_TYPE_LABEL,
						     "use-underline", TRUE,
						     "label", _("Font _size"),
						     "mnemonic-widget", prefs_dialog->font_size_combo_box,
						     NULL);
	
	g_signal_connect(prefs_dialog->font_size_combo_box,
			 "changed",
			 G_CALLBACK(prefs_dialog_font_size_changed),
			 prefs_dialog->sensors_applet);

	prefs_dialog->globals_hbox[2] = g_object_new(GTK_TYPE_HBOX,
					     "homogeneous", FALSE,
					     "spacing", 0,
					     NULL);

	gtk_box_pack_end(prefs_dialog->globals_hbox[2], GTK_WIDGET(prefs_dialog->font_size_combo_box), FALSE, FALSE, 0);
	gtk_box_pack_end(prefs_dialog->globals_hbox[2], GTK_WIDGET(prefs_dialog->font_size_label), FALSE, FALSE, 3);
	
	prefs_dialog->view = g_object_new(GTK_TYPE_TREE_VIEW,
					  "model", GTK_TREE_MODEL(sensors_applet->sensors),		     
					  "rules-hint", TRUE,
					  "reorderable", FALSE,
					  "enable-search", TRUE,
					  "search-column", LABEL_COLUMN,
					  NULL);
	
	prefs_dialog->id_renderer = gtk_cell_renderer_text_new();
	prefs_dialog->label_renderer = gtk_cell_renderer_text_new();
	g_object_set(prefs_dialog->label_renderer,
		     "editable", TRUE,
		     NULL);
	
	g_signal_connect(prefs_dialog->label_renderer, "edited", G_CALLBACK(prefs_dialog_sensor_name_changed), sensors_applet);

	prefs_dialog->enable_renderer = gtk_cell_renderer_toggle_new();
	g_signal_connect(prefs_dialog->enable_renderer, "toggled", G_CALLBACK(prefs_dialog_sensor_toggled), sensors_applet);
	
	prefs_dialog->id_column = gtk_tree_view_column_new_with_attributes(_("Sensor"),
										 prefs_dialog->id_renderer,
										 "text", ID_COLUMN,
										 NULL);
	
	gtk_tree_view_column_set_min_width(prefs_dialog->id_column, 90);
	
	prefs_dialog->label_column = gtk_tree_view_column_new_with_attributes(_("Label"),
									      prefs_dialog->label_renderer,
									      "text", LABEL_COLUMN,
									       "visible", VISIBLE_COLUMN,
									      NULL);
	
	gtk_tree_view_column_set_min_width(prefs_dialog->label_column, 150);

	/* create the tooltip */
	prefs_dialog->view_tooltips = gtk_tooltips_new();
	gtk_tooltips_set_tip(prefs_dialog->view_tooltips,
			     GTK_WIDGET(prefs_dialog->view),
			     _("Labels can be edited directly by clicking on them."),
			     "");
			     
	prefs_dialog->enable_column = gtk_tree_view_column_new_with_attributes(_("Enabled"),
									       prefs_dialog->enable_renderer,
									       "active", ENABLE_COLUMN,
									       "visible", VISIBLE_COLUMN,
									       NULL);
	

	gtk_tree_view_append_column(prefs_dialog->view, prefs_dialog->id_column);
	gtk_tree_view_append_column(prefs_dialog->view, prefs_dialog->label_column);
	gtk_tree_view_append_column(prefs_dialog->view, prefs_dialog->enable_column);
	prefs_dialog->scrolled_window = g_object_new(GTK_TYPE_SCROLLED_WINDOW,
						     "hadjustment", NULL, 
						     "height-request", 200,
						     "hscrollbar-policy", GTK_POLICY_AUTOMATIC,
						     "vadjustment",NULL,
						     "vscrollbar-policy", GTK_POLICY_AUTOMATIC,
						     NULL);

	gtk_container_add(GTK_CONTAINER(prefs_dialog->scrolled_window), GTK_WIDGET(prefs_dialog->view));

	/* GtkTree Selection */
	prefs_dialog->selection = gtk_tree_view_get_selection(prefs_dialog->view);
	/* allow user to only select one row at a time at most */
	gtk_tree_selection_set_mode(prefs_dialog->selection, GTK_SELECTION_SINGLE);
	/* when selection is changed, make sure sensor_config button is
	   activated */

	/* Sensor Config button */
	/* initially make button insensitive until user selects a row
	   from the sensors tree */
	prefs_dialog->sensor_config_button = g_object_new(GTK_TYPE_BUTTON,
						  "label", _("Configure _Sensor"),
						  "use-underline", TRUE,
						  "sensitive", FALSE,
						  NULL);
	

	g_signal_connect(prefs_dialog->selection,
			 "changed",
			 G_CALLBACK(prefs_dialog_selection_changed),
			 sensors_applet);
	
	/* pass selection to signal handler so we can give user a
	   dialog with the selected rows alarm value and enable */
	g_signal_connect(prefs_dialog->sensor_config_button, "clicked", G_CALLBACK(prefs_dialog_sensor_config_button_clicked), sensors_applet);

	prefs_dialog->sensors_vbox = g_object_new(GTK_TYPE_VBOX,
					  "border-width", 5,
					  "homogeneous", FALSE,
					  "spacing", 0,
					  NULL);

	/* pack sensors_vbox */
	gtk_box_pack_start(prefs_dialog->sensors_vbox, GTK_WIDGET(prefs_dialog->scrolled_window), FALSE, FALSE, 0);
	prefs_dialog->sensors_hbox = g_object_new(GTK_TYPE_HBOX,
						  "border-width", 5,
						  "homogeneous", FALSE,
						  "spacing", 0,
						  NULL);
	gtk_box_pack_end(prefs_dialog->sensors_hbox, GTK_WIDGET(prefs_dialog->sensor_config_button), FALSE, FALSE, 0);
	gtk_box_pack_start(prefs_dialog->sensors_vbox, GTK_WIDGET(prefs_dialog->sensors_hbox), FALSE, FALSE, 0);

	prefs_dialog->notebook = g_object_new(GTK_TYPE_NOTEBOOK,
					      NULL);
	prefs_dialog->globals_vbox = g_object_new(GTK_TYPE_VBOX,
						  "border-width", 5,
						  "homogeneous", FALSE,
						  "spacing", 0,
						  NULL);
	
	gtk_notebook_append_page(prefs_dialog->notebook, GTK_WIDGET(prefs_dialog->globals_vbox), gtk_label_new(_("Global Options")));
	gtk_notebook_append_page(prefs_dialog->notebook, GTK_WIDGET(prefs_dialog->sensors_vbox), gtk_label_new(_("Sensors")));

	/* pack notebook into prefs_dialog */
	gtk_box_pack_start_defaults(GTK_BOX(prefs_dialog->dialog->vbox), GTK_WIDGET(prefs_dialog->notebook));

	/* pack globals_vbox */
	gtk_box_pack_start(prefs_dialog->globals_vbox, GTK_WIDGET(prefs_dialog->show_labels_checkbutton), FALSE, FALSE, 0);
	gtk_box_pack_start(prefs_dialog->globals_vbox, GTK_WIDGET(prefs_dialog->labels_inline_checkbutton), FALSE, FALSE, 0);
	gtk_box_pack_start(prefs_dialog->globals_vbox, GTK_WIDGET(prefs_dialog->show_units_checkbutton), FALSE, FALSE, 0);
	gtk_box_pack_start(prefs_dialog->globals_vbox, GTK_WIDGET(prefs_dialog->fahrenheit_checkbutton), FALSE, FALSE, 0);
	gtk_box_pack_end(prefs_dialog->globals_vbox, GTK_WIDGET(prefs_dialog->globals_hbox[2]), FALSE, FALSE, 0);
	gtk_box_pack_end(prefs_dialog->globals_vbox, GTK_WIDGET(prefs_dialog->globals_hbox[1]), FALSE, FALSE, 0);
	gtk_box_pack_end(prefs_dialog->globals_vbox, GTK_WIDGET(prefs_dialog->globals_hbox[0]), FALSE, FALSE, 0);

}

								   
