/*
 * Copyright (C) 2005 Alex Murray <pragmatine@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include "ibm-acpi-sensors-interface.h"
#include "sensors-applet.h"

#define IBM_ACPI_TEMPERATURE_FILE "/proc/acpi/ibm/thermal"
#define IBM_ACPI_FAN_FILE "/proc/acpi/ibm/fan"

/* sensor id's */
#define CPU "CPU"
#define MPCI "mPCI"
#define HDD "HDD"
#define GPU "GPU"
#define BAT0 "BAT0"
#define BAT1 "BAT1"
#define FAN "FAN"


/* for error handling */
#define IBM_ACPI_FILE_ERROR (ibm_acpi_sensors_interface_device_file_error_quark())

enum {
        IBM_ACPI_FILE_OPEN_ERROR,
        IBM_ACPI_FILE_READ_ERROR
};


static void ibm_acpi_sensors_interface_setup_manually(SensorsApplet *sensors_applet) {
	gchar *label;

        if (g_file_test(IBM_ACPI_TEMPERATURE_FILE, G_FILE_TEST_EXISTS)) {
                /* with Ibm_Acpi have 8 fixed sensors, all accessed
                   from the IBM_ACPI_TEMPERATURE_FILE */
                sensors_applet_add_sensor(sensors_applet,
                                          IBM_ACPI_TEMPERATURE_FILE,
                                          CPU,
                                          _(CPU),
                                          IBM_ACPI,
                                          TRUE,
                                          TEMP_SENSOR,
					  CPU_ICON);
                sensors_applet_add_sensor(sensors_applet,
                                          IBM_ACPI_TEMPERATURE_FILE,
                                          MPCI,
                                          _("MiniPCI"),
                                          IBM_ACPI,
                                          FALSE,
                                          TEMP_SENSOR,
					  GENERIC_ICON);
                sensors_applet_add_sensor(sensors_applet,
                                          IBM_ACPI_TEMPERATURE_FILE,
                                          HDD,
                                          _(HDD),
                                          IBM_ACPI,
                                          FALSE,
                                          TEMP_SENSOR,
					  HDD_ICON);
                sensors_applet_add_sensor(sensors_applet,
                                          IBM_ACPI_TEMPERATURE_FILE,
                                          GPU,
                                          _(GPU),
                                          IBM_ACPI,
                                          FALSE,
                                          TEMP_SENSOR,
					  GPU_ICON);
		label = g_strdup_printf("%s %d", _("Battery"), 0);
                sensors_applet_add_sensor(sensors_applet,
                                          IBM_ACPI_TEMPERATURE_FILE,
                                          BAT0,
                                          label,
                                          IBM_ACPI,
                                          FALSE,
                                          TEMP_SENSOR,
					  BATTERY_ICON);
		g_free(label);
		label = g_strdup_printf("%s %d", _("Battery"), 0);
                sensors_applet_add_sensor(sensors_applet,
                                          IBM_ACPI_TEMPERATURE_FILE,
                                          BAT1,
                                          label,
                                          IBM_ACPI,
                                          FALSE,
                                          TEMP_SENSOR,
					  BATTERY_ICON);
		g_free(label);
	}

        if (g_file_test(IBM_ACPI_FAN_FILE, G_FILE_TEST_EXISTS)) {
                sensors_applet_add_sensor(sensors_applet,
                                          IBM_ACPI_FAN_FILE,
                                          FAN,
                                          _("Fan"),
                                          IBM_ACPI,
                                          FALSE,
                                          FAN_SENSOR,
					  FAN_ICON);
        }
}
/* to be called externally to setup for ibm_acpi sensors */
void ibm_acpi_sensors_interface_init(SensorsApplet *sensors_applet) {
        sensors_applet_register_sensors_interface(sensors_applet,
                                                  IBM_ACPI,
                                                  ibm_acpi_sensors_interface_get_sensor_value);
        ibm_acpi_sensors_interface_setup_manually(sensors_applet);
                
}

/* for error handling */
static GQuark ibm_acpi_sensors_interface_device_file_error_quark(void) {
        static GQuark quark = 0;
        gchar *string;

        if (quark == 0) {
                string = g_strdup_printf("%s-device-file-error", sensor_interface[IBM_ACPI]);
                quark = g_quark_from_string(string);
                g_free(string);
        }

        return quark;
}

gdouble ibm_acpi_sensors_interface_get_sensor_value(const gchar *path, 
                                              const gchar *id, 
                                              SensorType type,
                                              GError **error) {

        /* to open and access the value of each sensor */
        FILE *fp;
        gint sensor_value;
        gint cpu_temp, minipci_temp, hdd_temp, gpu_temp, bat0_temp, bat1_temp, unknown0, unknown1;
        gint fan_speed;
        gchar fan_enabled[9];

        if (NULL == (fp = fopen(path, "r"))) {
                g_set_error(error, IBM_ACPI_FILE_ERROR, IBM_ACPI_FILE_OPEN_ERROR, "Error opening sensor device file %s", path);
                return;
        }

        switch (type) {
        case TEMP_SENSOR:
                if (fscanf(fp, "temperatures:   %d %d %d %d %d %d %d %d", &cpu_temp, &minipci_temp, &hdd_temp, &gpu_temp, &bat0_temp, &unknown0, &bat1_temp, &unknown1) != 8) {
                        g_set_error(error, IBM_ACPI_FILE_ERROR, IBM_ACPI_FILE_READ_ERROR, "Error reading from sensor device file %s", path);
                        fclose(fp);
                        return;
                }

                if (strcmp(id, CPU) == 0) {
                        sensor_value = cpu_temp;
                } else if (strcmp(id, MPCI) == 0) {
                        sensor_value = minipci_temp;
                } else if (strcmp(id, HDD) == 0) {
                        sensor_value = hdd_temp;
                } else if (strcmp(id, GPU) == 0) {
                        sensor_value = gpu_temp;
                } else if (strcmp(id, BAT0) == 0) {
                        sensor_value = bat0_temp;
                } else if (strcmp(id, BAT1) == 0) {
                        sensor_value = bat1_temp;
                }
                break;
        case FAN_SENSOR:
                fscanf(fp, "%*[^\n]");   /* Skip to the End of the Line */
                fscanf(fp, "%*1[\n]");   /* Skip One Newline */
                if (fscanf(fp, "speed:   %d", &fan_speed) != 1) {
                        g_set_error(error, IBM_ACPI_FILE_ERROR, IBM_ACPI_FILE_READ_ERROR, "Error reading from sensor device file %s", path);
                        fclose(fp);
                        return;
                }

                if (strcmp(id, "FAN") == 0) {
                        sensor_value = fan_speed;
                }
                break;

        default:
                g_assert_not_reached();                 
        }

        fclose(fp);

        return (gdouble)sensor_value;

}




