/*
 * Copyright (C) 2005-2006 Alex Murray <pragmatine@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include "omnibook-sensors-interface.h"
#include "sensors-applet.h"

#define OMNIBOOK_DEVICE_FILE "/proc/omnibook/temperature"

/* for error handling */
#define OMNIBOOK_DEVICE_FILE_ERROR (omnibook_sensors_interface_device_file_error_quark())

enum {
	OMNIBOOK_DEVICE_FILE_OPEN_ERROR,
	OMNIBOOK_DEVICE_FILE_READ_ERROR
};

static void omnibook_sensors_interface_setup_manually(SensorsApplet *sensors_applet) {
	/* omnibook only has the one device file with one temp in it */
	if (g_file_test(OMNIBOOK_DEVICE_FILE, G_FILE_TEST_EXISTS)) {

		sensors_applet_add_sensor(sensors_applet,
					  OMNIBOOK_DEVICE_FILE,
					  _("temperature"),
					  _("CPU"),
					  OMNIBOOK,
					  TRUE,
					  TEMP_SENSOR,
					  CPU_ICON);
	}
}	


/* to be called to setup for sys sensors */
void omnibook_sensors_interface_init(SensorsApplet *sensors_applet) {
	sensors_applet_register_sensors_interface(sensors_applet,
						  OMNIBOOK,
						  omnibook_sensors_interface_get_sensor_value);
	/* call function to recursively look for sensors
	   starting at the defined base directory */
	omnibook_sensors_interface_setup_manually(sensors_applet);
		
}


/* for error handling */
static GQuark omnibook_sensors_interface_device_file_error_quark(void) {
	static GQuark quark = 0;
	gchar *string;

	if (quark == 0) {
		string = g_strdup_printf("%s-device-file-error", sensor_interface[OMNIBOOK]);
		quark = g_quark_from_string(string);
		g_free(string);
	}

	return quark;
}


gdouble omnibook_sensors_interface_get_sensor_value(const gchar *path, 
						  const gchar *id, 
						  SensorType type,
						  GError **error) {

	/* to open and access the value of each sensor */
	FILE *fp;
	gfloat sensor_value;

	if (NULL == (fp = fopen(path, "r"))) {
		g_set_error(error, OMNIBOOK_DEVICE_FILE_ERROR, OMNIBOOK_DEVICE_FILE_OPEN_ERROR, "Error opening sensor device file %s", path);
		return -1.0;
	}

	if (fscanf(fp, "CPU temperature: %f", &sensor_value) != 1) {
		g_set_error(error, OMNIBOOK_DEVICE_FILE_ERROR, OMNIBOOK_DEVICE_FILE_READ_ERROR, "Error reading from sensor device file %s", path);
		fclose(fp);
		return -1.0;
	}

	fclose(fp);

	return (gdouble)sensor_value;
}
