/*
 * Copyright (C) 2005-2006 Alex Murray <pragmatine@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include "smu-sys-sensors-interface.h"
#include "sensors-applet.h"

#define SMU_SYS_BASE_DIR "/sys/devices/platform/windfarm.0"

#define SENSOR1 "sensor1"
#define CPU "cpu"
#define SENSOR2 "sensor2"
#define GPU "gpu"
#define TEMPERATURE "-temp"
#define FAN_SPEED "-fan"



/* for error handling */
#define SMU_SYS_DEVICE_FILE_ERROR (smu_sys_sensors_interface_device_file_error_quark())

enum {
	SMU_SYS_DEVICE_FILE_OPEN_ERROR,
	SMU_SYS_DEVICE_FILE_READ_ERROR
};

static void smu_sys_sensors_interface_add_sensor(SensorsApplet *sensors_applet, const gchar *path) {
	gchar *filename;
	gchar *label = NULL;
	gboolean enable;
	SensorType sensor_type;
	IconType icon_type = GENERIC_ICON;

	filename = g_path_get_basename(path);

	if (g_ascii_strcasecmp(filename, SENSOR1 TEMPERATURE) == 0 || g_ascii_strcasecmp(filename, CPU TEMPERATURE) == 0) {
		label = g_strdup(_("CPU"));
		sensor_type = TEMP_SENSOR;
		enable = TRUE;
		icon_type = CPU_ICON;
	} else if (g_ascii_strcasecmp(filename, SENSOR2 TEMPERATURE) == 0 || g_ascii_strcasecmp(filename, GPU TEMPERATURE) == 0) {
		label = g_strdup(_("GPU"));
		sensor_type = TEMP_SENSOR;
		enable = TRUE;
		icon_type = GPU_ICON;
	} else if (g_ascii_strcasecmp(filename, SENSOR1 FAN_SPEED) == 0|| g_ascii_strcasecmp(filename, CPU FAN_SPEED) == 0) { 
		label = g_strdup(_("FAN"));
		sensor_type = FAN_SENSOR;
		enable = TRUE;
		icon_type = FAN_ICON;
	} else {
		/* disable all other sensors */
		enable = FALSE;
	}

	/* only add these 3 sensors */
	if (enable) {
		sensors_applet_add_sensor(sensors_applet,
					  path,
					  filename,
					  label,
					  SMU_SYS,
					  enable,
					  sensor_type,
					  icon_type);

	}
	g_free(filename);
	if (label != NULL) {
		g_free(label);
	}
}	

static void smu_sys_sensors_interface_test_sensor(SensorsApplet *sensors_applet, const gchar *path) {
	gchar *filename;

        filename = g_path_get_basename(path);
        if (g_ascii_strcasecmp(filename, SENSOR1 TEMPERATURE) == 0 ||
            g_ascii_strcasecmp(filename, SENSOR2 TEMPERATURE) == 0 ||
            g_ascii_strcasecmp(filename, SENSOR1 FAN_SPEED) == 0 ||
            g_ascii_strcasecmp(filename, CPU TEMPERATURE) == 0 ||
            g_ascii_strcasecmp(filename, GPU TEMPERATURE) == 0 ||
            g_ascii_strcasecmp(filename, CPU FAN_SPEED) == 0) {
                smu_sys_sensors_interface_add_sensor(sensors_applet, path);
        }
        g_free(filename);
}

/* to be called to setup for sys sensors */
void smu_sys_sensors_interface_init(SensorsApplet *sensors_applet) {
	sensors_applet_register_sensors_interface(sensors_applet,
						  SMU_SYS,
						  smu_sys_sensors_interface_get_sensor_value);
	/* call function to recursively look for sensors
	   starting at the defined base directory */
	sensors_applet_find_sensors(sensors_applet, 
                                    SMU_SYS_BASE_DIR, 
                                    smu_sys_sensors_interface_test_sensor);
	
}


/* for error handling */
static GQuark smu_sys_sensors_interface_device_file_error_quark(void) {
	static GQuark quark = 0;
	gchar *string;

	if (quark == 0) {
		string = g_strdup_printf("%s-device-file-error", sensor_interface[SMU_SYS]);
		quark = g_quark_from_string(string);
		g_free(string);
	}

	return quark;
}

/* returns the value of the sensor_list at the given iter, or if an
   error occurs, instatiates error with an error message */
gdouble smu_sys_sensors_interface_get_sensor_value(const gchar *path, 
						  const gchar *id, 
						  SensorType type,
						  GError **error) {

	/* to open and access the value of each sensor */
	FILE *fp;
	gfloat sensor_value = -1.0;

	if (NULL == (fp = fopen(path, "r"))) {
		g_set_error(error, SMU_SYS_DEVICE_FILE_ERROR, SMU_SYS_DEVICE_FILE_OPEN_ERROR, "Error opening sensor device file %s", path);
		return -1.0;
	}
	switch(type) {
	case FAN_SENSOR:
		if (fscanf(fp, "%f", &sensor_value) != 1) {
			g_set_error(error, SMU_SYS_DEVICE_FILE_ERROR, SMU_SYS_DEVICE_FILE_READ_ERROR, "Error reading from sensor device file %s", path);
			fclose(fp);
			return -1.0;
		}
		break;

	case TEMP_SENSOR:
		if (fscanf(fp, "%f", &sensor_value) != 1) {
			g_set_error(error, SMU_SYS_DEVICE_FILE_ERROR, SMU_SYS_DEVICE_FILE_READ_ERROR, "Error reading from sensor device file %s", path);
			fclose(fp);
			return -1.0;
		}
		break;

	default:
		/* should only have added temp or fan sensors */
                g_error("Unknown sensor type passed as parameter to smu-sys sensor interface, cannot get value for this sensor");
                g_set_error(error, SMU_SYS_DEVICE_FILE_ERROR, SMU_SYS_DEVICE_FILE_READ_ERROR, "Error reading from sensor device file %s", path);
	}
	fclose(fp);

	return (gdouble)sensor_value;
}
