//----------------------------------------------------------------------------
//
//  This file is part of seq24.
//
//  seq24 is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  seq24 is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with seq24; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//-----------------------------------------------------------------------------

#include "seqedit.h" 
#include "sequence.h"
#include "midibus.h"
#include "controllers.h"
#include "event.h"
#include "options.h"

#include "play.xpm"
#include "rec.xpm"
#include "thru.xpm"
#include "bus.xpm"
#include "midi.xpm"
#include "snap.xpm"
#include "zoom.xpm"
#include "length.xpm"
#include "scale.xpm"
#include "key.xpm"   
#include "down.xpm"

int seqedit::m_initial_zoom = 4;
int seqedit::m_initial_snap = c_ppqn / 4;
int seqedit::m_initial_scale = 0;
int seqedit::m_initial_key = 0;

/* connects to a menu item, tells the performance 
   to launch the timer thread */
void 
seqedit::menu_action_quantise( void )
{
}



seqedit::seqedit( sequence *a_seq, 
		  perform *a_perf,  
		  mainwid *a_mainwid, 
		  int a_pos  ) 

: Gtk::Window( GTK_WINDOW_TOPLEVEL )
{
    /* set the performance */
    m_seq = a_seq;

    m_zoom =  m_initial_zoom;
    m_snap =  m_initial_snap;
    m_scale = m_initial_scale;
    m_key =   m_initial_key;

    m_mainperf = a_perf;
    m_mainwid = a_mainwid;
    m_pos = a_pos;

    /* main window */
    set_title ( m_seq->get_name());
    set_default_size(500, 540);

    m_seq->set_editing( true );

    /* get some new objects */

    m_seqkeys_wid  = manage( new seqkeys(  m_seq ));
    m_seqdata_wid  = manage( new seqdata(  m_seq, m_zoom ));
    m_seqevent_wid = manage( new seqevent( m_seq, m_zoom, m_snap, m_seqdata_wid ));
    m_seqroll_wid  = manage( new seqroll(  m_seq, m_zoom, m_snap, m_seqdata_wid, m_seqevent_wid, m_seqkeys_wid, m_mainwid, m_pos  ));
    m_seqtime_wid  = manage( new seqtime(  m_seq, m_zoom ));
    
    /* menus */
    m_menubar   =  manage( new MenuBar());
    m_menu_action = manage( new Menu());
    m_menu_zoom =  manage( new Menu());
    m_menu_snap =   manage( new Menu());
    m_menu_length = manage( new Menu());
    m_menu_bpm = manage( new Menu() );
    m_menu_bw = manage( new Menu() );

    m_menu_midich = manage( new Menu());
    m_menu_midibus = manage( new Menu());
    m_menu_data = manage( new Menu());  

    m_menu_key = manage( new Menu());
    m_menu_scale = manage( new Menu());

    

    create_menus(); 

    /* tooltips */
    m_tooltips = manage( new Tooltips( ) );

 
    /* init table, viewports and scroll bars */
    m_table     = manage( new Table( 6, 3, false));
    m_vbox      = manage( new VBox( false, 2 ));
    m_hbox      = manage( new HBox( false, 2 ));
    m_hbox2     = manage( new HBox( false, 2 ));
    HBox *dhbox = manage( new HBox( false, 2 ));

    m_vbox->set_border_width( 2 );

    m_keysview   =  manage( new Viewport());
    m_rollview   =  manage( new Viewport());
    m_dataview   =  manage( new Viewport());
    m_timeview   =  manage( new Viewport());
    m_eventview  =  manage( new Viewport());

    m_vscroll   =  manage(new VScrollbar());
    m_hscroll   =  manage(new HScrollbar());

    /* init keys view */
    m_keysview->set_shadow_type( GTK_SHADOW_NONE );
    m_keysview->add( *m_seqkeys_wid );
    m_keysview->set_usize( c_keyarea_x + 2, 10 );
    m_keysview->set_vadjustment( *(m_vscroll->get_adjustment()));

    /* init paino roll view */
    m_rollview->set_shadow_type( GTK_SHADOW_NONE );
    m_rollview->add( *m_seqroll_wid );
    m_rollview->set_usize( 30, 30 );
    m_rollview->set_hadjustment( *(m_hscroll->get_adjustment()));
    m_rollview->set_vadjustment( *(m_vscroll->get_adjustment()));

    /* init event view */
    m_eventview->set_shadow_type( GTK_SHADOW_NONE );
    m_eventview->add( *m_seqevent_wid );
    m_eventview->set_usize( 10, c_eventarea_y + 1 );
    m_eventview->set_hadjustment( *(m_hscroll->get_adjustment()));
  
    /* data */
    m_dataview->set_shadow_type( GTK_SHADOW_NONE );
    m_dataview->add( *m_seqdata_wid );
    m_dataview->set_usize( 10, c_dataarea_y + 2 );
    m_dataview->set_hadjustment( *(m_hscroll->get_adjustment()));

    /* time */
    m_timeview->set_shadow_type( GTK_SHADOW_NONE );
    m_timeview->add( *m_seqtime_wid );
    m_timeview->set_usize( 10, c_timearea_y + 1 );
    m_timeview->set_hadjustment( *(m_hscroll->get_adjustment()));

    /* fill table */
    m_table->attach( *m_keysview,    0, 1, 1, 2, 0, GTK_FILL );

    m_table->attach( *m_timeview, 1, 2, 0, 1, GTK_FILL, 0 );
    m_table->attach( *m_rollview , 1, 2, 1, 2,
    	     GTK_FILL | GTK_SHRINK,  
    	     GTK_FILL | GTK_SHRINK );

    m_table->attach( *m_eventview, 1, 2, 2, 3, GTK_FILL, 0 );
    m_table->attach( *m_dataview, 1, 2, 3, 4, GTK_FILL, 0 );
    m_table->attach( *dhbox,      1, 2, 4, 5, GTK_FILL | GTK_EXPAND, 0, 0, 2 );

    m_table->attach( *m_vscroll, 2, 3, 1, 2, 0, GTK_FILL | GTK_EXPAND  );
    m_table->attach( *m_hscroll, 1, 2, 5, 6, GTK_FILL | GTK_EXPAND, 0  );

    /* no expand, just fit the widgets */
    /* m_vbox->pack_start(*m_menubar, false, false, 0); */
    m_vbox->pack_start(*m_hbox,  false, false, 0);
    m_vbox->pack_start(*m_hbox2, false, false, 0);

    /* exapand, cause rollview expands */
    m_vbox->pack_start(*m_table, true, true, 0);


    /* data button */
    m_button_data = manage( new Button( " Event " ));
    m_button_data->clicked.connect( bind<Menu *>( slot( this, &seqedit::popup_menu), m_menu_data ));

    m_entry_data = manage( new Entry( ));
    m_entry_data->set_usize(40,-1);
    m_entry_data->set_editable( false );

    dhbox->pack_start( *m_button_data, false, false );
    dhbox->pack_start( *m_entry_data, true, true );

    /* play, rec, thru */
    m_toggle_play = manage( new ToggleButton() );
    m_toggle_play->add( *manage( new Pixmap( play_xpm )));
    m_toggle_play->clicked.connect( slot( this, &seqedit::play_change_callback));
    m_tooltips->set_tip( *m_toggle_play, "Sequence dumps data to midi bus." );
    
    m_toggle_record = manage( new ToggleButton(  ));
    m_toggle_record->add( *manage( new Pixmap( rec_xpm )));
    m_toggle_record->clicked.connect( slot( this, &seqedit::record_change_callback));
    m_tooltips->set_tip( *m_toggle_record, "Records incoming midi data." );

    m_toggle_thru = manage( new ToggleButton(  ));
    m_toggle_thru->add( *manage( new Pixmap( thru_xpm )));
    m_toggle_thru->clicked.connect( slot( this, &seqedit::thru_change_callback));
    m_tooltips->set_tip( *m_toggle_thru, "Incoming midi data passes thru to sequences midi bus and channel." );

    m_toggle_play->set_active( m_seq->get_playing());
    m_toggle_record->set_active( m_seq->get_recording());
    m_toggle_thru->set_active( m_seq->get_thru());
 
    dhbox->pack_end( *m_toggle_record, false, false, 4);
    dhbox->pack_end( *m_toggle_thru, false, false, 4);
    dhbox->pack_end( *m_toggle_play, false, false, 4);
    dhbox->pack_end( *(manage(new VSeparator( ))), false, false, 4);

    fill_top_bar();


    /* add table */
    this->add( *m_vbox );
    /* show everything */
    show_all();

    /* sets scroll bar to the middle */
    gfloat middle = m_vscroll->get_adjustment()->get_upper() / 3;
    m_vscroll->get_adjustment()->set_value(middle);

    set_zoom( m_zoom );
    set_snap( m_snap );


    set_bpm( m_seq->get_bpm() );
    set_bw( m_seq->get_bw() );
    set_measures( get_measures() );

    set_midi_channel( m_seq->get_midi_channel() );
    set_midi_bus( m_seq->get_midi_bus() );
    set_data_type( EVENT_NOTE_ON );

    set_scale( m_scale );
    set_key( m_key );

}




void 
seqedit::create_menus( void )
{
    using namespace Menu_Helpers;

    /* zoom */
    m_menu_zoom->items().push_back(MenuElem("1:1",  bind(slot(this,&seqedit::set_zoom), 1 )));
    m_menu_zoom->items().push_back(MenuElem("1:2",  bind(slot(this,&seqedit::set_zoom), 2 )));
    m_menu_zoom->items().push_back(MenuElem("1:4",  bind(slot(this,&seqedit::set_zoom), 4 )));
    m_menu_zoom->items().push_back(MenuElem("1:8",  bind(slot(this,&seqedit::set_zoom), 8 )));
    m_menu_zoom->items().push_back(MenuElem("1:16", bind(slot(this,&seqedit::set_zoom), 16 )));
    m_menu_zoom->items().push_back(MenuElem("1:32", bind(slot(this,&seqedit::set_zoom), 32 )));

    /* note snap */
    m_menu_snap->items().push_back(MenuElem("1",     bind(slot(this,&seqedit::set_snap), c_ppqn * 4  )));
    m_menu_snap->items().push_back(MenuElem("1/2",   bind(slot(this,&seqedit::set_snap), c_ppqn * 2  )));
    m_menu_snap->items().push_back(MenuElem("1/4",   bind(slot(this,&seqedit::set_snap), c_ppqn * 1  )));
    m_menu_snap->items().push_back(MenuElem("1/8",   bind(slot(this,&seqedit::set_snap), c_ppqn / 2  )));
    m_menu_snap->items().push_back(MenuElem("1/16",  bind(slot(this,&seqedit::set_snap), c_ppqn / 4  )));
    m_menu_snap->items().push_back(MenuElem("1/32",  bind(slot(this,&seqedit::set_snap), c_ppqn / 8  )));
    m_menu_snap->items().push_back(MenuElem("1/64",  bind(slot(this,&seqedit::set_snap), c_ppqn / 16 )));
    m_menu_snap->items().push_back(MenuElem("1/128", bind(slot(this,&seqedit::set_snap), c_ppqn / 32 )));
	m_menu_snap->items().push_back(SeparatorElem());
    m_menu_snap->items().push_back(MenuElem("1/3",   bind(slot(this,&seqedit::set_snap), c_ppqn * 4  / 3 )));
    m_menu_snap->items().push_back(MenuElem("1/6",   bind(slot(this,&seqedit::set_snap), c_ppqn * 2  / 3 )));
    m_menu_snap->items().push_back(MenuElem("1/12",  bind(slot(this,&seqedit::set_snap), c_ppqn * 1  / 3 )));
    m_menu_snap->items().push_back(MenuElem("1/24",  bind(slot(this,&seqedit::set_snap), c_ppqn / 2  / 3 )));
    m_menu_snap->items().push_back(MenuElem("1/48",  bind(slot(this,&seqedit::set_snap), c_ppqn / 4  / 3 )));
    m_menu_snap->items().push_back(MenuElem("1/96",  bind(slot(this,&seqedit::set_snap), c_ppqn / 8  / 3 )));
	m_menu_snap->items().push_back(MenuElem("1/192", bind(slot(this,&seqedit::set_snap), c_ppqn / 16 / 3 )));
	
    /* Key */
    m_menu_key->items().push_back(MenuElem( c_key_text[0],  bind(slot(this,&seqedit::set_key), 0 )));
    m_menu_key->items().push_back(MenuElem( c_key_text[1],  bind(slot(this,&seqedit::set_key), 1 )));
    m_menu_key->items().push_back(MenuElem( c_key_text[2],  bind(slot(this,&seqedit::set_key), 2 )));
    m_menu_key->items().push_back(MenuElem( c_key_text[3],  bind(slot(this,&seqedit::set_key), 3 )));
    m_menu_key->items().push_back(MenuElem( c_key_text[4],  bind(slot(this,&seqedit::set_key), 4 )));
    m_menu_key->items().push_back(MenuElem( c_key_text[5],  bind(slot(this,&seqedit::set_key), 5 )));
    m_menu_key->items().push_back(MenuElem( c_key_text[6],  bind(slot(this,&seqedit::set_key), 6 )));
    m_menu_key->items().push_back(MenuElem( c_key_text[7],  bind(slot(this,&seqedit::set_key), 7 )));
    m_menu_key->items().push_back(MenuElem( c_key_text[8],  bind(slot(this,&seqedit::set_key), 8 )));
    m_menu_key->items().push_back(MenuElem( c_key_text[9],  bind(slot(this,&seqedit::set_key), 9 )));
    m_menu_key->items().push_back(MenuElem( c_key_text[10], bind(slot(this,&seqedit::set_key), 10 )));
    m_menu_key->items().push_back(MenuElem( c_key_text[11], bind(slot(this,&seqedit::set_key), 11 )));

    /* bw */
    m_menu_bw->items().push_back(MenuElem("1", bind(slot(this,&seqedit::set_bw), 1  )));
    m_menu_bw->items().push_back(MenuElem("2", bind(slot(this,&seqedit::set_bw), 2  )));
    m_menu_bw->items().push_back(MenuElem("4", bind(slot(this,&seqedit::set_bw), 4  )));
    m_menu_bw->items().push_back(MenuElem("8", bind(slot(this,&seqedit::set_bw), 8  )));
    m_menu_bw->items().push_back(MenuElem("16", bind(slot(this,&seqedit::set_bw), 16 )));


    /* music scale */
    m_menu_scale->items().push_back(MenuElem(c_scales_text[0], bind(slot(this,&seqedit::set_scale), c_scale_off )));
    m_menu_scale->items().push_back(MenuElem(c_scales_text[1], bind(slot(this,&seqedit::set_scale), c_scale_major )));
    m_menu_scale->items().push_back(MenuElem(c_scales_text[2], bind(slot(this,&seqedit::set_scale), c_scale_minor )));

    /* temp */
    char b[20];

    m_menu_data->items().push_back( MenuElem( "Note On Velocity",  bind(slot(this,&seqedit::set_data_type), (unsigned char) EVENT_NOTE_ON, 0 )));
    m_menu_data->items().push_back( SeparatorElem( )); 
    m_menu_data->items().push_back( MenuElem( "Note Off Velocity", bind(slot(this,&seqedit::set_data_type), (unsigned char) EVENT_NOTE_OFF, 0 )));
    m_menu_data->items().push_back( MenuElem( "AfterTouch",        bind(slot(this,&seqedit::set_data_type), (unsigned char) EVENT_AFTERTOUCH, 0 )));
    m_menu_data->items().push_back( MenuElem( "Program Change",    bind(slot(this,&seqedit::set_data_type), (unsigned char) EVENT_PROGRAM_CHANGE, 0 )));
    m_menu_data->items().push_back( MenuElem( "Channel Pressure",  bind(slot(this,&seqedit::set_data_type), (unsigned char) EVENT_CHANNEL_PRESSURE, 0 )));
    m_menu_data->items().push_back( MenuElem( "Pitch Wheel",       bind(slot(this,&seqedit::set_data_type), (unsigned char) EVENT_PITCH_WHEEL , 0 )));
    m_menu_data->items().push_back( SeparatorElem( )); 

    /* create control change */
    for ( int i=0; i<8; i++ ){

	sprintf( b, "Controls %d-%d", (i*16), (i*16)+15 );
	Menu *menu_cc = manage( new Menu() ); 

	for( int j=0; j<16; j++ ){
	    menu_cc->items().push_back( MenuElem( c_controller_names[i*16+j], 
						  bind(slot(this,&seqedit::set_data_type), 
						       (unsigned char) EVENT_CONTROL_CHANGE, i*16+j)));
	}
	m_menu_data->items().push_back(MenuElem( string(b) , *menu_cc ));
    }

    /* midi channel menu */
    for( int i=0; i<16; i++ ){

	sprintf( b, "%d", i+1 );
	m_menu_midich->items().push_back(MenuElem(b, 
						  bind(slot(this,&seqedit::set_midi_channel), 
						       i )));
	/* length */
	m_menu_length->items().push_back(MenuElem(b, 
						  bind(slot(this,&seqedit::set_measures),   
						       i+1 )));
	/* length */
	m_menu_bpm->items().push_back(MenuElem(b, 
					       bind(slot(this,&seqedit::set_bpm),   
						    i+1 )));
    }



    /* midi buses */
    mastermidibus *masterbus = m_mainperf->get_master_midi_bus();
    for ( int i=0; i< masterbus->get_num_out_buses(); i++ ){
	m_menu_midibus->items().push_back(MenuElem(masterbus->get_midi_bus_name(i),
						  bind(slot(this,&seqedit::set_midi_bus), i)));
    }
}



void  
seqedit::fill_top_bar( void )
{

     /* name */
    m_entry_name = manage( new Entry( 14 ));
    m_entry_name->set_usize( 40, -1 );
    m_entry_name->set_text( m_seq->get_name());
    m_entry_name->changed.connect( slot( this, &seqedit::name_change_callback));

    m_hbox->pack_start( *m_entry_name, true, true );

    m_hbox->pack_start( *(manage(new VSeparator( ))), false, false, 4);

    /* key */
    m_button_key = manage( new Button());
    m_button_key->add( *manage( new Pixmap( key_xpm  )));
    m_button_key->clicked.connect(  bind<Menu *>( slot( this, &seqedit::popup_menu), m_menu_key  ));
    m_tooltips->set_tip( *m_button_key, "Key of Sequence" );
    m_entry_key = manage( new Entry());
    m_entry_key->set_usize( 42, -1 );
    m_entry_key->set_editable( false );

    m_hbox->pack_start( *m_button_key , false, false );
    m_hbox->pack_start( *m_entry_key , false, false );

    /* music scale */
    m_button_scale = manage( new Button());
    m_button_scale->add( *manage( new Pixmap( scale_xpm  )));
    m_button_scale->clicked.connect(  bind<Menu *>( slot( this, &seqedit::popup_menu), m_menu_scale  ));
    m_tooltips->set_tip( *m_button_scale, "Musical Scale" );
    m_entry_scale = manage( new Entry());
    m_entry_scale->set_usize( 42, -1 );
    m_entry_scale->set_editable( false );

    m_hbox->pack_start( *m_button_scale , false, false );
    m_hbox->pack_start( *m_entry_scale , false, false );

    m_hbox->pack_start( *(manage(new VSeparator( ))), false, false, 4);

 
    /* beats per measure */ 
    m_button_bpm = manage( new Button());
    m_button_bpm->add( *manage( new Pixmap( down_xpm  )));
    m_button_bpm->clicked.connect(  bind<Menu *>( slot( this, &seqedit::popup_menu), m_menu_bpm  ));
    m_tooltips->set_tip( *m_button_bpm, "Time Signature. Beats per Measure" );
    m_entry_bpm = manage( new Entry());
    m_entry_bpm->set_usize( 20, -1 );
    m_entry_bpm->set_editable( false );

    m_hbox->pack_start( *m_button_bpm , false, false );
    m_hbox->pack_start( *m_entry_bpm , false, false );
 
    m_hbox->pack_start( *(manage(new Label( "/" ))), false, false, 4);

    /* beat width */
    m_button_bw = manage( new Button());
    m_button_bw->add( *manage( new Pixmap( down_xpm  )));
    m_button_bw->clicked.connect(  bind<Menu *>( slot( this, &seqedit::popup_menu), m_menu_bw  ));
    m_tooltips->set_tip( *m_button_bw, "Time Signature.  Length of Beat" );
    m_entry_bw = manage( new Entry());
    m_entry_bw->set_usize( 20, -1 );
    m_entry_bw->set_editable( false );

    m_hbox->pack_start( *m_button_bw , false, false );
    m_hbox->pack_start( *m_entry_bw , false, false );


    /* length */
    m_button_length = manage( new Button());
    m_button_length->add( *manage( new Pixmap( length_xpm  )));
    m_button_length->clicked.connect(  bind<Menu *>( slot( this, &seqedit::popup_menu), m_menu_length  ));
    m_tooltips->set_tip( *m_button_length, "Sequence length in Quarter Notes" );
    m_entry_length = manage( new Entry());
    m_entry_length->set_usize( 20, -1 );
    m_entry_length->set_editable( false );

    m_hbox->pack_start( *m_button_length , false, false );
    m_hbox->pack_start( *m_entry_length , false, false );



    /* snap */
    m_button_snap = manage( new Button());
    m_button_snap->add( *manage( new Pixmap( snap_xpm  )));
    m_button_snap->clicked.connect(  bind<Menu *>( slot( this, &seqedit::popup_menu), m_menu_snap  ));
    m_tooltips->set_tip( *m_button_snap, "Grid snap." );
    m_entry_snap = manage( new Entry());
    m_entry_snap->set_usize( 36, -1 );
    m_entry_snap->set_editable( false );
    
    m_hbox2->pack_start( *m_button_snap , false, false );
    m_hbox2->pack_start( *m_entry_snap , false, false );

    /* zoom */
    m_button_zoom = manage( new Button());
    m_button_zoom->add( *manage( new Pixmap( zoom_xpm  )));
    m_button_zoom->clicked.connect(  bind<Menu *>( slot( this, &seqedit::popup_menu), m_menu_zoom  ));
    m_tooltips->set_tip( *m_button_zoom, "Zoom. Pixels to Ticks" );
    m_entry_zoom = manage( new Entry());
    m_entry_zoom->set_usize( 30, -1 );
    m_entry_zoom->set_editable( false );

    m_hbox2->pack_start( *m_button_zoom , false, false );
    m_hbox2->pack_start( *m_entry_zoom , false, false );

  
    m_hbox2->pack_start( *(manage(new VSeparator( ))), false, false, 4);


    /* midi bus */
    m_button_bus = manage( new Button());
    m_button_bus->add( *manage( new Pixmap( bus_xpm  )));
    m_button_bus->clicked.connect(  bind<Menu *>( slot( this, &seqedit::popup_menu), m_menu_midibus ));
    m_tooltips->set_tip( *m_button_bus, "Select Output Bus." );
    m_entry_bus = manage( new Entry(30));
    m_entry_bus->set_usize( 40, -1 );
    m_entry_bus->set_editable( false );

    m_hbox2->pack_start( *m_button_bus , false, false );
    m_hbox2->pack_start( *m_entry_bus , true, true );

    /* midi channel */
    m_button_channel = manage( new Button());
    m_button_channel->add( *manage( new Pixmap( midi_xpm  )));
    m_button_channel->clicked.connect(  bind<Menu *>( slot( this, &seqedit::popup_menu), m_menu_midich  ));
    m_tooltips->set_tip( *m_button_channel, "Select Midi channel." );
    m_entry_channel = manage( new Entry());
    m_entry_channel->set_usize( 20, -1 );
    m_entry_channel->set_editable( false );

    m_hbox2->pack_start( *m_button_channel , false, false );
    m_hbox2->pack_start( *m_entry_channel , false, false );

}

void
seqedit::popup_menu(Menu *a_menu)
{
    a_menu->popup(0,0);
}


    //m_option_midich->set_history( m_seq->getMidiChannel() );
    //m_option_midibus->set_history( m_seq->getMidiBus()->getID() );

void 
seqedit::set_midi_channel( int a_midichannel  )
{
    char b[10];
    sprintf( b, "%d", a_midichannel+1 );
    m_entry_channel->set_text(b);
    m_seq->set_midi_channel( a_midichannel );
    m_mainwid->update_sequence_on_window( m_pos );
}


void 
seqedit::set_midi_bus( int a_midibus )
{
    m_seq->set_midi_bus( a_midibus );
    m_entry_bus->set_text( m_mainperf->get_master_midi_bus()->get_midi_bus_name( a_midibus ));
    m_mainwid->update_sequence_on_window( m_pos );
}


void 
seqedit::set_zoom( int a_zoom  )
{
    char b[10];
    sprintf( b, "1:%d", a_zoom );
    m_entry_zoom->set_text(b);

    m_zoom = a_zoom;
    m_initial_zoom = a_zoom;
    m_seqroll_wid->set_zoom( m_zoom );
    m_seqtime_wid->set_zoom( m_zoom );
    m_seqdata_wid->set_zoom( m_zoom ); 
    m_seqevent_wid->set_zoom( m_zoom );
}


void 
seqedit::set_snap( int a_snap  )
{
    char b[10];
    sprintf( b, "1/%d",   c_ppqn * 4 / a_snap );
    m_entry_snap->set_text(b);
    
    m_snap = a_snap;
    m_initial_snap = a_snap;
    m_seqroll_wid->set_snap( m_snap );
    m_seqevent_wid->set_snap( m_snap );
}





void 
seqedit::set_scale( int a_scale )
{
  m_entry_scale->set_text( c_scales_text[a_scale] );

  m_scale = m_initial_scale = a_scale;

  m_seqroll_wid->set_scale( m_scale );


}

void 
seqedit::set_key( int a_note )
{ 
  m_entry_key->set_text( c_key_text[a_note] );

  m_key = m_initial_key = a_note;
  
  m_seqroll_wid->set_key( m_key );

}
 
void
seqedit::apply_length( int a_bpm, int a_bw, int a_measures )
{
  m_seq->set_length( a_measures * a_bpm * ((c_ppqn * 4) / a_bw) );

  m_seqroll_wid->reset();
  m_seqtime_wid->reset();
  m_seqdata_wid->reset();  
  m_seqevent_wid->reset();
    
}

long
seqedit::get_measures( void )
{
  long measures = ( m_seq->get_length() / ((m_seq->get_bpm() * (c_ppqn * 4)) /  m_seq->get_bw() ));
 
  return measures;
}

void 
seqedit::set_measures( int a_length_measures  )
{

    char b[10];
    sprintf( b, "%d", a_length_measures );
    m_entry_length->set_text(b);

    if ( a_length_measures > 16  )
	 a_length_measures = 16;

    m_measures = a_length_measures;
    apply_length( m_seq->get_bpm(), m_seq->get_bw(), a_length_measures );
    
}



void 
seqedit::set_bpm( int a_beats_per_measure )
{
  char b[4];
  sprintf( b, "%d", a_beats_per_measure );
  m_entry_bpm->set_text(b);

  if ( a_beats_per_measure != m_seq->get_bpm() ){
    
    long length = get_measures();
    m_seq->set_bpm( a_beats_per_measure );
    apply_length( a_beats_per_measure, m_seq->get_bw(), length );
  }
}



void 
seqedit::set_bw( int a_beat_width  )
{
  char b[4];
  sprintf( b, "%d",   a_beat_width );
  m_entry_bw->set_text(b);
  
  if ( a_beat_width != m_seq->get_bw()){
    
    long length = get_measures();
    m_seq->set_bw( a_beat_width );
    apply_length( m_seq->get_bpm(), a_beat_width, length );
  }
}




void 
seqedit::name_change_callback( void )
{
    m_seq->set_name( m_entry_name->get_text());
    m_mainwid->update_sequence_on_window( m_pos );
}


void 
seqedit::play_change_callback( void )
{
    m_seq->set_playing( m_toggle_play->get_active() );
    m_mainwid->update_sequence_on_window( m_pos );
}


void 
seqedit::record_change_callback( void )
{
    m_seq->set_recording( m_toggle_record->get_active() );
}


void 
seqedit::thru_change_callback( void )
{
    m_seq->set_thru( m_toggle_thru->get_active() );
}

void 
seqedit::set_data_type( unsigned char a_status, unsigned char a_control  )
{
    m_editing_status = a_status;
    m_editing_cc = a_control;

    m_seqevent_wid->set_data_type( a_status, a_control );
    m_seqdata_wid->set_data_type( a_status, a_control );

    char text[100];
    char hex[20];
    char type[80];

    sprintf( hex, "[0x%02X]", a_status );  
    
    if ( a_status ==  EVENT_NOTE_OFF )         
	sprintf( type, "Note Off" );
    if ( a_status ==  EVENT_NOTE_ON )          
	sprintf( type, "Note On" );  
    if ( a_status ==  EVENT_AFTERTOUCH )       
	sprintf( type, "Aftertouch" );           
    if ( a_status ==  EVENT_CONTROL_CHANGE )   
	sprintf( type, "Control Change - %s", c_controller_names[a_control].c_str() );  
    if ( a_status ==    EVENT_PROGRAM_CHANGE )   
	sprintf( type, "Program Change" );       
    if ( a_status ==    EVENT_CHANNEL_PRESSURE ) 
	sprintf( type, "Channel Pressure" );     
    if ( a_status ==  EVENT_PITCH_WHEEL )      
	sprintf( type, "Pitch Wheel" );          

    sprintf( text, "%s %s", hex, type );

    m_entry_data->set_text( text );

}


void 
seqedit::realize_impl()
{
    // we need to do the default realize
    Gtk::Window::realize_impl();

    Gtk::Main::timeout.connect(slot(this,&seqedit::timeout ), c_redraw_ms);
 
}

int
seqedit::timeout( void )
{

    if (m_seq->is_dirty() ){

	m_seqroll_wid->idle_redraw();
	m_seqevent_wid->idle_redraw();
	m_seqdata_wid->idle_redraw();
    }

    m_seqroll_wid->draw_progress_on_window();
    
    return true;
}

seqedit::~seqedit()
{
    m_seq->set_editing( false );
}




int 
seqedit::focus_in_event_impl( GdkEventFocus* a_ev )
{
    m_mainperf->get_master_midi_bus()->set_sequence_input( true, m_seq ); 
    return false;
}


int 
seqedit::focus_out_event_impl( GdkEventFocus* a_ev )
{
    return false;
}


int 
seqedit::delete_event_impl(GdkEventAny *a_event)
{
    m_seq->set_recording( false );
    m_mainperf->get_master_midi_bus()->set_sequence_input( false, NULL );
    return false;
}
