//----------------------------------------------------------------------------
//
//  This file is part of seq24.
//
//  seq24 is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  seq24 is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with seq24; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//-----------------------------------------------------------------------------
#include "event.h"
#include "seqdata.h"


seqdata::seqdata(sequence *a_seq, int a_zoom): DrawingArea() 
{     
    m_seq = a_seq;
    m_zoom = a_zoom;

    add_events( GDK_BUTTON_PRESS_MASK | 
		GDK_BUTTON_RELEASE_MASK |
		GDK_POINTER_MOTION_MASK |
		GDK_LEAVE_NOTIFY_MASK );

    // in the construor you can only allocate colors, 
    // get_window() returns 0 because we have not be realized
    Gdk_Colormap colormap = get_default_colormap();

    m_black = Gdk_Color( "black" );
    m_white = Gdk_Color( "white" );
    m_grey  = Gdk_Color( "grey" );

    colormap.alloc( m_black );
    colormap.alloc( m_white );
    colormap.alloc( m_grey );

    m_dragging = false;

    set_flags(GTK_CAN_FOCUS );
} 

void 
seqdata::update_sizes()
{
    /* set default size */
    size( m_seq->get_length() / m_zoom , c_dataarea_y );

    /* set these for later */
    m_window_x = m_seq->get_length() / m_zoom;
    m_window_y = c_dataarea_y;

    m_pixmap = Gdk_Pixmap( m_window,
			   m_window_x,
			   m_window_y );
}

void 
seqdata::reset()
{
    update_sizes();
    update_pixmap();
    draw_pixmap_on_window();
}


void 
seqdata::realize_impl()
{
    // we need to do the default realize
    Gtk::DrawingArea::realize_impl();

    // Now we can allocate any additional resources we need
    m_window = get_window();
    m_gc.create( m_window );
    m_window.clear();

    update_sizes();
    update_pixmap();
}

void 
seqdata::set_zoom( int a_zoom )
{
    if ( m_zoom != a_zoom ){
	m_zoom = a_zoom;
	update_sizes();
	update_pixmap();
	draw_pixmap_on_window();
    }
}



void 
seqdata::set_data_type( unsigned char a_status, unsigned char a_control = 0  )
{
    m_status = a_status;
    m_cc = a_control;

    this->reset();
}
   

void 
seqdata::update_pixmap()
{
    /* clear background */
    m_gc.set_foreground(m_white);
    m_pixmap.draw_rectangle(m_gc,true,
			    0,
			    0, 
			    m_window_x, 
			    m_window_y );

    /* draw horz grey lines */
    m_gc.set_foreground(m_black);
    m_pixmap.draw_line(m_gc,0,0,m_window_x,0 );

    draw_events_on_pixmap();
}

void 
seqdata::draw_events_on( Gdk_Drawable *a_draw,
			 bool a_flush )
{
    long tick;

    unsigned char d0,d1;

    int event_x;
    int event_width;
    int event_height;

    bool selected;


    /* draw boxes from sequence */
    if ( !a_flush )
	m_gc.set_foreground( m_black );
    else
	m_gc.set_foreground( m_white );

    m_seq->reset_draw_marker();
    while ( m_seq->get_next_event( m_status,
				   m_cc,
				   &tick, &d0, &d1, 
				   &selected ) == true )
    {
	/* turn into screen corrids */
	    
	event_x = tick / m_zoom;
	event_width  = c_data_x;

	/* generate the value */
	event_height = d1;
	
	if ( m_status == EVENT_PROGRAM_CHANGE ||
	     m_status == EVENT_CHANNEL_PRESSURE  ){

	    event_height = d0;
	}

	/* redraw ?, then set to full */
	if ( a_flush ){

	    event_height = c_dataarea_y - 1;
	    event_width = c_data_x + 8;
	}
	
	/* draw vert lines */
	a_draw->draw_rectangle(m_gc,true,
			       event_x,
			       c_dataarea_y - event_height, 
			       event_width, 
			       event_height );
	
	char val[4];
	sprintf( val, "%3d", event_height );

	a_draw->draw_text( c_text_font_5_7, m_gc, event_x + c_data_x + 1 ,
			    c_dataarea_y - 14, &val[0], 1 );
	a_draw->draw_text( c_text_font_5_7, m_gc, event_x + c_data_x + 1 ,
			    c_dataarea_y - 7, &val[1], 1 );
	a_draw->draw_text( c_text_font_5_7, m_gc, event_x + c_data_x + 1 ,
			    c_dataarea_y - 0, &val[2], 1 );
    }
}




void 
seqdata::draw_events_on_pixmap()
{
    draw_events_on( &m_pixmap, false );
}

void 
seqdata::draw_pixmap_on_window()
{
    queue_draw();
}


int 
seqdata::idle_redraw()
{
    /* no flicker, redraw */
    if ( !m_dragging ){
	draw_events_on( &m_window, false );
	draw_events_on( &m_pixmap, false );
    }
	return true;
}

int 
seqdata::expose_event_impl(GdkEventExpose* a_e)
{
    m_window.draw_pixmap(m_gc, 
    			 m_pixmap, 
    			 a_e->area.x,
			 a_e->area.y,
			 a_e->area.x,
			 a_e->area.y,
			 a_e->area.width,
			 a_e->area.height );
    return true;
}

/* takes screen corrdinates, give us notes and ticks */
void 
seqdata::convert_x( int a_x, long *a_tick )
{
    *a_tick = a_x * m_zoom; 
}

int 
seqdata::button_press_event_impl(GdkEventButton* a_p0)
{

    if ( a_p0-> type == GDK_BUTTON_PRESS ){



		if ( a_p0->button == 4 || a_p0->button == 5   ){	
		
			if ( a_p0->button == 4 ){
				m_seq->increment_selected( m_status, m_cc );
			}
			if ( a_p0->button == 5 ){
				m_seq->decrement_selected( m_status, m_cc );
			}

			return true;
		}
    
		/* set values for line */
		m_drop_x = (int) a_p0->x;
		m_drop_y = (int) a_p0->y;
		
		/* reset box that holds dirty redraw spot */
		m_old.x = 0;
		m_old.y = 0;
		m_old.width = 0;
		m_old.height = 0;
		
		m_dragging = true;
    }

    return true;
}

int 
seqdata::button_release_event_impl(GdkEventButton* a_p0)
{
    m_current_x = (int) a_p0->x;
    m_current_y = (int) a_p0->y;

    if ( m_dragging ){

	long tick_s, tick_f;

	if ( m_current_x < m_drop_x ){
	    swap( m_current_x, m_drop_x );
	    swap( m_current_y, m_drop_y );
	}

	convert_x( m_drop_x, &tick_s );
	convert_x( m_current_x, &tick_f );


	if ( m_drop_y < 0 ) m_drop_y = 0;
	if ( m_drop_y > 127 ) m_drop_y = 127;

	if ( m_current_y < 0 ) m_current_y = 0;
	if ( m_current_y > 127 ) m_current_y = 127;

  	m_seq->change_event_data_range( tick_s, tick_f,
					m_status,
					m_cc,
					c_dataarea_y - m_drop_y -1, 
					c_dataarea_y - m_current_y-1 );

	/* convert x,y to ticks, then set events in range */
	m_dragging = false;
    }

    update_pixmap();
    draw_pixmap_on_window();	    
    return true;
}


// Takes two points, returns a Xwin rectangle 
void 
seqdata::xy_to_rect(  int a_x1,  int a_y1,
		      int a_x2,  int a_y2,
		      int *a_x,  int *a_y,
		      int *a_w,  int *a_h )
{
    /* checks mins / maxes..  the fills in x,y
       and width and height */

    if ( a_x1 < a_x2 ){
	*a_x = a_x1; 
	*a_w = a_x2 - a_x1;
    } else {
	*a_x = a_x2; 
	*a_w = a_x1 - a_x2;
    }

    if ( a_y1 < a_y2 ){
	*a_y = a_y1; 
	*a_h = a_y2 - a_y1;
    } else {
	*a_y = a_y2; 
	*a_h = a_y1 - a_y2;
    }
}

int 
seqdata::motion_notify_event_impl(GdkEventMotion* a_p0)
{
    if ( m_dragging ){

	m_current_x = (int) a_p0->x;
	m_current_y = (int) a_p0->y;

	long tick_s, tick_f;

	int adj_x_min, adj_x_max,
	    adj_y_min, adj_y_max;

	if ( m_current_x < m_drop_x ){

	    adj_x_min = m_current_x;
	    adj_y_min = m_current_y;
	    adj_x_max = m_drop_x;
	    adj_y_max = m_drop_y;

	} else {

	    adj_x_max = m_current_x;
	    adj_y_max = m_current_y;
	    adj_x_min = m_drop_x;
	    adj_y_min = m_drop_y;
	}

	convert_x( adj_x_min, &tick_s );
	convert_x( adj_x_max, &tick_f );

	if ( adj_y_min < 0   ) adj_y_min = 0;
	if ( adj_y_min > 127 ) adj_y_min = 127;
	if ( adj_y_max < 0   ) adj_y_max = 0;
	if ( adj_y_max > 127 ) adj_y_max = 127;	

	m_seq->change_event_data_range( tick_s, tick_f,
					m_status,
					m_cc,
					c_dataarea_y - adj_y_min -1, 
					c_dataarea_y - adj_y_max -1 );
	
	/* convert x,y to ticks, then set events in range */
	update_pixmap();
	//draw_pixmap_on_window();  

	draw_events_on( &m_window, true );
	draw_events_on( &m_window, false );

	draw_line_on_window();
    }

    return true;
}


int
seqdata::leave_notify_event_impl(GdkEventCrossing* p0)
{
    m_dragging = false;
    update_pixmap();
    draw_pixmap_on_window();	    
    return true;
}




void 
seqdata::draw_line_on_window( void )
{
    int x,y,w,h;
    m_gc.set_line_style( GDK_LINE_SOLID );

   /* replace old */
    m_window.draw_pixmap(m_gc, 
			 m_pixmap, 
			 m_old.x,
			 m_old.y,
			 m_old.x,
			 m_old.y,
			 m_old.width + 1,
			 m_old.height + 1 );

    xy_to_rect ( m_drop_x,
		 m_drop_y,
		 m_current_x,
		 m_current_y,
		 &x, &y,
		 &w, &h );

    m_old.x = x;
    m_old.y = y;
    m_old.width = w;
    m_old.height = h;

    m_gc.set_foreground(m_black);
    m_window.draw_line(m_gc, m_current_x, m_current_y, m_drop_x, m_drop_y );

}
