# Copyright (C) 2004 Tiago Cogumbreiro <cogumbreiro@users.sf.net>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#
# Authors: Tiago Cogumbreiro <cogumbreiro@users.sf.net>

"""
This is the window widget which will contain the audio mastering widget
defined in audio_widgets.AudioMastering. 
"""

import os
import os.path
import gtk
import gtk.glade
import gobject
import statvfs

from gettext import gettext as _

# Private modules
from serpentine import operations
from serpentine import urlutil

from serpentine.operations import (CallbackListener, SuccessfulOperation,
                                   OperationsQueue, Listenable,
                                   CallableOperation)
from serpentine.mastering import GtkMusicList, MusicListGateway
from serpentine.mastering import ErrorTrapper
from serpentine.recording import ConvertAndWrite
from serpentine.preferences import RecordingPreferences
from serpentine.components import Component
from serpentine.mainwindow import SerpentineWindow
from serpentine.common import SerpentineNotSupportedError, SerpentineCacheError
from serpentine.plugins import plugins

PATTERN_SEPARATOR = ";"

def _validate_music_list (music_list, preferences):
    """Check if we have space available in our cache dir"""
    secs = 0
    missing_musics = []
    for music in music_list:
        if not urlutil.exists(music["location"]):
            missing_musics.append("%s - %s" % (music["title"], music["artist"]))
            
        # When music is not available it will have to be converted
        if not preferences.pool.is_available (music["location"]):
            secs += music["duration"]
            
    if len(missing_musics) > 0:
        raise SerpentineCacheError(SerpentineCacheError.INVALID,
                "%s\n%s" % (
                    _("The following musics have missing files:"),
                    "\n".join(missing_musics)
                ))
                 
    # 44100hz * 16bit * 2channels / 8bits = 176400 bytes per sec
    size_needed = secs * 176400L
    
    # Now check if cache location is ok
    for tmp in preferences.temporary_dirs:
        try:
            stat = os.statvfs(tmp)
        except OSError:
            # skip it if there's an error
            continue
        
        # Now check for available size
        size_avail = stat[statvfs.F_BAVAIL] * long(stat[statvfs.F_BSIZE])
        if (size_avail - size_needed) < 0:
            preferences.pool.clear ()
        
        size_avail = stat[statvfs.F_BAVAIL] * long(stat[statvfs.F_BSIZE])
        
        if size_avail - size_needed >= 0:
            # Yay we have the needed space
            return
            
    raise SerpentineCacheError(SerpentineCacheError.INVALID, _("Please "
                            "check if the cache location exists and "
                            "has writable permissions."))



class SavePlaylistRegistry(Component, Listenable):
    """Component responsible for keeping a registry of objects
    that can save playlists."""
    def __init__(self, parent):
        Component.__init__(self, parent)
        Listenable.__init__(self)
        
        # all files filter
        all_files = gtk.FileFilter()
        all_files.set_name(_("All files"))
        all_files.add_pattern("*")
        
        self.__global_filter = gtk.FileFilter()
        self.__global_filter.set_name(_("Supported playlists"))
        self.__file_filters = [self.__global_filter, all_files]
        self.__factories = {}
        
    file_filters = property(lambda self: self.__file_filters)
    
    def register (self, factory, extension, description):
        self.__global_filter.add_pattern ("*" + extension)
        
        file_filter = gtk.FileFilter ()
        file_filter.set_name (description)
        file_filter.add_pattern ("*" + extension)
        
        self.file_filters.append (file_filter)

        self._notify('on_registred', factory, extension, description)
        
        self.__factories[extension] = factory
        self.__global_filter.add_pattern ("*" + extension)
    
    def save(self, filename, extension=None):
        if extension is None:
            fname, extension = os.path.splitext(filename)

        try:
            return  self.__factories[extension](self.parent.music_list,
                                                filename)
        except KeyError:
            raise SerpentineNotSupportedError (extension)
    
class LeaseMusicList(object):
    def __init__(self, music_list, pool):
        self.music_list = music_list
        self.pool = pool
        
    def request_music_list(self):
        self.requested = list(row['location']
                              for row in self.music_list)
        for location in self.requested:
            self.pool.request_music(location)

    def release_music_list(self):
        for location in self.requested:
            self.pool.release_music(location)
            

class Application(operations.Operation, Component):
    components = ()
    def __init__ (self, locations):
        operations.Operation.__init__(self)
        Component.__init__(self, None)
        self.savePlaylist = SavePlaylistRegistry(self)
        self.locations = locations
        self.__preferences = RecordingPreferences(locations)
        self.__running_ops = []
        
        self._music_file_patterns = {}
        self._playlist_file_patterns = {}
        self._music_file_filters = None
        self._playlist_file_filters = None
        self.register_music_file_pattern("MPEG Audio Stream, Layer III",
                                         "*.mp3")
        self.register_music_file_pattern("Ogg Vorbis Codec Compressed WAV File",
                                         "*.ogg")
        self.register_music_file_pattern("Free Lossless Audio Codec", "*.flac")
        self.register_music_file_pattern("MPEG 4 Audio", "*.m4a;*.mp4")
        self.register_music_file_pattern("PCM Wave audio", "*.wav")
        self._writing_started = CallbackListener(self._on_writing_started,
                                                 ('on_finished',))
        self._writing_finished = CallbackListener(self._on_writing_finished,
                                                  ('on_finished',))
    
    def _load_plugins (self):
        # Load Plugins
        self.__plugins = []
        for plug in plugins:
            try:
                self.__plugins.append(plugins[plug].create_plugin(self))
            except Exception, e:
                print "Error loading plugin", plug
                import traceback
                traceback.print_exc()

    cache_pool = property(lambda self: self.__preferences.pool)

    preferences = property(lambda self: self.__preferences)

    running_ops = property(lambda self: self.__running_ops)

    can_stop = property(lambda self: len(self.running_ops) == 0)
    
    # The window is only none when the operation has finished
    can_start = property(lambda self: self.__window is not None)

    def get_music_list(self):
        raise NotImplementedError()

    music_list = property(get_music_list)

    def get_music_list_gw(self):
        raise NotImplementedError()

    music_list_gw = property(get_music_list_gw)

    def on_finished (self, event):
        # We listen to operations we contain, when they are finished
        # we remove them
        self.__running_ops.remove(event.source)
        if self.can_stop:
            self.stop()

    def stop(self):
        assert self.can_stop, "Check if you can stop the operation first."
        self.preferences.savePlaylist(self.music_list)
        self.preferences.pool.clear()
        # Warn listeners
        self._send_finished_event(operations.SUCCESSFUL)
            
        # Cleanup plugins
        del self.__plugins

    validated = False
    def validate_files(self):
        """Call this before calling write files"""
        music_list = self.music_list_gw.music_list
        _validate_music_list(music_list, self.preferences)
        self.validated = True

    def write_files(self, window=None):
        # TODO: we should add a confirmation dialog
        if not self.validated:
            raise ValueError("Call validate_files first")

        start = SuccessfulOperation()
        start.add_listener(self._writing_started)

        lease = LeaseMusicList(self.music_list, self.cache_pool)

        request = CallableOperation(lease.request_music_list)

        release = CallableOperation(lease.release_music_list)
        
        conv_write = ConvertAndWrite(self.music_list, self.preferences, window)

        queue = OperationsQueue([start, request, conv_write, release])
        queue.abort_on_failure = False

        # remove this operation from the running ops
        self.running_ops.append(queue)
        queue.add_listener(self)
        
        queue.add_listener(self._writing_finished)
        
        self.validated = False
        
        return queue
    
    write_files = operations.async(write_files)

    def _on_writing_started(self, *args):
        """Called by '_writing_started'"""
        self._notify('on_writing_started', self)
        
    def _on_writing_finished(self, *args):
        """Called by '_writing_finished'"""
        self._notify('on_writing_finished', self)
    
    # TODO: should these be moved to MusicList object?
    # TODO: have a better definition of a MusicList
    def add_hints_filter (self, location_filter):
        self.music_list_gw.add_hints_filter (location_filter)
        
    def remove_hints_filter (self, location_filter):
        self.music_list_gw.remove_hints_filter (location_filter)
    
    def register_music_file_pattern (self, name, pattern):
        """
        Music patterns are meant to be used in the file dialog for adding
        musics to the playlist.
        
        To associate more then one pattern to the same name use the ';' char
        as the pattern separator.
        """
        
        self._music_file_patterns[pattern.strip(PATTERN_SEPARATOR)] = name
        self._music_file_filters = None

    def register_playlist_file_pattern (self, name, pattern):
        """Playlist patterns are meant to be used in the file dialog for adding
        playlist contents to the current playlist."""
        self._playlist_file_patterns[pattern] = name
        self._playlist_file_filters = None
    
    def __gen_file_filters(self, patterns, all_filters_name):
        all_files = gtk.FileFilter()
        all_files.set_name(_("All files"))
        all_files.add_pattern("*")
        
        all_musics = gtk.FileFilter()
        all_musics.set_name(all_filters_name)
        
        filters = [all_musics]
        
        for pattern, name in patterns.iteritems():
            patterns = pattern.split(PATTERN_SEPARATOR)
            for patt in patterns:
                all_musics.add_pattern(patt)
            
            filter = gtk.FileFilter()
            filter.set_name(name)
            for patt in patterns:
                filter.add_pattern(patt)
            
            filters.append(filter)
        
        filters.append(all_files)
        return filters
    
    def __get_file_filter (self, filter_attr, pattern_attr, name):
        file_filter = getattr (self, filter_attr)
        if file_filter is not None:
            return file_filter
        
        setattr (
            self,
            filter_attr,
            self.__gen_file_filters (
                getattr (self, pattern_attr),
                name
            )
        )
        
        return getattr (self, filter_attr)
        
    music_file_filters = property (
        lambda self: self.__get_file_filter (
            "_music_file_filters",
            "_music_file_patterns",
            _("Common media files")
        )
    )

    playlist_file_filters = property (
        lambda self: self.__get_file_filter (
            "_playlist_file_filters",
            "_playlist_file_patterns",
            _("Supported playlists")
        )
    )

    
    # a list of filenames, can be URI's
    def add_files(self, files):
        """Returns an operation"""
        return self.music_list_gw.add_files (files)

    add_files = operations.async(add_files)

    def clear_files(self):
        self.music_list.clear()


class HeadlessApplication (Application):

    
    class Gateway (MusicListGateway):
        def __init__ (self, app):
            MusicListGateway.__init__(self, app)
            self.music_list = GtkMusicList ()
        
        class Handler:
            def prepare_queue (self, gateway, queue):
                self.trapper = ErrorTrapper (None)
            
            def prepare_add_file(self, gateway, add_file):
                add_file.add_listener(self.trapper)
                
            def finish_queue (self, gateway, queue):
                queue.append (self.trapper)
                del self.trapper
        

    def __init__ (self, locations):
        Application.__init__ (self, locations)
        self.music_list_gw = HeadlessApplication.Gateway(self)
        self._load_plugins ()

    def get_music_list(self):
        return self.music_list_gw.music_list

    music_list = property(get_music_list)

    music_list_gw = None


class SerpentineApplication(Application):
    """When no operations are left in SerpentineApplication it will exit.
    An operation can be writing the files to cd or showing the main window.
    This enables us to close the main window and continue showing the progress
    dialog. This object should be simple enough for D-Bus export.
    """
    def __init__(self, locations):
        Application.__init__(self, locations)
        self.__window = SerpentineWindow(self)
        self.preferences.dialog.set_transient_for(self.window_widget)
        self.__window.add_listener(self)
        self._load_plugins()


    window_widget = property (lambda self: self.__window)

    def get_music_list(self):
        return self.window_widget.music_list

    music_list = property(get_music_list)

    def get_music_list_gw(self):
        return self.window_widget.masterer.music_list_gateway

    music_list_gw = property(get_music_list_gw)
    
    def write_files(self, window=None):
        """Writes the files onto the Audio-CD."""
        super_self = super(SerpentineApplication, self)
        return super_self.write_files(self.window_widget)
    
    # TODO: decouple the window from SerpentineApplication ?
    def show_window(self):
        """
        Opens the main window.
        """
        # Starts the window operation
        self.__window.start ()
        self.running_ops.append (self.__window)
    
    def close_window(self):
        """
        Closes the main window.
        """
        # Stops the window operation
        if self.__window.running:
            self.__window.stop ()
    
    def stop(self):
        """
        Stops the application.
        """
        # Clean window object
        Application.stop(self)
        self.__window.destroy()
        del self.__window
        

    


gobject.type_register (SerpentineWindow)

