
/*
#    Sfront, a SAOL to C translator    
#    This file: coremidi control driver for sfront
#    Copyright (C) 2002 Regents of the University of California
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License (Version 2) as
#    published by the Free Software Foundation.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#    Maintainer: John Lazzaro, lazzaro@cs.berkeley.edu
*/

/****************************************************************/
/****************************************************************/
/*                coremidi control driver for sfront            */ 
/****************************************************************/

#include <sys/types.h>
#include <sys/socket.h>
#include <unistd.h>
#include <sys/uio.h>
#include <fcntl.h>

#if !(defined(ASYS_OUTDRIVER_COREAUDIO)||defined(ASYS_INDRIVER_COREAUDIO))
#include <CoreFoundation/CFString.h>
#endif

#include <CoreMIDI/CoreMIDI.h>

/***************************/
/* graceful_exit constants */
/***************************/

#define CSYSI_GE_RUNNING  0
#define CSYSI_GE_DOEXIT   1
#define CSYSI_GE_EXITED   2

#define CSYSI_RETRY_MAX   256    /* retry counter for read() EINTR */

/*************************/
/* MIDI source constants */
/*************************/

#define CSYSI_MAXPORTS 4       /* must match ../../control.c  */

/**********************************/
/* buffer constants -- needs work */
/**********************************/

#define CSYSI_BUFFSIZE   1024  /* size of the holding buffer   */
#define CSYSI_SENDSIZE   256   /* maximum size of one packet[] */

/**********************/
/* preamble constants */
/**********************/

/*

 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 
-------------------------------------------------------------------
|    0xFD       |T|R|R|R|  EXT  |  tstamp (T = 1)  ... | MIDI ... |
-------------------------------------------------------------------

  T is set to 1 if a non-NOW timestamp follows the header octet.
  R's are reserved.
  EXT codes which of 16 16-channel MIDI sources.

*/

#define CSYSI_PREAMBLE_SIZE    6  /* read_proc preamble w/ timestamp */
#define CSYSI_PREAMBLE_NOWSIZE 2  /* read_proc preamble w/o timestamp */

#define CSYSI_PREAMBLE_POS_MARKER  0  /* first octet is the marker: 0xFF     */
#define CSYSI_PREAMBLE_POS_CONTROL 1  /* second octet holds control bits     */
#define CSYSI_PREAMBLE_POS_TSTAMP  2  /* last 4 (optional) octets for tstamp */

#define CSYSI_PREAMBLE_FLAGS_MARKER  0xFD  /* marker value: for easy parsing */
#define CSYSI_PREAMBLE_FLAGS_T       0x80  /* timestamp follows this octet   */
#define CSYSI_PREAMBLE_FLAGS_R2      0x40  /* reserved bits                  */
#define CSYSI_PREAMBLE_FLAGS_R1      0x20  
#define CSYSI_PREAMBLE_FLAGS_R0      0x10  
#define CSYSI_PREAMBLE_FLAGS_EXT     0x0F  /* the EXT nibble                 */
 
/************************/
/* input port variables */
/************************/

MIDIClientRef csysi_inclient;  /* the input client  */
MIDIPortRef   csysi_inport;    /* it's port         */

typedef struct csysi_portinfo {
  int ext;               /* EXT field for preamble, also array index */
  int valid;             /* 0 if unused, 1 if online, -1 if offline  */
  MIDIEndpointRef src;   /* current src  (may be invalid if offline) */
  int id;                /* unique ID of src (currently unused)      */
} csysi_portinfo;

csysi_portinfo csysi_ports[CSYSI_MAXPORTS];

/**********************************/
/* read_proc pipe data structures */
/**********************************/

int csysi_readproc_pipepair[2];
int csysi_graceful_exit;

/************************/
/* command buffer state */
/************************/

unsigned char csysi_buffer[CSYSI_BUFFSIZE];
int csysi_len, csysi_cnt;
unsigned short csysi_currext;

/****************************************************************/
/*          generic error-checking wrappers                     */
/****************************************************************/

#define  CSYSI_ERROR_RETURN(x) do {\
      fprintf(stderr, "  Error: %s.\n", x);\
      fprintf(stderr, "  Errno Message: %s\n\n", strerror(errno));\
      return CSYS_ERROR; } while (0)

#define  CSYSI_ERROR_TERMINATE(x) do {\
      fprintf(stderr, "  Runtime Errno Message: %s\n", strerror(errno));\
      epr(0,NULL,NULL, "Soundcard error -- " x );}  while (0)

/*******************/
/* forward externs */
/*******************/

extern void csysi_read_proc(const MIDIPacketList * pktlist,
			    void * readProcRefCon, 
			    void * srcConnRefCon);

extern void csysi_notify_proc(const MIDINotification * message,
			      void * refcon);

/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
/*     high-level functions: called by sfront engine            */
/*______________________________________________________________*/

/****************************************************************/
/*             initialization routine for control               */
/****************************************************************/

int csys_setup(void)
     
{
  int connected, numsrc, flags, i, numvsrc;
  MIDIEndpointRef src;
  MIDIEntityRef ent;
  char name[256];
  CFStringRef cf_name;

  /********************************/
  /* create input client and port */
  /********************************/

  if (MIDIClientCreate(CFSTR("sfront_inclient"), NULL, 
		       (void *) NULL, &csysi_inclient))
    CSYSI_ERROR_RETURN("Could not create CoreMIDI client");

  if (MIDIInputPortCreate(csysi_inclient, CFSTR("sfront_inport"),
			  csysi_read_proc, (void *) NULL, 
			  &csysi_inport))
    CSYSI_ERROR_RETURN("Could not create CoreMIDI inport");

  /********************************************************/
  /* open pipes, since data may start flowing immediately */
  /********************************************************/

  if (socketpair(AF_UNIX, SOCK_DGRAM, 0, csysi_readproc_pipepair))
    CSYSI_ERROR_RETURN("Cannot open readproc pipe");

  if ((flags = fcntl(csysi_readproc_pipepair[0], F_GETFL, 0)) == -1)
    CSYSI_ERROR_RETURN("Unknown flags for read end of readproc pipe");

  if (fcntl(csysi_readproc_pipepair[0], F_SETFL, flags | O_NONBLOCK) == -1) 
    CSYSI_ERROR_RETURN("Cannot O_NONBLOCK read end of readproc pipe");

  /********************************************************/
  /* for now, connect only one source, exit if no sources */
  /********************************************************/
  
  for (i = 0; i < CSYSI_MAXPORTS; i++)
    csysi_ports[i].ext = i;

  numvsrc = connected = 0;
  numsrc = (int) MIDIGetNumberOfSources();
  
  for (i = 0; i < numsrc; i++)
    if (src = MIDIGetSource((ItemCount) i))
      {
	if (MIDIEndpointGetEntity(src, &ent) || (ent == NULL))
	  {
	    numvsrc++;
	    continue;
	  }

	if (MIDIPortConnectSource(csysi_inport, src,
				  &(csysi_ports[connected].ext)))
	  continue;

	csysi_ports[connected].valid = 1;
	csysi_ports[connected].src = src;
	connected++;

	if ((connected == CSYSI_MAXPORTS) && (i + 1 != numsrc))
	  {
	    fprintf(stderr, 
		    "\nWarning: -cin coremidi configured for %i ports max,\n", 
		    CSYSI_MAXPORTS);
	    fprintf(stderr, 
		    "         skipping the last %i MIDI port(s) in your setup.\n", 
		    numsrc - i - 1);
	    break;
	  }
      }

  fprintf(stderr, "\nCoreMIDI setup info:\n");

  if (!connected)
    {
      fprintf(stderr, "   No CoreMIDI sources to connect with, exiting ...\n");
      if (numvsrc)
	fprintf(stderr, "   (Sfront doesn't support virtual sources yet.)\n");
      return CSYS_ERROR;
    }

  /*******************************/
  /* print out source identities */
  /*******************************/

  for (i = 0; i < connected; i++)
    {
      fprintf(stderr, "   SAOL MIDI channels %i-%i connect to ",
	      16*csysi_ports[i].ext, 16*(csysi_ports[i].ext + 1) - 1);

      src = csysi_ports[i].src;
      
      if (MIDIObjectGetStringProperty(src, kMIDIPropertyName, &cf_name))
	{
	  fprintf(stderr, "(unnamed source).\n");
	  continue;
	}

      if (!CFStringGetCString(cf_name, name, 256, CFStringGetSystemEncoding()))
	{
	  fprintf(stderr, "(unnamed source).\n");
	  CFRelease(cf_name);
	  continue;
	}

      CFRelease(cf_name);
      fprintf(stderr, "%s.\n", name);
    }

  if (numvsrc)
    fprintf(stderr, 
	    "   (Virtual sources also found; sfront doesn't support them yet.)\n");

  fprintf(stderr, "\nMIDI Preset Numbers ");

  i = 0;
  while (i < csys_sfront_argc)
    {
      if (!(strcmp(csys_sfront_argv[i],"-bitc") && 
	    strcmp(csys_sfront_argv[i],"-bit") &&
	    strcmp(csys_sfront_argv[i],"-orc")))
	{
	  i++;
	  fprintf(stderr, "for %s ", csys_sfront_argv[i]);
	  break;
	}
      i++;
    }

  fprintf(stderr, "(use MIDI controller to select):\n");

  for (i = 0; i < CSYS_PRESETNUM; i++)
    {
      fprintf(stderr, "%3i. %s", 
	      csys_presets[i].preset,
	      csys_instr[csys_presets[i].index].name);
      if ((i&1))
	fprintf(stderr, "\n");
      else
	{
	  fprintf(stderr, "\t\t");
	  if (i == (CSYS_PRESETNUM-1))
	    fprintf(stderr, "\n");
	}
    }

  return CSYS_DONE;
}

/****************************************************************/
/*             polling routine for new data                     */
/****************************************************************/

int csys_newdata(void)
     
{
  int len, retry;

  if (((len = read(csysi_readproc_pipepair[0], csysi_buffer, 
		   CSYSI_SENDSIZE)) < 0) && (errno == EAGAIN))
    return CSYS_NONE;  /* fast path */

  if (csysi_graceful_exit == CSYSI_GE_EXITED)
    return CSYS_NONE;

  csysi_len = 0;

  do 
    {
      if (len > 0)
	{
	  if ((csysi_len += len) > (CSYSI_BUFFSIZE - CSYSI_SENDSIZE))
	    break;
	}
      else
	{
	  if (len == 0)
	    {
	      fprintf(stderr, "-cin coremidi: writev() error.\n");
	      csysi_graceful_exit = CSYSI_GE_DOEXIT;
	      break;
	    }
	  else
	    if ((errno != EINTR) || (++retry > CSYSI_RETRY_MAX)) 
	      {
		fprintf(stderr, "-cin coremidi: pipe read() error.\n");
		csysi_graceful_exit = CSYSI_GE_DOEXIT;
		break;
	      }
	}

      len = read(csysi_readproc_pipepair[0], &(csysi_buffer[csysi_len]),
		 CSYSI_SENDSIZE);
    } 
  while ((len >= 0) || (errno != EAGAIN));

  csysi_cnt = 0;
  return CSYS_MIDIEVENTS;
}

/****************************************************************/
/*                 processes a MIDI event                       */
/****************************************************************/

int csys_midievent(unsigned char * cmd,   unsigned char * ndata, 
	           unsigned char * vdata, unsigned short * extchan,
		   float * fval)

{
  if (csysi_graceful_exit == CSYSI_GE_DOEXIT)
    {
      csysi_graceful_exit = CSYSI_GE_EXITED;
      *cmd = CSYS_MIDI_ENDTIME;
      *fval = scorebeats;
      return CSYS_NONE;
    }

  while (csysi_buffer[csysi_cnt] == CSYSI_PREAMBLE_FLAGS_MARKER)
    {
      /* later handle semantics of the timestamp bits */

      csysi_currext = 
	(csysi_buffer[++csysi_cnt] & CSYSI_PREAMBLE_FLAGS_EXT) << 4;

      /* skip over the rest of the preamble */

      if ((csysi_buffer[csysi_cnt] & CSYSI_PREAMBLE_FLAGS_T) == 0)
	csysi_cnt++;     
      else 
	csysi_cnt += 5;

      if (csysi_cnt == csysi_len)
	{
	  *cmd = CSYS_MIDI_NOOP;  
	  return CSYS_NONE;
	}
    }

  /* assumes:
   *   1 - integral number of commands
   *   2 - all system commands filtered out except 0xFF
   *   3 - input stream is perfect MIDI
   */

  if (csysi_buffer[csysi_cnt] != CSYS_MIDI_SYSTEM_RESET)
    {
      *cmd = 0xF0 & csysi_buffer[csysi_cnt];
      *extchan = (0x0F & csysi_buffer[csysi_cnt++]) + csysi_currext;
      *ndata = csysi_buffer[csysi_cnt++];

      if ((*cmd != CSYS_MIDI_PROGRAM) && (*cmd != CSYS_MIDI_CTOUCH))
	*vdata = csysi_buffer[csysi_cnt++];
    }
  else
    {
      *cmd = CSYS_MIDI_POWERUP;
      *extchan = csysi_currext;
      csysi_cnt++;
    }

  if (csysi_cnt == csysi_len)
    return CSYS_NONE;
  else
    return CSYS_MIDIEVENTS;
}

/****************************************************************/
/*                  closing routine for control                 */
/****************************************************************/

void csys_shutdown(void)
     
{
  close(csysi_readproc_pipepair[0]);
  close(csysi_readproc_pipepair[1]);
  return;
}

/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
/*       callback functions: called by coremidi directly        */
/*______________________________________________________________*/

extern int csysi_packet_shorten(MIDIPacket * p, 
				unsigned char * mdata,
				int * idx); 

/****************************************************************/
/*                  callback for MIDI input                     */
/****************************************************************/

void csysi_read_proc(const MIDIPacketList * pktlist,
		     void * readProcRefCon, void * srcConnRefCon)

{
  unsigned char preamble[CSYSI_PREAMBLE_NOWSIZE];
  unsigned char mdata[CSYSI_SENDSIZE];
  struct iovec vector[2];
  MIDIPacket * p;
  int num, idx, retry;

  /* initialize preamble */

  preamble[CSYSI_PREAMBLE_POS_MARKER] = CSYSI_PREAMBLE_FLAGS_MARKER;
  preamble[CSYSI_PREAMBLE_POS_CONTROL] = 
    CSYSI_PREAMBLE_FLAGS_EXT & ((unsigned char) *((int *) srcConnRefCon));

  /* initialize write vector -- update for timestamp support */

  vector[0].iov_base = (char *) preamble;
  vector[0].iov_len = (size_t) CSYSI_PREAMBLE_NOWSIZE;
  vector[1].iov_base = (char *) mdata;

  /* loop through packets, send DGRAMs */

  p = (MIDIPacket *) &(pktlist->packet[0]);
  num = pktlist->numPackets;

  while (num--)
    {
      retry = idx = 0;

      do {
	if ((vector[1].iov_len = csysi_packet_shorten(p, mdata, &idx)))
	  while (writev(csysi_readproc_pipepair[1], vector, 2) < 0)
	    if ((errno != EINTR) || (++retry >= CSYSI_RETRY_MAX))
	      {
		close(csysi_readproc_pipepair[0]);
		close(csysi_readproc_pipepair[1]);
		return;
	      }
      } while (idx < p->length);

      p = MIDIPacketNext(p);
    }
}

/****************************************************************/
/*                  callback for MIDI input                     */
/****************************************************************/

void csysi_notify_proc(const MIDINotification * message,
		       void * refcon)

{
  /* not currently registered with MIDIClientCreate() */
}

/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
/*   mid-level functions: called by top-level driver functions  */
/*______________________________________________________________*/

/****************************************************************/
/*               copy SA-safe commands into mdata               */
/****************************************************************/

int csysi_packet_shorten(MIDIPacket * p, unsigned char * mdata,
			 int * idx)
{
  int len = 0;

  do {
    switch (p->data[*idx] & 0xF0) {
    case CSYS_MIDI_NOTEOFF:
    case CSYS_MIDI_NOTEON:
    case CSYS_MIDI_PTOUCH:
    case CSYS_MIDI_WHEEL:
    case CSYS_MIDI_CC:
      if ((p->length >= (*idx + 3)) && (p->data[*idx + 1] < 128)
	  && (p->data[*idx + 2] < 128))
	{
	  memcpy(&(mdata[len]), &(p->data[*idx]), 3);
	  if (p->length == (*idx += 3))
	    return len + 3;
	  if ((len += 3) > (CSYSI_SENDSIZE - 3))
	    return len;
	}
      else
	{
	  mdata[len] = CSYS_MIDI_SYSTEM_RESET; /* CoreMIDI bug recovery */
	  *idx = p->length;
	  return len + 1;   
	}
      continue;
    case CSYS_MIDI_PROGRAM:
    case CSYS_MIDI_CTOUCH:
      if ((p->length >= (*idx + 2)) && (p->data[*idx + 1] < 128))
	{
	  memcpy(&(mdata[len]), &(p->data[*idx]), 2);
	  if (p->length == (*idx += 2))
	    return len + 2;
	  if ((len += 2) > (CSYSI_SENDSIZE - 3))
	    return len;     
	}
      else
	{
	  mdata[len] = CSYS_MIDI_SYSTEM_RESET; /* CoreMIDI bug recovery */
	  *idx = p->length;
	  return len + 1;   
	}
      continue;
    case CSYS_MIDI_SYSTEM:
      switch (p->data[*idx]) {
      case CSYS_MIDI_SYSTEM_RESET:        
	memcpy(&(mdata[len]), &(p->data[*idx]), 1);
	if (p->length == (*idx += 1))
	  return len + 1;
	if ((len += 1) > (CSYSI_SENDSIZE - 3))
	  return len;     
	continue;
      case CSYS_MIDI_SYSTEM_SONG_PP:                  /* skip 3-octet commands  */ 
	if (p->length == (*idx += 3))
	  return len;
	continue;
      case CSYS_MIDI_SYSTEM_QFRAME:                   /* skip 2-octet commands */ 
      case CSYS_MIDI_SYSTEM_SONG_SELECT: 
	if (p->length == (*idx += 2))
	  return len;
	continue;
      case CSYS_MIDI_SYSTEM_TUNE_REQUEST:             /* skip 1-octet commands */ 
      case CSYS_MIDI_SYSTEM_CLOCK:     
      case CSYS_MIDI_SYSTEM_TICK:
      case CSYS_MIDI_SYSTEM_START:
      case CSYS_MIDI_SYSTEM_CONTINUE:
      case CSYS_MIDI_SYSTEM_STOP:
      case CSYS_MIDI_SYSTEM_UNUSED3:
      case CSYS_MIDI_SYSTEM_SENSE:
      case CSYS_MIDI_SYSTEM_SYSEX_END:
	if (p->length == (*idx += 1))
	  return len;
	continue;
      case CSYS_MIDI_SYSTEM_SYSEX_START:          /* arbitrary-length commands */
      case CSYS_MIDI_SYSTEM_UNUSED1:   
      case CSYS_MIDI_SYSTEM_UNUSED2:
	*idx += 1;
	while ((*idx < p->length) && !(p->data[(*idx)++] & 0x80)); /* skip data */
	if (*idx == p->length)
	  return len;
	continue;
      }
      *idx = p->length;                                /* should never run     */
      return len;
    default:                                           /* a sysex continuation */
      while ((*idx < p->length) && !(p->data[(*idx)++] & 0x80));   
      if (*idx == p->length)
	return len;
      continue;
    }
  }
  while (1);
  
  *idx = p->length;  /* should never run, loop has no exit path */
  return len;
} 


/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
/*                 low-level functions                          */
/*______________________________________________________________*/




