/*
 *  Part of the shrinkta program, a dvd backup tool
 *
 *  Copyright (C) 2005  Daryl Gray
 *  E-Mail Daryl Gray darylgray1@dodo.com.au
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
*/

/* Some of the code in this file was adapted from various files within the transcode package,
 * particularly the code to read disk properties and extract the cell data - Thanks to the transcode programmers!
 * The transcode package contains license (GPL) notices similar to the one below (from extract_pcm.c)
 * 
 * 
 * snip ===================================================================================================
 *
 *  extract_pcm.c
 *
 *  Copyright (C) Thomas Östreich - June 2001
 *
 *  This file is part of transcode, a linux video stream processing tool
 *      
 *  transcode is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  transcode is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * end ====================================================================================================
 */
#include <inttypes.h>
#include <config.h>
#include <glib/gi18n.h>
#include <glib-object.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <dvdread/dvd_reader.h>
#include <dvdread/ifo_types.h>
#include <dvdread/ifo_read.h>
#include <dvdread/nav_read.h>


#include "dvd.h"
#include "dvd-time.h"
#include "dvd-marshal.h"
#include "pes.h"

enum {
	CHAPTER_OUTPUT_CELL,
	CHAPTER_TIME,
	CHAPTER_LAST_SIGNAL
};

guint dvd_chapter_disk_signals[CHAPTER_LAST_SIGNAL];
static DvdChapterClass *dvd_chapter_disk_parent_class = NULL;

static void     dvd_chapter_disk_class_init	(DvdChapterDiskClass	 *class);
static void     dvd_chapter_disk_instance_init	(GTypeInstance		 *instance,
						 gpointer		  g_class);
static void     dvd_chapter_disk_dispose	(GObject		 *object);
static gboolean	dvd_chapter_disk_read_cell	(DvdChapterDisk		 *chapter,
						 dvd_reader_t		 *dvd_handle,
						 guint			  cell_id,
						 GError			**error);
static void	demux_output_data		(DvdDemux		 *demux,
						 DvdStream		  stream_type,
						 gint			  track,
						 gint			  bytes,
						 guint8			 *buffer,
						 guint32		  pts,
						 guint64		  frame_clocks,
						 gpointer		  user_data);

/* Imported by dvd_title_disk.c */
gboolean	dvd_chapter_disk_read_properties(DvdChapterDisk		 *chapter,
						 guint8			  chapter_id,
						 dvd_reader_t		 *dvd_handle,
						 const gchar		 *device,
						 guint8			  title_id,
						 GError			**error);
gboolean	dvd_chapter_disk_read_chapter	(DvdChapterDisk		 *chapter,
						 dvd_reader_t		 *dvd_handle,
						 GError			**error);

static void
dvd_chapter_disk_class_init	(DvdChapterDiskClass *class)
{
	GObjectClass *object_class = (GObjectClass *) class;
	dvd_chapter_disk_signals[CHAPTER_OUTPUT_CELL] =
		g_signal_new ("output_cell",
			      G_TYPE_FROM_CLASS(class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (DvdChapterDiskClass, output_cell),
			      NULL, NULL,
			      dvd_marshal_VOID__INT_INT_INT_POINTER_UINT_UINT64,
			      G_TYPE_NONE,
			      6,
			      G_TYPE_INT,
			      G_TYPE_INT,
			      G_TYPE_INT,
			      G_TYPE_POINTER,
			      G_TYPE_UINT,
			      G_TYPE_UINT64);
	dvd_chapter_disk_signals[CHAPTER_TIME] =
		g_signal_new ("chapter_time",
			      G_TYPE_FROM_CLASS(class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (DvdChapterDiskClass, chapter_time),
			      NULL, NULL,
			      dvd_marshal_VOID__POINTER_POINTER,
			      G_TYPE_NONE,
			      2,
			      G_TYPE_POINTER,
			      G_TYPE_POINTER);
	dvd_chapter_disk_parent_class = g_type_class_ref (DVD_CHAPTER_TYPE);
	object_class->dispose = dvd_chapter_disk_dispose;
}

static void
dvd_chapter_disk_instance_init	(GTypeInstance	*instance,
				 gpointer	 g_class)
{
	DvdChapterDisk *chapter;
	
	chapter = DVD_CHAPTER_DISK (instance);
	chapter->op_mutex = g_mutex_new ();
	DVD_CHAPTER (chapter)->source = DVD_SOURCE_DISK;
}

static void
dvd_chapter_disk_dispose	(GObject	*object)
{
	DvdChapterDisk *chapter;
	GSList *iter;
	
	chapter = DVD_CHAPTER_DISK (object);
	for (iter = chapter->cells;
	     iter != NULL;
	     iter = g_slist_next (iter)) {
		DvdCell *cell;
		
		cell = (DvdCell *) iter->data;
		g_free (cell->time);
		g_free (cell);
	}
	g_slist_free (chapter->cells);
	g_free (chapter->device);
	g_mutex_free (chapter->op_mutex);
	
	G_OBJECT_CLASS (dvd_chapter_disk_parent_class)->dispose (G_OBJECT (chapter));
}

/**
 * dvd_title_get_type
 * @return The GType for the DvdTitle class.
 */
GType
dvd_chapter_disk_get_type	(void)
{
	static GType chapter_type = 0;

	if (chapter_type == 0) {
		GTypeInfo chapter_info = {
			sizeof (DvdChapterDiskClass),
			NULL,
			NULL,
			(GClassInitFunc) dvd_chapter_disk_class_init,
			NULL,
			NULL, /* class_data */
			sizeof (DvdChapterDisk),
			0, /* n_preallocs */
			(GInstanceInitFunc) dvd_chapter_disk_instance_init
	    	};
		chapter_type = g_type_register_static (DVD_CHAPTER_TYPE,
						       "DvdChapterDisk",
						       &chapter_info, 0);
	}
	return chapter_type;
}

/**
 * Create a new dvd chapter disk object with #DvdSource source.
 * You would normally create new chapters and then add them to a #DvdTitle.
 * You must call g_object_unref() when you are done with the returned object.
 * @returns A new #DvdChapterDisk object.
 */
DvdChapterDisk
*dvd_chapter_disk_new		(void)
{
	DvdChapterDisk *chapter;
	
	chapter = g_object_new (dvd_chapter_disk_get_type (), NULL);
	
	return chapter;
}


/* imported by dvd_title_disk */
gboolean
dvd_chapter_disk_read_properties	(DvdChapterDisk	 *chapter,
					 guint8		  chapter_id,
					 dvd_reader_t	 *dvd_handle,
					 const gchar	 *device,
					 guint8		  title_id,
					 GError		**error)
{
	dvd_file_t *title_file;
	ifo_handle_t *vmg_file;
	tt_srpt_t *tt_srpt;
	ifo_handle_t *vts_file;
	vts_ptt_srpt_t *vts_ptt_srpt;
	pgc_t *cur_pgc;
	gint ttn, pgn;
	guint8 title_set;
	gint angle;
	gint pgc_id;
	guint8 start_cell, cur_cell, last_cell, next_cell;
	guint32 chapter_ts = 0;
	guint cell_id;
	
	angle = 0;

	/* Load the video manager to find out the information about */
	/* the titles on this disc. */
	vmg_file = ifoOpen (dvd_handle, 0);
	if (vmg_file == NULL) {
		g_warning ("Can't open VMG info");
		g_set_error (error,
			     DVD_ERROR,
			     DVD_ERROR_DISK_READ,
			     _("Unable to open the DVD video manager. "
			       "The disk may be dirty, damaged, or may not be a DVD video disk."));
		return FALSE;
	}
	tt_srpt = vmg_file->tt_srpt;
	
	/* Make sure our title number is valid.*/
	if (title_id >= tt_srpt->nr_of_srpts) {
		g_warning ("dvd-chapter-disk Invalid title %d.", title_id + 1);
		ifoClose (vmg_file);
		g_set_error (error,
			     DVD_ERROR,
			     DVD_ERROR_INVALID_CHAPTER,
			     _("The DVD does not contain title %d"), title_id + 1);
		return FALSE;
	}
	/* Make sure the chapter number is valid for this title.*/
	if (chapter_id >= tt_srpt->title[title_id].nr_of_ptts) {
		g_warning ("Invalid chapter %d", chapter_id + 1);
		ifoClose (vmg_file);
		g_set_error (error,
			     DVD_ERROR,
			     DVD_ERROR_INVALID_CHAPTER,
			     _("The title does not contain chapter %d"), chapter_id + 1);
		return FALSE;
	}
	/* Make sure the angle number is valid for this title.*/
	if (angle < 0 || angle >= tt_srpt->title[title_id].nr_of_angles) {
		g_warning ("Invalid angle %d", angle + 1);
		ifoClose (vmg_file);
		g_set_error (error,
			     DVD_ERROR,
			     DVD_ERROR_INVALID_ANGLE,
			     _("The title does not contain angle %d"), angle + 1);
		return FALSE;
	}
	
	/* Load the VTS information for the title set our title is in. */
	title_set = tt_srpt->title[title_id].title_set_nr;
	ttn = tt_srpt->title[title_id].vts_ttn;
	vts_file = ifoOpen (dvd_handle, title_set);
	if (vts_file == NULL) {
		g_warning ("Can't open the title %d info file.", title_set);
		ifoClose (vmg_file);
		g_set_error (error,
			     DVD_ERROR,
			     DVD_ERROR_DISK_OPEN,
			     _("Unable to open DVD title set information file for reading. "
			       "The disk may be dirty or damaged."));
		return FALSE;
	}
	/* Determine which program chain we want to watch.  This is based on the */
	/* chapter number. */
	vts_ptt_srpt = vts_file->vts_ptt_srpt;
	pgc_id = vts_ptt_srpt->title[ttn - 1].ptt[chapter_id].pgcn;
	pgn = vts_ptt_srpt->title[ttn - 1].ptt[chapter_id].pgn;
	cur_pgc = vts_file->vts_pgcit->pgci_srp[pgc_id - 1].pgc;
	
	if (cur_pgc->program_map[pgn - 1] < 1) {
		g_warning ("start cell < 0, using 0");
		start_cell = 0;
	} else {
		start_cell = cur_pgc->program_map[pgn - 1] - 1;
	}
	if (chapter_id + 1 == tt_srpt->title[title_id].nr_of_ptts) {
		last_cell = cur_pgc->nr_of_cells;
	} else {
		if (cur_pgc->program_map[(vts_ptt_srpt->title[ttn - 1].ptt[chapter_id + 1].pgn) - 1] < 1) {
			g_warning ("last cell < 0, using 0");
			last_cell = 0;
		} else {
			last_cell = cur_pgc->program_map[(vts_ptt_srpt->title[ttn - 1].ptt[chapter_id + 1].pgn) - 1] - 1;
		}
	}
	
	/* We've got enough info, time to open the title set data. */
	title_file = DVDOpenFile (dvd_handle, title_set, DVD_READ_TITLE_VOBS);
	if (title_file == NULL) {
		g_warning ("Can't open title VOBS (VTS_%02d_1.VOB).", title_set);
		ifoClose (vts_file);
		ifoClose (vmg_file);
		g_set_error (error,
			     DVD_ERROR,
			     DVD_ERROR_DISK_OPEN,
			     _("Unable to open title set for reading. "
			       "The disk may be dirty or damaged."));
		return FALSE;
	}
	
	/* Playback the cells for our chapter.*/
	/*g_message ("start cell %d last cell %d", start_cell, last_cell);*/
	cell_id = 0;
	next_cell = start_cell;
	while (next_cell < last_cell) {
		DvdCell *cell;
		
		cur_cell = next_cell;
		/* Check if we're entering an angle block. */
		if (cur_pgc->cell_playback[cur_cell].block_type == BLOCK_TYPE_ANGLE_BLOCK) {
			gint i = 0;

			cur_cell += angle;
			while (1) {
				if (cur_pgc->cell_playback[cur_cell + i].block_mode == BLOCK_MODE_LAST_CELL) {
					next_cell = cur_cell + i + 1;
					break;
				}
				++i;
			}
		} else {
			next_cell = cur_cell + 1;
		}
		cell = g_malloc0 (sizeof (DvdCell));
		cell->time = dvd_time_new_from_dvd_read_time (&cur_pgc->cell_playback[cur_cell].playback_time);
		cell->first_cell_block = cur_pgc->cell_playback[cur_cell].first_sector;
		cell->last_cell_block = cur_pgc->cell_playback[cur_cell].last_sector;
		cell->cell_id = cell_id;
		cell->blocks = cell->last_cell_block - cell->first_cell_block;
		chapter->cells = g_slist_append (chapter->cells, cell);
		chapter->cell_count++;
		chapter->blocks += cell->blocks;
		chapter_ts += cell->time->ts;
		cell_id++;
		/*g_message ("cell length    = %dh:%dm:%d.%02ds - %u ts",
			   cell->time->bk_hours,
			   cell->time->bk_minutes,
			   cell->time->bk_seconds,
			   cell->time->bk_milliseconds,
			   cell->time->ts);*/
	}
	ifoClose (vts_file);
	ifoClose (vmg_file);
	DVDCloseFile (title_file);
	
	DVD_CHAPTER (chapter)->title_set = title_set;
	if (title_set < 1) g_warning ("chapter is a menu");
	DVD_CHAPTER (chapter)->time = dvd_time_new_from_time_stamp (chapter_ts);
	DVD_CHAPTER (chapter)->kbs = chapter->blocks * 2048;
	chapter->device = g_strdup (device);
	
	/*g_message ("chapter length = %dh:%dm:%d.%02ds - %u ts",
		   DVD_CHAPTER (chapter)->time->bk_hours,
		   DVD_CHAPTER (chapter)->time->bk_minutes,
		   DVD_CHAPTER (chapter)->time->bk_seconds,
		   DVD_CHAPTER (chapter)->time->bk_milliseconds,
		   DVD_CHAPTER (chapter)->time->ts);*/
	
	return TRUE;
}

/* Reading */
static void
demux_output_data		(DvdDemux	*demux,
				 DvdStream	 stream_type,
				 gint		 track,
				 gint		 bytes,
				 guint8		*buffer,
				 guint32	 pts,
				 guint64	 frame_clocks,
				 gpointer	 user_data)
{
	GObject *chapter;
	
	chapter = G_OBJECT (user_data);

	g_signal_emit (chapter,
		       dvd_chapter_disk_signals[CHAPTER_OUTPUT_CELL], 0,
		       stream_type,
		       track,
		       bytes,
		       buffer,
		       pts,
		       frame_clocks);
}

gboolean
dvd_chapter_disk_read_cell	(DvdChapterDisk	 *chapter,
				 dvd_reader_t	 *dvd_handle,
				 guint		  cell_id,
				 GError		**error)
{
	dvd_file_t *title_file;
	DvdCell *cell = NULL;
	guint block;
	guint8 *buffer;
	gint len;
	DvdTime *time, *chapter_time;
	guint32 elapsed, ts, start_ts;
	GSList *iter;
	gboolean ok = TRUE;
	gboolean canned = FALSE;
	gulong sig;
	DvdPackHeader pack_hdr;
	DvdPESHeader stream_hdr;
	guint64	scr = 0;
	
	g_assert (chapter != NULL);
	g_assert (chapter->cell_count > 0);
	g_assert (chapter->cell_count > cell_id);
	
	title_file = DVDOpenFile (dvd_handle,
				  DVD_CHAPTER (chapter)->title_set,
				  DVD_READ_TITLE_VOBS);
	if (title_file == NULL) {
		g_warning ("disk opened OK, but can not open title file");
		g_set_error (error,
			     DVD_ERROR,
			     DVD_ERROR_DISK_READ,
			     _("Unable to open title set for reading. "
			       "The disk may be dirty or damaged."));
		return FALSE;
	}
	start_ts = 0;
	ts = 0;
	for (iter = chapter->cells;
	     iter != NULL;
	     iter = g_slist_next (iter)) {
		cell = iter->data;
		g_assert (cell != NULL);
		if (cell->cell_id == cell_id) {
			break;
		}
		start_ts += cell->time->ts;
	}
	sig = g_signal_connect (chapter->demux,
				"output_data",
				G_CALLBACK (demux_output_data),
				(gpointer) chapter);
	g_assert (cell != NULL);
	chapter_time = dvd_time_new_from_time_stamp (start_ts);
	elapsed = 0;
	time = g_malloc (sizeof (DvdTime));
	buffer = g_malloc (DVD_VIDEO_LB_LEN * 1024);
	block = cell->first_cell_block;
	while (block < cell->last_cell_block) {
		dsi_t dsi_pack;
		guint next_vobu;
		guint8 *stream;
		guint stream_id;
		guint8 *end;
		
		nav_retry:	  
		len = DVDReadBlocks (title_file, (int) block, 1, buffer);
		if (len != 1) {
			g_warning ("Nav read failed for block %d", block);
			g_free (buffer);
			g_set_error (error,
				     DVD_ERROR,
				     DVD_ERROR_DISK_READ,
				     _("Could not read next navigation block. "
				       "The disk may be dirty or damaged."));
			g_signal_handler_disconnect (chapter->demux, sig);
			return FALSE;
		}
		stream = buffer;
		end = buffer + len * 2048;
		if (dvd_pes_read_pack_header (&pack_hdr, &stream) == FALSE) {
			g_warning ("no packet");
			block++;
			goto nav_retry;
		}

		/* pack header is 14 bytes + 1 stuffing byte */
		/* system header is 1009 bytes - ends at byte 1023 */
		stream += 1009;
		stream_id = dvd_pes_find_header (&stream, end);
		if (stream_id == DVD_PES_START_CODE_NOT_FOUND) {
			g_warning ("not a nav packet 0x%x", stream_id);
			block++;
			goto nav_retry;
		}
		if (dvd_pes_read_stream_header (&stream_hdr, &stream) == FALSE) {
			g_warning ("can't read stream header");
			block++;
			goto nav_retry;
		}
		if (stream_hdr.stream_id != DVD_PES_START_CODE_PRIV_STREAM2) {
			g_warning ("not a nav packet 0x%x", stream_hdr.stream_id);
			block++;
			goto nav_retry;
		}
		if (scr > pack_hdr.scr) {
			g_warning ("pack scr moves backwards");
		}
		scr = pack_hdr.scr;
		/*g_message ("pack scr=%llu", (guint64) scr / 300);
		g_message ("mux_rate = %u bytes/sec", pack_hdr.mux_rate * 50);*/
		navRead_DSI (&dsi_pack, &(buffer[DSI_START_BYTE]));
		
		/* Is this for displaying still images??
		   ==============================================================================
		   
		   all uint32_t, up to 8 audio streams [0-7]
		   
		   PTM [start point - play time?] of audio gap in stream
		   first_gap = dsi_pack.sml_pbi.vob_a[audio_stream_nr].stp_ptm1;
		   second_gap = dsi_pack.sml_pbi.vob_a[audio_stream_nr].stp_ptm2;
		   
		   Duration, in 90KHz clocks, of audio gap in stream
		   first_delay = dsi_pack.sml_pbi.vob_a[audio_stream_nr].gap_len1;
		   second_delay = dsi_pack.sml_pbi.vob_a[audio_stream_nr].gap_len2;
		*/

		if (block != dsi_pack.dsi_gi.nv_pck_lbn) {
			next_vobu = block + dsi_pack.dsi_gi.vobu_ea + 1;
		} else {
			next_vobu = block + (dsi_pack.vobu_sri.next_vobu & 0x7fffffff);
		}
		ts = dvd_read_time_to_time_stamp (&dsi_pack.dsi_gi.c_eltm);
		if (ts > elapsed + 1000) {
			elapsed = ts;
			time =  dvd_time_set_from_time_stamp (time, elapsed);
			chapter_time =  dvd_time_set_from_time_stamp (chapter_time, start_ts + elapsed);
			g_signal_emit (G_OBJECT (chapter),
				       dvd_chapter_disk_signals[CHAPTER_TIME],
				       0,
	 			       time,
	 			       chapter_time);
		}
		block++;
		/* Read in and output cursize packs. */
		len = DVDReadBlocks (title_file, (gint) block, dsi_pack.dsi_gi.vobu_ea, buffer);
		/*g_message ("read %d blocks", len);*/
		if (len != (gint) dsi_pack.dsi_gi.vobu_ea) {
			g_warning ("Read failed for %d blocks at %d",
				   dsi_pack.dsi_gi.vobu_ea, block);
			g_free (buffer);
			g_free (time);
			g_free (chapter_time);
			g_set_error (error,
				     DVD_ERROR,
				     DVD_ERROR_DISK_READ,
				     _("Could not read DVD media. "
				       "The disk may be dirty or damaged."));
			g_signal_handler_disconnect (chapter->demux, sig);
			return FALSE;
		}
		dvd_demux_vob_blocks (chapter->demux, buffer, len);
		g_signal_emit (G_OBJECT (chapter),
				dvd_chapter_disk_signals[CHAPTER_OUTPUT_CELL],
				0,
	 		 	DVD_STREAM_VOB,
				0,
				len * 2048,
	 			buffer,
	 			0);
		block = next_vobu;
		
		/* check to see if we have been canned */
		g_mutex_lock (chapter->op_mutex);
		if (chapter->op_canned == TRUE) {
			/* Should this be an error or just return TRUE? */
			ok = FALSE;
			canned = TRUE;
			/* dvd_chapter_disk_stop_read () is now in a loop */
			/* waiting for op_reading to be false */
			chapter->op_reading = FALSE;
			g_mutex_unlock (chapter->op_mutex);
			g_set_error (error,
				     DVD_ERROR,
				     DVD_ERROR_CANCELLED,
				     _("The operation has been cancelled"));
			break;
		}
		g_mutex_unlock (chapter->op_mutex);
	}
	
	DVDCloseFile (title_file);
	g_free (buffer);
	g_free (time);
	chapter_time = dvd_time_set_from_time_stamp (chapter_time, start_ts + ts);
	
	g_signal_emit (G_OBJECT (chapter),
			dvd_chapter_disk_signals[CHAPTER_OUTPUT_CELL], 0,
	 	 	DVD_STREAM_VOB,
			0,			
			-1,
	 		NULL,
	 		0);
	g_signal_emit (G_OBJECT (chapter),
		       dvd_chapter_disk_signals[CHAPTER_TIME],
		       0,
		       cell->time,
		       chapter_time);
	g_free (chapter_time);
	g_signal_handler_disconnect (chapter->demux, sig);
	return ok;
}

/* Imported by dvd-title-disk.c Note chapter->demux must be valid */
gboolean
dvd_chapter_disk_read_chapter	(DvdChapterDisk	 *chapter,
				 dvd_reader_t	 *dvd_handle,
				 GError		**error)
{
	guint cell, cells;
	gboolean ok = TRUE;
	
	g_assert (chapter->demux != NULL);
	cells = dvd_chapter_disk_get_cell_count (chapter);
	for (cell = 0;
	     cell < cells;
	     cell++) {
		ok = dvd_chapter_disk_read_cell (chapter, dvd_handle, cell, error);
		if (ok == FALSE) {
			break;
		}
	}
	
	return ok;
}

/**
 * Gets number of cells in a #DvdChapter.
 * @param chapter A #DvdChapter.
 * @returns Number of cells.
 */
guint
dvd_chapter_disk_get_cell_count	(DvdChapterDisk	*chapter)
{
	g_assert (chapter != NULL);
	
	return chapter->cell_count;
}

/**
 * Gets a constant pointer to the play time of the specified cell.
 * Cell id's start from 0.
 * @param chapter A #DvdChapter.
 * @param cell_id The id of the cell you want the time of.
 * @returns A constant #DvdTime.
 */
G_CONST_RETURN DvdTime
*dvd_chapter_disk_get_cell_time	(DvdChapterDisk	*chapter,
				 guint		 cell_id)
{
	DvdCell *cell;
	
	g_assert (chapter != NULL);
	g_assert (chapter->cell_count > 0);
	g_assert (chapter->cell_count > cell_id);
	
	cell = (DvdCell *) g_slist_nth_data (chapter->cells, cell_id);
	g_assert (cell != NULL);
	
	return cell->time;
}

/**
 * Reads the chapter data from disk and emits data through the #DvdChapterClass::output_cell signal handler.
 * Also outputs time information through the #DvdChapterClass::chapter_time signal handler.
 * It is best to run this in a separate thread to allow the gui to be updated.
 * You will need to use the timestamp in the output to determine the audio delay
 * when multiplexing/playing the output video and audio streams.
 * Audio packets may span accross chapters and those packets
 * will be discarded because the demuxer discards data untill the first packet header.
 * Therefore, if you intend to stream more than one chapter, read at the title level
 * which will stream the specified chapters without any data loss between those chapters.
 * @param chapter A #DvdChapter.
 * @param error The return location for an allocated GError, or NULL to ignore errors.
 * @returns TRUE if successful, FALSE on fail.
 */
gboolean
dvd_chapter_disk_read		(DvdChapterDisk	 *chapter,
				 GError		**error)
{
	gboolean ok = FALSE;
	dvd_reader_t *dvd_handle;
	
	g_assert (chapter != NULL);
	g_assert (chapter->device != NULL);
	
	g_mutex_lock (chapter->op_mutex);
	if (chapter->op_canned == TRUE) {
		chapter->op_reading = FALSE;
		g_mutex_unlock (chapter->op_mutex);
		g_set_error (error,
			     DVD_ERROR,
			     DVD_ERROR_CANCELLED,
			     _("The operation has been cancelled"));
		return FALSE;
	}
	chapter->op_reading = TRUE;
	g_mutex_unlock (chapter->op_mutex);
	chapter->demux = dvd_demux_new ();
	dvd_handle = DVDOpen (chapter->device);
	if (dvd_handle == NULL) {
		g_warning ("Can't open disc %s!", chapter->device);
		g_set_error (error,
			     DVD_ERROR,
			     DVD_ERROR_DISK_OPEN,
			     _("There may not be a disk in the drive or the disk is unreadable"));
		return FALSE;
	}
	ok = dvd_chapter_disk_read_chapter (chapter, dvd_handle, error);
	DVDClose (dvd_handle);
	g_object_unref (G_OBJECT (chapter->demux));
	chapter->demux = NULL;
	g_mutex_lock (chapter->op_mutex);
	chapter->op_reading = FALSE;
	g_mutex_unlock (chapter->op_mutex);
	
	return ok;
}


/**
 * Stops the current read process invoked by #dvd_chapter_disk_read ().
 * The read emits signals that may update your gui and will be 
 * running in a separate thread, so be sure to leave gdk threads first!
 * @param chapter A #DvdChapterDisk
 * @returns void.
 */
void
dvd_chapter_disk_stop_read	(DvdChapterDisk	*chapter)
{
	gboolean stopped;
	
	g_assert (chapter != NULL);
	
	g_mutex_lock (chapter->op_mutex);
	if (chapter->op_reading == FALSE) {
		g_warning ("dvd_chapter_disk_stop_read called and chapter is not reading");
		g_mutex_unlock (chapter->op_mutex);
		return;
	}
	g_assert (chapter->op_canned != TRUE);
	chapter->op_canned = TRUE;
	g_mutex_unlock (chapter->op_mutex);
	
	/* wait untill stopped */
	/*g_message ("chapter waiting");*/
	for (stopped = FALSE;
	     stopped == FALSE;
	     g_usleep (100)) {
		g_mutex_lock (chapter->op_mutex);
		if (chapter->op_reading == FALSE) {
			stopped = TRUE;
			chapter->op_canned = FALSE;
		}
		g_mutex_unlock (chapter->op_mutex);
	}
	/*g_message ("chapter stopped");*/
}
