/*
 *  Part of the shrinkta program, a dvd backup tool
 *
 *  Copyright (C) 2005  Daryl Gray
 *  E-Mail Daryl Gray darylgray1@dodo.com.au
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
*/

#if !defined (__DVD_H_INSIDE__) && !defined (LIBDVD_COMPILATION)
#error "Only <dvd.h> can be included directly."
#endif

#ifndef __PES_H__
#define __PES_H__


G_BEGIN_DECLS

/******************************************************
DVD block contains:
A PackHeader then,
1 or 2 packets of similar stream including
a) PESHeader
b) data depending on PES type

The data
-----------------------------------
Nav pack data includes 2 packets
1) a PCI packet
2) a DSI packet

Private stream 1 includes
1) DvdPS1Header
2) stream data

First MPEG2 video block includes
1) Sequence header, ext headers etc
2) GOP header
3) Picture header
4) Picture frame, etc

Next MPEG2 video blocks include
1) frames may span over several blocks, padded to end of a block
2) Sequence header, ext headers etc
3) Picture header
4) Picture frame, etc

work in progress...

 NOTE: some headers may be incomplete.

*******************************************************/

typedef enum _DvdPesStartCode {
	/* video start codes */
	DVD_PES_START_CODE_PICTURE	= 0x00,
	DVD_PES_START_CODE_SLICE	= 0x01, /* 176x 0x01 to 0xaf */
	DVD_PES_START_CODE_USER_DATA	= 0xb2,
	DVD_PES_START_CODE_SEQUENCE_HDR	= 0xb3,
	DVD_PES_START_CODE_SEQUENCE_ERR	= 0xb4,
	DVD_PES_START_CODE_EXTENSION	= 0xb5,
	DVD_PES_START_CODE_SEQUENCE_END	= 0xb7,
	DVD_PES_START_CODE_GROUP_OF_PIC	= 0xb8,
	/* system codes - from 0xb9 to 0xff */
	DVD_PES_START_CODE_STREAM_END	= 0xb9,
	DVD_PES_START_CODE_PACK_HDR	= 0xba,
	DVD_PES_START_CODE_SYSTEM_HDR	= 0xbb,
	DVD_PES_START_CODE_PS_MAP	= 0xbc,
	DVD_PES_START_CODE_PRIV_STREAM1	= 0xbd,
	DVD_PES_START_CODE_PADDING	= 0xbe,
	DVD_PES_START_CODE_PRIV_STREAM2	= 0xbf,
	DVD_PES_START_CODE_MPEG_AUDIO	= 0xc0, /* 32x 0xc0 to 0xdf */
	DVD_PES_START_CODE_MPEG_VIDEO	= 0xe0, /* 16x 0xe0 to 0xef */
	DVD_PES_START_CODE_ECM		= 0xf0,
	DVD_PES_START_CODE_EMM		= 0xf1,
	/* 0xf2 ITU-T Rec. H.222.0 | ISO/IEC 13818-1 Annex A or ISO/IEC 13818-6_DSMCC_stream */
	/* 0xf3 ISO/IEC_13522_stream */
	/* 0xf4 - 0xf9 ITU-T Rec. H.222.1 type A,B,C,D,E */
	DVD_PES_START_CODE_ANCILLARY	= 0xf9,
	/* 0xfa - 0xfe reserved */
	DVD_PES_START_CODE_PS_DIR	= 0xff,
	/* any other value is reserved */
	DVD_PES_START_CODE_NOT_FOUND	= 0xfe	/* use last reserved value */
} DvdPesStartCode;

typedef struct _DvdPackHeader {
	/* 0x000001 */
	/* DVD_PES_START_CODE_PACK_HDR */
	guint64	 scr		: 42;	/* Stream Clock Reference (2700kHz) */
	guint32	 mux_rate	: 22;	/* Rate at which the program stream target decoder */
					/* receives the Program Stream during the pack in which */
					/* it is included - measured in units of 50 bytes/second */
} DvdPackHeader;

/* Packetized Elementary Stream (PES) Header */
typedef struct _DvdPESHeader {
	/* 0x000001 */
	guint8	stream_id;		/* DvdPesStartCode various system codes */
	guint16	packet_length;
	
	/* extension flags - byte 7 */
	guint	reserved0	: 2;	/* just stuffing?? 01 */
	guint	scrambled	: 2;	/* 00 - not scrambled */
	guint	priority	: 1;
	guint	alignmentp	: 1;	/* 1 - PES packet header is immediately followed by the video start code or audio syncword */
	guint	copyb		: 1;
	guint	original	: 1;
	
	/* extension flags - byte 8 */
	guint	pts_dts		: 2;
	guint	escre		: 1;
	guint	es_ratee	: 1;
	guint	dsm_trick_mode	: 1;
	guint	addicopyb	: 1;
	guint	crce		: 1;	/* crc exists */
	guint	exte		: 1;	/* header extension exists */
	
	/* extension  - byte 9 */
	guint8	hdr_dlength;		/* header data length  including stuffing bytes (0xff) */
	
	/* extension data */
	guint32	pts;			/* (90kHz) if pts_dts is 10 or 11 */
	guint32	dts;			/* (90kHz) if pts_dts is 11 */
	guint64	escr		: 42;	/* Elementary Stream Clock Reference (2700kHz) if escre is 1 */
	guint32	es_rate		: 22;	/* stream data rate in units of 50 bytes/second if es_ratee = 1 */
	/* DSM trick not used by DVD */
	guint8	addicopy_info;		/* if addicopyb = 1 */
	guint16	prev_crc;		/* previous packet crc if crce = 1 */
	
	/* if exte = 1 */
	guint	priv_data	: 1;	/* PES private data exists */
	guint	pack_hdr	: 1;	/* pack header field exists */
	guint	pps_counter	: 1;	/* program packet sequence counter exists */
	guint	pstd_buf	: 1;	/* PES private data exists */
	/* 3 unused bits (111) */
	guint	pes_ext2	: 1;	/* PES private data exists */
	
	/* 16 bytes of user defined data if priv_data = 1 */
	guint8	pack_length;		/* if pack_hdr = 1 */
	guint8	pps;			/* if pps_counter = 1 */
	/* a single bit (1) */
	guint	mpeg_id		: 1;	/* mpeg1 or mpeg2 (1 or 0) - if pps_counter = 1 */
	guint	orig_stuffing	: 6;	/* original stuffing length */
	
	/* if pstd_buf = 1 */
	guint8	pstd_buf_scale	: 1;	/* if 1 multiply pstd_buf_size by 1024 else by 128 */
	guint	pstd_buf_size	: 13;	/* size of P-STD buffer - also see above */
	/* 2 byte pstd buffer if pstd_buf = 1 */
	/* pes extension 2 data if pes_ext2 = 1 */
	/* maybe some 0xff stuffing bytes */
} DvdPESHeader;

/* following headers after pes header */
/* private stream 1 header directly follows pes header - no start code */
typedef struct _DvdPS1Header {
	DvdStream  stream_type;
	guint8	   track;	/* audio track number / sub picture stream number */
	guint8	   frames;	/* number of frame headers  in data block */
	guint16	   pts_offset;	/* byte offset (after this header) to frame that coincides with PTS in stream header*/
} DvdPS1Header;

/* A 3 byte DvdLPCMHeader header follows a DvdPS1Header */
/* LPCM is usually 16bit x 48kHz, 320 bytes per frame or 24bit x 96 kHz, 960 bytes per frame */
/* LPCM frames are 150 clock ticks (90KHz) or 600 fps.  */
/* Bytes per frame = 48000 * 16 * 2 channels / 600 (fps) / 8 (bits) */
typedef struct _DvdLpcmHeader {
	guint	emphasis  : 1; /* audio_emphasis_flag [0 = off, 1 = on] */
	guint	mute	  : 1; /* audio_mute_flag [0 = off, 1 = on] */
	guint	reserved1 : 1;
	guint	frame	  : 5; /* Frame number within Group of Audio frames */
	/* byte 2 */
	guint	word_size : 2; /* quantisation_word_length [0 = 16, 1 = 20, 2 = 24, 3 = reserved	Bits per sample] */
	guint	samples	  : 2; /* audio_sample_frequency	[0 = 48, 1 = 96, other = reserved Sampling frequency in kHz] */
 	guint	reserved2 : 1;
 	guint	channels  : 3; /* number_of_audio_channels [0 = 1ch, 1 = 2ch etc] */
 	/* byte 3 */
	guint	dr_x	  : 3; /* dynamic_range X	High 3 bits of byte */
 	guint	dr_y	  : 5; /* dynamic_range Y	Lower 5 bits of byte */
 	/* not sure about the following... */
 	/* linear gain = 2 ^ (4 - ((float) dr_x + ((float) dr_y / 30))) */
 	/* (float) in dB gain = 24.082 - (6.0206 * (float) dr_x) - (0.2007 * (float) dr_y) */
} DvdLpcmHeader;

/* AC3 and DTS have their own headers */

gboolean	dvd_pes_check_start_code	(guint8			**stream);
guint8		dvd_pes_find_header		(guint8			**stream,
						 guint8			 *end);
gboolean	dvd_pes_read_pack_header	(DvdPackHeader		 *header,
						 guint8			**stream);
gboolean	dvd_pes_read_stream_header	(DvdPESHeader		 *header,
						 guint8			**stream);
gboolean	dvd_pes_read_ps1_header		(DvdPS1Header		 *header,
						 guint8			**stream);
void		dvd_pes_read_lpcm_header	(DvdLpcmHeader		 *header,
						 guint8			**stream);
/* no better place for this yet */
guint32		getbits				(guint8			**stream,
						 guint8			  bits,
						 guint8			 *cur_bit);

G_END_DECLS

#endif /*__PES_H__*/
