/**
 * HTTP/HTTPS protocol support 
 *
 * Copyright (C) 2000, 2001, 2002 by
 * Jeffrey Fulmer - <jdfulmer@armstrong.com>
 * This file is distributed as part of Siege 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <setup.h>
#include <http.h>
#include <stdio.h>
#include <stdarg.h>
#include <cookie.h>
#include <string.h>
#include <joedog/defs.h>

#define MAXFILE 10240
#define REQBUF  40960 

private char *__parse_pair(char **str);

/**
 * HTTPS tunnel; set up a secure tunnel with the
 * proxy server. CONNECT server:port HTTP/1.0
 */
BOOLEAN
https_tunnel_request(CONN *C, char *host, int port)
{
  size_t  rlen, n;
  char    request[256];

  if(C->prot == HTTPS && my.proxy.required){
    snprintf(
      request, sizeof(request),
      "CONNECT %s:%d HTTP/1.0\015\012"
      "User-agent: Proxy-User\015\012"
      "\015\012",
      host, port
    );    
    rlen = strlen(request); 
    if(my.debug){fprintf(stdout, "%s", request); fflush(stdout);}
    if((n = socket_write(C, request, rlen)) != rlen){
      joe_error( "socket_write: ERROR" );
      return FALSE;
    }
  } else {
    return FALSE; 
  }
  return TRUE;
}

int
https_tunnel_response(CONN *C)
{
  int  x, n;
  char c;
  char line[256];
  int  code = 100;

  while(TRUE){
    x = 0;
    memset( &line, 0, sizeof( line ));
    while((n = read(C->sock, &c, 1)) == 1){
      line[x] = c;
      if( my.debug ){ printf("%c", c ); fflush( stdout ); }
      if(( line[0] == '\n' ) || ( line[1] == '\n' )){
        return code;
      }
      if( line[x] == '\n' ) break;
      x ++;
    }
    line[x]=0;
    if( strncasecmp( line, "http", 4 ) == 0 ){
      code = atoi(line + 9);
    }
  }
}

/**
 * returns int, ( < 0 == error )
 * formats and sends an HTTP/1.0 request
 */
int
http_get(CONN *C, char *host, int port, char *path)
{
  int  rlen;
  char *protocol; 
  char *keepalive;
  char cookie[MAX_COOKIE_SIZE];
  char authwww[128];
  char authpxy[128];
  char request[REQBUF]; 

  /* Request path based on proxy settings */
  char fullpath[4096];
  if( my.proxy.required ){
    sprintf(
      fullpath, "%s://%s:%d%s", C->prot == 0?"http":"https", host, port, path 
    );
  } 
  else{
    sprintf( fullpath, "%s", path );
  }

  memset( cookie,  0, sizeof( cookie ));
  memset( request, 0, sizeof( request ));

  /* HTTP protocol string */
  protocol  = (my.protocol == TRUE)?"HTTP/1.1":"HTTP/1.0";
  keepalive = (C->connection.keepalive == TRUE)?"keep-alive":"close";
  get_cookie( (int)pthread_self(), host, cookie ); 
  if( C->auth.www ){
    rlen = snprintf(
      authwww, sizeof(authwww), 
      "Authorization: %s %s\015\012", 
      (C->auth.type.www==BASIC)?"Basic":"Digest", my.auth.encode
    );
  }
  if( C->auth.proxy ){
    rlen = snprintf(
      authpxy, sizeof(authpxy), 
      "Proxy-Authorization: %s %s\015\012", 
      (C->auth.type.www==BASIC)?"Basic":"Digest", my.proxy.encode
    );
  }

  /** 
   * build a request string to pass to the server       
   */
  rlen = snprintf(
    request, sizeof( request ),
    "GET %s %s\015\012"
    "Host: %s:%d\015\012" 
    "%s"
    "%s"
    "Cookie: %s\015\012"
    "Accept: */*\015\012"
    "Accept-Encoding: * \015\012"
    "User-Agent: %s\015\012%s"
    "Connection: %s\015\012\015\012",  
    fullpath, protocol, host, port,
    (C->auth.www==TRUE)?authwww:"",
    (C->auth.proxy==TRUE)?authpxy:"",
    cookie, my.uagent, my.extra, keepalive 
  );
  
  if( my.debug ){ printf("%s\n", request); fflush(stdout); }
  if( rlen < 0 || rlen > (int)sizeof(request) ) 
    joe_fatal("http_get: request buffer overrun!");

  if((socket_check(C, WRITE)) == FALSE){
    return -1;
  } 

  if((socket_write( C, request, rlen )) < 0){
    return -1;
  }
  
  return 0;
}

/**
 * returns int, ( < 0 == error )
 * formats and sends an HTTP/1.0 request
 */
int
http_post( CONN *C, char *host, int port, char *path, char *data, size_t len )
{
  int  rlen;
  char authwww[128];
  char authpxy[128]; 
  char request[REQBUF]; 
  char *protocol; 
  char *keepalive;

  /* cookie value  */
  char cookie[MAX_COOKIE_SIZE];

  /* Request path based on proxy settings */
  char fullpath[4096];
  if( my.proxy.required ){
    sprintf( fullpath, "%s://%s:%d%s", C->prot == 0?"http":"https", host, port, path );
  }
  else{
    sprintf( fullpath, "%s", path );
  }

  memset( cookie,  0, sizeof( cookie ));
  memset( request, 0, sizeof( request ));

  /* HTTP protocol string */
  protocol  = (my.protocol  == TRUE)?"HTTP/1.1":"HTTP/1.0";
  keepalive = (C->connection.keepalive == TRUE)?"keep-alive":"close";
  get_cookie( (int)pthread_self(), host, cookie );
  if( C->auth.www ){
    rlen = snprintf(
      authwww, sizeof(authwww),
      "Authorization: %s %s\015\012",
      (C->auth.type.www==BASIC)?"Basic":"Digest", my.auth.encode
    );
  }
  if( C->auth.proxy ){
    rlen = snprintf(
      authpxy, sizeof(authpxy),
      "Proxy-Authorization: %s %s\015\012",
      (C->auth.type.www==BASIC)?"Basic":"Digest", my.proxy.encode
    );
  } 
    
  /* build a request string to
     pass to the server       */
  rlen = snprintf(
    request, sizeof( request ),
    "POST %s %s\015\012"
    "Host: %s:%d\015\012"
    "%s"
    "%s"
    "Cookie: %s\015\012"
    "Accept: */*\015\012"
    "Accept-Encoding: * \015\012"
    "User-Agent: %s\015\012%s"
    "Connection: %s\015\012"
    "Content-type: application/x-www-form-urlencoded\015\012"
    "Content-length: %d\015\012\015\012",
    fullpath, protocol, host, port,
    (C->auth.www==TRUE)?authwww:"",
    (C->auth.proxy==TRUE)?authpxy:"",
    cookie, my.uagent, my.extra, keepalive, len
  ); 

  if( rlen + len < sizeof(request)){
    memcpy(request + rlen, data, len);
  }
  rlen += len;
  
  if( my.debug ){ printf("%s\n", request); fflush(stdout); }
  if( rlen<0 || rlen>(int)sizeof(request) )
    joe_fatal("http_post: request buffer overrun!"); 

  if((socket_check(C, WRITE)) == FALSE){
    return -1;
  } 

  if((socket_write( C, request, rlen )) < 0){
    return -1;
  }

  return 0;
}

/**
 * returns HEADERS struct
 * reads from http/https socket and parses
 * header information into the struct.
 */
HEADERS *
http_read_headers(CONN *C, char *host)
{ 
  int  x;           /* while loop index      */
  int  n;           /* assign socket_read    */
  char c;           /* assign char read      */
  char line[1536];  /* assign chars read     */
  HEADERS *h;       /* struct to hold it all */
  /*h = xmalloc(sizeof(HEADERS));*/
  h = xcalloc(sizeof(HEADERS), 1);
  
  if((socket_check(C, READ)) == FALSE){
    return NULL;
  } 

  h->redirection[0]=0;

  while( TRUE ){
    x = 0;
    memset( &line, 0, sizeof( line ));
    while(( n = socket_read( C, &c, 1 )) == 1 ){
      line[x] = c; 
      if( my.debug ){ printf("%c", c ); fflush( stdout ); }
      if(( line[0] == '\n' ) || ( line[1] == '\n' )){ 
        return h;
      }
      if( line[x] == '\n' ) break;
      x ++;
    }
    line[x]=0;
    /* strip trailing CR */
    if (x > 0 && line[x-1] == '\r') line[x-1]=0;
    if( strncasecmp( line, "http", 4 ) == 0 ){
      strncpy( h->head, line, 8 );
      h->code = atoi( line + 9 ); 
    }
    if( strncasecmp( line, "content-length: ", 16 ) == 0 ){ 
      C->content.length = atoi( line + 16 ); 
    }
    if( strncasecmp( line, "set-cookie: ", 12 ) == 0 ){
      if( my.cookies ){
        memset( h->cookie, 0, sizeof( h->cookie ));
        strncpy( h->cookie, line+12, strlen( line ));
        add_cookie( (int)pthread_self(), host, h->cookie );
      }
    }
    if( strncasecmp( line, "connection: ", 12 ) == 0 ){
      if( strncasecmp( line+12, "keep-alive", 10 ) == 0 ){
        h->keepalive = 1;
      }
      else if( strncasecmp( line+12, "close", 5 ) == 0 ){
        h->keepalive = 0;
      }
    }
    if( strncasecmp( line, "keep-alive: ", 12 ) == 0 ){
      char *tmp    = "";
      char *option = "", *value = "";
      char *newline = (char*)line;
      while(( tmp = __parse_pair( &newline )) != NULL ){
        option = tmp;
        while( *tmp && !ISSPACE( (int)*tmp ) && !ISSEPARATOR( *tmp ))
          tmp++;
        *tmp++=0;
        while( ISSPACE( (int)*tmp ) || ISSEPARATOR( *tmp ))
          tmp++;
        value  = tmp;
        while( *tmp )
          tmp++;  
        if( !strncasecmp( option, "timeout", 7 )){
          C->connection.timeout = atoi( value );
        }
        if( !strncasecmp( option, "max", 3 )){
          C->connection.max     = atoi( value );
        }
      }
    }
    if( strncasecmp(line, "location: ", 10) == 0) {
      if(strlen(line) - 10 > sizeof(h->redirection) - 1) {
	joe_warning( "redirection URL too long, ignored");
      }
      else {
        strcpy(h->redirection, line+10);
      }  
    }
    if( strncasecmp(line, "www-authenticate: ", 18 ) == 0 ){
      char *tmp;
      h->auth.www = TRUE;
      if( strncasecmp( line+19, "digest", 6 ) == 0 ){
        h->auth.type.www = DIGEST;
      } 
      else{
        h->auth.type.www = BASIC;
      }
      tmp = strchr( line, '=' );
      tmp++;
      if( tmp[0] == '"' ){ tmp++; tmp[strlen(tmp)-1] = '\0'; }
      strncpy( h->auth.realm.www, tmp, strlen( tmp )); 
    }
    if( strncasecmp(line, "proxy-authenticate: ", 20 ) == 0 ){
      char *tmp;
      h->auth.proxy = TRUE;
      if( strncasecmp( line+21, "digest", 6 ) == 0 ){
        h->auth.type.proxy = DIGEST;
      }
      else{
        h->auth.type.proxy = BASIC;
      }   
      tmp = strchr( line, '=' );
      tmp++;
      if( tmp[0] == '"' ){ tmp++; tmp[strlen(tmp)-1] = '\0'; }
      strncpy( h->auth.realm.proxy, tmp, strlen( tmp )); 
    }                      
    if( strncasecmp(line, "transfer-encoding: ", 19 ) == 0 ){
      if( strncasecmp( line+20, "chunked", 7 )){
        C->content.transfer = CHUNKED; 
      } 
      else if( strncasecmp( line+20, "trailer", 7 )){
        C->content.transfer = TRAILER; 
      } 
      else{
        C->content.transfer = NONE;
      }
    }
    if( n <  0 ){ 
      return( NULL ); 
    } /* socket closed */
  } /* end of while TRUE */

  return h;
}

int
http_chunk_size(CONN *C)
{
  int    n;
  char   line[256];
  char   *end;
  size_t length;

  memset( line, 0, sizeof( line ));
  if(( n = socket_readline( C, line, sizeof( line ))) < 1 ){
    joe_error("readline: unable to determine chunk size");
    return -1;
  }
  if((( line[0] == '\n' )||(strlen(line)==0)||(line[0] == '\r'))){
    return -1;
  }
 
  errno  = 0;
  if(!isxdigit(*line))
    return -1;
  length = strtoul(line, &end, 16);
  if(( errno == ERANGE ) || ( end == line )){
    joe_error( "bad chunk line %s\n", line );
    return 0;
  } else {
    return length;
  }
  return -1;
}
  
/**
 * returns int
 * reads a http/https socket
 * ( you know what I mean :)
 */
ssize_t
http_read( CONN *C )
{ 
  int    n = 0;
  int    chunk  = 0;
  size_t bytes  = 0;
  size_t length = 0;
  static char body[MAXFILE];

  if( C == NULL )
    joe_fatal("C is NULL!\n"); 

  if( C->content.length > 0 ){
    length = (C->content.length < MAXFILE)?C->content.length:MAXFILE;
    do {
      memset(body, 0, sizeof(body));
      if(( n = socket_read(C, body, length)) == 0 )
        break;
      bytes += n;
      length = (C->content.length - bytes < MAXFILE)?C->content.length-bytes:MAXFILE;
    } while( bytes <= C->content.length ); 
  }
  else if( C->content.transfer == CHUNKED ){
    int tries = 0;
    while(tries < 4) {
      chunk = http_chunk_size(C);
      if(chunk == 0)
        break;
      else if(chunk < 0) {
        tries ++;
        continue;
      }
      do {
        int n;
        memset(body, 0, MAXFILE);
        n = socket_read(C, body, (chunk>MAXFILE)?MAXFILE:chunk);
        chunk -= n;
        bytes += n;
      } while(chunk > 0);
    }
  }
  else{
    do {
      memset(body, 0, sizeof(body));
      if(( n = socket_read(C, body, sizeof(body))) == 0 )
        break;
      bytes += n;
    } while(TRUE);
  }

  return(bytes);
}


/**
 * parses option=value pairs from an
 * http header, see keep-alive: above
 * while(( tmp = __parse_pair( &newline )) != NULL ){
 *   do_something( tmp );
 * }
 */
private char *
__parse_pair(char **str)
{
  int  okay  = 0;
  char *p    = *str;
  char *pair = NULL;
 
  if( !str || !*str ) return NULL;
  /**
   * strip the header label
   */
  while( *p && *p != ' ' )
    p++;
  *p++=0;
  if( !*p ){
    *str   = p;
    return NULL;
  }
 
  pair = p;
  while( *p && *p != ';' && *p != ',' ){
    if( !*p ){
      *str = p;
      return NULL;
    }
    if( *p == '=' ) okay = 1;
    p++;
  }
  *p++ = 0;
  *str = p;
 
  if( okay )
    return pair;
  else
    return NULL;
} 
