/*
 * This file is part of Siril, an astronomy image processor.
 * Copyright (C) 2005-2011 Francois Meyer (dulle at free.fr)
 * Copyright (C) 2012-2014 team free-astro (see more in AUTHORS file)
 * Reference site is http://free-astro.vinvin.tf/index.php/Siril
 *
 * Siril is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Siril is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Siril. If not, see <http://www.gnu.org/licenses/>.
*/

#include <gsl/gsl_histogram.h>
#include <assert.h>
#include <string.h>
#include "siril.h"
#include <stdint.h>
#include "proto.h"
#include "histogram.h"
#include "callbacks.h"	// for lookup_widget()

/* The gsl_histogram, documented here:
 * http://linux.math.tifr.res.in/manuals/html/gsl-ref-html/gsl-ref_21.html
 * is able to group values into bins, it does not need to handle all values
 * separately. That is useful for display purposes, but not currently used.
 */

// margin between axis and drawing area border
//#define AXIS_MARGIN 10

// colors of layers histograms		R	G	B	RGB
static double histo_color_r[] = { 	1.0,	0.0,	0.0,	0.0	};
static double histo_color_g[] = { 	0.0,	1.0,	0.0,	0.0	};
static double histo_color_b[] = { 	0.0,	0.0,	1.0,	0.0	};
static double graph_height = 0.;
static uint32_t clipped[] = {0, 0};

static GtkToggleButton *toggles[MAXVPORT] = { NULL };

gsl_histogram* computeHisto_Selection(fits* fit, int layer, rectangle *selection){
	WORD *from;
	size_t size = (size_t)get_normalized_value(fit)+1;
	gsl_histogram* histo = gsl_histogram_alloc(size);
	int stridefrom, i, j;

	gsl_histogram_set_ranges_uniform (histo, 0, size);
	from = fit->pdata[layer] + (fit->ry - selection->y - selection->h) * fit->rx + selection->x;
	stridefrom = fit->rx - selection->w;
	for (i=0; i<selection->h; i++){
		for (j=0; j<selection->w; j++){
			gsl_histogram_increment (histo, (double)*from);
			from++;
		}
		from += stridefrom;
	}
	return histo;
}

gsl_histogram* computeHisto(fits* fit, int layer){
	unsigned int i, ndata;
	WORD *buf;
	if (layer >= 3)
		return NULL;
	size_t size = (size_t)get_normalized_value(fit)+1;
	gsl_histogram* histo = gsl_histogram_alloc(size);
	gsl_histogram_set_ranges_uniform (histo, 0, size);

	buf = fit->pdata[layer];
	ndata = fit->rx * fit->ry;
	for (i = 0; i < ndata; i++){
		gsl_histogram_increment (histo, (double)buf[i]);
	}
	return histo;
}

//compute histogram for all pixel values below backgroud value
gsl_histogram* histo_bg(fits* fit, int layer, double bg){
	unsigned int i, ndata;
	WORD *buf;
	if (layer >= 3)
		return NULL;
	size_t size = (size_t)bg + 1;
	gsl_histogram* histo = gsl_histogram_alloc(size);
	gsl_histogram_set_ranges_uniform (histo, 0, size);
	
	buf = fit->pdata[layer];
	ndata = fit->rx * fit->ry;
	for (i = 0; i < ndata; i++){
		if (buf[i] <= (WORD)bg) gsl_histogram_increment (histo, (double)buf[i]);
	}
	return histo;	
}

void compute_and_update_histo_for_fit(int force, fits *fit) {
	static GtkWidget *drawarea = NULL, *selarea = NULL;
	int nb_layers = 3;
	int i;
	if (fit->naxis == 2)
		nb_layers = 1;
	for (i = 0; i < nb_layers; i++) {
		if (force || !com.layers_hist[i])
			set_histogram(computeHisto(fit, i), i);
	}
	set_histo_toggles_names();
	if (!drawarea) {
		drawarea = lookup_widget("drawingarea_histograms");
		selarea = lookup_widget("drawingarea_histograms_selection");
	}
	gtk_widget_queue_draw(drawarea);
	gtk_widget_queue_draw(selarea);
}

void compute_histo_for_gfit(int force) {
	int nb_layers = 3;
	int i;
	if (gfit.naxis == 2)
		nb_layers = 1;
//~ #pragma omp parallel for private(i) schedule(dynamic,1)		// causes crashes ...
	for (i = 0; i < nb_layers; i++) {
		if (force || !com.layers_hist[i])
			set_histogram(computeHisto(&gfit, i), i);
	}
	set_histo_toggles_names();
}

void update_gfit_histogram_if_needed() {
	static GtkWidget *drawarea = NULL, *selarea = NULL;
	if (is_histogram_visible())
		compute_histo_for_gfit(1);
	if (!drawarea) {
		drawarea = lookup_widget("drawingarea_histograms");
		selarea = lookup_widget("drawingarea_histograms_selection");
	}
	gtk_widget_queue_draw(drawarea);
	gtk_widget_queue_draw(selarea);
}

void _histo_on_selection_changed() {
	static GtkWidget *selarea = NULL;
	if (!selarea)
		selarea = lookup_widget("drawingarea_histograms_selection");
	gtk_widget_queue_draw(selarea);
}

void set_histogram(gsl_histogram *histo, int layer) {
	assert(layer >= 0 && layer < MAXVPORT);
	if (com.layers_hist[layer])
		gsl_histogram_free(com.layers_hist[layer]);
	com.layers_hist[layer] = histo;
}

void clear_histograms() {
	int i;
	for (i = 0; i < MAXVPORT; i++)
		set_histogram(NULL, i);
}

void init_toggles() {
	if (!toggles[0]) {
		toggles[0] = GTK_TOGGLE_BUTTON(gtk_builder_get_object(builder, "check_histo_r"));
		toggles[1] = GTK_TOGGLE_BUTTON(gtk_builder_get_object(builder, "check_histo_g"));
		toggles[2] = GTK_TOGGLE_BUTTON(gtk_builder_get_object(builder, "check_histo_b"));
		toggles[3] = NULL;//GTK_TOGGLE_BUTTON(gtk_builder_get_object(builder, "check_histo_rgb"));
	}
}

// sets the channel names of the toggle buttons in the histogram window, based on
// the number of layers of gfit
void set_histo_toggles_names() {
	init_toggles();
	if (gfit.naxis == 2) {
		const char* test = gtk_button_get_label(GTK_BUTTON(toggles[0]));
		if (strcmp(test, "gray"))
			gtk_button_set_label(GTK_BUTTON(toggles[0]), "gray");
		gtk_toggle_button_set_active(toggles[0], TRUE);
		gtk_widget_set_visible(GTK_WIDGET(toggles[1]), FALSE);
		gtk_widget_set_visible(GTK_WIDGET(toggles[2]), FALSE);
		/* visible has no effect in GTK+ 3.12, trying sensitive too 
		 * Yes it does. The solution is to call the window (widget)
		 * with gtk_widget_show and not gtk_widget_show_all */
		gtk_widget_set_sensitive(GTK_WIDGET(toggles[1]), FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(toggles[2]), FALSE);
		if (toggles[3])
			gtk_widget_set_visible(GTK_WIDGET(toggles[3]), FALSE);
	} else {
		const char* test = gtk_button_get_label(GTK_BUTTON(toggles[0]));
		if (strcmp(test, "red"))
			gtk_button_set_label(GTK_BUTTON(toggles[0]), "red");
		gtk_toggle_button_set_active(toggles[0], TRUE);
		gtk_toggle_button_set_active(toggles[1], TRUE);
		gtk_toggle_button_set_active(toggles[2], TRUE);
		gtk_widget_set_sensitive(GTK_WIDGET(toggles[1]), TRUE);
		gtk_widget_set_sensitive(GTK_WIDGET(toggles[2]), TRUE);
		gtk_widget_set_visible(GTK_WIDGET(toggles[1]), TRUE);
		gtk_widget_set_visible(GTK_WIDGET(toggles[2]), TRUE);
		if (toggles[3]) {
			gtk_widget_set_visible(GTK_WIDGET(toggles[3]), TRUE);
			gtk_toggle_button_set_active(toggles[3], TRUE);
		}
	}
}

gboolean redraw_histo(GtkWidget *widget, cairo_t *cr, gpointer data) {
	int i, width, height;

	fprintf(stdout, "histogram redraw\n");
	init_toggles();
	width = gtk_widget_get_allocated_width(widget);
	height = gtk_widget_get_allocated_height(widget);
	if (height == 1) return FALSE;
	erase_histo_display(cr, width, height);
	graph_height = 0.0;
	for (i=0; i<MAXVPORT; i++) {
		if (com.layers_hist[i] && (!toggles[i] || gtk_toggle_button_get_active(toggles[i])))
			display_histo(com.layers_hist[i], cr, i, width, height);
	}
	return FALSE;
}

gboolean redraw_histo_selection(GtkWidget *widget, cairo_t *cr, gpointer data) {
	int i, width, height, nb_layers;
	double tmp_height = graph_height;
	
	width = gtk_widget_get_allocated_width(widget);
	height = gtk_widget_get_allocated_height(widget);
	if (height == 1) return FALSE;
	erase_histo_display(cr, width, height);
	
	if (!com.selection.w || !com.selection.h) return TRUE;
	if (sequence_is_loaded()) nb_layers = com.seq.nb_layers;
	else nb_layers = com.uniq->nb_layers;
	
	for(i = 0; i<nb_layers; i++) {
		gsl_histogram *histo = computeHisto_Selection(&gfit, i, &com.selection);
		graph_height = 0.0;
		if (histo) display_histo(histo, cr, i, width, height);
	}
	graph_height = tmp_height;
	return FALSE;
}

void on_histo_toggled(GtkToggleButton *togglebutton, gpointer user_data) {
	static GtkWidget *drawarea = NULL, *selarea = NULL;
	if (!drawarea) {
		drawarea = lookup_widget("drawingarea_histograms");
		selarea = lookup_widget("drawingarea_histograms_selection");
	}
	gtk_widget_queue_draw(drawarea);
	gtk_widget_queue_draw(selarea);
}

// erase image and redraw the background color and grid
void erase_histo_display(cairo_t *cr, int width, int height) {
	double dash_format[] = { 8.0, 4.0 };
	// clear all with background color
	cairo_set_source_rgb(cr, 0, 0, 0);
	cairo_rectangle (cr, 0,0, width, height);
	cairo_fill(cr);
	// draw axis
	/*cairo_set_line_width(cr, 1.0);
	cairo_move_to(cr, AXIS_MARGIN, AXIS_MARGIN);
	cairo_line_to(cr, AXIS_MARGIN, height-AXIS_MARGIN);
	cairo_line_to(cr, width-AXIS_MARGIN, height-AXIS_MARGIN);*/
	// draw grid 
	cairo_set_line_width (cr, 1.0);
	cairo_set_source_rgb(cr, 0.4, 0.4, 0.4);
	// quarters in solid, eights in dashed line
	cairo_set_dash(cr, NULL, 0, 0);
	cairo_move_to(cr, width*0.25, 0);
	cairo_line_to(cr, width*0.25, height);
	cairo_move_to(cr, width*0.5, 0);
	cairo_line_to(cr, width*0.5, height);
	cairo_move_to(cr, width*0.75, 0);
	cairo_line_to(cr, width*0.75, height);

	cairo_move_to(cr, 0, height*0.25);
	cairo_line_to(cr, width, height*0.25);
	cairo_move_to(cr, 0, height*0.5);
	cairo_line_to(cr, width, height*0.5);
	cairo_move_to(cr, 0, height*0.75);
	cairo_line_to(cr, width, height*0.75);

	cairo_stroke(cr);

	cairo_set_dash(cr, dash_format, 2, 0);
	cairo_move_to(cr, width*0.125, 0);
	cairo_line_to(cr, width*0.125, height);
	cairo_move_to(cr, width*0.375, 0);
	cairo_line_to(cr, width*0.375, height);
	cairo_move_to(cr, width*0.625, 0);
	cairo_line_to(cr, width*0.625, height);
	cairo_move_to(cr, width*0.875, 0);
	cairo_line_to(cr, width*0.875, height);

	cairo_move_to(cr, 0, height*0.125);
	cairo_line_to(cr, width, height*0.125);
	cairo_move_to(cr, 0, height*0.375);
	cairo_line_to(cr, width, height*0.375);
	cairo_move_to(cr, 0, height*0.625);
	cairo_line_to(cr, width, height*0.625);
	cairo_move_to(cr, 0, height*0.875);
	cairo_line_to(cr, width, height*0.875);
	cairo_stroke(cr);
}

void display_histo(gsl_histogram *histo, cairo_t *cr, int layer, int width, int height) {
	int current_bin;
	WORD norm = get_normalized_value(&gfit);
	
	float vals_per_px = (float)(norm+1) / (float)width;	// size of a bin
	size_t i, nb_orig_bins = gsl_histogram_bins(histo);

	// We need to store the binned histogram in order to find the binned maximum
	static double *displayed_values = NULL;
	static int nb_bins_allocated = 0;
	/* we create a bin for each pixel in the displayed width.
	 * nb_bins_allocated is thus equal to the width of the image */
	if (nb_bins_allocated != width) {
		double *tmp;
		nb_bins_allocated = width;
		tmp = realloc(displayed_values, nb_bins_allocated * sizeof(double));
		if (!tmp) {
			if (displayed_values)
				free(displayed_values);
			fprintf(stderr, "Failed to reallocate histogram bins\n");
			return;
		}
		displayed_values = tmp;
		memset(displayed_values, 0, nb_bins_allocated);
	}
	assert(displayed_values);

	if (gfit.naxis == 2)
		cairo_set_source_rgb(cr, 255.0, 255.0, 255.0);
	else cairo_set_source_rgb(cr, histo_color_r[layer],
			histo_color_g[layer], histo_color_b[layer]);
	cairo_set_dash(cr, NULL, 0, 0);
	cairo_set_line_width(cr, 1.5);

	// first loop builds the bins and finds the maximum
	i = 0; current_bin = 0;
	do {
		double bin_val = 0.0;
		while (i < nb_orig_bins &&
				(float)i / vals_per_px <= (float)current_bin+0.5f) { 
			bin_val += gsl_histogram_get (histo, i);
			i++;
		}
		displayed_values[current_bin] = bin_val;
		if (bin_val > graph_height)	// check for maximum
			graph_height = bin_val;
		current_bin++;
	} while (i < nb_orig_bins && current_bin < nb_bins_allocated) ;

	//graph_height *= 1.05;	// margin, but grid is not aligned anymore
	//cairo_move_to(cr, 0, height);	// first line_to will act as move_to
	for (i=0; i < nb_bins_allocated; i++) {
		double bin_height = height - height * displayed_values[i] / graph_height;
		cairo_line_to(cr, i, bin_height);
	}
	cairo_stroke(cr);
}

void on_histogram_window_show(GtkWidget *object, gpointer user_data) {
	register_selection_update_callback(_histo_on_selection_changed);
}

void on_histogram_window_hide(GtkWidget *object, gpointer user_data) {
	unregister_selection_update_callback(_histo_on_selection_changed);
}

void on_button_histo_close_clicked(){
	graph_height=0.;
	gtk_widget_hide (lookup_widget("histogram_window"));
	reset_curors_and_values();
}

void initialize_clip_text() {
	static GtkEntry *clip_high = NULL, *clip_low = NULL;
	
	if (clip_high == NULL) {
		clip_high = GTK_ENTRY(gtk_builder_get_object(builder, "clip_highlights"));
		clip_low = GTK_ENTRY(gtk_builder_get_object(builder, "clip_shadows"));
	}
	gtk_entry_set_text(clip_low, "0.0000%");
	gtk_entry_set_text(clip_high, "0.0000%");
}

void reset_curors_and_values() {
	gtk_range_set_value(GTK_RANGE(gtk_builder_get_object(builder, "scale_midtones")), 0.5);
	gtk_range_set_value(GTK_RANGE(gtk_builder_get_object(builder, "scale_shadows")), 0.0);
	gtk_range_set_value(GTK_RANGE(gtk_builder_get_object(builder, "scale_highlights")), 1.0);
	clipped[0] = 0;
	clipped[1] = 0;
	initialize_clip_text();
	update_gfit_histogram_if_needed();
}

void on_button_histo_reset_clicked() {
	set_cursor_waiting(TRUE);
	reset_curors_and_values();
	set_cursor_waiting(FALSE);
}

//~ void on_histogram_window_key_release_event(GtkWidget *widget, GdkEventKey *event, gpointer user_data){
	//~ if (event->keyval == GDK_KEY_Escape) {
		//~ graph_height=0.;
		//~ gtk_widget_hide (lookup_widget("histogram_window"));
	//~ }
	//~ 
//~ }

WORD midtones_transfer_function(double coeff, WORD pixel_in, double lo, double hi) {
	double in, out, pente;
	double max_norm = (double)gsl_histogram_bins(com.layers_hist[0]);
	
	pente = 1.0/(hi-lo);
	if (coeff==0.5) {
		out = ((double)pixel_in * pente) - lo * max_norm;
		if (out > (double)USHRT_MAX) clipped[1]++;
		if (out < 0.0) clipped[0]++;
		return round_to_WORD(out);
	}
	if (coeff==0.0) return 0;                               
	if (coeff==1.0) {
		out = (max_norm * pente) - lo * max_norm;
		if (out > (double)USHRT_MAX) clipped[1]++;
		if (out < 0.0) clipped[0]++;
		return round_to_WORD(out);
	}
	in = pixel_in/max_norm;
	
	out = ((coeff - 1.0) * in)/((2 * coeff - 1.0) * in - coeff);
	out -= lo;
	out *= max_norm * pente;
	
	if (out > (double)USHRT_MAX) clipped[1]++;
	if (out < 0.0) clipped[0]++;
	
	return round_to_WORD(out);
}

void midtones_adjust_curve(fits *fit) {
	double m, lo, hi;
	int i, chan, nb_chan, ndata;
	WORD *buf[3] = {fit->pdata[RLAYER], fit->pdata[GLAYER], fit->pdata[BLAYER]};
	static GtkRange *scale_transfert_function[3] = {NULL, NULL, NULL};
	
	if (scale_transfert_function[0] == NULL) {
		scale_transfert_function[0] = GTK_RANGE(lookup_widget("scale_shadows"));
		scale_transfert_function[1] = GTK_RANGE(lookup_widget("scale_midtones"));
		scale_transfert_function[2] = GTK_RANGE(lookup_widget("scale_highlights"));
	}
	
	assert(fit->naxes[2] == 1 || fit->naxes[2] == 3);
	nb_chan = fit->naxes[2];
	ndata = fit->rx*fit->ry;
	
	m = gtk_range_get_value(scale_transfert_function[1]);
	lo = gtk_range_get_value(scale_transfert_function[0]);
	hi = gtk_range_get_value(scale_transfert_function[2]);
		
	for (chan=0; chan < nb_chan; chan++){
#pragma omp parallel for private(i) schedule(dynamic, 1)
		for (i=0; i < ndata; i++) {
			buf[chan][i] = midtones_transfer_function(m, buf[chan][i], lo, hi);
		}
	}
}

void update_transfer_function() {
	double hi, lo, tmp;
	char buffer[16];
	int nbdata = gfit.rx * gfit.ry * gfit.naxes[2];
	static GtkEntry *clip_high = NULL, *clip_low = NULL;
	static GtkRange *scale_transfert_function[3] = {NULL, NULL, NULL};

	
	if (clip_high == NULL) {
		clip_high = GTK_ENTRY(lookup_widget("clip_highlights"));
		clip_low = GTK_ENTRY(lookup_widget("clip_shadows"));
	}
	if (scale_transfert_function[0] == NULL) {
		scale_transfert_function[0] = GTK_RANGE(lookup_widget("scale_shadows"));
		scale_transfert_function[1] = GTK_RANGE(lookup_widget("scale_midtones"));
		scale_transfert_function[2] = GTK_RANGE(lookup_widget("scale_highlights"));
	}

	lo = gtk_range_get_value(scale_transfert_function[0]);
	hi = gtk_range_get_value(scale_transfert_function[2]);
	
	gtk_range_set_range (scale_transfert_function[0], 0.0, hi);
	gtk_range_set_range (scale_transfert_function[2], lo, 1.0);
	
	if (hi < lo) {
		fprintf(stderr, "Should not happend\n");
		return ;
	}
	
	set_cursor_waiting(TRUE);
	copyfits(&gfit, &wfit[0], CP_ALLOC|CP_FORMAT|CP_COPYA, 0);
	midtones_adjust_curve(&wfit[0]);
	compute_and_update_histo_for_fit(1, &wfit[0]);
	tmp = (double)clipped[1] * 100.0 / nbdata;
	sprintf(buffer, "%.4lf%%", tmp);
	gtk_entry_set_text(clip_high, buffer);
	tmp = (double)clipped[0] * 100.0 / nbdata;
	sprintf(buffer, "%.4lf%%", tmp);
	gtk_entry_set_text(clip_low, buffer);
	clearfits(&wfit[0]);
	clipped[0] = 0;
	clipped[1] = 0;
	set_cursor_waiting(FALSE);
}

gboolean on_scale_button_release_event (GtkWidget *widget, GdkEvent *event, gpointer user_data) {	
	update_transfer_function();
	return FALSE;
}

gboolean on_scale_key_release_event (GtkWidget *widget, GdkEvent *event, gpointer user_data) {
	update_transfer_function();
	return FALSE;
}

void on_button_histo_apply_clicked (GtkButton *button, gpointer user_data) {
	set_cursor_waiting(TRUE);
	midtones_adjust_curve(&gfit);
	clipped[0] = 0;
	clipped[1] = 0;
	update_gfit_histogram_if_needed();
	reset_curors_and_values();
	redraw(com.cvport, REMAP_ALL);
	redraw_previews();
	set_cursor_waiting(FALSE);
}
