/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin for unit management.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgunitplugin.h"
#include "skgunitpluginwidget.h"
#include "skrooge_unit.h"
#include "skgtraces.h"
#include "skgmainpanel.h"
#include "skgtransactionmng.h"

#include <kinputdialog.h>
#include <kactioncollection.h>
#include <kaction.h>

K_PLUGIN_FACTORY(SKGUnitPluginFactory, registerPlugin<SKGUnitPlugin>();)
K_EXPORT_PLUGIN(SKGUnitPluginFactory("skrooge_unit", "skrooge_unit"))

SKGUnitPlugin::SKGUnitPlugin(QObject* iParent, const QVariantList& /*iArg*/) : SKGInterfacePlugin(iParent)
{
        SKGTRACEIN(10, "SKGUnitPlugin::SKGUnitPlugin");
}

SKGUnitPlugin::~SKGUnitPlugin()
{
        SKGTRACEIN(10, "SKGUnitPlugin::~SKGUnitPlugin");
        parent=NULL;
        currentBankDocument=NULL;
}

void SKGUnitPlugin::setupActions(SKGMainPanel* iParent, SKGDocument* iDocument, const QStringList& iArgument)
{
        SKGTRACEIN(10, "SKGUnitPlugin::setupActions");
        Q_UNUSED(iArgument);

        currentBankDocument=iDocument;
        parent=iParent;

        setComponentData( SKGUnitPluginFactory::componentData() );

        setXMLFile("skrooge_unit.rc");

        //Menu
        splitShareAction = new KAction(KIcon("skrooge_split_stock"), i18n("&Split share..."), this);
        connect(splitShareAction, SIGNAL(triggered(bool)), this, SLOT(actionSplitShare()));
        actionCollection()->addAction( QLatin1String("edit_split_stock"), splitShareAction );
        splitShareAction->setShortcut(Qt::ALT+Qt::Key_Slash);

        iParent->registedGlobalAction("edit_split_stock", splitShareAction);
}

void SKGUnitPlugin::close()
{
        SKGTRACEIN(10, "SKGUnitPlugin::close");
}

void SKGUnitPlugin::refresh()
{
        SKGTRACEIN(10, "SKGUnitPlugin::refresh");
        //Automatic download
        if (currentBankDocument->getDatabase()!=NULL) {
                QString doc_id=currentBankDocument->getParameter("SKG_UNIQUE_ID");
                if (docUniqueIdentifier!=doc_id) {
                        docUniqueIdentifier=doc_id;

                        //Read Setting
                        KSharedConfigPtr config=KSharedConfig::openConfig("skrooge_unitrc");
                        KConfigGroup pref=config->group("skrooge_unit");
                        bool download_on_open = pref.readEntry("download_on_open",false);
                        if (download_on_open) {
                                SKGError err;
                                QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

                                //download all units
                                SKGObjectBase::SKGListSKGObjectBase selection;
                                err=SKGObjectBase::getObjects(currentBankDocument, "unit","", selection);
                                int nb=selection.count();
                                for (int i=0; err.isSucceeded() && i<nb; ++i) {
                                        SKGUnitObject unit=selection.at(i);
                                        err=SKGUnitPluginWidget::downloadUnitValue(unit, parent);
                                }
                                QApplication::restoreOverrideCursor();

                                //Display error
                                parent->displayErrorMessage(err);
                        }
                }

                SKGObjectBase::SKGListSKGObjectBase selection=parent->getSelectedObjects();
                if (selection.count()==1 && selection.at(0).getRealTable()=="unit") {
                        SKGUnitObject unit=selection.at(0);
                        splitShareAction->setEnabled(unit.getType()==SKGUnitObject::SHARE);
                } else {
                        splitShareAction->setEnabled(false);
                }
        }
}

SKGTabWidget* SKGUnitPlugin::getWidget()
{
        SKGTRACEIN(10, "SKGUnitPlugin::getWidget");
        return new SKGUnitPluginWidget(parent, (SKGDocumentBank*) currentBankDocument);
}

QWidget* SKGUnitPlugin::getPreferenceWidget()
{
        SKGTRACEIN(10, "SKGUnitPlugin::getPreferenceWidget");
        QWidget* widget=new QWidget();
        ui.setupUi(widget);

        return widget;
}

KConfigSkeleton* SKGUnitPlugin::getPreferenceSkeleton()
{
        return skrooge_unit::self();
}

QString SKGUnitPlugin::title() const
{
        return i18n("Units");
}

QString SKGUnitPlugin::icon() const
{
        return "skrooge_unit";
}

QString SKGUnitPlugin::statusTip () const
{
        return i18n("Unit management (creation, update ...)");
}

QString SKGUnitPlugin::toolTip () const
{
        return i18n("Unit management");
}

QStringList SKGUnitPlugin::tips() const
{
        QStringList output;
        output.push_back(i18n("<p>... you can download units.</p>"));
        output.push_back(i18n("<p>... units can be downloaded automatically when a document is opened.</p>"));
        output.push_back(i18n("<p>... you can split a share.</p>"));
        return output;
}

int SKGUnitPlugin::getOrder() const
{
        return 60;
}

bool SKGUnitPlugin::isInContext() const
{
        return true;
}

void SKGUnitPlugin::actionSplitShare()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGUnitPlugin::actionSplitShare",err);

        //Get Selection
        SKGObjectBase::SKGListSKGObjectBase selection=parent->getSelectedObjects();
        int nb=selection.count();
        if (nb==1) {
                bool ok=false;
                double ratio=KInputDialog::getDouble  (i18n("Split share"),
                                                       i18n("Ratio (2 means 2-for-1, 0.5 means 1-for-2) :"), 2.0,
                                                       0, DBL_MAX, 2, &ok, parent);
                if (ok) {
                        SKGUnitObject unit=selection.at(0);
                        SKGBEGINTRANSACTION(*currentBankDocument, i18n("Split strock [%1] by [%2]", unit.getName(), ratio), err);
                        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
                        err=unit.split(ratio);
                        QApplication::restoreOverrideCursor();
                }
        }

		//status
		if (err.isSucceeded())  err=SKGError(0, i18n("Strock splitted."));
		else err.addError(ERR_FAIL, i18n("Split strock failed"));

        //Display error
        parent->displayErrorMessage(err);
}

#include "skgunitplugin.moc"
