/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is part of Skrooge and implements classes SKGOperationObject.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgoperationobject.h"
#include "skgrecurrentoperationobject.h"
#include "skgsuboperationobject.h"
#include "skgaccountobject.h"
#include "skgunitobject.h"
#include "skgservices.h"
#include "skgdocument.h"

SKGOperationObject::SKGOperationObject(const SKGDocument* iDocument, int iID): SKGObjectBase(iDocument, "v_operation", iID)
{
}

SKGOperationObject::~SKGOperationObject()
{
}

SKGOperationObject::SKGOperationObject(const SKGOperationObject& iObject)
                : SKGObjectBase(iObject)
{
}

SKGOperationObject::SKGOperationObject(const SKGObjectBase& iObject)
{
        if (iObject.getRealTable()=="operation") {
                copyFrom(iObject);
        } else {
                *this=SKGObjectBase::SKGObjectBase(iObject.getDocument(), "v_operation", iObject.getID());
        }
}

const SKGOperationObject& SKGOperationObject::operator= (const SKGObjectBase& iObject)
{
        copyFrom(iObject);
        return *this;
}

SKGError SKGOperationObject::duplicate(SKGOperationObject& oOperation, const QDate& iDate)
{
        SKGError err;
        //Create the duplicated operation
        oOperation=*this;
        err=oOperation.resetID();
        if (err.isSucceeded()) err=oOperation.setDate(iDate);
        if (err.isSucceeded()) err=oOperation.setStatus(SKGOperationObject::NONE);
        if (err.isSucceeded()) err=oOperation.setImported(false);
        if (err.isSucceeded()) err=oOperation.setImportID("");
        if (err.isSucceeded()) err=oOperation.bookmark(false);
        if (err.isSucceeded()) err=oOperation.setNumber(0);
        if (err.isSucceeded()) err=oOperation.setGroupOperation(oOperation); //We have to ungroup first to be able to group after
        if (err.isSucceeded()) err=oOperation.save(false, false); //Just an insert

        //Duplicate subop
        if (err.isSucceeded()) {
                SKGListSKGObjectBase subops;
                err=getSubOperations(subops);
                int nbsupops=subops.count();
                for (int i=0; err.isSucceeded() && i<nbsupops; ++i) {
                        SKGSubOperationObject subop=subops.at(i);
                        err=subop.resetID();
                        if (err.isSucceeded()) err=subop.setParentOperation(oOperation);
                        if (err.isSucceeded()) err=subop.save(false); //Just an insert
                }
        }

        //Duplicate grouped operation to support recurrent transfers
        if (err.isSucceeded()) {
                SKGListSKGObjectBase goupops;
                err=getGroupedOperations(goupops);
                int nbgoupops=goupops.count();
                for (int i=0; err.isSucceeded() && i<nbgoupops; ++i) {
                        SKGOperationObject groupop=goupops.at(i);
                        if (groupop!=*this) {
                                //Create the duplicated operation
                                SKGOperationObject newgroupop=groupop;
                                err=newgroupop.resetID();
                                if (err.isSucceeded()) err=newgroupop.setDate(iDate);
                                if (err.isSucceeded()) err=newgroupop.setStatus(SKGOperationObject::NONE);
                                if (err.isSucceeded()) err=newgroupop.setImported(false);
                                if (err.isSucceeded()) err=newgroupop.setImportID("");
                                if (err.isSucceeded()) err=newgroupop.bookmark(false);
                                if (err.isSucceeded()) err=newgroupop.setNumber(0);
                                if (err.isSucceeded()) err=newgroupop.setGroupOperation(newgroupop); //We have to ungroup first to be able to group after
                                if (err.isSucceeded()) err=newgroupop.setGroupOperation(oOperation);
                                if (err.isSucceeded()) err=newgroupop.save(false); //Just an insert

                                //Duplicate subop
                                if (err.isSucceeded()) {
                                        SKGListSKGObjectBase subops;
                                        err=groupop.getSubOperations(subops);
                                        int nbsupops=subops.count();
                                        for (int i=0; err.isSucceeded() && i<nbsupops; ++i) {
                                                SKGSubOperationObject subop=subops.at(i);
                                                err=subop.resetID();
                                                if (err.isSucceeded()) err=subop.setParentOperation(newgroupop);
                                                if (err.isSucceeded()) err=subop.save(false); //Just an insert
                                        }
                                }
                        }
                }
        }

        if (err.isSucceeded()) err=save();
        return err;
}


SKGError SKGOperationObject::getParentAccount(SKGAccountObject& oAccount) const
{
        SKGObjectBase objTmp;
        SKGError err = getObject(getDocument(), "v_account", "id=" + getAttribute("rd_account_id"), objTmp);
        oAccount=objTmp;
        return err;
}

SKGError SKGOperationObject::setParentAccount(const SKGAccountObject& iAccount)
{
        SKGError err;
        QString currentAccount=getAttribute("rd_account_id");
        QString newAccount=SKGServices::intToString(iAccount.getID());
        if (newAccount=="0") {
                err=SKGError(ERR_FAIL, tr("%1 failed because linked object is not yet saved in the database.").arg("SKGOperationObject::setParentAccount"));
        } else {
                if (newAccount!=currentAccount) {
                        if (iAccount.isClosed()) {
                                err=SKGError(ERR_FAIL, tr("Impossible to add an operation in a closed account"));
                        } else {
                                err=setAttribute("rd_account_id", newAccount);
                        }
                }
        }
        return err;
}

SKGError SKGOperationObject::setMode(const QString& iMode)
{
        return setAttribute("t_mode", iMode);
}

QString SKGOperationObject::getMode() const
{
        return getAttribute("t_mode");
}

SKGError SKGOperationObject::setPayee(const QString& iPayee)
{
        return setAttribute("t_payee", iPayee);
}

QString SKGOperationObject::getPayee() const
{
        return getAttribute("t_payee");
}

SKGError SKGOperationObject::setComment(const QString& iComment)
{
        return setAttribute("t_comment", iComment);
}

QString SKGOperationObject::getComment() const
{
        return getAttribute("t_comment");
}

SKGError SKGOperationObject::setNumber(int iNumber)
{
        return setAttribute("i_number", SKGServices::intToString(iNumber));
}

int SKGOperationObject::getNumber() const
{
        return SKGServices::stringToInt(getAttribute("i_number"));
}

SKGOperationObject::OperationStatus SKGOperationObject::getStatus() const
{
        QString t_status=getAttribute("t_status");
        if (t_status=="C") return SKGOperationObject::CHECKED;
        else if (t_status=="P") return SKGOperationObject::POINTED;
        return SKGOperationObject::NONE;
}

SKGError SKGOperationObject::setStatus(SKGOperationObject::OperationStatus iStatus)
{
        return setAttribute("t_status", (iStatus==SKGOperationObject::CHECKED ? "C" : (iStatus==SKGOperationObject::POINTED ? "P" : "N")));
}

SKGError SKGOperationObject::setDate(const QDate& iDate)
{
        return setAttribute("d_date", SKGServices::dateToSqlString(QDateTime(iDate)));
}

QDate SKGOperationObject::getDate() const
{
        return SKGServices::stringToTime(getAttribute("d_date")).date();
}

SKGError SKGOperationObject::getUnit(SKGUnitObject& oUnit) const
{
        SKGError err = getObject(getDocument(), "v_unit", "id=" + getAttribute("rc_unit_id"), oUnit);
        return err;
}

SKGError SKGOperationObject::setUnit(const SKGUnitObject& iUnit)
{
        return setAttribute("rc_unit_id", SKGServices::intToString(iUnit.getID()));
}

bool SKGOperationObject::isInGroup() const
{
        return (getAttribute("i_group_id")!="0");
}

SKGError SKGOperationObject::getGroupedOperations(SKGListSKGObjectBase& oGroupedOperations) const
{
        SKGError err;
        QString gpId1=getAttribute("i_group_id");
        if (gpId1=="0") oGroupedOperations.clear();
        else  err=SKGObjectBase::getObjects(getDocument(), "v_operation", "i_group_id="+ gpId1, oGroupedOperations);
        return err;
}

SKGError SKGOperationObject::getGroupOperation(SKGOperationObject& oOperation) const
{
        SKGError err = getObject(getDocument(), "v_operation", "id=" + getAttribute("i_group_id"), oOperation);
        return err;
}

SKGError SKGOperationObject::setGroupOperation(const SKGOperationObject& iOperation)
{
        SKGError err;

        //Is it a remove group ?
        if (iOperation==*this) {
                //Yes
                err=setAttribute("i_group_id", "0");
        } else {
                //No
                //Is a group already existing ?
                QString existingGroup=getAttribute("i_group_id");
                if (existingGroup=="0" || existingGroup.isEmpty()) existingGroup=iOperation.getAttribute("i_group_id");

                if (existingGroup=="0" || existingGroup.isEmpty()) {
                        //No, we have to create a new group
                        SKGStringListList result;
                        err=SKGServices::executeSelectSqliteOrder(getDocument(), "SELECT max(i_group_id) from operation", result);
                        if (err.isSucceeded()) {
                                //Compute new group id
                                QString newIdGroup="1";
                                if (result.count()==2) {
                                        newIdGroup=SKGServices::intToString(SKGServices::stringToInt(result.at(1).at(0))+1);
                                }

                                //Set group id
                                SKGOperationObject op1=iOperation;
                                err=op1.setAttribute("i_group_id", newIdGroup);
                                if (err.isSucceeded()) err=op1.save();

                                if (err.isSucceeded()) err=setAttribute("i_group_id", newIdGroup);

                        }
                } else {
                        //Yes, all operation must be grouped
                        SKGOperationObject op1=iOperation;
                        err=op1.setAttribute("i_group_id", existingGroup);
                        if (err.isSucceeded()) err=op1.save();

                        if (err.isSucceeded()) err=setAttribute("i_group_id", existingGroup);
                }

        }

        return err;
}

SKGError SKGOperationObject::bookmark(bool iBookmark)
{
        return setAttribute("t_bookmarked", iBookmark ? "Y" :"N");

}

bool SKGOperationObject::isBookmarked() const
{
        return (getAttribute("t_bookmarked")=="Y" ? true : false);
}

SKGError SKGOperationObject::setImported(bool iImported)
{
        return setAttribute("t_imported", iImported ? "Y" :"N");

}

bool SKGOperationObject::isImported() const
{
        return (getAttribute("t_imported")!="N" ? true : false);
}

SKGError SKGOperationObject::setImportID(const QString& iImportID)
{
        return setAttribute("t_import_id", iImportID);
}

QString SKGOperationObject::getImportID() const
{
        return getAttribute("t_import_id");
}

int SKGOperationObject::getNbSubOperations() const
{
        return SKGServices::stringToInt(getAttribute("i_NBSUBCATEGORY"));
}

SKGError SKGOperationObject::addSubOperation(SKGSubOperationObject& oSubOperation)
{
        SKGError err;
        if (getID() == 0) err=SKGError(ERR_FAIL, tr("%1 failed because linked object is not yet saved in the database.").arg("SKGOperationObject::addSubOperation"));
        else {
                oSubOperation = SKGSubOperationObject(getDocument());
                err = oSubOperation.setParentOperation(*this);
        }
        return err;
}

SKGError SKGOperationObject::getSubOperations(SKGListSKGObjectBase& oSubOperations) const
{
        SKGError err;
        if (getID() == 0) err=SKGError(ERR_FAIL, tr("%1 failed because linked object is not yet saved in the database.").arg("SKGOperationObject::getSubOperations"));
        else {

                err=SKGObjectBase::getObjects(getDocument(), "v_suboperation",
                                              "rd_operation_id="+ SKGServices::intToString(getID()), oSubOperations);
        }
        return err;
}

double SKGOperationObject::getCurrentAmount() const
{
        return SKGServices::stringToDouble(getAttribute("f_CURRENTAMOUNT"));
}

double SKGOperationObject::getAmount(const QDate& iDate) const
{
        //Get quantity
        double quantity=SKGServices::stringToDouble(getAttribute("f_QUANTITY"));

        //Is the unit value already in cache ?
        double coef=1;
        QString val=getDocument()->getCachedValue("unitvalue-"+getAttribute("rc_unit_id"));
        if (!val.isEmpty()) {
                //Yes
                coef=SKGServices::stringToDouble(val);
        } else {
                //No
                SKGUnitObject unit;
                if (getUnit(unit).isSucceeded()) coef=unit.getAmount(iDate);
        }

        return coef*quantity;
}

SKGError SKGOperationObject::addRecurrentOperation(SKGRecurrentOperationObject& oRecurrentOperation)
{
        SKGError err;
        if (getID() == 0) err=SKGError(ERR_FAIL, tr("%1 failed because linked object is not yet saved in the database.").arg("SKGOperationObject::addRecurrentOperation"));
        else {
                oRecurrentOperation = SKGRecurrentOperationObject(getDocument());
                err = oRecurrentOperation.setParentOperation(*this);
                if (err.isSucceeded()) oRecurrentOperation.setDate(getDate());
        }
        return err;
}

SKGError SKGOperationObject::getRecurrentOperations(SKGListSKGObjectBase& oRecurrentOperation) const
{
        SKGError err;
        if (getID() == 0) err=SKGError(ERR_FAIL, tr("%1 failed because linked object is not yet saved in the database.").arg("SKGOperationObject::getRecurrentOperation"));
        else {
                err=SKGObjectBase::getObjects(getDocument(), "v_recurrentoperation",
                                              "rd_operation_id="+ SKGServices::intToString(getID()), oRecurrentOperation);
        }
        return err;
}
#include "skgoperationobject.moc"

