/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is part of Skrooge and defines classes skgmainpanel.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgmainpanel.h"
#include "skgtraces.h"
#include "skgservices.h"
#include "skginterfaceplugin.h"
#include "skrooge.h"
#include "skgtransactionmng.h"

#include <KTemporaryFile>
#include <ktip.h>
#include <ksplashscreen.h>
#include <kconfigdialog.h>
#include <kstatusbar.h>
#include <ktoolbar.h>
#include <kmenubar.h>
#include <kpushbutton.h>
#include <KDE/KLocale>
#include <kglobal.h>
#include <kparts/componentfactory.h>
#include <kaction.h>
#include <kactionmenu.h>
#include <kactioncollection.h>
#include <kmessagebox.h>
#include <ktabwidget.h>
#include <kservice.h>
#include <kxmlguifactory.h>
#include <knotification.h>
#include <kfiledialog.h>
#include <kmenu.h>
#include <kcombobox.h>
#include <kselectaction.h>
#include <kstandarddirs.h>

#include <QtGui/QProgressBar>
#include <QtGui/QHeaderView>
#include <QApplication>
#include <QDir>
#include <QResizeEvent>

bool SKGMainPanel::currentActionCanceled=false;
SKGMainPanel::SKGMainPanel ( KSplashScreen* iSplashScreen, SKGDocument* iDocument, const QStringList& iArguments )
                : KParts::MainWindow(),
                tabView ( new KTabWidget ( this ) ),
                spScreen ( iSplashScreen ),
                currentBankDocument ( iDocument ),
                tipOfDayAction ( NULL ),closePageAction ( NULL ), closeAllPagesAction ( NULL ), closeAllOtherPagesAction ( NULL ),
                tipDatabase ( NULL ), kSystemTrayIcon ( NULL ), kNormalMessage(NULL)
{
        SKGTRACEIN ( 1, "SKGMainPanel::SKGMainPanel" );

        // accept dnd
        setAcceptDrops ( true );

        ui.setupUi ( this );

        // tell the KXmlGuiWindow that this is indeed the main widget
        setCentralWidget ( tabView );

        tabView->setTabReorderingEnabled ( true );
        tabView->setHoverCloseButton ( true );
        tabView->setHoverCloseButtonDelayed ( true );
        tabView->setTabCloseActivatePrevious ( true );
        tabView->setAutomaticResizeTabs ( true );
        tabView->setTabBarHidden ( false );
        tabView->setUsesScrollButtons ( true );

        tabView->show();

        //System tray
        if ( KSystemTrayIcon::isSystemTrayAvailable() ) {
                kSystemTrayIcon=new KSystemTrayIcon ( qApp->windowIcon() );
                KMenu* systemTrayMenu = new KMenu ( this );
                QAction* quit = systemTrayMenu->addAction ( KIcon ( "application-exit" ), i18n ( "Quit" ) );
                connect ( quit, SIGNAL ( triggered ( bool ) ), this, SLOT ( onQuitAction() ) );

                kSystemTrayIcon->setContextMenu ( systemTrayMenu );
                connect ( kSystemTrayIcon, SIGNAL ( activated ( QSystemTrayIcon::ActivationReason ) ), SLOT ( onTrayActivated ( QSystemTrayIcon::ActivationReason ) ) );
                if ( iconInSystemtray() ) kSystemTrayIcon->show();
        }

        // then, setup our actions
        setupActions();

        //Search plugins
        KService::List offers=KServiceTypeTrader::self()->query ( QLatin1String ( "Skrooge/Plugin" ) );

        //Create temporary file for tips
        KTemporaryFile tipsFile;
        tipsFile.setSuffix ( ".tips" );

        bool tipsFileOpened=tipsFile.open();
        bool tipsFound=false;

        QTextStream stream ( &tipsFile );

        //Load plugins
        int nb=offers.count();
        SKGTRACEL ( 1 ) << nb << " plugins found" << endl;
        if ( spScreen ) spScreen->showMessage ( i18n ( "Loading plugins..." ), Qt::AlignLeft, QColor ( 221, 130, 8 ) ); // krazy:exclude=doublequote_chars
        SKGError err;
        for ( int i=0; i<nb; ++i ) {
                KService::Ptr service = offers.at ( i );
                QString name=service->name();
                QString msg=i18n ( "Loading plugin %1/%2: %3...", i+1, nb, name );
                SKGTRACEL ( 1 ) << msg << endl;
                if ( spScreen ) spScreen->showMessage ( msg, Qt::AlignLeft, QColor ( 221, 130, 8 ) );
                KPluginLoader loader ( service->library() );
                KPluginFactory *factory =loader.factory();
                if ( factory ) {
                        SKGInterfacePlugin *pluginInterface = factory->create<SKGInterfacePlugin> ( this );
                        if ( pluginInterface ) {
                                if ( pluginInterface->isEnabled() ) {
                                        pluginInterface->setupActions ( this, currentBankDocument, iArguments );

                                        //Store plugin
                                        int nbplugin=pluginsList.count();
                                        int pos=nbplugin;
                                        for ( int j=nbplugin-1; j>=0; --j ) {
                                                if ( pluginInterface->getOrder() <pluginsList.at ( j )->getOrder() ) {
                                                        pos=j;
                                                }
                                        }

                                        pluginsList.insert ( pos, pluginInterface );
                                        pluginInterface->setObjectName ( name );

                                        //Build temporary tips file
                                        if ( tipsFileOpened ) {
                                                QStringList tipsList=pluginInterface->tips();
                                                int nbtips=tipsList.count();
                                                for ( int t=0; t<nbtips; ++t ) {
                                                        tipsFound=true;
                                                        stream << "<tip category=\""
                                                        << name
                                                        << "\"><html>"<< tipsList.at ( t )
                                                        << "</html></tip>"
                                                        << endl;
                                                }
                                        }
                                }
                        } else {
                                err.addError ( ERR_FAIL, i18n ( "Plugin's interface %1 not found in %2", name, service->library() ) );
                        }
                } else {
                        SKGTRACEL ( 1 ) << loader.errorString() << endl;
                        err.addError ( ERR_FAIL, loader.errorString() );
                        err.addError ( ERR_FAIL, i18n ( "Loading plugin %1 failed because the factory could not be found in %2", name, service->library() ) );
                }
        }

        spScreen=NULL;

        displayErrorMessage ( err );

        //Display tips
        if ( !tipDatabase && tipsFound ) {
                //Tip of day is available
                //Creation of the database
                tipDatabase=new KTipDatabase ( tipsFile.fileName() );
        }

        if ( tipsFileOpened ) {
                tipsFile.close();
        }

        //Add a status bar
        statusBar()->show();

        //Add action to control hide / display of Context Chooser
        ui.kDockContext->toggleViewAction()->setText ( i18n ( "Context Chooser" ) );
        ui.kDockContext->toggleViewAction()->setShortcut ( Qt::SHIFT+Qt::Key_F9 );
        actionCollection()->addAction ( "view_context", ui.kDockContext->toggleViewAction() );

        //Add style sheet menu
        QDir dirTheme ( KStandardDirs::locate ( "data", QString::fromLatin1 ( "skrooge/theme/" ) ) );
        dirTheme.setFilter ( QDir::Files );
        QStringList filters;
        filters << "*.css";
        dirTheme.setNameFilters ( filters );
        QStringList listThemes = dirTheme.entryList();
        int nbt=listThemes.size();
        if ( nbt>1 ) {
                defaultStyle=styleSheet();
                KSelectAction* styleSheetMenu = new KSelectAction ( KIcon ( "view-media-visualization" ), i18n ( "Theme" ), this );
                actionCollection()->addAction ( QLatin1String ( "view_stylesheet" ), styleSheetMenu );

                KSharedConfigPtr config=KSharedConfig::openConfig ( "skroogerc" );
                KConfigGroup pref=config->group ( "skrooge" );
                QString lastTheme=pref.readEntry ( "last_theme", "default" );

                for ( int i=0; i<nbt ; ++i ) {
                        QString themeName=listThemes.at ( i );
                        themeName.remove ( ".css" );

                        KAction* action=styleSheetMenu->addAction ( themeName );
                        if ( themeName==lastTheme ) styleSheetMenu->setCurrentAction ( action );
                }
                onChangeStyleSheet ( lastTheme );
                connect ( styleSheetMenu, SIGNAL ( triggered ( QString ) ), this, SLOT ( onChangeStyleSheet ( QString ) ), Qt::QueuedConnection );
        } else onChangeStyleSheet ( "default" );


        KSelectAction* contextMenu = new KSelectAction ( KIcon ( "tab-new" ), i18n ( "Context" ), this );
        actionCollection()->addAction ( "view_contextmenu" , contextMenu );

        //Add plugin in client in right order
        int shortCutIndex=0;
        int nbplugin=pluginsList.count();
        for ( int j=0; j<nbplugin; ++j ) {
                SKGInterfacePlugin *pluginInterface=pluginsList.at ( j );
                if ( pluginInterface ) {
                        //Creation of the item
                        QString title=pluginInterface->title();
                        SKGTRACEL ( 1 ) << "Add plugin (" << pluginInterface->getOrder() << ") : " << title << endl;
                        if ( title.length() >0 ) {
                                //Create icon to launch this plugin
                                KIcon icon(pluginInterface->icon());
                                bool visible=pluginInterface->isInContext();
                                QListWidgetItem *contextItem = new QListWidgetItem ( icon, title );
                                contextItem->setStatusTip ( pluginInterface->statusTip() );
                                contextItem->setToolTip ( pluginInterface->toolTip() );
                                contextItem->setData(12, j);

                                ui.kContextList->addItem ( contextItem );
                                ui.kContextList->setItemHidden ( contextItem, !visible );

                                //Add menu item with shortcuts
                                if (visible) {
                                        KAction* action=contextMenu->addAction ( icon, title );
                                        if (action) {
                                                action->setCheckable(false);
                                                action->setData(j);
                                                ++shortCutIndex;
                                                action->setShortcut("Ctrl+"+SKGServices::intToString(shortCutIndex));
                                                connect(action, SIGNAL(triggered(bool)), this, SLOT(onOpenContext()));
                                        }
                                }

                                //Create dock if needed
                                QDockWidget* dockWidget=pluginInterface->getDockWidget();
                                if ( dockWidget )  this->addDockWidget ( Qt::LeftDockWidgetArea, dockWidget );
                        }
                }
        }

        // a call to KXmlGuiWindow::setupGUI() populates the GUI
        // with actions, using KXMLGUI.
        // It also applies the saved mainwindow settings, if any, and ask the
        // mainwindow to automatically save settings if changed: window size,
        // toolbar position, icon size, etc.
        setupGUI ( Default, "skrooge.rc" );

        for ( int j=0; j<nbplugin; ++j ) {
                SKGInterfacePlugin *pluginInterface=pluginsList.at ( j );
                if ( pluginInterface ) {
                        QString title=pluginInterface->title();
                        SKGTRACEL ( 1 ) << "Add plugin client (" << pluginInterface->getOrder() << ") : " << title << endl;
                        guiFactory()->addClient ( pluginInterface );
                }
        }

        //Set status bar
        kNormalMessage=new QLabel(this);

        QProgressBar *kProgressBar=new QProgressBar(this);
        kProgressBar->setObjectName ( QString::fromUtf8 ( "kProgressBar" ) );
        kProgressBar->setValue ( 0 );
        kProgressBar->setFixedSize ( 100, 22 );
        QSizePolicy sizePolicy ( QSizePolicy::Fixed, QSizePolicy::Expanding );
        kProgressBar->setSizePolicy ( sizePolicy );
        kProgressBar->setToolTip ( i18n ( "Progress of the current action" ) );
        kProgressBar->setStatusTip ( i18n ( "Progress of the current action" ) );
        kProgressBar->setWhatsThis ( i18n ( "Progress of the current action" ) );

        statusBar()->addWidget ( kNormalMessage );
        statusBar()->addPermanentWidget ( kProgressBar );

        //Set Cancel button
        kCancelButton=new KPushButton();
        kCancelButton->setObjectName ( QString::fromUtf8 ( "kCancelButton" ) );
        kCancelButton->setIcon ( KIcon ( "process-stop" ) );
        kCancelButton->setEnabled ( false );
        kCancelButton->setToolTip ( i18n ( "Cancel the current action" ) );
        kCancelButton->setStatusTip ( i18n ( "Cancel the current action" ) );
        kCancelButton->setWhatsThis ( i18n ( "Cancel the current action" ) );
        connect ( kCancelButton, SIGNAL ( clicked() ), this, SLOT ( onCancelCurrentAction() ) );

        statusBar()->addPermanentWidget ( kCancelButton );

        //Set progress bar call back
        if ( currentBankDocument ) {
                progressObjects.p1= ( void* ) kProgressBar;
                progressObjects.p2= ( void* ) kCancelButton;
                currentBankDocument->setProgressCallback ( &SKGMainPanel::progressBarCallBack, ( void* ) &progressObjects );
        }

        //Connection
        if ( currentBankDocument ) {
                connect ( ( const QObject* ) currentBankDocument, SIGNAL ( transactionSuccessfullyEnded ( int ) ), this, SLOT ( refresh() ), Qt::QueuedConnection );
                connect ( ( const QObject* ) currentBankDocument, SIGNAL ( transactionSuccessfullyEnded ( int ) ), this, SLOT ( notify ( int ) ), Qt::QueuedConnection );
        }
        connect ( ui.kContextList, SIGNAL ( itemPressed ( QListWidgetItem* ) ), this, SLOT ( onBeforeOpenContext () ) );
        connect ( ui.kContextList, SIGNAL ( itemClicked ( QListWidgetItem* ) ), this, SLOT ( onOpenContext () ) );
        connect ( tabView, SIGNAL ( closeRequest ( QWidget* ) ), this, SLOT ( closeTab ( QWidget* ) ) );
        connect ( tabView, SIGNAL ( mouseMiddleClick ( QWidget* ) ), this, SLOT ( closeTab ( QWidget* ) ) );
        connect ( tabView, SIGNAL ( mouseDoubleClick() ), this, SLOT ( addTab() ) );
        connect ( tabView, SIGNAL ( currentChanged ( int ) ), this, SLOT ( refresh() ) );

        //Refresh
        refresh();

        //Tip of day
        if ( tipDatabase ) {
                //Tip of day is available
                KSharedConfigPtr config=KSharedConfig::openConfig ( "skroogerc" );
                KConfigGroup pref=config->group ( "TipOfDay" );
                if ( pref.readEntry ( "RunOnStart", true ) ) onTipOfDay();
        } else {
                //Remove useless menu
                delete tipOfDayAction;
                tipOfDayAction=NULL;
        }

        ui.kContextList->installEventFilter ( this );
}

SKGMainPanel::~SKGMainPanel()
{
        SKGTRACEIN ( 1, "SKGMainPanel::~SKGMainPanel" );
        disconnect ( ( const QObject* ) currentBankDocument, 0, this, 0 );

        //close plugins
        int nb=pluginsList.count();
        for ( int i = 0; i < nb; ++i ) {
                getPluginByIndex ( i )->close();
        }


        if ( tipDatabase ) {
                delete tipDatabase;
                tipDatabase=NULL;
        }

        if ( currentBankDocument ) {
                currentBankDocument->close();
        }
        kNormalMessage=NULL;
}

void SKGMainPanel::registedGlobalAction ( const QString& iIdentifier, KAction* iAction )
{
        registeredGlogalAction[iIdentifier]=iAction;
}

KAction* SKGMainPanel::getGlobalAction ( const QString& iIdentifier )
{
        return registeredGlogalAction[iIdentifier];
}

SKGInterfacePlugin * SKGMainPanel::getPluginByIndex ( int iIndex )
{
        SKGInterfacePlugin * output=NULL;
        if ( iIndex>=0 && iIndex<pluginsList.count() ) {
                output=pluginsList[iIndex];
        }

        return output;
}

SKGInterfacePlugin * SKGMainPanel::getPluginByName ( const QString& iName )
{
        SKGInterfacePlugin * output=NULL;
        int nbplugin=pluginsList.count();
        for ( int j=0; !output && j<nbplugin; ++j ) {
                if ( pluginsList[j]->objectName() ==iName ) {
                        output=pluginsList[j];
                }
        }

        return output;
}

void SKGMainPanel::setupActions()
{
        SKGTRACEIN ( 1, "SKGMainPanel::setupActions" );

        //Std File
        KStandardAction::quit ( this, SLOT ( onQuitAction() ), actionCollection() );

        //Preferences
        KStandardAction::preferences ( this, SLOT ( optionsPreferences() ), actionCollection() );

        // New Tab
        addTabAction = new KAction ( KIcon ( "tab-new-background" ), i18n ( "New &Tab" ), this );
        actionCollection()->addAction ( QLatin1String ( "new_tab" ), addTabAction );
        addTabAction->setShortcut ( Qt::CTRL + Qt::Key_T );
        connect ( addTabAction,SIGNAL ( triggered() ), this, SLOT ( addTab() ) );

        registedGlobalAction ( "tab-new-background", addTabAction );

        closePageAction = new KAction ( KIcon ( "window-close" ), i18n ( "Close" ), this );
        actionCollection()->addAction ( QLatin1String ( "tab_close" ), closePageAction );
        closePageAction->setShortcut ( Qt::CTRL + Qt::Key_W );
        connect ( closePageAction,SIGNAL ( triggered() ), this, SLOT ( closeTab() ) );

        registedGlobalAction ( "tab_close", closePageAction );

        closeAllPagesAction = new KAction ( KIcon ( "window-close" ), i18n ( "Close All" ), this );
        actionCollection()->addAction ( QLatin1String ( "tab_closeall" ), closeAllPagesAction );
        closeAllPagesAction->setShortcut ( Qt::ALT +Qt::Key_W );
        connect ( closeAllPagesAction,SIGNAL ( triggered() ), this, SLOT ( closeAllTabs() ) );

        registedGlobalAction ( "tab_closeall", closeAllPagesAction );

        closeAllOtherPagesAction = new KAction ( KIcon ( "window-close" ), i18n ( "Close All Other" ), this );
        actionCollection()->addAction ( QLatin1String ( "tab_closeallother" ), closeAllOtherPagesAction );
        closeAllOtherPagesAction->setShortcut ( Qt::CTRL + Qt::ALT +Qt::Key_W );
        connect ( closeAllOtherPagesAction,SIGNAL ( triggered() ), this, SLOT ( closeAllOtherTabs() ) );

        registedGlobalAction ( "tab_closeallother", closeAllOtherPagesAction );

        saveDefaultStateAction = new KAction ( KIcon ( "document-save" ), i18n ( "Save state as default" ), this );
        actionCollection()->addAction ( QLatin1String ( "tab_savedefaultstate" ), saveDefaultStateAction );
        saveDefaultStateAction->setShortcut ( Qt::CTRL + Qt::ALT +Qt::Key_S );
        connect ( saveDefaultStateAction,SIGNAL ( triggered() ), this, SLOT ( saveDefaultState() ) );

        registedGlobalAction ( "tab_savedefaultstate", saveDefaultStateAction );

        tipOfDayAction=KStandardAction::tipOfDay ( this, SLOT ( onTipOfDay() ), actionCollection() );

        //Contextual menu
        tabView->setContextMenuPolicy ( Qt::ActionsContextMenu );
        tabView->insertAction ( 0, getGlobalAction ( "tab-new-background" ) );
        tabView->insertAction ( 0, getGlobalAction ( "tab_close" ) );
        tabView->insertAction ( 0, getGlobalAction ( "tab_closeall" ) );
        tabView->insertAction ( 0, getGlobalAction ( "tab_closeallother" ) );
        QAction* sep=new QAction ( this );
        sep->setSeparator ( true );
        tabView->insertAction ( 0, sep );
        tabView->insertAction ( 0, getGlobalAction ( "tab_savedefaultstate" ) );
}

bool SKGMainPanel::queryClose()
{
        SKGTRACEIN ( 1, "SKGMainPanel::queryClose" );
        //Bug 2777697: To be sure that all page modifications are closed
        closeAllTabs();
        //Bug 2777697:

        bool output=queryFileClose();

        if ( output && kSystemTrayIcon ) {
                kSystemTrayIcon->hide();
                delete kSystemTrayIcon;
                kSystemTrayIcon=NULL;
        }

        return output;
}

bool SKGMainPanel::queryFileClose()
{
        SKGTRACEIN ( 1, "SKGMainPanel::queryFileClose" );
        bool output=true;
        if ( currentBankDocument->getCurrentTransaction() ) {
                QApplication::setOverrideCursor ( QCursor ( Qt::ArrowCursor ) );
                KMessageBox::information ( this,i18n ( "Application can be closed until an operation is running." ) );
                QApplication::restoreOverrideCursor();
                output=false;
        } else if ( currentBankDocument->isFileModified() && currentBankDocument->getDatabaseMode() !=SKGDocument::DirectAccess ) {
                QApplication::setOverrideCursor ( QCursor ( Qt::ArrowCursor ) );
                int code=KMessageBox::Cancel;
                QString fileName=currentBankDocument->getCurrentFileName();
                KAction* save=getGlobalAction ( fileName.isEmpty() ? "file_save_as" : "file_save" );
                if ( save ) {
                        code=KMessageBox::questionYesNoCancel ( this, i18n ( "The document has been modified.\nDo you want to save it before closing?" ),
                                                                QString(),
                                                                KGuiItem ( fileName.isEmpty() ?  i18n ( "Save as" ) : i18n ( "Save" ),
                                                                           KIcon ( fileName.isEmpty() ?  "document-save-as" : "document-save" ) ),
                                                                KGuiItem ( i18n ( "Do not save" ) )
                                                              );
                        if ( code==KMessageBox::Yes ) save->trigger();
                        output= ( code==KMessageBox::No || code==KMessageBox::Yes );
                } else {
                        code=KMessageBox::questionYesNo ( this,i18n ( "Current modifications will not be saved.\nDo you want to continue ?" ) );
                        output= ( code==KMessageBox::Yes );
                }
                QApplication::restoreOverrideCursor();
        }

        return output;
}

SKGObjectBase::SKGListSKGObjectBase SKGMainPanel::getSelectedObjects()
{
        SKGObjectBase::SKGListSKGObjectBase selection;

        SKGTabWidget* currentPage= ( SKGTabWidget* ) tabView->currentWidget();
        if ( currentPage ) {
                selection=currentPage->getSelectedObjects();
        }
        return selection;
}

int SKGMainPanel::getNbSelectedObjects()
{
        int nb=0;

        SKGTabWidget* currentPage= ( SKGTabWidget* ) tabView->currentWidget();
        if ( currentPage ) {
                nb=currentPage->getNbSelectedObjects();
        }
        return nb;
}

void SKGMainPanel::optionsPreferences()
{
        SKGTRACEIN ( 1, "SKGMainPanel::optionsPreferences" );
        //Synchronize setting with confirmation panel
        KMessageBox::ButtonCode confirm;
        int ask=KMessageBox::shouldBeShownYesNo ( "updateBookmarkOnClose", confirm );

        KSharedConfigPtr config=KSharedConfig::openConfig ( "skroogerc" );
        KConfigGroup pref=config->group ( "skrooge" );
        if ( ask ) {
                pref.writeEntry ( "update_modified_bookmarks", 0 );
        } else if ( confirm==KMessageBox::Yes ) {
                pref.writeEntry ( "update_modified_bookmarks", 1 );
        } else {
                pref.writeEntry ( "update_modified_bookmarks", 2 );
        }

        skrooge::self()->readConfig();

        // The preference dialog is derived from prefs_base.ui
        //
        // compare the names of the widgets in the .ui file
        // to the names of the variables in the .kcfg file
        //avoid to have 2 dialogs shown
        if ( KConfigDialog::showDialog ( "settings" ) )  {
                return;
        }
        KConfigDialog *dialog = new KConfigDialog ( this, "settings", skrooge::self() );

        //Add main
        QWidget* widget=new QWidget();
        uipref.setupUi ( widget );
        dialog->addPage ( widget, skrooge::self(), i18n ( "General" ), "preferences-other" );

        //Add plugin in client in right order
        int nbplugin=pluginsList.count();
        for ( int j=0; j<nbplugin; ++j ) {
                SKGInterfacePlugin *pluginInterface=getPluginByIndex ( j );
                if ( pluginInterface ) {
                        QWidget* w=pluginInterface->getPreferenceWidget();
                        if ( w ) {
                                dialog->addPage ( w, pluginInterface->getPreferenceSkeleton(), pluginInterface->title(), pluginInterface->icon() );
                        }
                }
        }
        connect ( dialog, SIGNAL ( settingsChanged ( const QString& ) ), this, SLOT ( onSettingsChanged() ) );

        dialog->setAttribute ( Qt::WA_DeleteOnClose );
        dialog->show();

        //Refresh
        refresh();
}

void SKGMainPanel::onSettingsChanged()
{
        SKGError err;
        SKGTRACEINRC ( 1, "SKGMainPanel::onSettingsChanged", err );
        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
        {
                int nb=pluginsList.count();
                SKGBEGINPROGRESSTRANSACTION ( *currentBankDocument, i18n ( "Save settings" ), err, nb );

                //Refresh plugins
                for ( int i = 0; err.isSucceeded() && i < nb; ++i ) {
                        err=getPluginByIndex ( i )->savePreferences();
                        if ( err.isSucceeded() ) err=currentBankDocument->stepForward ( i+1 );
                }

                //Setting for icon
                if ( iconInSystemtray() ) kSystemTrayIcon->show();
                else kSystemTrayIcon->hide();

                //
                KSharedConfigPtr config=KSharedConfig::openConfig ( "skroogerc" );
                KConfigGroup pref=config->group ( "skrooge" );
                int option=pref.readEntry ( "update_modified_bookmarks", 0 );

                KConfigGroup prefNotificationMessages=config->group ( "Notification Messages" );
                if ( option==0 ) {
                        //ASK: remove following setting
                        //[Notification Messages]
                        //updateBookmarkOnClose=false
                        prefNotificationMessages.deleteEntry ( "updateBookmarkOnClose" );
                } else if ( option==1 ) {
                        //ALWAYS: set following setting
                        //[Notification Messages]
                        //updateBookmarkOnClose=true
                        prefNotificationMessages.writeEntry ( "updateBookmarkOnClose", true );
                } else {
                        //NEVER: set following setting
                        //[Notification Messages]
                        //updateBookmarkOnClose=false
                        prefNotificationMessages.writeEntry ( "updateBookmarkOnClose", false );
                }
        }
        QApplication::restoreOverrideCursor();

        //Display error
        displayErrorMessage ( err );
}

void SKGMainPanel::refresh()
{
        SKGTRACEIN ( 1, "SKGMainPanel::refresh" );

        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );

        //Refresh plugins
        int nb=pluginsList.count();
        for ( int i = 0; i < nb; ++i ) {
                getPluginByIndex ( i )->refresh();
        }

        //Enable addTabeAction
        bool atLeastOnePageOpened= ( tabView->count() >0 );
        addTabAction->setEnabled ( atLeastOnePageOpened );
        closePageAction->setEnabled ( atLeastOnePageOpened );
        closeAllPagesAction->setEnabled ( atLeastOnePageOpened );
        closeAllOtherPagesAction->setEnabled ( tabView->count() >1 );

        SKGTabWidget* toSave= (SKGTabWidget*) tabView->currentWidget();
        saveDefaultStateAction->setEnabled(toSave && !(toSave->getDefaultStateAttibute().isEmpty()));

        //Set window title
        QString modified;
        if ( currentBankDocument->isFileModified() ) modified=i18n ( " [modified]" );
        QString fileName=currentBankDocument->getCurrentFileName();
        if ( fileName.isEmpty() ) fileName=i18n ( "Untitled" );
        setWindowTitle ( i18n ( "%1%2 - %3", fileName, modified, KGlobal::activeComponent ().componentName () ) );

        QApplication::restoreOverrideCursor();
}

SKGTabWidget* SKGMainPanel::setNewTabContent ( SKGInterfacePlugin* plugin, int index, const QString& parameters, const QString& title )
{
        SKGTRACEIN ( 1, "SKGMainPanel::setNewTabContent" );
        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
        SKGTabWidget* w=NULL;

        if ( index!=-1 ) {
                int currentIndex=tabView->currentIndex();
                if ( currentIndex>=0 ) {
                        SKGTabWidget* currentPage= ( SKGTabWidget* ) tabView->currentWidget();
                        tabView->removeTab ( currentIndex );
                        if ( currentPage ) closeTab ( currentPage );
                }
        }

        if ( plugin ) {
                w=plugin->getWidget();
                if ( w ) {
                        //Title
                        QString title2= ( title.isEmpty() ? plugin->title() : title );
                        w->setObjectName ( plugin->objectName() );

                        QString param=parameters;
                        if (param.isEmpty()) {
                                QString def=w->getDefaultStateAttibute();
                                if (!def.isEmpty()) {
                                        param=currentBankDocument->getParameter(def);
                                }
                        }
                        w->setState ( param );
                        connect ( w, SIGNAL ( selectionChanged() ), SLOT ( refresh() ) );

                        if ( index==-1 ) {
                                tabView->addTab ( w, KIcon ( plugin->icon() ), title2 );
                                tabView->setCurrentWidget ( w );

                        } else {
                                tabView->insertTab ( index, w, KIcon ( plugin->icon() ), title2 );
                                tabView->setCurrentWidget ( w );

                                refresh();
                        }
                        SKGTRACEL ( 1 ) << "opening plugin [" << plugin->objectName () << ']' << endl;
                }
        } else {
                displayErrorMessage ( SKGError ( 10, i18n ( "Impossible to open the page because the plugin was not found" ) ) );
        }
        QApplication::restoreOverrideCursor();
        return w;
}

int SKGMainPanel::currentTabIndex() const
{
        return tabView->currentIndex();
}

SKGTabWidget* SKGMainPanel::currentTab() const
{
        return ( SKGTabWidget* ) tabView->currentWidget();
}

KSplashScreen* SKGMainPanel::splashScreen() const
{
        return spScreen;
}

int SKGMainPanel::countTab() const
{
        return tabView->count();
}

void SKGMainPanel::setCurrentTabIndex ( int iIndex )
{
        tabView->setCurrentIndex ( iIndex );
}

void SKGMainPanel::onBeforeOpenContext ()
{
        middleClick=QApplication::keyboardModifiers() &Qt::ControlModifier || QApplication::mouseButtons () &Qt::MidButton;
}

void SKGMainPanel::onOpenContext ()
{
        SKGTRACEIN ( 1, "SKGMainPanel::onOpenContext" );
        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
        int index=-1;
        QAction* sender=dynamic_cast<QAction*> ( this->sender() );
        if (sender) {
                index=sender->data().toInt();
        } else {
                QList<QListWidgetItem *> selection=ui.kContextList->selectedItems ();
                if (selection.count()>0) {
                        index=selection.at(0)->data(12).toInt();
                }
        }
        if (index!=-1) setNewTabContent ( getPluginByIndex ( index ), middleClick ? -1 : tabView->currentIndex() );
        QApplication::restoreOverrideCursor();
}

void SKGMainPanel::closeTab ( QWidget* iWidget )
{
        SKGTRACEIN ( 1, "SKGMainPanel::closeTab" );
        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
        QWidget* toRemove=iWidget;
        if ( toRemove==NULL )  toRemove=tabView->currentWidget();
        if ( toRemove ) {
                ( ( SKGTabWidget* ) toRemove )->close();
                delete toRemove;
        }
        QApplication::restoreOverrideCursor();
}

void SKGMainPanel::closeAllTabs()
{
        SKGTRACEIN ( 1, "SKGMainPanel::closeAllTabs" );
        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
        int nb=tabView->count();
        for ( int i=nb-1; i>=0; --i ) {
                closeTab ( tabView->widget ( i ) );
        }
        QApplication::restoreOverrideCursor();
}

void SKGMainPanel::saveDefaultState()
{
        SKGTRACEIN ( 1, "SKGMainPanel::saveDefaultState" );
        SKGError err;

        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
        SKGTabWidget* toSave= (SKGTabWidget*) tabView->currentWidget();
        if (toSave) {
                QString name=toSave->getDefaultStateAttibute();
                if (!name.isEmpty()) {
                        SKGBEGINTRANSACTION ( *currentBankDocument, i18n ( "Save default parameters" ), err );
                        err=currentBankDocument->setParameter ( name, toSave->getState() );
                }
        }
        QApplication::restoreOverrideCursor();

        //status bar
        if (err.isSucceeded()) err=SKGError(0, i18n("Defaut state saved"));
        displayErrorMessage ( err );
}

void SKGMainPanel::closeAllOtherTabs ( QWidget* iWidget )
{
        SKGTRACEIN ( 1, "SKGMainPanel::closeAllOtherTabs" );
        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
        QWidget* toKeep=iWidget;
        if ( toKeep==NULL )  toKeep=tabView->currentWidget();

        int nb=tabView->count();
        for ( int i=nb-1; i>=0; --i ) {
                QWidget* w=tabView->widget ( i );
                if ( w!=toKeep ) closeTab ( w );
        }
        QApplication::restoreOverrideCursor();
}

void SKGMainPanel::addTab()
{
        SKGTRACEIN ( 1, "SKGMainPanel::addTab" );
        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
        SKGTabWidget* currentPage= ( SKGTabWidget* ) tabView->currentWidget();
        if ( currentPage ) {
                setNewTabContent ( getPluginByName ( currentPage->objectName() ) );
        }
        QApplication::restoreOverrideCursor();
}

bool SKGMainPanel::eventFilter ( QObject *object, QEvent *event )
{
        if ( event && object ) {
                QResizeEvent *resizeEvent = dynamic_cast<QResizeEvent*> ( event );
                if ( resizeEvent ) {
                        QSize newSize=resizeEvent->size ();

                        //Compute icon size
                        int s=qMax ( qMin ( newSize.width() /5,64 ),16 );
                        ui.kContextList->setIconSize ( QSize ( s, s ) );
                }
        }
        return false;
}

void SKGMainPanel::onTipOfDay()
{
        SKGTRACEIN ( 1, "SKGMainPanel::onTipOfDay" );
        if ( tipDatabase ) {
                //Tip of day is available
                //Creation of the manu
                KTipDialog* tipsDialog=new KTipDialog ( tipDatabase, this );
                tipsDialog->setAttribute ( Qt::WA_DeleteOnClose );
                tipsDialog->show();
        }
}

void SKGMainPanel::onTrayActivated ( KSystemTrayIcon::ActivationReason reason )
{
        SKGTRACEIN ( 1, "SKGMainPanel::onTrayActivated" );
        if ( reason==KSystemTrayIcon::Trigger ) {
                setVisible ( !isVisible() );
        }
}

void SKGMainPanel::notify ( int iTransaction )
{
        SKGTRACEIN ( 1, "SKGMainPanel::notify" );
        SKGTRACEL ( 1 ) << "iTransaction=" << iTransaction << endl;

        //Notify
        if ( iTransaction ) {
                SKGObjectBase transaction ( currentBankDocument, "doctransaction", iTransaction );
                if ( transaction.getAttribute ( "t_mode" ) !="R" ) {
                        QStringList msg;
                        currentBankDocument->getMessages ( iTransaction, msg, false );
                        int nbMessages=msg.count();
                        if ( nbMessages ) {
                                //Build message
                                QString message;
                                for ( int i=0; i<nbMessages; ++i ) {
                                        if ( i!=0 ) message+="<br>";
                                        message+=msg.at ( i );
                                }

                                KNotification *notify = new KNotification ( "skrooge_info_event" , this );
                                notify->setText ( message );
                                notify->sendEvent();
                        }
                }
        }
}

void SKGMainPanel::changeEvent ( QEvent * e )
{
        KParts::MainWindow::changeEvent ( e );
}

bool SKGMainPanel::iconInSystemtray()
{
        KSharedConfigPtr config=KSharedConfig::openConfig ( "skroogerc" );
        KConfigGroup pref=config->group ( "skrooge" );
        return pref.readEntry ( "icon_in_system_tray",false );
}

QLabel* SKGMainPanel::statusNormalMessage() const
{
        return kNormalMessage;
}

void SKGMainPanel::displayErrorMessage ( const SKGError& iError )
{
        SKGTRACEIN ( 1, "SKGMainPanel::displayErrorMessage" );
        QWidget* parent= ( SKGMainPanel* ) QApplication::activeWindow();

        if ( iError.isFailed() ) {
                QApplication::setOverrideCursor ( QCursor ( Qt::ArrowCursor ) );
                int rc=KMessageBox::warningContinueCancel ( parent, iError.getFullMessage(),
                                i18n ( "Warning" ),
                                KStandardGuiItem::cont(),
                                KGuiItem ( i18n ( "History" ), KIcon ( "dialog-information" ) ) );
                QApplication::restoreOverrideCursor();
                if ( rc==KMessageBox::Cancel ) {
                        KMessageBox::information ( parent, iError.getFullMessageWithHistorical(), i18n ( "History" ) );
                }
        } else {
                //Status bar
                SKGMainPanel *parent2 = qobject_cast<SKGMainPanel *> ( parent );
                if ( parent2 ) {
                        // parent2->statusBar()->showMessage ( iError.getMessage(), 3000 );
                        QLabel* label=parent2->statusNormalMessage();
                        QString message=iError.getMessage();
                        if (label && !message.isEmpty()) label->setText(message);
                }
        }
}

void SKGMainPanel::onCancelCurrentAction()
{
        SKGMainPanel::currentActionCanceled=true;
}

void SKGMainPanel::onQuitAction()
{
        //Bug 2777697: To be sure that all page modifications are closed
        closeAllTabs();
        //Bug 2777697:

        qApp->closeAllWindows();
}

void SKGMainPanel::onChangeStyleSheet ( const QString& iNewStryle )
{
        QFile file ( KStandardDirs::locate ( "data", QString::fromLatin1 ( "skrooge/theme/" ) +iNewStryle+".css" ) );
        if ( file.open ( QIODevice::ReadOnly | QIODevice::Text ) ) {
                QTextStream in ( &file );
                setStyleSheet ( defaultStyle+'\n'+in.readAll() );

                file.close();

                KSharedConfigPtr config=KSharedConfig::openConfig ( "skroogerc" );
                KConfigGroup pref=config->group ( "skrooge" );
                pref.writeEntry ( "last_theme", iNewStryle );
        } else {
                SKGTRACE << "WARNING: " << file.fileName() << " does not exits for language [" << KGlobal::locale()->language() << "]" << endl;
        }
}

int SKGMainPanel::progressBarCallBack ( int iPos, void* iProgressBar )
{
        QProgressBar* progressBar=NULL;
        QPushButton* button=NULL;
        doublePointer* pointers= ( doublePointer* ) iProgressBar;
        if ( pointers ) {
                progressBar= ( QProgressBar* ) pointers->p1;
                button= ( QPushButton* ) pointers->p2;
        }

        if ( progressBar ) progressBar->setValue ( iPos );
        if ( button ) button->setEnabled ( iPos>0 && iPos<100 );

        SKGMainPanel::currentActionCanceled=false;
        qApp->processEvents();
        return ( SKGMainPanel::currentActionCanceled ? 1 : 0 );
}

QString SKGMainPanel::getSaveFileName ( const KUrl & startDir, const QString & filter, QWidget * parent, const QString & caption )
{
        QString fileName=KFileDialog::getSaveFileName ( startDir, filter , parent, caption );
        if ( fileName.isEmpty() ) return "";
        QFile f ( fileName );
        if ( f.exists() && KMessageBox::warningContinueCancel ( parent,
                        i18n ( "File <b>%1</b> already exists. Do you really want to overwrite it?", fileName ),
                        i18n ( "Warning" ),
                        KGuiItem (i18n ( "Save" ), KIcon ( "document-save" ))) !=KMessageBox::Continue ) return "";

        return fileName;

}

void SKGMainPanel::fillComboWithDistinctValue (
        KComboBox* iComboBox,
        const SKGDocument* iDoc,
        const QString& iTable,
        const QString& iAttribut,
        const QString& iWhereClause )
{
        SKGTRACEIN ( 10, "SKGMainPanel::fillComboWithDistinctValue" );

        if ( iComboBox ) {
                //Get list
                QStringList list;
                SKGServices::getDistinctValues ( iDoc, iTable, iAttribut, iWhereClause, list );
                if ( list.count() && !list.at ( 0 ).isEmpty() ) list.insert ( 0, "" );

                //Fill combo
                iComboBox->clear();
                iComboBox->addItems ( list );

                //Fill completion
                KCompletion *comp = iComboBox->completionObject();
                comp->clear ();
                comp->insertItems ( list );
        }
}
#include "skgmainpanel.moc"
