/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin for operation management.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgoperationplugin.h"
#include "skgoperationpluginwidget.h"
#include "skgoperationboardwidget.h"
#include "skgtraces.h"
#include "skgoperationobject.h"
#include "skgsuboperationobject.h"
#include "skgtransactionmng.h"
#include "skgmainpanel.h"
#include "skrooge_operation.h"

#include <kactioncollection.h>
#include <kstandardaction.h>

#include <QDomDocument>

K_PLUGIN_FACTORY ( SKGOperationPluginFactory, registerPlugin<SKGOperationPlugin>(); )
K_EXPORT_PLUGIN ( SKGOperationPluginFactory ( "skrooge_operation", "skrooge_operation" ) )

SKGOperationPlugin::SKGOperationPlugin ( QObject* iParent, const QVariantList& /*iArg*/ ) : SKGInterfacePlugin ( iParent )
{
        SKGTRACEIN ( 10, "SKGOperationPlugin::SKGOperationPlugin" );
}

SKGOperationPlugin::~SKGOperationPlugin()
{
        SKGTRACEIN ( 10, "SKGOperationPlugin::~SKGOperationPlugin" );
        parent=NULL;
        currentBankDocument=NULL;

        duplicateAction=NULL;
        switchToPointedAction=NULL;
        switchBookmark=NULL;
        fastEditionAction=NULL;
        openBookmarks=NULL;
}

void SKGOperationPlugin::setupActions ( SKGMainPanel* iParent, SKGDocument* iDocument, const QStringList& iArgument )
{
        SKGTRACEIN ( 10, "SKGOperationPlugin::setupActions" );
        Q_UNUSED ( iArgument );

        currentBankDocument=iDocument;
        parent=iParent;

        setComponentData ( SKGOperationPluginFactory::componentData() );
        setXMLFile ( "skrooge_operation.rc" );

        //Menu
        duplicateAction = new KAction ( KIcon ( "skrooge_duplicate" ), i18n ( "&Duplicate" ), this );
        connect ( duplicateAction, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionDuplicate() ) );
        actionCollection()->addAction ( QLatin1String ( "edit_duplicate_operation" ), duplicateAction );
        duplicateAction->setShortcut ( Qt::CTRL+Qt::Key_D );

        if ( parent ) parent->registedGlobalAction ( "edit_duplicate_operation", duplicateAction );

        switchToPointedAction = new KAction ( KIcon ( "dialog-ok" ), i18n ( "&Point" ), this );
        connect ( switchToPointedAction, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionSwitchToPointed() ) );
        actionCollection()->addAction ( QLatin1String ( "edit_point_selected_operation" ), switchToPointedAction );
        switchToPointedAction->setShortcut ( Qt::CTRL+Qt::Key_R );

        if ( parent ) parent->registedGlobalAction ( "edit_point_selected_operation", switchToPointedAction );

        fastEditionAction = new KAction ( KIcon ( "games-solve" ), i18n ( "&Fast edit" ), this );
        actionCollection()->addAction ( QLatin1String ( "fast_edition" ), fastEditionAction );
        fastEditionAction->setEnabled ( false );
        fastEditionAction->setShortcut ( Qt::Key_F10 );

        if ( parent ) parent->registedGlobalAction ( "fast_edition", fastEditionAction );

        switchBookmark = new KAction ( KIcon ( "rating" ), i18n ( "Switch &bookmark" ), this );
        connect ( switchBookmark, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionSwitchBookmark() ) );
        actionCollection()->addAction ( QLatin1String ( "edit_switch_bookmark" ), switchBookmark );
        switchBookmark->setShortcut ( Qt::CTRL+Qt::Key_B );

        if ( parent ) parent->registedGlobalAction ( "edit_switch_bookmark", switchBookmark );

        openBookmarks = new KAction ( KIcon ( "rating" ), i18n ( "Open &bookmarks" ), this );
        connect ( openBookmarks, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionOpenBookmarks() ) );
        actionCollection()->addAction ( QLatin1String ( "view_open_bookmarks" ), openBookmarks );
        openBookmarks->setShortcut ( Qt::CTRL+Qt::SHIFT+Qt::Key_B );

        if ( parent ) parent->registedGlobalAction ( "edit_open_bookmarks", openBookmarks );
}

void SKGOperationPlugin::refresh()
{
        SKGTRACEIN ( 10, "SKGOperationPlugin::refresh" );
        if ( currentBankDocument && parent ) {
                bool test=(currentBankDocument->getDatabase()!=NULL);
                if ( openBookmarks ) openBookmarks->setEnabled ( test );

                SKGObjectBase::SKGListSKGObjectBase selection=parent->getSelectedObjects();
                if ( selection.count() >0 ) {
                        bool onOperation= ( selection.at ( 0 ).getRealTable() =="operation" );
                        if ( duplicateAction ) duplicateAction->setEnabled ( onOperation );
                        if ( switchBookmark ) switchBookmark->setEnabled ( onOperation );
                        if ( switchToPointedAction ) switchToPointedAction->setEnabled ( onOperation );
                } else {
                        if ( duplicateAction ) duplicateAction->setEnabled ( false );
                        if ( switchBookmark ) switchBookmark->setEnabled ( false );
                        if ( switchToPointedAction ) switchToPointedAction->setEnabled ( false );
                }
        }
}

void SKGOperationPlugin::close()
{
        SKGTRACEIN ( 10, "SKGOperationPlugin::close" );
}

SKGWidget* SKGOperationPlugin::getDashboardWidget()
{
        return new SKGOperationBoardWidget(currentBankDocument);
}

SKGTabWidget* SKGOperationPlugin::getWidget()
{
        SKGTRACEIN ( 10, "SKGOperationPlugin::getWidget" );
        return new SKGOperationPluginWidget ( parent, ( SKGDocumentBank* ) currentBankDocument, fastEditionAction );
}

QWidget* SKGOperationPlugin::getPreferenceWidget()
{
        SKGTRACEIN ( 10, "SKGMonthlyPlugin::getPreferenceWidget" );
        QWidget* widget=new QWidget();
        ui.setupUi ( widget );
        return widget;
}

KConfigSkeleton* SKGOperationPlugin::getPreferenceSkeleton()
{
        return skrooge_operation::self();
}

SKGError SKGOperationPlugin::savePreferences() const
{
        return SKGError();
}

QString SKGOperationPlugin::title() const
{
        return i18n ( "Operations" );
}

QString SKGOperationPlugin::icon() const
{
        return "view-pim-tasks";
}

QString SKGOperationPlugin::statusTip () const
{
        return i18n ( "Operation management (creation, update, ...)" );
}

QString SKGOperationPlugin::toolTip () const
{
        return i18n ( "Operation management" );
}


int SKGOperationPlugin::getOrder() const
{
        return 15;
}

QStringList SKGOperationPlugin::tips() const
{
        QStringList output;
        output.push_back ( i18n ( "<p>... you can press <strong>+</strong>, <strong>-</strong>, <strong>CTRL +</strong> or <strong>CTRL -</strong> to quickly change dates.</p>" ) );
        output.push_back ( i18n ( "<p>... you can update many operations in one shot.</p>" ) );
        output.push_back ( i18n ( "<p>... you can double click on an operation to show or edit sub operations.</p>" ) );
        output.push_back ( i18n ( "<p>... you can duplicate an operation including complex operations (split, grouped, ...).</p>" ) );
        return output;
}

bool SKGOperationPlugin::isInContext() const
{
        return true;
}

void SKGOperationPlugin::actionSwitchToPointed()
{
        SKGError err;
        SKGTRACEINRC ( 10, "SKGOperationPlugin::actionSwitchToPointed",err );
        //Get Selection
        if ( parent && currentBankDocument ) {
                SKGObjectBase::SKGListSKGObjectBase selection=parent->getSelectedObjects();
                int nb=selection.count();
                {
                        SKGBEGINPROGRESSTRANSACTION ( *currentBankDocument, i18n ( "Switch to pointed" ), err, nb );

                        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
                        for ( int i=0; err.isSucceeded() && i<nb; ++i ) {
                                SKGOperationObject operationObj=selection.at ( i );
                                if ( err.isSucceeded() ) err=operationObj.setStatus ( SKGOperationObject::POINTED );
                                if ( err.isSucceeded() ) err=operationObj.save();

                                if ( err.isSucceeded() ) err=currentBankDocument->stepForward ( i+1 );
                        }
                        QApplication::restoreOverrideCursor();
                }

                //status bar
                if ( err.isSucceeded() ) err=SKGError ( 0, i18n ( "Operation pointed." ) );
                else err.addError ( ERR_FAIL, i18n ( "Switch failed" ) );

                //Display error
                parent->displayErrorMessage ( err );
        }
}

void SKGOperationPlugin::actionSwitchBookmark()
{
        SKGError err;
        SKGTRACEINRC ( 10, "SKGOperationPlugin::actionSwitchBookmark",err );
        //Get Selection
        if ( parent && currentBankDocument ) {
                SKGObjectBase::SKGListSKGObjectBase selection=parent->getSelectedObjects();
                int nb=selection.count();
                {
                        SKGBEGINPROGRESSTRANSACTION ( *currentBankDocument, i18n ( "Operation bookmark" ), err, nb );

                        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
                        for ( int i=0; err.isSucceeded() && i<nb; ++i ) {
                                SKGOperationObject operationObj=selection.at ( i );
                                if ( err.isSucceeded() ) err=operationObj.bookmark ( !operationObj.isBookmarked() );
                                if ( err.isSucceeded() ) err=operationObj.save();

                                if ( err.isSucceeded() ) err=currentBankDocument->stepForward ( i+1 );
                        }
                        QApplication::restoreOverrideCursor();
                }

                //status bar
                if ( err.isSucceeded() )  err=SKGError ( 0, i18n ( "Operation bookmarked." ) );
                else err.addError ( ERR_FAIL, i18n ( "Operation bookmark failed" ) );

                //Display error
                parent->displayErrorMessage ( err );
        }
}

void SKGOperationPlugin::actionDuplicate()
{
        SKGError err;
        SKGTRACEINRC ( 10, "SKGOperationPlugin::actionDuplicate",err );
        //Get Selection
        if ( parent && currentBankDocument ) {
                QStringList listUUID;
                SKGObjectBase::SKGListSKGObjectBase selection=parent->getSelectedObjects();
                int nb=selection.count();
                {
                        SKGBEGINPROGRESSTRANSACTION ( *currentBankDocument, i18n ( "Duplicate operation" ), err, nb );

                        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
                        for ( int i=0; err.isSucceeded() && i<nb; ++i ) {
                                SKGOperationObject operationObj=selection.at ( i );
                                SKGOperationObject dup;
                                if ( err.isSucceeded() ) err=operationObj.duplicate ( dup );
                                if ( err.isSucceeded() ) err=currentBankDocument->stepForward ( i+1 );

                                listUUID.push_back ( dup.getUniqueID() );
                        }
                        QApplication::restoreOverrideCursor();
                }

                //status bar
                if ( err.isSucceeded() ) {
                        err=SKGError ( 0, i18n ( "Operation duplicated." ) );
                        //TODO ui.kOperationView->selectObjects(listUUID, true);
                } else err.addError ( ERR_FAIL, i18n ( "Duplicate operation failed" ) );

                //Display error
                parent->displayErrorMessage ( err );
        }
}

void SKGOperationPlugin::actionOpenBookmarks()
{
        SKGError err;
        SKGTRACEINRC ( 10, "SKGOperationPlugin::actionOpenBookmarks",err );
        QString wc="t_bookmarked='Y'";
        QString title=i18n ( "Operations bookmarked" );

        //Call operation plugin
        QDomDocument doc ( "SKGML" );
        QDomElement root = doc.createElement ( "parameters" );
        doc.appendChild ( root );
        root.setAttribute ( "operationTable", "v_operation_consolidated" );
        root.setAttribute ( "operationWhereClause", wc );
        root.setAttribute ( "title", title );
        root.setAttribute ( "title_icon", "view-statistics" );

        if ( parent ) parent->setNewTabContent ( parent->getPluginByName ( "Skrooge operation plugin" ), -1, doc.toString() );
}

#include "skgoperationplugin.moc"
