/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
#ifndef SKGIMPORTEXPORTMANAGER_H
#define SKGIMPORTEXPORTMANAGER_H
/** @file
 * This file defines classes SKGImportExportManager.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include <QObject>
#include <QStringList>
#include <QMap>

#include "skgerror.h"
#include "skgbankmodeler_export.h"

class SKGDocumentBank;
class SKGAccountObject;
class SKGUnitObject;
class QDate;

/**
 *Manage import and export
 */
class SKGBANKMODELER_EXPORT SKGImportExportManager : public QObject
{
    Q_OBJECT;
    /**
     * This enumerate defines dump modes
     */
    Q_ENUMS(ImportExportMode);

public:
    /**
     * Constructor.
     * @param iDocument the document
     * @param iFileName the file name
     * @param iMode the mode
     */
    explicit SKGImportExportManager(SKGDocumentBank* iDocument,
                                    const QString& iFileName = "");

    /**
     * Destructor
     */
    virtual ~SKGImportExportManager();

    /**
     * Set the codec used for imports.
     * @param iCodec the codec name.
     */
    virtual inline void setCodec(const QString& iCodec) {
        m_codec = iCodec;
    }

    /**
     * Get the codec used for imports.
     * @return code.
     */
    virtual inline QString getCodec() const {
        return m_codec;
    }

    /**
     * Set the default account for import in case of account is not detected in imported file.
     * @param iAccount the account where to import. NULL if you want to create a specific account for that.
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError setDefaultAccount(SKGAccountObject* iAccount);

    /**
     * Set the default unit for import in case of unit is not detected in imported file.
     * @param iUnit the unit where to import. NULL if you want to create a specific unit for that.
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError setDefaultUnit(SKGUnitObject* iUnit);

    /**
     * Enable/disable the automatically validation of imported operations.
     * @param iValidation the mode.
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual void setAutomaticValidation(bool iValidation);
    /**
     * Enable/disable the automatically rules application on imported operations.
     * @param iApply the mode.
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual void setAutomaticApplyRules(bool iApply);

    /**
     * Set the CSV mapping rule.
     * A mapping rule is a list of regular expression allowing to automatically map each attribute:
     * the operation object.
     * List of supported attribute:
     * date
     * number
     * mode
     * payee
     * comment
     * status
     * bookmarked
     * account
     * category
     * amount
     * quantity
     * sign + debit
     * unit
     * This list is a list of operation attributes.
     * @param iCSVMappingRules the mapping. NULL to build automatically the CSV mapping.
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError setCSVMappingRules(const QMap<QString, QString>& iCSVMappingRules);

    /**
     * Set the CSV mapping rule.
     * A mapping rule is a list of regular expression allowing to automatically map each attribute:
     * the operation object.
     * List of supported attribute:
     * date
     * number
     * mode
     * payee
     * comment
     * status
     * bookmarked
     * account
     * category
     * amount
     * quantity
     * sign + debit
     * unit
     * This list is a list of operation attributes.
     * @return the mapping rules.
     */
    virtual  QMap<QString, QString> getCSVMappingRules();

    /**
     * Set the CSV mapping.
     * A mapping is ordered list to described the mapping between the csv file and
     * the operation object.
     * List of supported key word:
     * date
     * number
     * mode
     * payee
     * comment
     * status
     * bookmarked
     * account
     * category
     * amount
     * quantity
     * sign
     * unit
     * This list is a list of operation attributes.
     * @param iCSVMapping the mapping. NULL to build automatically the CSV mapping.
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError setCSVMapping(const QStringList* iCSVMapping = NULL);

    /**
     * Get the CSV mapping.
     * @return the CSV mapping
     */
    virtual QStringList getCSVMapping() const;

    /**
     * Set the index of the header in the CSV file.
     * @param iIndex the index. -1 to search automatically the index of the header.
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError setCSVHeaderIndex(int iIndex = -1);

    /**
     * Get the index of the header in the CSV file.
     * @return the index
     */
    virtual int getCSVHeaderIndex();

    /**
     * Get the CSV separator
     * @return the separator
     */
    virtual QChar getCSVSeparator();

    /**
     * Get the mime type filter for import
     * @return the mime type filter
     */
    static QString getImportMimeTypeFilter();

    /**
     * Import the file in the document
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError importFile();

    /**
     * Get the mime type filter for export
     * @return the mime type filter
     */
    static QString getExportMimeTypeFilter();

    /**
     * Export the file in the document
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError exportFile();

    /**
     * Anonymize the document.
     * This function must not be launched into a transaction
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError anonymize();

    /**
     * Find and group operations
     * @param oNbOperationsMerged returns the number of operations merged.
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError findAndGroupTransfers(int& oNbOperationsMerged);

    /**
     * Clean operations after an import coming from bank's web sites
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError cleanBankImport();

    /**
     * Get the document
     * @return the document.
     */
    virtual inline SKGDocumentBank* getDocument() {
        return m_document;
    }
    /**
     * Get file name
     * @return the file name.
     */
    virtual inline QString getFileName() {
        return m_fileName;
    }
    /**
     * Return the default account for import
     * @param oAccount the default account for import.
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError getDefaultAccount(SKGAccountObject& oAccount);

    /**
     * Return the preferred unit for a date for import
     * @param oUnit the default unit for import.
     * @param iDate the date.
     * @brief
     * If @see setDefaultUnit is used then getDefaultUnit will return this unit.
     * else return the unit compatible with entry date and with a value nearest than 1
     * else a new unit is created and returned
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError getDefaultUnit(SKGUnitObject& oUnit, const QDate* iDate = NULL);

    /**
     * Import units
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError importCSVUnit();

private:
    Q_DISABLE_COPY(SKGImportExportManager);


    SKGError finalizedImportedOperations();

    QStringList getCSVMappingFromLine(const QString& iLine);

    SKGDocumentBank* m_document;
    QString m_fileName;
    QStringList m_csvMapping;
    QMap<QString, QString>  m_csvMapper;
    QChar m_csvSeparator;
    int m_csvHeaderIndex;
    SKGAccountObject* m_defaultAccount;
    SKGUnitObject* m_defaultUnit;
    QString m_codec;
    bool m_automaticValidationOfImportedOperation;
    bool m_automaticApplyRulesOfImportedOperation;
};

#endif
