/****************************************************************/
/* Memory unit                                                  */
/* (c) Christophe CALMEJANE (Ze KiLleR) - 1999-03               */
/****************************************************************/

/*
    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include "skyutils.h"
#include <time.h>

#define SU_DEFAULT_MALLOC_CHECK 2
#define SU_MALLOC_KEY 0x5c
#define SU_MALLOC_KEY2 0xa7
#define SU_MALLOC_BOUND_VALUE 0x66AA55CC
#define SU_MALLOC_REUSE_VALUE 0x11CC77BB
#define SU_MALLOC_REUSE_SIZE 64

#ifdef _REENTRANT
SU_SEM_HANDLE SU_alloc_trace_sem; /* Semaphore to protect the use of SU_alloc_trace_list in MT environment */
#endif /* _REENTRANT */
bool SU_sem_init=false;
int SU_env_check = SU_DEFAULT_MALLOC_CHECK;
int SU_env_trace = 0;
int SU_env_print = 0;

SU_PList SU_alloc_trace_list = NULL; /* SU_PAlloc */

typedef struct
{
  void *ptr;
  long int size;
  long int time;
  char file[512];
  int line;
  bool freed;
} SU_TAlloc, *SU_PAlloc;

#undef malloc
#undef calloc
#undef realloc
#undef strdup
#undef free

SU_PList SU_AddElementHead_no_trace(SU_PList List,void *Elem)
{
  SU_PList El;

  El = (SU_PList) malloc(sizeof(SU_TList));
  El->Next = List;
  El->Data = Elem;
  return El;
}

SU_PList SU_DelElementHead_no_trace(SU_PList List)
{
  SU_PList Ptr;

  if(List == NULL)
    return NULL;
  Ptr = List->Next;
  free(List);
  return Ptr;
}

/* MEMORY ALIGNEMENT FUNCTIONS */
void *SU_malloc(long int size)
{
  unsigned char pad;
  void *memblock,*retblock;

  memblock = malloc(size+SU_MALLOC_ALIGN_SIZE);
  if(memblock == NULL)
    return NULL;
  pad = ((int)memblock)%SU_MALLOC_ALIGN_SIZE;
  if(pad == 0)
    pad = SU_MALLOC_ALIGN_SIZE;
  retblock = (unsigned char *)memblock+pad;
  *((unsigned char *)retblock-1) = pad;
  *((unsigned char *)retblock-2) = SU_MALLOC_KEY;
  return retblock;
}

void *SU_calloc(long int nbelem,long int size)
{
  void *ptr;

  ptr = SU_malloc(nbelem*size);
  if(ptr == NULL)
    return NULL;
  memset(ptr,0,nbelem*size);
  return ptr;
}

void SU_free(void *memblock)
{
  unsigned char pad;
  if(*((unsigned char *)memblock-2) == SU_MALLOC_KEY2)
  {
    printf("SkyUtils_SU_free Warning : bloc already freed\n");
    return;
  }
  else if(*((unsigned char *)memblock-2) != SU_MALLOC_KEY)
  {
    printf("SkyUtils_SU_free Warning : bloc might have been underwritten\n");
    return;
  }
  *((unsigned char *)memblock-2) = SU_MALLOC_KEY2;
  pad = *((unsigned char *)memblock-1);
  free((unsigned char *)memblock-pad);
}

/* TRACE DEBUG FUNCTIONS */
void SU_printf_trace_debug(char *func,char *Str,void *memblock,char *file,int line,char *file2,int line2)
{
  if(SU_env_check > 0)
  {
    if(file2 == NULL)
      printf("SkyUtils_%s Warning : bloc %p %s (%s:%d)\n",func,memblock,Str,file,line);
    else
      printf("SkyUtils_%s Warning : bloc %p %s %s:%d (%s:%d)\n",func,memblock,Str,file,line,file2,line2);
  }
  if(SU_env_check == 2)
    abort();
}

void *SU_malloc_trace(long int size,char *file,int line)
{
  SU_PAlloc Al = NULL;
  void *ptr;
  SU_PList Ptr;
//#ifdef __unix__
  char *s;
//#endif /* __unix__ */

  ptr = malloc(size+16); /* 8 before, 8 after */
  if(ptr == NULL)
  {
    printf("SkyUtils_SU_malloc_trace Warning : malloc returned NULL\n");
    return NULL;
  }
  *((unsigned long int *)((char *)ptr+4)) = SU_MALLOC_BOUND_VALUE;
  *((unsigned long int *)((char *)ptr+size+8)) = SU_MALLOC_BOUND_VALUE;
  if(!SU_sem_init)
  {
#ifdef _REENTRANT
    if(!SU_CreateSem(&SU_alloc_trace_sem,1,1,"SU_alloc_trace_sem"))
      printf("SkyUtils_SU_malloc_trace Warning : Couldn't allocate semaphore\n");
#endif /* _REENTRANT */
    SU_sem_init = true;
//#ifdef __unix__
    s = getenv("MALLOC_CHECK_");
    SU_env_check = (s==NULL)?SU_DEFAULT_MALLOC_CHECK:atoi(s);
    s = getenv("SU_MALLOC_TRACE");
    SU_env_trace = (s==NULL)?0:atoi(s);
    s = getenv("SU_MALLOC_PRINT");
    SU_env_print = (s==NULL)?0:atoi(s);
//#endif /* __unix__ */
    printf("SkyUtils Information : Using SU_MALLOC_TRACE hooks : MALLOC_CHECK_=%d SU_MALLOC_TRACE=%d SU_MALLOC_PRINT=%d\n",SU_env_check,SU_env_trace,SU_env_print);
  }
#ifdef _REENTRANT
  SU_SEM_WAIT(SU_alloc_trace_sem);
#endif /* _REENTRANT */
  Ptr = SU_alloc_trace_list;
  while(Ptr != NULL)
  {
    Al = (SU_PAlloc)Ptr->Data;
    if(Al->ptr == ptr)
      break;
    Ptr = Ptr->Next;
  }
  if(Ptr == NULL)
  {
    Al = (SU_PAlloc) malloc(sizeof(SU_TAlloc));
    if(Al == NULL)
    {
      free(ptr);
      return NULL;
    }
    SU_alloc_trace_list = SU_AddElementHead_no_trace(SU_alloc_trace_list,Al);
  }
  Al->ptr = ptr;
  Al->size = size;
  Al->time = time(NULL);
  SU_strcpy(Al->file,file,sizeof(Al->file));
  Al->line = line;
  Al->freed = false;
  if(SU_env_print)
    printf("SU_malloc_trace Information : Allocating bloc %p (%ld bytes) in pid %d (%s:%d)\n",(char *)ptr+8,size,getpid(),file,line);
#ifdef _REENTRANT
  SU_SEM_POST(SU_alloc_trace_sem);
#endif /* _REENTRANT */
  return (void *)((char *)ptr+8);
}

void *SU_calloc_trace(long int nbelem,long int size,char *file,int line)
{
  void *ptr;

  ptr = SU_malloc_trace(nbelem*size,file,line);
  if(ptr == NULL)
    return NULL;
  memset(ptr,0,nbelem*size);
  return ptr;
}

void *SU_realloc_trace(void *memblock,long int size,char *file,int line)
{
  SU_PList Ptr;
  void *new_ptr;

  if(memblock == NULL) /* If memblock is NULL -> malloc */
    return SU_malloc_trace(size,file,line);
#ifdef _REENTRANT
  SU_SEM_WAIT(SU_alloc_trace_sem);
#endif /* _REENTRANT */
  Ptr = SU_alloc_trace_list;
  while(Ptr != NULL)
  {
    if(((SU_PAlloc)Ptr->Data)->ptr == (void *)((char *)memblock-8))
      break;
    Ptr = Ptr->Next;
  }
#ifdef _REENTRANT
  SU_SEM_POST(SU_alloc_trace_sem);
#endif /* _REENTRANT */
  if(Ptr == NULL)
  {
    SU_printf_trace_debug("SU_realloc_trace","already freed, or never allocated",memblock,file,line,NULL,0);
    return NULL;
  }
  if(((SU_PAlloc)Ptr->Data)->freed)
  {
    SU_printf_trace_debug("SU_realloc_trace","was freed at",memblock,((SU_PAlloc)Ptr->Data)->file,((SU_PAlloc)Ptr->Data)->line,file,line);
    return NULL;
  }
  if(size == 0) /* If size is 0 -> free */
  {
    SU_free_trace(memblock,file,line);
    return NULL;
  }
  if(size > ((SU_PAlloc)Ptr->Data)->size)
  {
    new_ptr = SU_malloc_trace(size,file,line);
    if(new_ptr != NULL)
    {
      memcpy(new_ptr,memblock,((SU_PAlloc)Ptr->Data)->size);
      SU_free_trace(memblock,file,line);
    }
    return new_ptr;
  }
  else
  {
    SU_strcpy(((SU_PAlloc)Ptr->Data)->file,file,sizeof(((SU_PAlloc)Ptr->Data)->file));
    ((SU_PAlloc)Ptr->Data)->line = line;
    return memblock;
  }
}

char *SU_strdup_trace(const char *in,char *file,int line)
{
  char *s;
  long int len;

  len = strlen(in) + 1;
  s = (char *) SU_malloc_trace(len,file,line);
  if(s == NULL)
    return NULL;
  SU_strcpy(s,in,len);
  return s;
}

void SU_free_trace(void *memblock,char *file,int line)
{
  SU_PList Ptr,Ptr2;

  if(SU_env_print)
    printf("SU_free_trace Information : Freeing bloc %p in pid %d (%s:%d)\n",memblock,getpid(),file,line);
#ifdef _REENTRANT
  SU_SEM_WAIT(SU_alloc_trace_sem);
#endif /* _REENTRANT */
  Ptr = SU_alloc_trace_list;
  Ptr2 = NULL;
  while(Ptr != NULL)
  {
    if(((SU_PAlloc)Ptr->Data)->ptr == (void *)((char *)memblock-8))
      break;
    Ptr2 = Ptr;
    Ptr = Ptr->Next;
  }
  if(Ptr == NULL)
  {
    if(SU_env_trace)
      SU_printf_trace_debug("SU_free_trace","was never allocated",memblock,file,line,NULL,0);
    else
      SU_printf_trace_debug("SU_free_trace","already freed, or never allocated",memblock,file,line,NULL,0);
#ifdef _REENTRANT
    SU_SEM_POST(SU_alloc_trace_sem);
#endif /* _REENTRANT */
    return;
  }
  if(((SU_PAlloc)Ptr->Data)->freed)
  {
    SU_printf_trace_debug("SU_free_trace","was freed at",memblock,((SU_PAlloc)Ptr->Data)->file,((SU_PAlloc)Ptr->Data)->line,file,line);
#ifdef _REENTRANT
    SU_SEM_POST(SU_alloc_trace_sem);
#endif /* _REENTRANT */
    return;
  }
  if(*((unsigned long int *)((char *)((SU_PAlloc)Ptr->Data)->ptr+4)) != SU_MALLOC_BOUND_VALUE)
    SU_printf_trace_debug("SU_free_trace","might have been pre-written",memblock,file,line,NULL,0);
  if(*((unsigned long int *)((char *)((SU_PAlloc)Ptr->Data)->ptr+((SU_PAlloc)Ptr->Data)->size+8)) != SU_MALLOC_BOUND_VALUE)
    SU_printf_trace_debug("SU_free_trace","might have been post-written",memblock,file,line,NULL,0);

  if(SU_env_trace)
  {
    SU_PList Ptr2;

    if(((SU_PAlloc)Ptr->Data)->size <= SU_MALLOC_REUSE_SIZE)
    {
      unsigned int i;
      for(i=0;i<((SU_PAlloc)Ptr->Data)->size/sizeof(unsigned long int);i++)
        *((unsigned long int *)memblock+i) = SU_MALLOC_REUSE_VALUE;
    }
    else
      *((unsigned long int *)memblock) = SU_MALLOC_REUSE_VALUE;
    ((SU_PAlloc)Ptr->Data)->freed = true;
    SU_strcpy(((SU_PAlloc)Ptr->Data)->file,file,sizeof(((SU_PAlloc)Ptr->Data)->file));
    ((SU_PAlloc)Ptr->Data)->line = line;
    /* Check for reused blocs */
    Ptr2 = SU_alloc_trace_list;
    while(Ptr2 != NULL)
    {
      if(((SU_PAlloc)Ptr2->Data)->freed)
      {
        bool reused = false;
        if(((SU_PAlloc)Ptr2->Data)->size <= SU_MALLOC_REUSE_SIZE)
        {
          unsigned int i;
          for(i=0;i<((SU_PAlloc)Ptr2->Data)->size/sizeof(unsigned long int);i++)
          {
            reused = *((unsigned long int *)((char *)((SU_PAlloc)Ptr2->Data)->ptr+8)+i) != SU_MALLOC_REUSE_VALUE;
            if(reused)
              break;
          }
        }
        else
          reused = (*((unsigned long int *)((char *)((SU_PAlloc)Ptr2->Data)->ptr+8)) != SU_MALLOC_REUSE_VALUE);
        if(reused)
        {
          SU_printf_trace_debug("SU_free_trace","might have been reused",(char *)((SU_PAlloc)Ptr2->Data)->ptr+8,((SU_PAlloc)Ptr2->Data)->file,((SU_PAlloc)Ptr2->Data)->line,NULL,0);
        }
      }
      Ptr2 = Ptr2->Next;
    }
  }
  else
  {
    free(((SU_PAlloc)Ptr->Data)->ptr); /* Actually frees the bloc */
    free(Ptr->Data);
    if(Ptr2 == NULL)
      SU_alloc_trace_list = SU_DelElementHead_no_trace(SU_alloc_trace_list);
    else
      Ptr2->Next = SU_DelElementHead_no_trace(Ptr);
  }
#ifdef _REENTRANT
  SU_SEM_POST(SU_alloc_trace_sem);
#endif /* _REENTRANT */
}

void SU_alloc_trace_print(bool detail)
{
  SU_PList Ptr;
  int count = 0;

#ifdef _REENTRANT
  SU_SEM_WAIT(SU_alloc_trace_sem);
#endif /* _REENTRANT */
  Ptr = SU_alloc_trace_list;
  while(Ptr != NULL)
  {
    if(!((SU_PAlloc)Ptr->Data)->freed)
    {
      count++;
      if(detail)
        printf("SkyUtils_SU_alloc_trace_print : %ld %p %ld -> %s:%d\n",((SU_PAlloc)Ptr->Data)->time,((SU_PAlloc)Ptr->Data)->ptr,((SU_PAlloc)Ptr->Data)->size,((SU_PAlloc)Ptr->Data)->file,((SU_PAlloc)Ptr->Data)->line);
    }
    Ptr = Ptr->Next;
  }
  printf("SkyUtils_SU_alloc_trace_print : %d blocks\n",count);
#ifdef _REENTRANT
  SU_SEM_POST(SU_alloc_trace_sem);
#endif /* _REENTRANT */
}

void SU_check_memory(void)
{
  SU_PList Ptr;

#ifdef _REENTRANT
  SU_SEM_WAIT(SU_alloc_trace_sem);
#endif /* _REENTRANT */
  Ptr = SU_alloc_trace_list;
  while(Ptr != NULL)
  {
    if(*((unsigned long int *)((char *)((SU_PAlloc)Ptr->Data)->ptr+4)) != SU_MALLOC_BOUND_VALUE)
      SU_printf_trace_debug("SU_check_memory","might have been pre-written",(char *)((SU_PAlloc)Ptr->Data)->ptr+4,((SU_PAlloc)Ptr->Data)->file,((SU_PAlloc)Ptr->Data)->line,NULL,0);
    if(*((unsigned long int *)((char *)((SU_PAlloc)Ptr->Data)->ptr+((SU_PAlloc)Ptr->Data)->size+8)) != SU_MALLOC_BOUND_VALUE)
      SU_printf_trace_debug("SU_check_memory","might have been post-written",(char *)((SU_PAlloc)Ptr->Data)->ptr+4,((SU_PAlloc)Ptr->Data)->file,((SU_PAlloc)Ptr->Data)->line,NULL,0);

    if(!((SU_PAlloc)Ptr->Data)->freed)
    {
      bool reused = false;
      if(((SU_PAlloc)Ptr->Data)->size <= SU_MALLOC_REUSE_SIZE)
      {
        unsigned int i;
        for(i=0;i<((SU_PAlloc)Ptr->Data)->size/sizeof(unsigned long int);i++)
        {
          reused = *((unsigned long int *)((char *)((SU_PAlloc)Ptr->Data)->ptr+8)+i) != SU_MALLOC_REUSE_VALUE;
          if(reused)
            break;
        }
      }
      else
        reused = (*((unsigned long int *)((char *)((SU_PAlloc)Ptr->Data)->ptr+8)) != SU_MALLOC_REUSE_VALUE);
      if(reused)
        SU_printf_trace_debug("SU_check_memory","might have been reused",(char *)((SU_PAlloc)Ptr->Data)->ptr+8,((SU_PAlloc)Ptr->Data)->file,((SU_PAlloc)Ptr->Data)->line,NULL,0);
    }
    Ptr = Ptr->Next;
  }
#ifdef _REENTRANT
  SU_SEM_POST(SU_alloc_trace_sem);
#endif /* _REENTRANT */
}
