/****************************************************************/
/* Win32 registry functions                                     */
/* (c) Christophe CALMEJANE (Ze KiLleR) - 1999-03               */
/****************************************************************/

/*
    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include "skyutils.h"
static int SU_RB_LastError = SU_RB_ERR_SUCCESS;

#ifndef SU_TRACE_INTERNAL
#undef malloc
#undef calloc
#undef realloc
#undef strdup
#undef free
#endif /* !SU_TRACE_INTERNAL */

#ifdef _WIN32

#include <winreg.h>

HKEY SU_RB_OpenKeys(const char Key[],int Access)
{
  HKEY H,H2;
  LONG res;
  char *tmp,*p,*q;

  if(Key == NULL)
  {
    SU_RB_LastError = SU_RB_ERR_INVALID_KEY;
    return 0;
  }
  tmp = strdup(Key);
  p = SU_strparse(tmp,'\\');
  if(p == NULL)
  {
    SU_RB_LastError = SU_RB_ERR_INVALID_KEY;
    free(tmp);
    return 0;
  }
  if(strcmp(p,"HKEY_CLASSES_ROOT") == 0)
    H = HKEY_CLASSES_ROOT;
  else if(strcmp(p,"HKEY_CURRENT_USER") == 0)
    H = HKEY_CURRENT_USER;
  else if(strcmp(p,"HKEY_LOCAL_MACHINE") == 0)
    H = HKEY_LOCAL_MACHINE;
  else if(strcmp(p,"HKEY_USERS") == 0)
    H = HKEY_USERS;
  else
  {
    SU_RB_LastError = SU_RB_ERR_INVALID_KEY;
    free(tmp);
    return 0;
  }

  p = SU_strparse(NULL,'\\');
  if(p == NULL)
  {
    SU_RB_LastError = SU_RB_ERR_INVALID_KEY;
    free(tmp);
    return 0;
  }
  q = SU_strparse(NULL,'\\');
  while(q != NULL)
  {
    res = RegOpenKeyEx(H,p,0,KEY_READ | Access,&H2);
    RegCloseKey(H);
    H = H2;
    if(res != ERROR_SUCCESS)
    {
      free(tmp);
      if(res == ERROR_ACCESS_DENIED)
        SU_RB_LastError = SU_RB_ERR_ACCESS_DENIED;
      else
        SU_RB_LastError = SU_RB_ERR_INVALID_KEY;
      return 0;
    }
    p = q;
    q = SU_strparse(NULL,'\\');
  }
  free(tmp);
  SU_RB_LastError = SU_RB_ERR_SUCCESS;
  return H;
}

HKEY SU_RB_CreateKeys(const char Key[])
{
  HKEY H,H2;
  LONG res;
  DWORD Ret;
  char *tmp,*p,*q;

  if(Key == NULL)
  {
    SU_RB_LastError = SU_RB_ERR_INVALID_KEY;
    return 0;
  }
  tmp = strdup(Key);
  p = SU_strparse(tmp,'\\');
  if(p == NULL)
  {
    SU_RB_LastError = SU_RB_ERR_INVALID_KEY;
    free(tmp);
    return 0;
  }
  if(strcmp(p,"HKEY_CLASSES_ROOT") == 0)
    H = HKEY_CLASSES_ROOT;
  else if(strcmp(p,"HKEY_CURRENT_USER") == 0)
    H = HKEY_CURRENT_USER;
  else if(strcmp(p,"HKEY_LOCAL_MACHINE") == 0)
    H = HKEY_LOCAL_MACHINE;
  else if(strcmp(p,"HKEY_USERS") == 0)
    H = HKEY_USERS;
  else
  {
    SU_RB_LastError = SU_RB_ERR_INVALID_KEY;
    free(tmp);
    return 0;
  }

  p = SU_strparse(NULL,'\\');
  if(p == NULL)
  {
    SU_RB_LastError = SU_RB_ERR_INVALID_KEY;
    free(tmp);
    return 0;
  }
  q = SU_strparse(NULL,'\\');
  while(q != NULL)
  {
    res = RegCreateKeyEx(H,p,0,"",REG_OPTION_NON_VOLATILE,KEY_ALL_ACCESS,NULL,&H2,&Ret);
    RegCloseKey(H);
    H = H2;
    if(res != ERROR_SUCCESS)
    {
      free(tmp);
      if(res == ERROR_ACCESS_DENIED)
        SU_RB_LastError = SU_RB_ERR_ACCESS_DENIED;
      else
        SU_RB_LastError = SU_RB_ERR_INVALID_KEY;
      return 0;
    }
    p = q;
    q = SU_strparse(NULL,'\\');
  }
  free(tmp);
  SU_RB_LastError = SU_RB_ERR_SUCCESS;
  return H;
}

void SU_RB_GetStrValue(const char Key[],char *buf,int buf_len,const char Default[])
{
  HKEY Handle;
  char *p;
  DWORD VType;
  DWORD Size;

  SU_strcpy(buf,Default,buf_len);
  Handle = SU_RB_OpenKeys(Key,0);
  if(Handle == 0)
    return;
  p = strrchr(Key,'\\');
  if(p == NULL)
  {
    RegCloseKey(Handle);
    SU_RB_LastError = SU_RB_ERR_INVALID_KEY;
    return;
  }
  p++;
  Size = buf_len;
  RegQueryValueEx(Handle,p,NULL,&VType,buf,&Size);
  RegCloseKey(Handle);
}

int SU_RB_GetIntValue(const char Key[],int Default)
{
  HKEY Handle;
  char *p;
  int Value;
  DWORD VType;
  DWORD Size;
  long R;

  Handle = SU_RB_OpenKeys(Key,0);
  if(Handle == 0)
    return Default;
  p = strrchr(Key,'\\');
  if(p == NULL)
  {
    RegCloseKey(Handle);
    SU_RB_LastError = SU_RB_ERR_INVALID_KEY;
    return Default;
  }
  p++;
  Size = sizeof(Value);
  R = RegQueryValueEx(Handle,p,NULL,&VType,(BYTE *)&Value,&Size);
  RegCloseKey(Handle);
  if(R != ERROR_SUCCESS)
    return Default;
  return Value;
}

bool SU_RB_SetStrValue(const char Key[],const char Value[])
{
  HKEY Handle;
  char *p;
  long R;

  Handle = SU_RB_CreateKeys(Key);
  if(Handle == 0)
    return false;
  p = strrchr(Key,'\\');
  if(p == NULL)
  {
    RegCloseKey(Handle);
    SU_RB_LastError = SU_RB_ERR_INVALID_KEY;
    return false;
  }
  p++;
  R = RegSetValueEx(Handle,p,0,REG_SZ,Value,strlen(Value)+1);
  RegCloseKey(Handle);
  if(R == ERROR_SUCCESS)
  {
    SU_RB_LastError = SU_RB_ERR_SUCCESS;
    return true;
  }
  else
  {
    SU_RB_LastError = SU_RB_ERR_ACCESS_DENIED;
    return false;
  }
}

bool SU_RB_SetIntValue(const char Key[],int Value)
{
  HKEY Handle;
  char *p;
  long R;

  Handle = SU_RB_CreateKeys(Key);
  if(Handle == 0)
    return false;
  p = strrchr(Key,'\\');
  if(p == NULL)
  {
    RegCloseKey(Handle);
    SU_RB_LastError = SU_RB_ERR_INVALID_KEY;
    return false;
  }
  p++;
  R = RegSetValueEx(Handle,p,0,REG_DWORD,(BYTE *)&Value,sizeof(Value));
  RegCloseKey(Handle);
  if(R == ERROR_SUCCESS)
  {
    SU_RB_LastError = SU_RB_ERR_SUCCESS;
    return true;
  }
  else
  {
    SU_RB_LastError = SU_RB_ERR_ACCESS_DENIED;
    return false;
  }
}

bool SU_RB_DelKey(const char Key[])
{
  HKEY Handle;
  char *p;
  long R;

  Handle = SU_RB_OpenKeys(Key,KEY_SET_VALUE);
  if(Handle == 0)
    return false;
  p = strrchr(Key,'\\');
  if(p == NULL)
  {
    RegCloseKey(Handle);
    SU_RB_LastError = SU_RB_ERR_INVALID_KEY;
    return false;
  }
  p++;
  R = RegDeleteKey(Handle,p);
  RegCloseKey(Handle);
  if(R == ERROR_SUCCESS)
  {
    SU_RB_LastError = SU_RB_ERR_SUCCESS;
    return true;
  }
  else
  {
    SU_RB_LastError = SU_RB_ERR_ACCESS_DENIED;
    return false;
  }
}

bool SU_RB_DelValue(const char Key[])
{
  HKEY Handle;
  char *p;
  long R;

  Handle = SU_RB_OpenKeys(Key,KEY_SET_VALUE);
  if(Handle == 0)
    return false;
  p = strrchr(Key,'\\');
  if(p == NULL)
  {
    RegCloseKey(Handle);
    SU_RB_LastError = SU_RB_ERR_INVALID_KEY;
    return false;
  }
  p++;
  R = RegDeleteValue(Handle,p);
  RegCloseKey(Handle);
  if(R == ERROR_SUCCESS)
  {
    SU_RB_LastError = SU_RB_ERR_SUCCESS;
    return true;
  }
  else
  {
    SU_RB_LastError = SU_RB_ERR_ACCESS_DENIED;
    return false;
  }
}

void SU_RB_CloseKey(HKEY Key)
{
  RegCloseKey(Key);
}

bool SU_RB_EnumStrValue(HKEY Key,int Idx,char *Name,int name_len,char *Value,int value_len) /* True on Success. False when no more values available */
{
  int len = name_len;
  int len2 = value_len;
  LONG ret = RegEnumValue(Key,Idx,Name,&len,NULL,NULL,Value,&len2);

  return(ret == ERROR_SUCCESS);
}

bool SU_RB_EnumIntValue(HKEY Key,int Idx,char *Name,int name_len,int *Value) /* True on Success. False when no more values available */
{
  int len = name_len;
  int len2 = sizeof(Value);
  LONG ret = RegEnumValue(Key,Idx,Name,&len,NULL,NULL,Value,&len2);

  return(ret == ERROR_SUCCESS);
}

void SU_RB_FlushRegistry(void) {}

#else /* !_WIN32 */

#define SU_RB_TYPE_INT 1
#define SU_RB_TYPE_STR 2

typedef struct
{
  SU_PList Values; /* SU_RB_PValue */
  SU_PList Nodes; /* SU_RB_PNode */
} SU_RB_TNode, *SU_RB_PNode;

typedef struct
{
  int Type;
  void *Value;
} SU_RB_TValue, *SU_RB_PValue;

HKEY SU_RB_OpenKeys(const char Key[],int Access) /* Opened HKEY or 0 on error */
{
}

HKEY SU_RB_CreateKeys(const char Key[]) /* Created HKEY or 0 on error */
{
}

void SU_RB_FlushRegistry(void)
{
}

#endif /* _WIN32 */

int SU_RB_GetLastError(void)
{
  return SU_RB_LastError;
}

