/****************************************************************/
/* Socket unit - TCP/UDP                                        */
/* (c) Christophe CALMEJANE (Ze KiLleR) - 1999-03               */
/****************************************************************/

/*
    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include "skyutils.h"

#ifndef SU_TRACE_INTERNAL
#undef malloc
#undef calloc
#undef realloc
#undef strdup
#undef free
#endif /* !SU_TRACE_INTERNAL */

int SU_GetPortByName(char *port,char *proto)
{
  struct servent *PSE;

  PSE = getservbyname(port,proto);
  if(PSE == NULL)
    return atoi(port);
  return ntohs(PSE->s_port);
}

char *SU_GetMachineName(char *RemoteHost)
{
  char *tmp,*tmp2;
  tmp = strchr(RemoteHost,'.');
  if(tmp == NULL)
  {
    return strdup(RemoteHost);
  }
  tmp2 = (char *)malloc(tmp-RemoteHost+1);
  SU_strcpy(tmp2,RemoteHost,tmp-RemoteHost+1);
  return tmp2;
}

char *SU_NameOfPort(char *Host)
{
  struct hostent *hp;
  struct in_addr inp;

  inp.s_addr = inet_addr(Host);
  if(inp.s_addr == INADDR_NONE)
    return NULL;
  hp = gethostbyaddr((char *)&inp,4,AF_INET);
  if(hp == NULL)
    return NULL;
  return (char *)hp->h_name;
}

char *SU_AdrsOfPort(char *Host)
{
  struct hostent *hp;
  struct in_addr inp;

  hp = gethostbyname(Host);
  if(hp == NULL)
    return NULL;
  memcpy((void *)&inp, hp->h_addr, hp->h_length);
  return inet_ntoa(inp);
}

/*------------------------------------------------------------*/
SU_PServerInfo SU_CreateServer(int port,int type,bool ReUseAdrs)
{
  SU_PServerInfo SI;
  int len;

  SI = (SU_PServerInfo) malloc(sizeof(SU_TServerInfo));
  memset(SI,0,sizeof(SU_TServerInfo));
  if( type == SOCK_STREAM )
    SI->sock = socket(AF_INET,type,getprotobyname("tcp")->p_proto);
  else if( type == SOCK_DGRAM )
    SI->sock = socket(AF_INET,type,getprotobyname("udp")->p_proto);
  else
    return NULL;
  if(SI->sock == -1)
  {
    free(SI);
    return NULL;
  }
  memset(&(SI->SAddr),0,sizeof(struct sockaddr_in));
#ifdef __unix__
  if(ReUseAdrs)
  {
    len = sizeof(struct sockaddr_in);
    if(getsockname(SI->sock,(struct sockaddr *)&(SI->SAddr),&len) == -1)
    {
      SU_CLOSE_SOCKET(SI->sock);
      free(SI);
      return NULL;
    }
    len = 1;
    setsockopt(SI->sock,SOL_SOCKET,SO_REUSEADDR,(char *)&len,sizeof(len));
  }
#endif /* __unix__ */
  SI->SAddr.sin_family = AF_INET;
  SI->SAddr.sin_port = htons(port);
  SI->SAddr.sin_addr.s_addr = 0;
  if(bind(SI->sock,(struct sockaddr *)&(SI->SAddr), sizeof(SI->SAddr)) == -1)
  {
    SU_CLOSE_SOCKET(SI->sock);
    free(SI);
    return NULL;
  }

#ifdef _WIN32
  if(ReUseAdrs)
  {
    len = sizeof(struct sockaddr_in);
    if(getsockname(SI->sock,(struct sockaddr *)&(SI->SAddr),&len) == -1)
    {
      SU_CLOSE_SOCKET(SI->sock);
      free(SI);
      return NULL;
    }
    len = 1;
    setsockopt(SI->sock,SOL_SOCKET,SO_REUSEADDR,(char *)&len,sizeof(len));
  }
#endif /* _WIN32 */

  return SI;
}

int SU_ServerListen(SU_PServerInfo SI)
{
  if(SI == NULL)
    return SOCKET_ERROR;
  if(listen(SI->sock,10) == -1)
    return SOCKET_ERROR;
  return 0;
}

SU_PClientSocket SU_ServerAcceptConnection(SU_PServerInfo SI)
{
  struct sockaddr sad;
  int len;
  int tmpsock;
  SU_PClientSocket CS;

  if(SI == NULL)
    return NULL;
  len = sizeof(sad);
  tmpsock = accept(SI->sock,&sad,&len);
  if(tmpsock == -1)
    return NULL;
  CS = (SU_PClientSocket) malloc(sizeof(SU_TClientSocket));
  memset(CS,0,sizeof(SU_TClientSocket));
  CS->sock = tmpsock;
  memcpy(&CS->SAddr,&sad,sizeof(CS->SAddr));
  return CS;
}

void SU_ServerDisconnect(SU_PServerInfo SI)
{
  if(SI == NULL || SI->sock == -1)
    return;
  SU_CLOSE_SOCKET(SI->sock);
  SI->sock = -1;
}

void SU_FreeSI(SU_PServerInfo SI)
{
  SU_ServerDisconnect(SI);
  free(SI);
}

/*------------------------------------------------------------*/
SU_PClientSocket SU_ClientConnect(char *adrs,char *port,int type)
{
  struct servent *SE;
  struct sockaddr_in sin;
  struct hostent *HE;
  SU_PClientSocket CS;

  CS = (SU_PClientSocket) malloc(sizeof(SU_TClientSocket));
  memset(CS,0,sizeof(SU_TClientSocket));
  if(type == SOCK_STREAM)
    CS->sock = socket(AF_INET,SOCK_STREAM,getprotobyname("tcp")->p_proto);
  else if(type == SOCK_DGRAM)
    CS->sock = socket(AF_INET,SOCK_DGRAM,getprotobyname("udp")->p_proto);
  else
    return NULL;
  if(CS->sock == -1)
  {
    free(CS);
    return NULL;
  }
  sin.sin_family = AF_INET;
  if(type == SOCK_STREAM)
    SE = getservbyname(port,"tcp");
  else if(type == SOCK_DGRAM)
    SE = getservbyname(port,"udp");
  else
    return NULL;
  if(SE == NULL)
    sin.sin_port = htons(atoi(port));
  else
    sin.sin_port = SE->s_port;
  sin.sin_addr.s_addr = inet_addr(adrs);
  if(sin.sin_addr.s_addr == INADDR_NONE)
  {
    HE = gethostbyname(adrs);
    if(HE == NULL)
    {
      printf("SkyUtils_ClientConnect Warning : Unknown Host : %s\n",adrs);
      return NULL;
    }
    sin.sin_addr = *(struct in_addr *)(HE->h_addr_list[0]);
  }
  if(connect(CS->sock,(struct sockaddr *)(&sin),sizeof(sin)) == -1)
  {
    SU_CLOSE_SOCKET(CS->sock);
    free(CS);
    return NULL;
  }
  memcpy(&CS->SAddr,&sin,sizeof(CS->SAddr));
  return CS;
}

int SU_ClientSend(SU_PClientSocket CS,char *msg)
{
  if(CS == NULL)
    return SOCKET_ERROR;
  return send(CS->sock,msg,strlen(msg),SU_MSG_NOSIGNAL);
}

int SU_ClientSendBuf(SU_PClientSocket CS,char *buf,int len)
{
  if(CS == NULL)
    return SOCKET_ERROR;
  return send(CS->sock,buf,len,SU_MSG_NOSIGNAL);
}

void SU_ClientDisconnect(SU_PClientSocket CS)
{
  if(CS == NULL || CS->sock == -1)
    return;
  SU_CLOSE_SOCKET(CS->sock);
  CS->sock = -1;
}

void SU_FreeCS(SU_PClientSocket CS)
{
  SU_ClientDisconnect(CS);
  free(CS);
}

/*------------------------------------------------------------*/
int SU_UDPSendBroadcast(SU_PServerInfo SI,char *Text,int len,char *port)
{
  struct sockaddr_in sin;
  int i;
  int total = 0,packet;

  if(SI == NULL)
    return SOCKET_ERROR;
#ifdef DEBUG
  {
    int si = sizeof(int);
    if(getsockopt(SI->sock,SOL_SOCKET,SO_BROADCAST,&i,&si) == SOCKET_ERROR)
      return SOCKET_ERROR;
    if(i == 0)
    {
      printf("SkyUtils_SendBroadcast Warning : SO_BROADCAST is OFF... must have been set ON before\n");
      return SOCKET_ERROR;
    }
  }
#endif /* DEBUG */
  sin.sin_family = AF_INET;
  sin.sin_port = htons(atoi(port));
  sin.sin_addr.s_addr = INADDR_BROADCAST;

  i = 0;
  while(len > 0)
  {
    packet = len;
    if(packet > SU_UDP_MAX_LENGTH)
      packet = SU_UDP_MAX_LENGTH;
    i += sendto(SI->sock,Text+total,packet,0,(struct sockaddr *)&sin,sizeof(sin));
    total += packet;
    len -= packet;
    if(len != 0)
      SU_USLEEP(500); /* Sleep for 500 msec */
  }
  return i;
}

int SU_UDPSendToAddr(SU_PServerInfo SI,char *Text,int len,char *Addr,char *port)
{
  int i;
  struct sockaddr_in sin;
  struct hostent *PHE;
  int total = 0,packet;

  if(SI == NULL)
    return SOCKET_ERROR;
  sin.sin_addr.s_addr = inet_addr(Addr);
  if(sin.sin_addr.s_addr == INADDR_NONE)
  {
    PHE = gethostbyname(Addr);
    if(PHE == NULL)
      return SOCKET_ERROR;
    sin.sin_addr = *(struct in_addr *)(PHE->h_addr_list[0]);
  }
  sin.sin_family = AF_INET;
  sin.sin_port = htons(atoi(port));

  i = 0;
  while(len > 0)
  {
    packet = len;
    if(packet > SU_UDP_MAX_LENGTH)
      packet = SU_UDP_MAX_LENGTH;
    i += sendto(SI->sock,Text+total,packet,0,(struct sockaddr *)&sin,sizeof(sin));
    total += packet;
    len -= packet;
    if(len != 0)
      SU_USLEEP(500); /* Sleep for 500 msec */
  }
  return i;
}

int SU_UDPSendToSin(SU_PServerInfo SI,char *Text,int len,struct sockaddr_in sin)
{
  int i = 0;
  int total = 0,packet;

  if(SI == NULL)
    return SOCKET_ERROR;
  while(len > 0)
  {
    packet = len;
    if(packet > SU_UDP_MAX_LENGTH)
      packet = SU_UDP_MAX_LENGTH;
    i += sendto(SI->sock,Text+total,packet,0,(struct sockaddr *)&sin,sizeof(struct sockaddr_in));
    total += packet;
    len -= packet;
    if(len != 0)
      SU_USLEEP(500); /* Sleep for 500 msec */
  }
  return i;
}

int SU_UDPReceiveFrom(SU_PServerInfo SI,char *Text,int len,char **ip,int Blocking)
{
   struct sockaddr_in sin;
   int ssin;
   int i;
   struct hostent *hp;

  if(SI == NULL)
    return SOCKET_ERROR;
  if(!Blocking)
  {
#ifdef _WIN32
    i = 1;
    ioctlsocket(SI->sock,FIONBIO,&i);
#else /* !_WIN32 */
    fcntl(SI->sock,F_SETFL,O_NONBLOCK);
#endif /* _WIN32 */
  }
  ssin = sizeof(sin);
  i = recvfrom(SI->sock,Text,len,SU_MSG_NOSIGNAL,(struct sockaddr *)&sin,&ssin);
  if(i == SOCKET_ERROR)
    return SOCKET_ERROR;

  hp = gethostbyaddr((char *)&sin.sin_addr,4,AF_INET);
  if(hp == NULL)
    return i;
  *ip = (char *)hp->h_name;

  return i;
}

int SU_UDPReceiveFromSin(SU_PServerInfo SI,char *Text,int len,struct sockaddr_in *ret_sin,int Blocking)
{
   struct sockaddr_in sin;
   int i;
   int ssin;

  if(SI == NULL)
    return SOCKET_ERROR;
  if(!Blocking)
  {
#ifdef _WIN32
    i = 1;
    ioctlsocket(SI->sock,FIONBIO,&i);
#else /* !_WIN32 */
    fcntl(SI->sock,F_SETFL,O_NONBLOCK);
#endif /* _WIN32 */
  }
  ssin = sizeof(sin);
  i = recvfrom(SI->sock,Text,len,SU_MSG_NOSIGNAL,(struct sockaddr *)&sin,&ssin);
  if(i == SOCKET_ERROR)
    return SOCKET_ERROR;

  memcpy(ret_sin,&sin,sizeof(sin));
  return i;
}

int SU_SetSocketOpt(SU_SOCKET sock,int Opt,int value)
{
  return setsockopt(sock,SOL_SOCKET,Opt,(char *)&value,sizeof(value));
}

/* Code a tester !!! */
bool SU_SetSocketBlocking(SU_SOCKET sock,bool Block)
{
#ifdef _WIN32
  int v = Block;
	return ioctlsocket(sock,FIONBIO,&v) != SOCKET_ERROR;
#else /* !_WIN32 */
  return fcntl(sock,F_SETFL,Block?0:O_NONBLOCK) != SOCKET_ERROR;
#endif /* _WIN32 */
}

#ifdef _WIN32
bool SU_WSInit(int Major,int Minor)
{
  WORD wVersionRequested;
  WSADATA wsaData;
  int err;
  static bool SU_Sock_Init = false;

  if(!SU_Sock_Init)
  {
    SU_Sock_Init = true;
    wVersionRequested = MAKEWORD( Major, Minor );
    err = WSAStartup(wVersionRequested,&wsaData);
    if(err != 0)
      return false;
    if(LOBYTE( wsaData.wVersion ) != 2 || HIBYTE( wsaData.wVersion ) != 2)
    {
      WSACleanup();
      return false;
    }
  }
  return true;
}

void SU_WSUninit(void)
{
  static bool SU_Sock_UnInit = false;

  if(!SU_Sock_UnInit)
  {
    SU_Sock_UnInit = true;
    WSACleanup();
  }
}
#endif /* _WIN32 */
