#include <stdlib.h>
#include <dlfcn.h>
#include <assert.h>

#include "slatevm.h"
#include "extprim.h"

#if defined(__CYGWIN__)
  #define DLL_FILE_NAME_EXTENSION ".dll"
#else
  #define DLL_FILE_NAME_EXTENSION ".so"
#endif 

#define BYTEARRAY_LEN(x)	(PSObject_payloadSize((struct Object *) (x)))

static char *safe_string(struct ByteArray *s, char const *suffix) {
  size_t len = BYTEARRAY_LEN(s);
  char *result = malloc(strlen(suffix) + len + 1);
  if (result == NULL)
    return NULL;
  memcpy(result, s->elements, len);
  strcpy(result + len, suffix);
  return result;
}

Bool openExternalLibrary(struct ByteArray *libname,
			struct ByteArray *handle)
{
  char *fullname;
  void *h;

  assert(BYTEARRAY_LEN(handle) >= sizeof(h));

  fullname = safe_string(libname, DLL_FILE_NAME_EXTENSION);
  if (fullname == NULL)
    return False;

  h = dlopen(fullname, RTLD_NOW);
  free(fullname);

  if (h == NULL) {
    return False;
  } else {
    memcpy(handle->elements, &h, sizeof(h));
    return True;
  }
}

Bool closeExternalLibrary(struct ByteArray *handle) {
  void *h;

  assert(BYTEARRAY_LEN(handle) >= sizeof(h));
  memcpy(&h, handle->elements, sizeof(h));

  return (dlclose(h) == 0) ? True : False;
}

Bool lookupExternalLibraryPrimitive(struct ByteArray *handle,
				   struct ByteArray *symname,
				   struct ByteArray *ptr)
{
  void *h;
  void *fn;
  char *symbol;

  assert(BYTEARRAY_LEN(handle) >= sizeof(h));
  assert(BYTEARRAY_LEN(ptr) >= sizeof(fn));

  symbol = safe_string(symname, "");
  if (symbol == NULL)
    return False;

  memcpy(&h, handle->elements, sizeof(h));
  fn = (void *) dlsym(h, symbol);
  free(symbol);

  if (fn == NULL) {
    return False;
  } else {
    memcpy(ptr->elements, &fn, sizeof(fn));
    return True;
  }
}

Bool readExternalLibraryError(struct ByteArray *messageBuffer) {
  char *message = dlerror();
  if (message == NULL)
    return False;
  int len = strlen(message);
  assert(BYTEARRAY_LEN(messageBuffer) >= len);
  memcpy(messageBuffer->elements, message, len);
  return True;
}
