/*
** The Sleuth Kit 
**
** Brian Carrier [carrier@sleuthkit.org]
** Copyright (c) 2004 Brian Carrier.  All rights reserved 
**
** 
*/

#include "fs_tools.h"
#include "fs_types.h"
#include "error.h"
#include "mymalloc.h"


/**************************************************************************
 *
 * INODE WALKING
 *
 **************************************************************************/

/* rawfs_inode_walk - inode iterator 
 *
 */
void
rawfs_inode_walk(FS_INFO *fs, INUM_T start, INUM_T last, int flags,
		               FS_INODE_WALK_FN action, char *ptr)
{
	error ("rawfs: Illegal analysis method for raw data");
	return;
}

static FS_INODE *
rawfs_inode_lookup(FS_INFO *fs, INUM_T inum)
{
	error ("rawfs: Illegal analysis method for raw data");
	return NULL;
}


/**************************************************************************
 *
 * BLOCK WALKING
 *
 **************************************************************************/

/* rawpfs_block_walk - block iterator 
 *
 * flags used: ALIGN, META, ALLOC, UNALLOC
 */

void
rawfs_block_walk(FS_INFO *fs, DADDR_T start, DADDR_T last, int flags,
		               FS_BLOCK_WALK_FN action, char *ptr)
{
    char   *myname = "rawfs_block_walk";
    FS_BUF *fs_buf = fs_buf_alloc(fs->block_size);
    DADDR_T addr;

    /*
     * Sanity checks.
     */
    if (start < fs->first_block || start > fs->last_block)
		error("%s: invalid start block number: %lu", myname, (ULONG) start);

    if (last < fs->first_block || last > fs->last_block || last < start)
		error("%s: invalid last block number: %lu", myname, (ULONG) last);

	if (!(flags & FS_FLAG_DATA_ALLOC))
		return;

    for (addr = start; addr <= last; addr ++) {
		fs_read_block(fs, fs_buf, fs->block_size, addr, 
		  "block_walk: data block");

		if (WALK_STOP == action(fs, addr, fs_buf->data,
		  FS_FLAG_DATA_ALLOC | FS_FLAG_DATA_CONT, ptr)) {
    		fs_buf_free(fs_buf);
			return;
		}
    }

    /*
     * Cleanup.
     */
    fs_buf_free(fs_buf);
	return;
}

/**************************************************************************
 *
 * FILE WALKING
 *
 **************************************************************************/


/*  
 */ 
void
rawfs_file_walk(FS_INFO *fs, FS_INODE *inode, u_int32_t type, u_int16_t id,
    int flags, FS_FILE_WALK_FN action, char *ptr)
{    
	error ("rawfs: Illegal analysis method for raw data");
	return;
}

void
rawfs_dent_walk(FS_INFO *fs, INUM_T inode, int flags,
  FS_DENT_WALK_FN action, char *ptr)
{
	error ("rawfs: Illegal analysis method for raw data");
	return;
}


static void
rawfs_fsstat(FS_INFO *fs, FILE *hFile)
{
	fprintf(hFile, "Raw Data\n");
	fprintf(hFile, "Block Size: %d\n", fs->block_size);
	return;
}


/************************* istat *******************************/

static void
rawfs_istat (FS_INFO *fs, FILE *hFile, INUM_T inum, int numblock,
  int32_t sec_skew)
{
	error ("rawfs: Illegal analysis method for raw data");
	return;
}





void
rawfs_jopen (FS_INFO *fs, u_int32_t inum)
{
        fprintf(stderr, "Error: RAW does not have a journal\n"); 
	exit(1);
}

void
rawfs_jentry_walk (FS_INFO *fs, int flags, FS_JENTRY_WALK_FN action, char *ptr)
{
        fprintf(stderr, "Error: RAW does not have a journal\n");
        exit(1);
}


void
rawfs_jblk_walk (FS_INFO *fs, u_int32_t start, u_int32_t end, int flags, FS_JBLK_WALK_FN action, char *ptr)
{       
        fprintf(stderr, "Error: RAW does not have a journal\n");
        exit(1);
}



/* rawfs_close - close a fast file system */
static void 
rawfs_close(FS_INFO *fs)
{
    close(fs->fd);
    free(fs);
}


/* rawfs_open - open a fast file system */

FS_INFO *
rawfs_open(const char *name, unsigned char ftype)
{
    char   *myname = "rawfs_open";
    FS_INFO *fs = (FS_INFO *) mymalloc(sizeof(FS_INFO));
	OFF_T len;

	if ((ftype & FSMASK) != RAWFS_TYPE)
		error ("Invalid FS Type in rawfs_open");


	/* All we need to set are the block sizes and max bloc size etc. */
	if ((fs->fd = open(name, O_RDONLY)) < 0)
		error("%s: open %s: %m", myname, name);

	fs->ftype = ftype;
	fs->flags = 0;


	fs->inum_count = 0;
	fs->root_inum = 0;
	fs->first_inum = 0;
	fs->last_inum = 0;

	len = LSEEK(fs->fd, 0, SEEK_END);
	if (len < 0)
		error ("error getting size of raw file");

	fs->block_count = len / 512;
	if (len % 512) 
		fs->block_count++;

	fs->first_block = 0;
	fs->last_block = fs->block_count - 1;
	fs->block_size = 512;
	fs->dev_bsize = 512;
	fs->seek_pos = -1;

	fs->inode_walk = rawfs_inode_walk;
	fs->block_walk = rawfs_block_walk;
	fs->inode_lookup = rawfs_inode_lookup;
	fs->dent_walk = rawfs_dent_walk;
	fs->file_walk = rawfs_file_walk;
	fs->fsstat = rawfs_fsstat;
	fs->istat = rawfs_istat;
	fs->close = rawfs_close;
        fs->jblk_walk = rawfs_jblk_walk;
        fs->jentry_walk = rawfs_jentry_walk;
        fs->jopen = rawfs_jopen;
	fs->journ_inum = 0;


    return (fs);
}

