/*
** The Sleuth Kit 
**
** Brian Carrier [carrier@sleuthkit.org]
** Copyright (c) 2004 Brian Carrier.  All rights reserved 
**
** 
*/

#include "fs_tools.h"
#include "fs_types.h"
#include "error.h"
#include "mymalloc.h"


/**************************************************************************
 *
 * INODE WALKING
 *
 **************************************************************************/

/* swapfs_inode_walk - inode iterator 
 *
 */
void
swapfs_inode_walk(FS_INFO *fs, INUM_T start, INUM_T last, int flags,
		               FS_INODE_WALK_FN action, char *ptr)
{
	error ("swapfs: Illegal analysis method for swap space data");
	return;
}

static FS_INODE *
swapfs_inode_lookup(FS_INFO *fs, INUM_T inum)
{
	error ("swapfs: Illegal analysis method for swap space data");
	return NULL;
}


/**************************************************************************
 *
 * BLOCK WALKING
 *
 **************************************************************************/

/* swapfs_block_walk - block iterator 
 *
 * flags used: ALLOC
 */

void
swapfs_block_walk(FS_INFO *fs, DADDR_T start, DADDR_T last, int flags,
		               FS_BLOCK_WALK_FN action, char *ptr)
{
    char   *myname = "swapfs_block_walk";
    FS_BUF *fs_buf = fs_buf_alloc(fs->block_size);
    DADDR_T addr;

    /*
     * Sanity checks.
     */
    if (start < fs->first_block || start > fs->last_block)
		error("%s: invalid start block number: %lu", myname, (ULONG) start);

    if (last < fs->first_block || last > fs->last_block || last < start)
		error("%s: invalid last block number: %lu", myname, (ULONG) last);

	/* All we have is ALLOC */
	if (!(flags & FS_FLAG_DATA_ALLOC))
		return;

    for (addr = start; addr <= last; addr++) {

		fs_read_block(fs, fs_buf, fs->block_size, addr, 
		  "block_walk: data block");

		if (WALK_STOP == action(fs, addr, fs_buf->data,
		  FS_FLAG_DATA_ALLOC | FS_FLAG_DATA_CONT, ptr)) {
    		fs_buf_free(fs_buf);
			return;
		}
    }

    /*
     * Cleanup.
     */
    fs_buf_free(fs_buf);
	return;
}

/**************************************************************************
 *
 * FILE WALKING
 *
 **************************************************************************/


/*  
 */ 
void
swapfs_file_walk(FS_INFO *fs, FS_INODE *inode, u_int32_t type, u_int16_t id,
    int flags, FS_FILE_WALK_FN action, char *ptr)
{    
	error ("swapfs: Illegal analysis method for swap space data");
	return;
}

void
swapfs_dent_walk(FS_INFO *fs, INUM_T inode, int flags,
  FS_DENT_WALK_FN action, char *ptr)
{
	error ("swapfs: Illegal analysis method for swap space data");
	return;
}


static void
swapfs_fsstat(FS_INFO *fs, FILE *hFile)
{
	fprintf(hFile, "Swap Space\n");
	fprintf(hFile, "Page Size: %d\n", fs->block_size);
	return;
}


/************************* istat *******************************/

static void
swapfs_istat (FS_INFO *fs, FILE *hFile, INUM_T inum, int numblock,
  int32_t sec_skew)
{
	error ("swapfs: Illegal analysis method for swap space data");
	return;
}


void
swapfs_jopen (FS_INFO *fs, u_int32_t inum)
{
        fprintf(stderr, "Error: SWAP does not have a journal\n"); 
	exit(1);
}

void
swapfs_jentry_walk (FS_INFO *fs, int flags, FS_JENTRY_WALK_FN action, char *ptr)
{
        fprintf(stderr, "Error: SWAP does not have a journal\n");
        exit(1);
}


void
swapfs_jblk_walk (FS_INFO *fs, u_int32_t start, u_int32_t end, int flags, FS_JBLK_WALK_FN action, char *ptr)
{       
        fprintf(stderr, "Error: SWAP does not have a journal\n");
        exit(1);
}



/* swapfs_close - close a fast file system */
static void 
swapfs_close(FS_INFO *fs)
{
    close(fs->fd);
    free(fs);
}


/* swaps_open - open a fast file system */

FS_INFO *
swapfs_open(const char *name, unsigned char ftype)
{
    char   *myname = "swapfs_open";
    FS_INFO *fs = (FS_INFO *) mymalloc(sizeof(*fs));
	OFF_T len;

	if ((ftype & FSMASK) != SWAPFS_TYPE)
		error ("Invalid FS Type in swapfs_open");


	/* All we need to set are the block sizes and max bloc size etc. */

    if ((fs->fd = open(name, O_RDONLY)) < 0)
		error("%s: open %s: %m", myname, name);

	fs->ftype = ftype;
	fs->flags = 0;


	fs->inum_count = 0;
    fs->root_inum = 0;
	fs->first_inum = 0;
    fs->last_inum = 0;

	len = LSEEK(fs->fd, 0, SEEK_END);
	if (len < 0)
		error ("error getting size of swap file");

	fs->block_count = len / 4096;
	if (len % 4096) 
		fs->block_count++;

    fs->first_block = 0;
    fs->last_block = fs->block_count - 1;
    fs->block_size = 4096;
	fs->dev_bsize = 512;
    fs->seek_pos = -1;

	fs->inode_walk = swapfs_inode_walk;
	fs->block_walk = swapfs_block_walk;
	fs->inode_lookup = swapfs_inode_lookup;
	fs->dent_walk = swapfs_dent_walk;
	fs->file_walk = swapfs_file_walk;
	fs->fsstat = swapfs_fsstat;
	fs->istat = swapfs_istat;
	fs->close = swapfs_close;
	fs->jblk_walk = swapfs_jblk_walk;
	fs->jentry_walk = swapfs_jentry_walk;
	fs->jopen = swapfs_jopen;
	fs->journ_inum = 0;


    return (fs);
}

