/*
 * Brian Carrier [carrier@sleuthkit.org]
 * Copyright (c) 2003 Brian Carrier.  All rights reserved
 *
 * mmls - list media management structure contents
 *
 * This file is part of mmtools
 *
 * mmtools is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * mmtools is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with mactime; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR ANY PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, LOSS OF USE, DATA, OR PROFITS OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "mm_tools.h"
#include "error.h"

extern char *progname;

FILE *logfp;

void
usage()
{
	fprintf(stderr, "%s [-o offset] [-v] -t mmtype disk\n", progname);
	fprintf(stderr, "\t-t mmtype: media type\n");
	fprintf(stderr, "\t[-o offset]: sector offset to start reading from\n");
	fprintf(stderr, "\t-v: verbose output\n");
	fprintf(stderr, "\t-V: print the version\n");
	fprintf(stderr, "Supported media types:\n");
	mm_print_types();
	exit(1);
}

/*
 * The callback action for the part_walk
 * */
u_int8_t 
part_act (MM_INFO *mm, int mnum, MM_PART *part, int flag, char *ptr) 
{
	/* Neither table or slot were given */
	if ((part->table_num == -1) && (part->slot_num == -1) )
		printf ("%.2d:  -----   ", mnum);

	/* Table was not given, but slot was */
	else if ((part->table_num == -1) && (part->slot_num != -1) )
		printf ("%.2d:  %.2d      ", mnum, part->slot_num);

	/* The Table was given, but slot wasn't */
	else if ((part->table_num != -1) && (part->slot_num == -1) )
		printf ("%.2d:  -----   ", mnum);

	/* Both table and slot were given */
	else if ((part->table_num != -1) && (part->slot_num != -1) )
		printf ("%.2d:  %.2d:%.2d   ", mnum, part->table_num, 
		  part->slot_num);

	/* Print the layout */
	printf ("%.10lu   %.10lu   %.10lu   %s\n",
	  (unsigned long)part->start,
	  (unsigned long)(part->start + part->len - 1),
	  (unsigned long)part->len,
	  part->desc);

	return WALK_CONT;
}


int 
main(int argc, char **argv)
{
	MM_INFO *mm;
	char *mmtype = NULL;
	int 	ch;
	daddr_t	sect_offset = 0;
	u_int8_t flags = 0;
	logfp = stderr;

	progname = argv[0];

	while ((ch = getopt(argc, argv, "o:t:vV")) > 0) {
		char *cp;
		switch (ch) {
			case 'o':
				sect_offset = STRTOUL(optarg, &cp, 0);
				if (*cp || cp == optarg) {
					fprintf (stderr, "Bad sector offset\n");
					usage();
				}

				break;
			case 't':
				mmtype = optarg;
				break;
			case 'v':
				verbose++;
				break;
			case 'V':
				print_version();
				exit (0);
			default:
				usage();
		}
	}

	if (optind + 1 != argc) {
		usage();
	}

	if (mmtype == NULL) {
		fprintf(stderr, "Missing media type\n");
		usage();
	}


	mm = mm_open(argv[optind], mmtype, sect_offset);

	/* Print the header */
	printf ("%s\n", mm->str_type);
	printf ("Units are in 512-byte sectors\n\n");
	printf ("     Slot    Start        End          Length       Description\n");

	mm->part_walk(mm, mm->first_part, mm->last_part, flags, 
	  part_act, "mmls");

	mm->close(mm);
	exit(0);
}
