/*
 * ewfexport
 * Export media data from EWF files to a file
 *
 * Copyright (c) 2006, Joachim Metz <forensics@hoffmannbv.nl>,
 * Hoffmann Investigations. All rights reserved.
 *
 * Refer to AUTHORS for acknowledgements.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 * - Neither the name of the creator, related organisations, nor the names of
 *   its contributors may be used to endorse or promote products derived from
 *   this software without specific prior written permission.
 * - All advertising materials mentioning features or use of this software
 *   must acknowledge the contribution by people stated in the acknowledgements.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER, COMPANY AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "libewf_common.h"

#include <sys/stat.h>
#include <inttypes.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>

#include "libewf.h"

#include "ewfcommon.h"

/* Prints the executable usage information
 */
void usage( void )
{
	fprintf( stderr, "Usage: ewfexport [ -hsqvV ] ewf_files\n" );

	fprintf( stderr, "\t-h: shows this help\n" );
	fprintf( stderr, "\t-q: quiet shows no status information\n" );
	fprintf( stderr, "\t-s: swap byte pairs of the media data (from AB to BA)\n" );
	fprintf( stderr, "\t    (use this for big to little endian conversion and vice versa)\n" );
	fprintf( stderr, "\t-v: verbose output to stderr\n" );
	fprintf( stderr, "\t-V: print version\n" );
}

/* Prints the executable version information
 */
void version( void )
{
	fprintf( stderr, "Copyright 2006 Joachim Metz, Hoffmann Investigations <%s> and contributors.\n", PACKAGE_BUGREPORT );
	fprintf( stderr, "This is free software; see the source for copying conditions. There is NO\n" );
	fprintf( stderr, "warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n" );
}

/* Print the status of the export process
 */
int8_t last_percentage = -1;

void print_percentage_callback( uint64_t bytes_read, uint64_t bytes_total )
{
	char *bytes_read_string  = NULL;
	char *bytes_total_string = NULL;
	int8_t new_percentage    = ( bytes_total > 0 ) ? ( ( bytes_read * 100 ) / bytes_total ) : 1;

	if( new_percentage > last_percentage )
	{
		last_percentage    = new_percentage;
		bytes_read_string  = determine_human_readable_size_string( bytes_read );
		bytes_total_string = determine_human_readable_size_string( bytes_total );

		if( ( bytes_read_string != NULL ) && ( bytes_total_string != NULL ) )
		{
			fprintf( stderr, "Status: bytes read: %s (%" PRIu64 " bytes) of total: %s (%" PRIu64 " bytes) (%" PRIi8 "%%).\n", bytes_read_string, bytes_read, bytes_total_string, bytes_total, last_percentage );
		}
		else
		{
			fprintf( stderr, "Status: bytes read: %" PRIu64 " of total: %" PRIu64 " (%" PRIi8 "%%).\n", bytes_read, bytes_total, last_percentage );
		}

		if( bytes_read_string != NULL )
		{
			libewf_common_free( bytes_read_string );
		}
		if( bytes_total_string != NULL )
		{
			libewf_common_free( bytes_total_string );
		}
	}
}

/* The main program
 */
int main( int argc, const char **argv )
{
	LIBEWF_HANDLE *handle   = NULL;
	void *callback          = &print_percentage_callback;
	int64_t count           = 0;
	uint64_t size           = 0;
	uint64_t export_offset  = 0;
	uint64_t export_size    = 0;
	uint8_t swap_byte_pairs = 0;
	int option              = 0;

	fprintf( stderr, "ewfexport version: %s.\n\n", VERSION );

	while( ( option = getopt( argc, (char **) argv, "hsqvV" ) ) > 0 )
	{
		switch( option )
		{
			case '?':
			default:
				fprintf( stderr, "Invalid argument: %s.\n", argv[ optind ] );

				usage();

				return( EXIT_FAILURE );

			case 'h':
				usage();

				return( EXIT_SUCCESS );

			case 's':
				swap_byte_pairs = 1;

				break;

			case 'q':
				callback = NULL;

				break;

			case 'v':
				libewf_verbose = 1;

				break;

			case 'V':
				version();

				return( EXIT_SUCCESS );
		}
	}
	if( optind == argc )
	{
		fprintf( stderr, "Missing EWF image file(s).\n" );

		usage();

		return( EXIT_FAILURE );
	}
	handle = libewf_open( &argv[ optind ], ( argc - optind ), LIBEWF_OPEN_READ );

	if( handle == NULL )
	{
		fprintf( stderr, "Unable to open EWF image file(s).\n" );

		return( EXIT_FAILURE );
	}
	handle->swap_byte_pairs = swap_byte_pairs;

	size = libewf_data_size( handle );

	if( size == 0 )
	{
		fprintf( stderr, "Error exporting data from EWF file(s) - media size is 0.\n" );

		return( EXIT_FAILURE );
	}
	/* Request the necessary case data
	 */
	fprintf( stdout, "Information for export required, please provide the necessary input\n" );

	export_offset = get_user_input_size_variable( stderr, "Start export at offset", 0, size, 0 );
	export_size   = get_user_input_size_variable( stderr, "Amount of bytes to export", 0, size, size );

	count = libewf_read_to_file_descriptor( handle, 1, export_size, export_offset, callback );

	libewf_close( handle );

	if( count <= -1 )
	{
		fprintf( stderr, "Error exporting data from EWF file(s).\n" );

		return( EXIT_FAILURE );
	}
	else
	{
		fprintf( stderr, "Success: bytes written: %" PRIi64 ".\n", count );

		return( EXIT_SUCCESS );
	}
}

